/*-----------------------------------------------------------------------------
	[MainBoard.c]
		C{[h܂B

	Copyright (C) 2004 Ki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
**---------------------------------------------------------------------------*/
#include <stdio.h>
#include <malloc.h>
#include "MainBoard.h"

#include "ScreenInterface.h"
#include "InputInterface.h"
#include "Mouse.h"
#include "Printf.h"


#define CYCLES_PER_LINE		456


// debug 
#ifdef DEBUGBUILD
	#include "WinMain.h"
	#include "MemoryEditor.h"
	#include "CpuMemoryEditor.h"
	#include "RegisterEditor.h"
	#include "DisasmWindow.h"
#endif

/*-----------------------------------------------------------------------------
** C{[h̋L̈錾܂B
**---------------------------------------------------------------------------*/
static Uint8*		_pMainRam;
static Uint8*		_pBackupRam;
static Uint8*		_pRom;

// ӁF͖{ CD-ROM^2 VXeɑ݂B 
static Uint8*		_pBufferRam;

static Uint32		_RomMask;

static BOOL			_bSystemInit = FALSE;

static Uint8*		_pRomMap[256];

static BOOL			_bStretched  = FALSE;
static Sint32		_Magnification = 1;
static BOOL			_bPause = FALSE;

static Uint16		_ScreenBuf[512*256];	// 512 * 256 * sizeof(Uint16) = 262144

static Uint8		_Buffer;


/*
	All the Arcade Card I/O code is taken from Hu-Go!
*/
static Uint32		_AcShift;
static Uint32		_AcShiftBits;
static Uint8		_0x1ae5;
static Uint8		_AcRam[0x200000];


typedef struct
{
	Uint8		control;
	Uint32		base;
	Uint16		offset;
	Uint16		increment;
} ACIO;


static ACIO			_Ac[4];


/* memory editors for debug purpose */
static HANDLE	_RegEdit    = NULL;

static HANDLE	_CpuMemEdit = NULL;
static HANDLE	_VramEdit   = NULL;
static HANDLE	_RomEdit    = NULL;
static HANDLE	_AcRamEdit  = NULL;
static HANDLE	_CdRamEdit  = NULL;
static HANDLE	_BramEdit	= NULL;

static HANDLE	_DisasmWindow = NULL;


static
inline
void
increment_acaddr(
	ACIO*		port)
{
	if (port->control & 1)		// CONFIRMED:  D0 enables base / offset increment
	{
		if (port->control & 0x10)	// CONFIRMED: D4 selects base / offset to be incremented
		{
			port->base += port->increment;
			port->base &= 0xffffff;
		}
		else
		{
			port->offset += port->increment;
		}
	}
}


static
Uint8
ac_read(
	Uint32		physAddr)
{
	ACIO		*port = &_Ac[(physAddr >> 4) & 3];
	Uint8		ret;

	if ((physAddr & 0x1ae0) == 0x1ae0)
	{
		switch (physAddr & 0x1aef)
		{
			case 0x1ae0:
				return (Uint8)_AcShift;
			case 0x1ae1:
				return (Uint8)(_AcShift >> 8);
			case 0x1ae2:
				return (Uint8)(_AcShift >> 16);
			case 0x1ae3:
				return (Uint8)(_AcShift >> 24);
			case 0x1ae4:
				return (Uint8)(_AcShiftBits);
			case 0x1ae5:
				return _0x1ae5;
			case 0x1aee:
				return 0x10;
			case 0x1aef:
				return 0x51;
		}
		return 0xff;
	}

	switch (physAddr & 0xf)
	{
		case 0x0:
		case 0x1:
			if (port->control & 2)
				ret = _AcRam[(port->base + port->offset) & 0x1fffff];
			else
				ret = _AcRam[port->base & 0x1fffff];
			increment_acaddr(port);
			return ret;

		case 0x2:	return (Uint8)(port->base);
		case 0x3:	return (Uint8)(port->base >> 8);
		case 0x4:	return (Uint8)(port->base >> 16);
		case 0x5:	return (Uint8)(port->offset);
		case 0x6:	return (Uint8)(port->offset >> 8);
		case 0x7:	return (Uint8)(port->increment);
		case 0x8:	return (Uint8)(port->increment >> 8);
		case 0x9:	return port->control;
		case 0xa:	return 0;
		default:
			break;
	}
	return 0xff;
}


static
void
ac_write(
	Uint32		physAddr,
	Uint8		data)
{
	if ((physAddr & 0x1ae0) == 0x1ae0)
	{
		switch (physAddr & 0xf)
		{
			case 0:
				_AcShift = (_AcShift & ~0xff) | data;
				return;
			case 1:
				_AcShift = (_AcShift & ~0xff00) | (data << 8);
				return;
			case 2:
				_AcShift = (_AcShift & ~0xff0000) | (data << 16);
				return;
			case 3:
				_AcShift = (_AcShift & ~0xff000000) | (data << 24);
				return;
			case 4:
				if ((_AcShiftBits = data & 0xf) != 0)
				{
					if (_AcShiftBits < 8)
						_AcShift <<= _AcShiftBits;
					else
						_AcShift >>= 16 - _AcShiftBits;
				}
				return;
			case 5:
				_0x1ae5 = data;
				return;
		}
	}
	else
	{
		ACIO		*port = &_Ac[(physAddr >> 4) & 3];

		switch (physAddr & 0xf)
		{
			case 0x0:
			case 0x1:
				if (port->control & 2)
					_AcRam[(port->base + port->offset) & 0x1fffff] = data;
				else
					_AcRam[port->base & 0x1fffff] = data;
				increment_acaddr(port);
				return;

			case 0x2:
				port->base = (port->base & ~0xff) | data;
				return;
			case 0x3:
				port->base = (port->base & ~0xff00) | (data << 8);
				return;
			case 0x4:
				port->base = (port->base & ~0xff0000) | (data << 16);
				return;
			case 0x5:
				port->offset = (port->offset & ~0xff) | data;
				return;
			case 0x6:
				port->offset = (port->offset & ~0xff00) | (data << 8);
				if ((port->control & 0x60) == 0x40)
				{
					if (port->control & 0x08)
						port->base += port->offset + 0xff0000;
					else
						port->base += port->offset;
					port->base &= 0xffffff;
				}
				return;
			case 0x7:
				port->increment = (port->increment & ~0xff) | data;
				return;
			case 0x8:
				port->increment = (port->increment & ~0xff00) | (data << 8);
				return;
			case 0x9:
				port->control = data & 0x7f;		// D7 is not used
				return;
			case 0xa:
				// value written is not used 
				if ((port->control & 0x60) == 0x60)
				{
					port->base += port->offset;
					port->base &= 0xffffff;
				}
				return;
		}
	}
}


/*-----------------------------------------------------------------------------
** ǂݏo֐(ǂݏoɊւI/O}bv)`܂B
**---------------------------------------------------------------------------*/
static
Uint8
cpu_read_func_with_cdrom2_system(
	Uint32		physAddr)
{
	switch (physAddr >> 13)
	{
		case 0x40:
			return ac_read(0x1a00);
		case 0x41:
			return ac_read(0x1a10);
		case 0x42:
			return ac_read(0x1a20);
		case 0x43:
			return ac_read(0x1a30);

		case 0x68:	// SUPER CD-ROM^2 buffer RAM (192K bytes)
		case 0x69:
		case 0x6a:
		case 0x6b:
		case 0x6c:
		case 0x6d:
		case 0x6e:
		case 0x6f:
		case 0x70:
		case 0x71:
		case 0x72:
		case 0x73:
		case 0x74:
		case 0x75:
		case 0x76:
		case 0x77:
		case 0x78:
		case 0x79:
		case 0x7a:
		case 0x7b:
		case 0x7c:
		case 0x7d:
		case 0x7e:
		case 0x7f:

		case 0x80:	// CD-ROM^2 buffer RAM (64K bytes)
		case 0x81:
		case 0x82:
		case 0x83:
		case 0x84:
		case 0x85:
		case 0x86:
		case 0x87:
			return _pBufferRam[physAddr & 0x3ffff];

		case 0xf7:
			if (CDROM_IsBRAMEnabled())
			{
				return _pBackupRam[physAddr & 0x1fff];
			}
			return 0xff;
		case 0xf8:
		case 0xf9:
		case 0xfa:
		case 0xfb:
			return _pMainRam[physAddr & 0x1fff];
		case 0xfc:
		case 0xfd:
		case 0xfe:
			return 0xff;
		case 0xff:
			switch (physAddr & 0x1e00)
			{
				case 0x0000: // VDC
					CPU_DelayClock(1);
					return VDC_Read(physAddr & 3);
				case 0x0400: // VCE
					CPU_DelayClock(1);
					return VCE_Read(physAddr & 7);
/*
				case 0x0800: // PSG
					return PSG_Read(physAddr & 15);
				case 0x0c00: // Timer
					return TIMER_Read(physAddr & 1);
				case 0x1000: // PAD
					return JOYPAD_Read(physAddr);
				case 0x1400: // INT
					return INTCTRL_Read(physAddr & 3);
*/
				case 0x0800: // PSG
					return _Buffer; //PSG_Read(physAddr & 15);
				case 0x0c00: // Timer
					_Buffer = (_Buffer & 0x80) | (TIMER_Read(physAddr & 1) & 0x7f);
					return _Buffer;
				case 0x1000: // PAD
					_Buffer = JOYPAD_Read(physAddr);
					return _Buffer;
				case 0x1400: // INT
					if (physAddr & 2)	_Buffer = (_Buffer & 0xf8) | (INTCTRL_Read(physAddr & 3) & 7);
					return _Buffer;
				case 0x1800: // CD-ROM^2
					return CDROM_Read(physAddr);
				case 0x1a00: // ARCADE CARD
					return ac_read(physAddr);
			}
			return 0xff;
		default:
			return _pRom[physAddr & _RomMask];
	}
}


static
Uint8
cpu_read_func_without_cdrom2_system(
	Uint32		physAddr)
{
	switch (physAddr >> 13)
	{
		case 0xf7:
			if (CDROM_IsBRAMEnabled())
			{
				return _pBackupRam[physAddr & 0x1fff];
			}
			return 0xff;
		case 0xf8:
		case 0xf9:
		case 0xfa:
		case 0xfb:
			return _pMainRam[physAddr & 0x1fff];
		case 0xfc:
		case 0xfd:
		case 0xfe:
			return 0xff;
		case 0xff:
			switch (physAddr & 0x1c00)
			{
				case 0x0000: // VDC
					CPU_DelayClock(1);
					return VDC_Read(physAddr & 3);
				case 0x0400: // VCE
					CPU_DelayClock(1);
					return VCE_Read(physAddr & 7);
				case 0x0800: // PSG
					return _Buffer; //PSG_Read(physAddr & 15);
				case 0x0c00: // Timer
					_Buffer = (_Buffer & 0x80) | (TIMER_Read(physAddr & 1) & 0x7f);
					return _Buffer;
				case 0x1000: // PAD
					_Buffer = JOYPAD_Read(physAddr);
					return _Buffer;
				case 0x1400: // INT
					if (physAddr & 2)	_Buffer = (_Buffer & 0xf8) | (INTCTRL_Read(physAddr & 3) & 7);
					return _Buffer;
				case 0x1800: // CD-ROM^2 (for BRAM)
					return CDROM_Read(physAddr & 15);
			}
			return 0xff;
		default:
			return _pRomMap[physAddr >> 13][physAddr & 0x1fff];
	}
}


/*-----------------------------------------------------------------------------
** ݊֐(݂ɊւI/O}bv)`܂B
**---------------------------------------------------------------------------*/
static
void
cpu_write_func_with_cdrom2_system(
	Uint32		physAddr,
	Uint8		data)
{
	switch (physAddr >> 13)
	{
		case 0x40:
			ac_write(0x1a00, data);
			break;
		case 0x41:
			ac_write(0x1a10, data);
			break;
		case 0x42:
			ac_write(0x1a20, data);
			break;
		case 0x43:
			ac_write(0x1a30, data);
			break;

		case 0x68:	// SUPER CD-ROM^2 buffer RAM (192K bytes)
		case 0x69:
		case 0x6a:
		case 0x6b:
		case 0x6c:
		case 0x6d:
		case 0x6e:
		case 0x6f:
		case 0x70:
		case 0x71:
		case 0x72:
		case 0x73:
		case 0x74:
		case 0x75:
		case 0x76:
		case 0x77:
		case 0x78:
		case 0x79:
		case 0x7a:
		case 0x7b:
		case 0x7c:
		case 0x7d:
		case 0x7e:
		case 0x7f:

		case 0x80:	// CD-ROM^2 buffer RAM (64K bytes)
		case 0x81:
		case 0x82:
		case 0x83:
		case 0x84:
		case 0x85:
		case 0x86:
		case 0x87:
			_pBufferRam[physAddr & 0x3ffff] = data;
			break;

		case 0xf7:
			if (CDROM_IsBRAMEnabled())
			{
				_pBackupRam[physAddr & 0x1fff] = data;
			}
			break;

		case 0xf8:
		case 0xf9:
		case 0xfa:
		case 0xfb:
			_pMainRam[physAddr & 0x1fff] = data;
			break;
		case 0xff:
			switch (physAddr & 0x1e00)
			{
				case 0x0000: // VDC
					CPU_DelayClock(1);
					VDC_Write(physAddr & 3, data);
					break;
				case 0x0400: // VCE
					CPU_DelayClock(1);
					VCE_Write(physAddr & 7, data);
					break;
				case 0x0800: // PSG
					_Buffer = data;
					PSG_Write(physAddr & 15, data);
					break;
				case 0x0c00: // Timer
					_Buffer = data;
					TIMER_Write(physAddr & 1, data);
					break;
				case 0x1000: // PAD
					_Buffer = data;
					JOYPAD_Write(physAddr, data);
					break;
				case 0x1400: // INT
					_Buffer = data;
					INTCTRL_Write(physAddr & 3, data);
					break;
				case 0x1800: // CD-ROM^2
					CDROM_Write(physAddr, data);
					break;
				case 0x1a00: // ARCADE CARD
					ac_write(physAddr, data);
					break;
			}
	}
}


static
void
cpu_write_func_without_cdrom2_system(
	Uint32		physAddr,
	Uint8		data)
{
	switch (physAddr >> 13)
	{
		case 0xf7:
			if (CDROM_IsBRAMEnabled())
			{
				_pBackupRam[physAddr & 0x1fff] = data;
			}
			break;
		case 0xf8:
		case 0xf9:
		case 0xfa:
		case 0xfb:
			_pMainRam[physAddr & 0x1fff] = data;
			break;
		case 0xff:
			switch (physAddr & 0x1c00)
			{
				case 0x0000: // VDC
					CPU_DelayClock(1);
					VDC_Write(physAddr & 3, data);
					break;
				case 0x0400: // VCE
					CPU_DelayClock(1);
					VCE_Write(physAddr & 7, data);
					break;
				case 0x0800: // PSG
					_Buffer = data;
					PSG_Write(physAddr & 15, data);
					break;
				case 0x0c00: // Timer
					_Buffer = data;
					TIMER_Write(physAddr & 1, data);
					break;
				case 0x1000: // PAD
					_Buffer = data;
					JOYPAD_Write(physAddr, data);
					break;
				case 0x1400: // INT
					_Buffer = data;
					INTCTRL_Write(physAddr & 3, data);
					break;
				case 0x1800: // CD-ROM^2 (for BRAM)
					CDROM_Write(physAddr & 15, data);
					break;
			}
//#ifdef SF2
#if 1
		default:
			if ((physAddr & 0x1ffc) == 0x1ff0)
			{
				int		i;
				for (i = 0; i < 0x40; i++)
				{
					_pRomMap[0x40+i] = &_pRom[(((physAddr & 3)+1) * 0x80000 + i*0x2000) & _RomMask];
				}
			}
#endif
	}
}


#ifdef DEBUGBUILD
/*-----------------------------------------------------------------------------
** fobKpǂݏo֐`܂B
**---------------------------------------------------------------------------*/
static
Uint8
debugger_read_func(
	Uint32		physAddr)
{
	switch (physAddr >> 13)
	{
		case 0x68:	// SUPER CD-ROM^2 buffer RAM (192K bytes)
		case 0x69:
		case 0x6a:
		case 0x6b:
		case 0x6c:
		case 0x6d:
		case 0x6e:
		case 0x6f:
		case 0x70:
		case 0x71:
		case 0x72:
		case 0x73:
		case 0x74:
		case 0x75:
		case 0x76:
		case 0x77:
		case 0x78:
		case 0x79:
		case 0x7a:
		case 0x7b:
		case 0x7c:
		case 0x7d:
		case 0x7e:
		case 0x7f:

		case 0x80:	// CD-ROM^2 buffer RAM (64K bytes)
		case 0x81:
		case 0x82:
		case 0x83:
		case 0x84:
		case 0x85:
		case 0x86:
		case 0x87:
			return _pBufferRam[physAddr & 0x3ffff];

		case 0xf7:
				return _pBackupRam[physAddr & 0x7ff];
		case 0xf8:
		case 0xf9:
		case 0xfa:
		case 0xfb:
			return _pMainRam[physAddr & 0x1fff];
		case 0xfc:
		case 0xfd:
		case 0xfe:
		case 0xff:
			return 0xff;
		default:
			return _pRom[physAddr & _RomMask];
	}
}


/*-----------------------------------------------------------------------------
** fobKp݊֐`܂B
**---------------------------------------------------------------------------*/
static
void
debugger_write_func(
	Uint32		physAddr,
	Uint8		data)
{
	switch (physAddr >> 13)
	{
		case 0x68:	// SUPER CD-ROM^2 buffer RAM (192K bytes)
		case 0x69:
		case 0x6a:
		case 0x6b:
		case 0x6c:
		case 0x6d:
		case 0x6e:
		case 0x6f:
		case 0x70:
		case 0x71:
		case 0x72:
		case 0x73:
		case 0x74:
		case 0x75:
		case 0x76:
		case 0x77:
		case 0x78:
		case 0x79:
		case 0x7a:
		case 0x7b:
		case 0x7c:
		case 0x7d:
		case 0x7e:
		case 0x7f:

		case 0x80:	// CD-ROM^2 buffer RAM (64K bytes)
		case 0x81:
		case 0x82:
		case 0x83:
		case 0x84:
		case 0x85:
		case 0x86:
		case 0x87:
			_pBufferRam[physAddr & 0x3ffff] = data;
			break;

		case 0xf7:
			_pBackupRam[physAddr & 0x7ff] = data;
			break;

		case 0xf8:
		case 0xf9:
		case 0xfa:
		case 0xfb:
			_pMainRam[physAddr & 0x1fff] = data;
			break;
		case 0xfc:
		case 0xfd:
		case 0xfe:
		case 0xff:
			break;
		default:
			// allow writing to ROM on debugger
			_pRom[physAddr & _RomMask] = data;
	}
}

Uint8
MAINBOARD_ReadByte(
	Uint32		physAddr)
{
	return debugger_read_func(physAddr);
}

void
MAINBOARD_WriteByte(
	Uint32		physAddr,
	Uint8		data)
{
	debugger_write_func(physAddr, data);
}

#endif /* DEBUGBUILD */



/******************************************************************************
** OJ֐Lq܂B
******************************************************************************/


/*-----------------------------------------------------------------------------
** [Init]
**   n[hEFA̎s𐮂܂B
**---------------------------------------------------------------------------*/
BOOL
MAINBOARD_Init(
	const char*		pGameName)
{
	int			i;

	// at least pGameName needs to be valid
	if (pGameName == NULL)
		return FALSE;

	// mۂB
	_pMainRam   = (Uint8*)malloc(0x2000);
	_pBackupRam   = (Uint8*)malloc(0x2000);
	_pBufferRam = (Uint8*)malloc(0x40000);
	if (_pMainRam == NULL || _pBackupRam == NULL || _pBufferRam == NULL)
	{
		free(_pMainRam);
		free(_pBackupRam);
		free(_pBufferRam);
		return FALSE;
	}

	memset(_pMainRam, 0, 0x2000);
	memset(_pBufferRam, 0, 0x40000);

	// J[gbWǂݍށB
	if (pGameName != NULL)
	{
		if ((_RomMask = LoadCartridge(pGameName, &_pRom)) == 0)
		{
			free(_pMainRam);
			free(_pBackupRam);
			free(_pBufferRam);
			return FALSE;
		}
	}

	// 
	for (i = 0; i < sizeof(_pRomMap)/sizeof(_pRomMap[0]); i++)
	{
		_pRomMap[i] = _pRom + ((i * 0x2000) & _RomMask);
	}

	// bot̃[h^Cg֐ƃ[y[Wo^B
	if (_RomMask > 524288-1)
	{
		// J[gbW̃TCY 512KB 傫ꍇ
		// h^n CD-ROM^2 VXe؂藣B
		CPU_SetReadFunction(cpu_read_func_without_cdrom2_system);
		CPU_SetWriteFunction(cpu_write_func_without_cdrom2_system);
	}
	else
	{
		CPU_SetReadFunction(cpu_read_func_with_cdrom2_system);
		CPU_SetWriteFunction(cpu_write_func_with_cdrom2_system);
	}
	CPU_SetZeroPageMemory(_pMainRam);

	CPU_Reset();

	// 荞݃Rg[ 
	INTCTRL_Init();

	// ^C}[ 
	TIMER_Init();

	// ͂ 
	JOYPAD_Init();

	// ucb 
	if (VDC_Init() != 0)
	{
		FreeCartridge(_pRom);
		free(_pMainRam);
		free(_pBackupRam);
		free(_pBufferRam);
		return FALSE;
	}

	// ubd
	if (VCE_Init() != 0)
	{
		FreeCartridge(_pRom);
		free(_pMainRam);
		free(_pBackupRam);
		free(_pBufferRam);
		return FALSE;
	}

	// bc|qnl̏
	if (CDROM_Init() != 0)
	{
		PRINTF("MAINBOARD_Init: CDROM_Init failed.");
	}

	// `otftHgŏ
	if (!APU_Init(44100, 2560))
	{
		PRINTF("(NOTICE: MAINBOARD_Init: APU_Init failed.)");
	}

	_RegEdit       = NULL;
	_CpuMemEdit    = NULL;
	_RomEdit       = NULL;
	_AcRamEdit     = NULL;
	_CdRamEdit     = NULL;
	_VramEdit      = NULL;
	_BramEdit      = NULL;
	_DisasmWindow  = NULL;

#ifdef DEBUGBUILD
	_RegEdit       = REGED_Init("CPU REGISTERS / WP / BP", WINMAIN_GetHInstance());
	_CpuMemEdit    = CPUMEM_Init("[CPURAM] / ROM / VRAM / CDRAM / ACRAM / BRAM (TAB key to switch)",
									debugger_read_func, debugger_write_func, WINMAIN_GetHInstance());
	_DisasmWindow  = DISASMWND_Init("DISASSEMBLY LIST", WINMAIN_GetHInstance());
	_bPause			= TRUE;
#endif

	_bSystemInit = TRUE;
	PRINTF("MAINBOARD_Init: done.");

	return TRUE;
}


/*-----------------------------------------------------------------------------
** [MAINBOARD_Deinit]
**   n[hEFAj܂B
**---------------------------------------------------------------------------*/
void
MAINBOARD_Deinit()
{
	if (!_bSystemInit)
		return;

#ifdef DEBUGBUILD
	DISASMWND_Deinit(_DisasmWindow);

	REGED_Deinit(_RegEdit);
	CPUMEM_Deinit(_CpuMemEdit);
	MEMED_Deinit(_BramEdit);
	MEMED_Deinit(_VramEdit);
	MEMED_Deinit(_RomEdit);
	MEMED_Deinit(_AcRamEdit);
#endif

	APU_Deinit();

	//  
	free(_pMainRam);
	free(_pBackupRam);
	free(_pBufferRam);

	// J[gbW 
	FreeCartridge(_pRom);

	JOYPAD_Deinit();

	VDC_Deinit();
	VCE_Deinit();
	CDROM_Deinit();
	INTCTRL_Deinit();
	TIMER_Deinit();

	_bSystemInit = FALSE;
}


/*-----------------------------------------------------------------------------
	[SaveBRAM]
		BRAM w̃t@Cɕۑ܂B
-----------------------------------------------------------------------------*/
BOOL
MAINBOARD_SaveBRAM(
	const char*		pathName)
{
	FILE* bram = fopen(pathName, "wb");

	if (bram != NULL && _pBackupRam != NULL)
	{
		fwrite(_pBackupRam, sizeof(Uint8), 0x2000, bram);
		fclose(bram);
		return TRUE;
	}

	return FALSE;
}


/*-----------------------------------------------------------------------------
	[LoadBRAM]
		BRAM w̃t@Cǂݍ݂܂B
-----------------------------------------------------------------------------*/
BOOL
MAINBOARD_LoadBRAM(
	const char*		pathName)
{
	FILE* bram = fopen(pathName, "rb");

	if (bram != NULL && _pBackupRam != NULL)
	{
		fread(_pBackupRam, sizeof(Uint8), 0x2000, bram);
		fclose(bram);
		return TRUE;
	}

	return FALSE;
}


/*-----------------------------------------------------------------------------
	[MAINBOARD_AdvanceFrame]
		Pt[̃G~[Vs܂B

	[DEV NOTE]

	IRQ ɂ

		EIRQ  VDC, TIMER, CD-ROM^2 VXe甭sB
		EVDC  IRQ ͑S IRQ1 gpB
		ETIMER  IRQ  TIRQ gpB
		ECD-ROM^2  IRQ  IRQ2 gpB

		EMAINBOARD_AdvanceFrame ł́A܂botP߂i߁A
		  ̏NbNɂƂÂ VDC, TIMER, PSG, CD-ROM 
		  ӑȕi߂ (XXX_AdvanceClock)B
			(݂͂PXLCPʁj

		Eӑu IRQ vƂ́AMAINBOARD_AdvanceFrame ̏
		  ̒ʒm󂯎A Interrupt Controller (IntCtrl)
		  ɓnB

		EIntCtrl ́Aʒmꂽ IRQ bot acknowledge ܂
		  LĂACLI sꂽ瑦 IRQ 𔭓ł悤
		  ԂɂB

-----------------------------------------------------------------------------*/
static
Sint32
advance_frame_stretched()
{
	Uint32		vdcRet;
	Uint32		line;
	Sint32		srcX = 0;
	Sint32		srcY = 0;
	Sint32		dstY = 0;
	Sint32		screenW = SCREEN_GetWidth();
	Sint32		screenH = SCREEN_GetHeight();
	Sint32		srcW = VDC_GetDisplayWidth();
	Sint32		dstW;
	Sint32		dstH;
	Uint16*		pLineBuf;

	if (!_bSystemInit)
		return 0;

	/* Xgb`[hł͓]͉̉ʂƓTCY */
	dstW = screenW;
	dstH = 224 * _Magnification;

	if (SCREEN_IsFullScreen())
	{
		switch (_Magnification)
		{
			case 1: dstH = 240; break;
			case 2: dstH = 480; break;
			case 3: dstH = 600; break;
			case 4: dstH = 768; break;
		}
	}

	/*  */
	if (srcW < 256)
	{
		srcX = (256 - srcW) / 2;
	}

	/* c */
	dstY = (screenH - dstH) / 2;
	if (dstY < 0)
	{
		dstY = 0;
		srcY = (dstH - screenH) / 2;
		dstH = screenH;
	}
	else if (dstY + dstH > screenH)		// VDC ɈuƂN肤 
	{
		dstY = 0;
		dstH = screenH;
	}

	do
	{
		CPU_AdvanceClock(CYCLES_PER_LINE);					// NbNx[X(PC)̏ꍇ
		APU_AdvanceClock(CYCLES_PER_LINE);
		CDROM_AdvanceClock(CYCLES_PER_LINE);
		vdcRet = VDC_AdvanceLine();
		MOUSE_AdvanceClock(CYCLES_PER_LINE);

		// PC̉摜XV 
		// {vOł͉ʂ̃ItZbglPXɐݒ肵Ă 
		//  C #19 - #258  240 Cʕ\̈Ƃ 
		//  ۂ͏㉺WC͂suʂ猩Ȃ̂
		//    C #27 - #250  224 Cʕ\̈Ƃ 
		line = VDC_GetScanLine()-1;
		if (line >= 19+8 && line < 224 + 19+8)
		{
			line -= 19+8;

			if (srcX > 0)
			{
				memset(&_ScreenBuf[line*512], 0, srcX*sizeof(Uint16));
				memset(&_ScreenBuf[line*512+srcX+srcW], 0, (256-(srcX+srcW))*sizeof(Uint16));
			}
			pLineBuf = &_ScreenBuf[line*512+srcX];

			// \̉摜f[^ RGB555/RGB565 ŏo͂Ă炤 
			VCE_EncodeLine(pLineBuf, VDC_GetLineBuffer(), VDC_GetSpBgBuffer(), VDC_GetDisplayWidth());
		}
	} while ((vdcRet & VDC_VBL_END) == 0);

	// ʕ\XV 
	if (SCREEN_WaitVBlank(FALSE))
	{
		if (SCREEN_Lock())
		{
			if (srcW < 256)
				srcW = 256;

			SCREEN_Blt(_ScreenBuf, 0, 0, 0, dstY, srcW, 224, dstW, dstH, 512, 0);
			SCREEN_Update(0, 0, screenW, screenH);
			SCREEN_Unlock();
		}

#ifdef DEBUGBUILD
		MAINBOARD_Update();
#endif
	}

	return 0;
}


Sint32
MAINBOARD_AdvanceFrame()
{
	static Sint32 prevW = 256;
	Sint32		cycles;
	Uint32		vdcRet;
	Uint32		line;
	Sint32		srcX = 0;
	Sint32		srcY = 0;
	Sint32		dstX = 0;
	Sint32		dstY = 0;
	Sint32		screenW = SCREEN_GetWidth();
	Sint32		screenH = SCREEN_GetHeight();
	Sint32		srcW = VDC_GetDisplayWidth();
	Sint32		dstW;
	Sint32		dstH;
	BOOL		bHires = VCE_IsHiresolutionMode();
	Uint16*		pLineBuf;

	if (!_bSystemInit || _bPause)
	{
		// if (!DISASMWND_TraceRunning())
		SCREEN_WaitVBlank(FALSE);
		return 0;
	}

	// Xgb`[ĥƂ͂ 
	if (_bStretched)
		return advance_frame_stretched();

	// ȉ̓Xgb`[hȊÔƂ̏ 
	pLineBuf = &_ScreenBuf[0];

	if (prevW != srcW/* || prevH != srcH*/)
	{
		prevW = srcW;
		SCREEN_Update(0, 0, 0, 0);
	}

	/* WԂł͓]Ɠ]͓TCY */
	dstW = srcW;
	dstH = 224; /*srcH;*/

	dstW *= _Magnification;
	dstH *= _Magnification;

	/*  */
	/* `敝ʃTCY傫Ƃ */
	if (dstW > screenW)
	{
		if (bHires)
		{
			srcX = 0;
			dstX = 0;
			srcW = srcW;
			if (_Magnification >= 2)
				dstW = screenW;
			else
			{
				dstW = srcW/2;
				dstX = (screenW - dstW) / 2;
			}
		}
		else
		{
			srcX = (dstW - screenW) / 2;
			if (_Magnification >= 2)
				srcX /= 2;

			dstX = 0;
			srcW = screenW / _Magnification;
			dstW = screenW;
		}
	}
	else if (screenW > dstW)	/* Ƃ */
	{
		srcX = 0;
		dstX = (screenW - dstW) / 2;
		srcW = srcW;
		dstW = dstW;
	}

	/* c */
	dstY = (screenH - dstH) / 2;
	if (dstY < 0)
	{
		dstY = 0;
		srcY = (dstH - screenH) / 2;
		dstH = screenH;
	}
	else if (dstY + dstH > screenH)		// VDC ɈuƂN肤 
	{
		dstY = 0;
		dstH = screenH;
	}

//	printf("srcX=%d dstX=%d srcW=%d dstW=%d scrW=%d scrH=%d\n", srcX, dstX, srcW, dstW, screenW, screenH);

	if (!SCREEN_Lock())
		return 0;

	cycles = 456;
	do
	{
		CPU_AdvanceClock(cycles);					// NbNx[X(PC)̏ꍇ
		APU_AdvanceClock(cycles);
		CDROM_AdvanceClock(cycles);
		vdcRet = VDC_AdvanceLine();
		MOUSE_AdvanceClock(cycles);

		// PC̉摜XV 
		// {vOł͉ʂ̃ItZbglPXɐݒ肵Ă 
		//  C #19 - #258  240 Cʕ\̈Ƃ 
		//  ۂ͏㉺WC͂suʂ猩Ȃ̂
		//    C #27 - #250  224 Cʕ\̈Ƃ 
		line = VDC_GetScanLine()-1;
		if (line >= 19+8 && line < 224 + 19+8)
		{
			line -= 19+8;

			// \̉摜f[^ RGB555/RGB565 ŏo͂Ă炤 
			VCE_EncodeLine(pLineBuf, VDC_GetLineBuffer(), VDC_GetSpBgBuffer(), VDC_GetDisplayWidth());

			// PC݂̏Ȃ̂ŎgCrc 
			SCREEN_Blt(pLineBuf, srcX, 0, dstX, dstY+line*_Magnification, srcW, 1, dstW, _Magnification, 512*_Magnification, 0);
		}
	} while ((vdcRet & VDC_VBL_END) == 0);

	SCREEN_Unlock();

	// ʕ\XV 
	if (SCREEN_WaitVBlank(TRUE))
	{
		SCREEN_Update(dstX, dstY, dstW, dstH);
#ifdef DEBUGBUILD
		MAINBOARD_Update();
#endif
	}

	return 0;
}


Sint32
MAINBOARD_AdvanceInstruction(Uint32 nInst)
{
	static Sint32	prevW = 256;
	static Uint32	prevLine = 0;
	static Sint32	cycles = 0;
	Uint32		line;
	Sint32		srcX = 0;
	Sint32		srcY = 0;
	Sint32		dstX = 0;
	Sint32		dstY = 0;
	Sint32		screenW = SCREEN_GetWidth();
	Sint32		screenH = SCREEN_GetHeight();
	Sint32		srcW = VDC_GetDisplayWidth();
	Sint32		dstW;
	Sint32		dstH;
	BOOL		bHires = VCE_IsHiresolutionMode();
	Uint16*		pLineBuf;
	Sint32		ret = 0;

	if (!_bSystemInit)
		return 0;

#ifndef DEBUGBUILD
	return 0;
#endif

	pLineBuf = &_ScreenBuf[0];

	if (prevW != srcW)
	{
		prevW = srcW;
		SCREEN_Update(0, 0, 0, 0);
	}

	/* WԂł͓]Ɠ]͓TCY */
	dstW = srcW;
	dstH = 224; /*srcH;*/

	dstW *= _Magnification;
	dstH *= _Magnification;

	/*  */
	/* `敝ʃTCY傫Ƃ */
	if (dstW > screenW)
	{
		if (bHires)
		{
			srcX = 0;
			dstX = 0;
			srcW = srcW;
			if (_Magnification >= 2)
				dstW = screenW;
			else
			{
				dstW = srcW/2;
				dstX = (screenW - dstW) / 2;
			}
		}
		else
		{
			srcX = (dstW - screenW) / 2;
			if (_Magnification >= 2)
				srcX /= 2;

			dstX = 0;

			srcW = screenW / _Magnification;

			dstW = screenW;
		}
	}
	else if (screenW > dstW)	/* Ƃ */
	{
		srcX = 0;
		dstX = (screenW - dstW) / 2;
		srcW = srcW;
		dstW = dstW;
	}

	/* c */
	dstY = (screenH - dstH) / 2;
	if (dstY < 0)
	{
		dstY = 0;
		srcY = (dstH - screenH) / 2;
		dstH = screenH;
	}
	else if (dstY + dstH > screenH)		// VDC ɈuƂN肤 
	{
		dstY = 0;
		dstH = screenH;
	}

	while ((nInst-- > 0) && (ret == 0))
	{
		cycles += CPU_ExecuteSingleInstruction();
		if (CPU_IsBreakPoint())		ret |= MAINBOARD_BREAKPOINT;
		if (CPU_IsWatchPoint())		ret |= MAINBOARD_WATCHPOINT;

		while (cycles >= CYCLES_PER_LINE)
		{
			cycles -= CYCLES_PER_LINE;
			APU_AdvanceClock(CYCLES_PER_LINE);
			CDROM_AdvanceClock(CYCLES_PER_LINE);
			MOUSE_AdvanceClock(CYCLES_PER_LINE);

			if (VDC_AdvanceClock(CYCLES_PER_LINE) & VDC_VBL_END)
			{
				INPUT_UpdateState();
			}

			// PC̉摜XV 
			// {vOł͉ʂ̃ItZbglPXɐݒ肵Ă 
			//  C #19 - #258  240 Cʕ\̈Ƃ 
			//  ۂ͏㉺WC͂suʂ猩Ȃ̂
			//    C #27 - #250  224 Cʕ\̈Ƃ 
			line = VDC_GetScanLine()-1;
			if (line != prevLine)
			{
				prevLine = line;

				if (line >= 19+8 && line < 224 + 19+8)
				{
					line -= 19+8;

					// 摜f[^ RGB555/RGB565 ŏo͂Ă炤 
					VCE_EncodeLine(pLineBuf, VDC_GetLineBuffer(), VDC_GetSpBgBuffer(), VDC_GetDisplayWidth());

					if (!SCREEN_Lock())
						return 0;

					// PC݂̏Ȃ̂ŎgCrc 
					SCREEN_Blt(pLineBuf, srcX, 0, dstX, dstY+line*_Magnification, srcW, 1, dstW, _Magnification, 512*_Magnification, 0);

					SCREEN_Unlock();
				}
				SCREEN_Update(dstX, dstY, dstW, dstH);
			}
		}
	}

	return ret;
}


void
MAINBOARD_Update()
{
#ifdef DEBUGBUILD
	// ʕ\XV 
	DISASMWND_Update(_DisasmWindow);
	REGED_Update(_RegEdit);
	MEMED_Update(_VramEdit);
	MEMED_Update(_RomEdit);
	MEMED_Update(_AcRamEdit);
	MEMED_Update(_CdRamEdit);
	CPUMEM_Update(_CpuMemEdit);
#endif
}


void
MAINBOARD_SwitchMemoryEditor()
{
#ifdef DEBUGBUILD
	static int		n = 0;

	CPUMEM_Deinit(_CpuMemEdit);
	MEMED_Deinit(_BramEdit);
	MEMED_Deinit(_VramEdit);
	MEMED_Deinit(_RomEdit);
	MEMED_Deinit(_AcRamEdit);
	MEMED_Deinit(_CdRamEdit);

	_CpuMemEdit = NULL;
	_BramEdit   = NULL;
	_VramEdit   = NULL;
	_RomEdit    = NULL;
	_AcRamEdit  = NULL;
	_CdRamEdit  = NULL;

	switch (++n)
	{
		case 0:
			_CpuMemEdit = CPUMEM_Init("[CPURAM] / ROM / VRAM / CDRAM / ACRAM / BRAM (TAB key to switch)",
										debugger_read_func, debugger_write_func, WINMAIN_GetHInstance());
			break;
		case 1:
			_RomEdit    = MEMED_Init("[ROM] / VRAM / CDRAM / ACRAM / BRAM / CPURAM (TAB key to switch)",
										_pRom,       _RomMask+1, WINMAIN_GetHInstance());
			break;
		case 2:
			_VramEdit   = MEMED_Init("[VRAM] / CDRAM / ACRAM / BRAM / CPURAM / ROM (TAB key to switch)",
										VDC_GetVideoRam(),  0x10000, WINMAIN_GetHInstance());
			break;
		case 3:
			_CdRamEdit  = MEMED_Init("[CDRAM] / ACRAM / BRAM / CPURAM / ROM / VRAM (TAB key to switch)",
										_pBufferRam,  0x40000, WINMAIN_GetHInstance());
			break;
		case 4:
			_AcRamEdit  = MEMED_Init("[ACRAM] / BRAM / CPURAM / ROM / VRAM / CDRAM (TAB key to switch)",
										_AcRam,      0x200000,   WINMAIN_GetHInstance());
			break;
		case 5:
			_BramEdit   = MEMED_Init("[BRAM] / CPURAM / ROM / VRAM / CDRAM / ACRAM (TAB key to switch)",
										_pBackupRam,  0x2000, WINMAIN_GetHInstance());
			n = -1;
			break;
	}
#endif
}


void
MAINBOARD_ActivateIRQ1()
{
	CPU_ActivateIRQ1();
}

void
MAINBOARD_ActivateIRQ2()
{
	CPU_ActivateIRQ2();
}

void
MAINBOARD_ActivateTIRQ()
{
	CPU_ActivateTIMER();
}


void
MAINBOARD_Pause(
	BOOL		bPause)
{
	_bPause = bPause;
	APU_Pause(bPause);
	CDROM_Pause(bPause);
}


/*-----------------------------------------------------------------------------
	[Reset]
-----------------------------------------------------------------------------*/
void
MAINBOARD_Reset()
{
	CPU_Reset();
	APU_Reset();
	CDROM_Reset();
	PRINTF("MAINBOARD_Reset()");
	MAINBOARD_Update();
}


/*-----------------------------------------------------------------------------
	[ChangeScreenMode]
		XN[̐ݒύX܂B
-----------------------------------------------------------------------------*/
BOOL
MAINBOARD_ChangeScreenMode(
	Sint32		screenWidth,
	Sint32		screenHeight,
	Sint32		magnification,
	Sint32		bitsPerPixel,
	BOOL		bFullScreen,
	BOOL		bScanLine,
	BOOL		bStretched,
	BOOL		bSync60HzScreen,
	BOOL		bHardwareAcceleration)
{
	Uint32		flags;

	_bStretched = bStretched;
	_Magnification = magnification;

	flags = SCREEN_FDEFAULT;
	if (bFullScreen)			flags |= SCREEN_FFULLSCREEN;
	if (bScanLine)				flags |= SCREEN_FSCANLINED;
	if (bSync60HzScreen)		flags |= SCREEN_FSYNCTO60HZFULLSCREEN;
	if (bHardwareAcceleration)	flags |= SCREEN_FHARDWAREACCELERATION;

	return SCREEN_ChangeMode(screenWidth, screenHeight, bitsPerPixel, flags);
}


/*-----------------------------------------------------------------------------
	[ChangeSoundMode]
		TEh̐ݒύX܂B
-----------------------------------------------------------------------------*/
BOOL
MAINBOARD_ChangeSoundMode(
	Uint32		bufferSize,
	Uint32		sampleRate,
	Uint32		overSampleRate,
	Uint32		masterVolume,
	BOOL		bReverb)
{
	APU_Deinit();

	if (!APU_Init(sampleRate, bufferSize))
	{
		PRINTF("MAINBOARD_ChangeSoundMode:  failed changing sound mode.");
		return FALSE;
	}

	APU_SetVolume(masterVolume);

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[ChangeButtonConnection]
		{^̐ڑύX܂B
-----------------------------------------------------------------------------*/
BOOL
MAINBOARD_ChangeButtonConnection(
	Sint32		joyID,
	Sint32		pceButton,
	Sint32		appButton)
{
	return JOYPAD_ChangeButtonConnection(joyID, pceButton, appButton);
}


/*-----------------------------------------------------------------------------
	[ChangeFPS]
		eor̐ݒύX܂B
-----------------------------------------------------------------------------*/
void
MAINBOARD_ChangeFPS(
	float		fps)
{
	SCREEN_SetFPS(fps);
}


/*-----------------------------------------------------------------------------
	[ChangeMemoryValue]
		̓eύX܂B
-----------------------------------------------------------------------------*/
void
MAINBOARD_ChangeMemoryValue(
	Sint32		ramType,
	Uint32		addr,
	Uint8		data)
{
	if (_pMainRam == NULL)
		return;

	if (_pBufferRam == NULL)
		return;

	switch (ramType)
	{
		case MAINBOARD_MAINRAM:
			_pMainRam[addr & 0x1fff] = data;
			break;
		case MAINBOARD_BUFFERRAM:
			_pBufferRam[addr & 0x3ffff] = data;
			break;
		case MAINBOARD_ARCADERAM:
			_AcRam[addr & 0x1fffff] = data;
			break;
	}
}


// save variable
#define SAVE_V(V)	if (fwrite(&V, sizeof(V), 1, p) != 1)	return FALSE
#define LOAD_V(V)	if (fread(&V, sizeof(V), 1, p) != 1)	return FALSE
// save array
#define SAVE_P(P, N)	if (fwrite(P, N, 1, p) != 1)	return FALSE
#define LOAD_P(P, N)	if (fread(P, N, 1, p) != 1)		return FALSE
/*-----------------------------------------------------------------------------
	[SaveState]
		Ԃt@Cɕۑ܂B 
-----------------------------------------------------------------------------*/
BOOL
MAINBOARD_SaveState(
	FILE*		p)
{
	BOOL		ret;
	int			i;

	if (p == NULL)
		return FALSE;

	SAVE_P(_pMainRam, 0x2000);
	SAVE_P(_pBackupRam, 0x2000);
	SAVE_P(_pBufferRam, 0x40000);
	SAVE_P(&_AcRam[0], 0x200000);
	SAVE_P(&_Ac[0], sizeof(_Ac));

	// |C^ItZbgϊ 
	for (i = 0; i < 256; i++)
	{
		Uint32	offset = _pRomMap[i] - _pRom;
		SAVE_V(offset);
	}

	SAVE_V(_bSystemInit);
	SAVE_V(_Buffer);
	SAVE_V(_AcShift);
	SAVE_V(_AcShiftBits);
	SAVE_V(_0x1ae5);

	ret =  CPU_SaveState(p);
	ret |= TIMER_SaveState(p);
	ret |= INTCTRL_SaveState(p);
	ret |= JOYPAD_SaveState(p);
	ret |= APU_SaveState(p);
	ret |= CDROM_SaveState(p);
	ret |= VDC_SaveState(p);
	ret |= VCE_SaveState(p);

	return ret;
}


/*-----------------------------------------------------------------------------
	[LoadState]
		Ԃt@Cǂݍ݂܂B 
-----------------------------------------------------------------------------*/
BOOL
MAINBOARD_LoadState(
	FILE*		p)
{
	BOOL		ret;
	int			i;

	if (p == NULL)
		return FALSE;

	LOAD_P(_pMainRam, 0x2000);
	LOAD_P(_pBackupRam, 0x2000);
	LOAD_P(_pBufferRam, 0x40000);
	LOAD_P(&_AcRam[0], 0x200000);
	LOAD_P(&_Ac[0], sizeof(_Ac));

	// ItZbg|C^ϊ 
	for (i = 0; i < 256; i++)
	{
		Uint32	offset;
		LOAD_V(offset);
		_pRomMap[i] = offset + _pRom;
	}

	LOAD_V(_bSystemInit);
	LOAD_V(_Buffer);
	LOAD_V(_AcShift);
	LOAD_V(_AcShiftBits);
	LOAD_V(_0x1ae5);

	ret =  CPU_LoadState(p);
	ret |= TIMER_LoadState(p);
	ret |= INTCTRL_LoadState(p);
	ret |= JOYPAD_LoadState(p);
	ret |= APU_LoadState(p);
	ret |= CDROM_LoadState(p);
	ret |= VDC_LoadState(p);
	ret |= VCE_LoadState(p);

	return TRUE;
}

#undef SAVE_V
#undef SAVE_P
#undef LOAD_V
#undef LOAD_P
