/*-----------------------------------------------------------------------------
	[APU.c]
		`ot܂B

	Copyright (C) 2004 Ki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
**---------------------------------------------------------------------------*/
#include "APU.h"
#include "AudioInterface.h"
#include "PSG.h"
#include "ADPCM.h"
#include "Printf.h"

#define APU_MAX_AMPLIFICATION		512


static Sint32		_SampleRate = 44100;	// [Hz]
static Sint32		_BufSize = 2560;	// [sample]

static Sint32		_nSamplesPerFrame;

static Sint32*		_pMixBuf;
static Sint32		_MixBufPos;
static double		_MixBufEndPos;

static Uint32		_ClockCount;

static Uint32		_Volume;				// APU volume (0-512)



#if defined(__GNUC__) && defined(USE_INLINE_ASM)
static
inline
Sint32
saturate(Sint32 val, Sint32 min, Sint32 max)
{
	__asm__(
			"cmpl		%1, %0			\n\t"
			"cmovll		%1, %0			\n\t"
			"cmpl		%2, %0			\n\t"
			"cmovgl		%2, %0			\n\t"
			: "+r" (val)
			: "r" (min), "r" (max)
	);
	return val;
}
#else
static
inline
Sint32
saturate(Sint32 val, Sint32 min, Sint32 max)
{
	if (val < min)	val = min;
	if (val > max)	val = max;

	return val;
}
#endif


static
inline
void
normalize_output(
	Sint16*		pDst,
	Sint32*		pSrc,
	Sint32		nSamples)
{
	const	Sint32		amp = (Sint32)_Volume;
	int		i;

	for (i = 0; i < nSamples; i++)
	{
		pSrc[0] *= amp;
		pSrc[0] = saturate(pSrc[0], -32768<<15, 32767<<15);

		pSrc[1] *= amp;
		pSrc[1] = saturate(pSrc[1], -32768<<15, 32767<<15);

		*pDst++ = (Sint16)(pSrc[0] >> 15);
		*pDst++ = (Sint16)(pSrc[1] >> 15);

		pSrc[0] = pSrc[1] = 0;
		pSrc += 2;
	}
}



/*-----------------------------------------------------------------------------
	[callback_mixer]
		̊֐ AudioInterface ̃R[obNƂēo^B
	̊֐Ăяo邽тɁAꂼ̃`l̏o͂
	Sint16 ɕϊ pDst ɏoB`l̏óiTvj
	Ȃꍇ́A̕[B
-----------------------------------------------------------------------------*/
static
void
callback_mixer(
	Sint16*			pDst,				// o͐obt@ 
	Sint32			nSamples)			// oTv 
{
	Sint32*			pSrc = _pMixBuf;

	AOUT_Lock();		// r 

	if (_MixBufPos < nSamples)
	{
		// Ȃ[B
		Uint32		n = nSamples - _MixBufPos;
		PSG_Mix(_pMixBuf+_MixBufPos*2, n);
		ADPCM_Mix(_pMixBuf+_MixBufPos*2, _SampleRate, n);
	}

	normalize_output(pDst, pSrc, nSamples);

	_MixBufPos = 0;
	_MixBufEndPos = 0.0;

	AOUT_Unlock();
}


/*-----------------------------------------------------------------------------
	[Init]
		APU܂B
-----------------------------------------------------------------------------*/
BOOL
APU_Init(
	Uint32		sampleRate,
	Uint32		bufSize)		// in samples
{
	// 8 [byte/sample] for Sint32 stereo buffer 
	if ((_pMixBuf = (Sint32*)malloc(bufSize*sizeof(Sint32)*2)) == NULL)
	{
		AOUT_Deinit(NULL);	return FALSE;
	}

	ZeroMemory(_pMixBuf, bufSize*sizeof(Sint32)*2);

	if (!AOUT_Init(bufSize, sampleRate, FALSE, callback_mixer))
		return FALSE;

	_BufSize		= bufSize;
	_SampleRate		= sampleRate;
	_ClockCount		= 0;
	_MixBufPos		= 0;
	_MixBufEndPos	= 0;
	_Volume			= APU_MAX_AMPLIFICATION;

	_nSamplesPerFrame = sampleRate / 60;

	PSG_Init(sampleRate);
	PSG_Reset();

	ADPCM_Init();
	ADPCM_Reset();

	AOUT_Play(TRUE);

	return TRUE;
}


void
APU_Pause(
	BOOL		bPause)
{
	if (bPause)
		AOUT_Play(FALSE);
	else
		AOUT_Play(TRUE);
}


void
APU_Deinit()
{
	AOUT_Deinit();
	PSG_Deinit();
	ADPCM_Deinit();
	free(_pMixBuf);
	_pMixBuf = NULL;
}


void
APU_Reset()
{
	PSG_Reset();
	ADPCM_Reset();
}


/*-----------------------------------------------------------------------------
	[SetSampleRate]
-----------------------------------------------------------------------------*/
BOOL
APU_SetSampleRate(
	Uint32			sampleRate)
{
	AOUT_Play(FALSE);
	AOUT_Deinit();

	ZeroMemory(_pMixBuf, _BufSize*sizeof(Sint32)*2);

	if (!AOUT_Init(_BufSize, sampleRate, FALSE, callback_mixer))
	{
		// ̐ݒɖ߂ 
		if (!AOUT_Init(_BufSize, _SampleRate, FALSE, callback_mixer))
		{
			// ł_Ȃ炠߂B 
			return FALSE;
		}
	}
	_nSamplesPerFrame = sampleRate / 60;
	_SampleRate = sampleRate;

	PSG_SetSampleRate(sampleRate);

	AOUT_Play(TRUE);

	PRINTF("APU_SetSampleRate(%ld)\n", _SampleRate);

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[SetBufferSize]
-----------------------------------------------------------------------------*/
BOOL
APU_SetBufferSize(
	Uint32			bufSize)
{
	AOUT_Play(FALSE);
	AOUT_Deinit();

	free(_pMixBuf);
	// 8 [byte/sample] for Sint32 stereo buffer 
	if ((_pMixBuf = (Sint32*)malloc(bufSize*sizeof(Sint32)*2)) == NULL)
	{
		AOUT_Deinit(NULL);	return FALSE;
	}

	ZeroMemory(_pMixBuf, bufSize*sizeof(Sint32)*2);

	if (!AOUT_Init(bufSize, _SampleRate, FALSE, callback_mixer))
	{
		// ̐ݒɖ߂ 
		free(_pMixBuf);
		if ((_pMixBuf = (Sint32*)malloc(_BufSize*sizeof(Sint32)*2)) == NULL)
		{
			AOUT_Deinit(NULL);	return FALSE;
		}

		if (!AOUT_Init(_BufSize, _SampleRate, FALSE, callback_mixer))
		{
			// ł_Ȃ炠߂B 
			return FALSE;
		}
	}

	_BufSize		= bufSize;
	_ClockCount		= 0;
	_MixBufPos		= 0;
	_MixBufEndPos	= 0;

	AOUT_Play(TRUE);

	PRINTF("APU_SetSoundBuffer(%ld)\n", _BufSize);

	return TRUE;
}


void
APU_SetVolume(
	Uint32		volume)		// 0 - 65535
{
	_Volume = volume * APU_MAX_AMPLIFICATION / 65535;
}


#define DIV		256


Uint32
APU_AdvanceClock(
	Sint32		clock)		// 456 ɌŒ
{
	Sint32		nSamples;
	Sint32		deltaClock;
	double		deltaPos;

	ADPCM_AdvanceClock(clock);

	_ClockCount += clock;

	deltaClock = 2 * PSG_FRQ / 60 / DIV;
	deltaPos   = (double)_SampleRate / 60.0 / (double)DIV;

	if (_ClockCount >= deltaClock)
	{
		_ClockCount -= deltaClock;

		AOUT_Lock();		// r 

		// J[\obt@͈͓̔Ȃ~bNXB 
		if (_MixBufPos < _BufSize)
		{
			_MixBufEndPos += deltaPos;

			nSamples = (Sint32)_MixBufEndPos - _MixBufPos;

			// nSamples Ȃꍇ͏邾
			if (_MixBufPos + nSamples > _BufSize)
			{
				nSamples = _BufSize - _MixBufPos;
			}
			// XeI _MixBufPos*2 
			PSG_Mix(_pMixBuf + _MixBufPos*2, nSamples);
			ADPCM_Mix(_pMixBuf + _MixBufPos*2, _SampleRate, nSamples);
			_MixBufPos += nSamples;
		}

		AOUT_Unlock();
	}
	return 0;
}

// save variable
#define SAVE_V(V)	if (fwrite(&V, sizeof(V), 1, p) != 1)	return FALSE
#define LOAD_V(V)	if (fread(&V, sizeof(V), 1, p) != 1)	return FALSE
/*-----------------------------------------------------------------------------
	[SaveState]
		Ԃt@Cɕۑ܂B 
-----------------------------------------------------------------------------*/
BOOL
APU_SaveState(
	FILE*		p)
{
	BOOL	ret;

	if (p == NULL)
		return FALSE;

	SAVE_V(_ClockCount);

	ret =  PSG_SaveState(p);
	ret |= ADPCM_SaveState(p);
	
	return ret;
}


/*-----------------------------------------------------------------------------
	[LoadState]
		Ԃt@Cǂݍ݂܂B 
-----------------------------------------------------------------------------*/
BOOL
APU_LoadState(
	FILE*		p)
{
	BOOL	ret;

	if (p == NULL)
		return FALSE;

	LOAD_V(_ClockCount);

	ret =  PSG_LoadState(p);
	ret |= ADPCM_LoadState(p);

	return TRUE;
}

#undef SAVE_V
#undef LOAD_V
