/*-----------------------------------------------------------------------------
	[ApuQueue.c]
		TEhWX^̃L[܂B

	Copyright (C) 2004 Ki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
-----------------------------------------------------------------------------*/
#include "ApuQueue.h"

#include "CriticalSection.h"

/*
	`otpw{w{L[̎dl

	WX^ɏ݂sȂ邲ƂɁA
	L[ɂ̓eǉB

	TEhobt@XVɌoߎԂ݂āA
	ߋɏ܂ꂽWX^e
	܂ꂽɃL[oA
	orfWX^XVB
	iȂorfWX^͑S write only Ƃ݂Ȃ)
	  vmF

	L[ɒǉƂɂ write index pA
	oƂɂ read index pB

	// ǉ
	queue[write index++] = written data

	// o
	data = queue[read index++]

	L[loƂ read index 
	write index ƈvƂ queue underflowB
	Ƃ肠ȂɂȂB

	L[ɒlǉƂ write index 
	read index ƈvƂ queue overflowB
	Ƃ肠Zbg邱ƂɂB
*/


#define APUQUEUE_SIZE	65536				// must be power of 2

typedef struct
{
	Uint32			clock;					// cpu cycles elapsed since previous write
	Uint8			reg;					// 0-15
	Uint8			data;					// written data
} ApuQueue;


static ApuQueue				_Queue[APUQUEUE_SIZE];
static Sint32				_QueueWriteIndex;
static Sint32				_QueueReadIndex;
static double				_ClockCounter;

/* L[ɍő剽NbÑf[^܂Ă邩 */
static Sint32				_TotalClockAdvanced;

static Sint32				_CriticalSection;

static BOOL					_bInit;


/*-----------------------------------------------------------------------------
	[Init]
-----------------------------------------------------------------------------*/
BOOL
APUQUEUE_Init()
{
	_bInit = FALSE;

	_TotalClockAdvanced = 0;

	_CriticalSection = CS_Create();

	if (_CriticalSection < 0)
		return FALSE;

	_bInit = TRUE;
	return TRUE;
}


/*-----------------------------------------------------------------------------
	[Deinit]
-----------------------------------------------------------------------------*/
void
APUQUEUE_Deinit()
{
	if (_bInit)
	{
		CS_Delete(_CriticalSection);
		_bInit = FALSE;
	}
}


/*-----------------------------------------------------------------------------
	[Lock]
-----------------------------------------------------------------------------*/
void
APUQUEUE_Lock()
{
	if (_bInit)
		CS_Enter(_CriticalSection);
}


/*-----------------------------------------------------------------------------
	[Unlock]
-----------------------------------------------------------------------------*/
void
APUQUEUE_Unlock()
{
	if (_bInit)
		CS_Leave(_CriticalSection);
}


/*-----------------------------------------------------------------------------
	[Reset]
-----------------------------------------------------------------------------*/
void
APUQUEUE_Reset()
{
	_Queue[0].clock = 0;
	_QueueWriteIndex = 0;
	_QueueReadIndex  = 0;
	_ClockCounter = 0.;
	_TotalClockAdvanced = 0;
}


/*-----------------------------------------------------------------------------
	[Add]
-----------------------------------------------------------------------------*/
BOOL
APUQUEUE_Add(
	Sint32	clock,
	Uint8	reg,
	Uint8	data)
{
	if (((_QueueWriteIndex + 1) & (APUQUEUE_SIZE-1)) == _QueueReadIndex)
	{
		// L[^ 
		return FALSE;
	}

	_Queue[_QueueWriteIndex].reg   = reg;
	_Queue[_QueueWriteIndex].data  = data;
	_QueueWriteIndex = (_QueueWriteIndex + 1) & (APUQUEUE_SIZE-1);
	_Queue[_QueueWriteIndex].clock = clock;

	_TotalClockAdvanced += clock;

//	printf("add: clock=%d, _TotalClockAdvanced=%d, writeIndex=%d, readIndex=%d\n", clock, _TotalClockAdvanced, _QueueWriteIndex, _QueueReadIndex);

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[Get]
-----------------------------------------------------------------------------*/
BOOL
APUQUEUE_Get(
	Uint8*	pRegNum,
	Uint8*	pData)
{
	if (_QueueReadIndex != _QueueWriteIndex)
	{
		Uint32	clock = _Queue[_QueueReadIndex].clock;
		Sint32	dc = (Sint32)(clock - (Uint32)_ClockCounter);

		if (dc <= 0)
		{
			_ClockCounter -= (double)clock;
			_TotalClockAdvanced -= clock;

			if (_TotalClockAdvanced < 0)
				_TotalClockAdvanced = 0;

//			printf("get: _ClockCounter=%d, clock=%d, _TotalClockAdvanced=%d, writeIndex=%d, readIndex=%d\n", _ClockCounter, _Queue[_QueueReadIndex].clock, _TotalClockAdvanced, _QueueWriteIndex, _QueueReadIndex);

			*pRegNum = _Queue[_QueueReadIndex].reg;
			*pData   = _Queue[_QueueReadIndex].data;
			_QueueReadIndex = (_QueueReadIndex + 1) & (APUQUEUE_SIZE-1);
			return TRUE;
		}
	}

	return FALSE;
}


/*-----------------------------------------------------------------------------
	[ForceRead]
-----------------------------------------------------------------------------*/
BOOL
APUQUEUE_ForceRead(
	Uint8*	pRegNum,
	Uint8*	pData)
{
	if (_QueueReadIndex != _QueueWriteIndex)
	{
		*pRegNum = _Queue[_QueueReadIndex].reg;
		*pData   = _Queue[_QueueReadIndex].data;
		_QueueReadIndex = (_QueueReadIndex + 1) & (APUQUEUE_SIZE-1);
		return TRUE;
	}

	return FALSE;
}


/*-----------------------------------------------------------------------------
	[AdvanceCounter]
-----------------------------------------------------------------------------*/
BOOL
APUQUEUE_AdvanceCounter(
	double		clock)
{
	_ClockCounter += clock;

	if (_ClockCounter >= 1.7e300)
	{
		return FALSE;
	}

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[GetNumData]
-----------------------------------------------------------------------------*/
Sint32
APUQUEUE_GetNumData()
{
	return (_QueueWriteIndex - _QueueReadIndex) & (APUQUEUE_SIZE-1);
}


/*-----------------------------------------------------------------------------
	[GetTotalClockAdvanced]
		Returns the total clock count in queue.
-----------------------------------------------------------------------------*/
Sint32
APUQUEUE_GetTotalClockAdvanced()
{
//	printf("_TotalClockAdvanced = %d\n", _TotalClockAdvanced);
	return _TotalClockAdvanced;
}


// save variable
#define SAVE_V(V)	if (fwrite(&V, sizeof(V), 1, p) != 1)	return FALSE
#define LOAD_V(V)	if (fread(&V, sizeof(V), 1, p) != 1)	return FALSE
// save array
#define SAVE_A(A)	if (fwrite(A, sizeof(A), 1, p) != 1)	return FALSE
#define LOAD_A(A)	if (fread(A, sizeof(A), 1, p) != 1)		return FALSE
/*-----------------------------------------------------------------------------
	[SaveState]
-----------------------------------------------------------------------------*/
BOOL
APUQUEUE_SaveState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	SAVE_A(_Queue);

	SAVE_V(_QueueWriteIndex);
	SAVE_V(_QueueReadIndex);
	SAVE_V(_ClockCounter);
	SAVE_V(_bInit);
	SAVE_V(_TotalClockAdvanced);

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[LoadState]
-----------------------------------------------------------------------------*/
BOOL
APUQUEUE_LoadState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	LOAD_A(_Queue);

	LOAD_V(_QueueWriteIndex);
	LOAD_V(_QueueReadIndex);
	LOAD_V(_ClockCounter);
	LOAD_V(_bInit);
	LOAD_V(_TotalClockAdvanced);

	return TRUE;
}

#undef SAVE_V
#undef SAVE_A
#undef LOAD_V
#undef LOAD_A
