/******************************************************************************
	[CdFader.c]
		Implement CD fader using mixer device.

	Copyright (C) 2004 Ki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
******************************************************************************/
#include <stdio.h>		// debug

#include "CdFader.h"
#include "WinMain.h"
#include "Printf.h"

#define N_SHIFT		8


static HMIXER					_hMix;

static Sint32					_MaxVolumeCd;
static Sint32					_MaxVolumeMaster;
static Sint32					_MaxVolumeWave;

static Sint32					_InitialCdVolume;
static Sint32					_CurrentCdVolume;
static Sint32					_VolumeStep;
static BOOL						_bFadeIn  = FALSE;
static BOOL						_bFadeOut = FALSE;

static Sint32					_ClockCount;
static Sint32					_FadeCycle;

static Sint32					_PreservedVolumeMaster;
static Sint32					_PreservedVolumeCd;
static Sint32					_PreservedVolumeWave;

static DWORD					_dwIDControlMaster;
static DWORD					_dwIDControlCd;
static DWORD					_dwIDControlWave;


static
void
set_volume(
	const HMIXER					hMix,
	const Uint32					controlID,
	const Sint32					volume)
{
	MIXERCONTROLDETAILS				mcd;
	MIXERCONTROLDETAILS_UNSIGNED	mcdu;

	if (hMix == INVALID_HANDLE_VALUE)
		return;

	mcd.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mcd.dwControlID = (DWORD)controlID;
	mcd.cChannels = 1;
	mcd.hwndOwner = NULL;
	mcd.cbDetails = sizeof(MIXERCONTROLDETAILS_UNSIGNED);
	mcd.paDetails = (LPMIXERCONTROLDETAILS_UNSIGNED)&mcdu;

	mcdu.dwValue = (DWORD)volume;
	mixerSetControlDetails((HMIXEROBJ)hMix, &mcd, MIXER_OBJECTF_HMIXER|MIXER_SETCONTROLDETAILSF_VALUE);
}


/* get volume by control ID */
static
Sint32
get_volume(
	const HMIXER					hMix,
	const Uint32					controlID)
{
	MIXERCONTROLDETAILS				mcd;
	MIXERCONTROLDETAILS_UNSIGNED	mcdu;

	if (hMix == INVALID_HANDLE_VALUE)
		return -1;

	mcd.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mcd.dwControlID = (DWORD)controlID;
	mcd.cChannels = 1;
	mcd.hwndOwner = NULL;
	mcd.cbDetails = sizeof(MIXERCONTROLDETAILS_UNSIGNED);
	mcd.paDetails = (LPMIXERCONTROLDETAILS_UNSIGNED)&mcdu;

	if (mixerGetControlDetails((HMIXEROBJ)hMix, &mcd, MIXER_OBJECTF_HMIXER|MIXER_GETCONTROLDETAILSF_VALUE) != MMSYSERR_NOERROR)
	{
		return -1;
	}

	return (Sint32)mcdu.dwValue;
}


/* get control id by component type */
static
DWORD
get_control_id(
	HMIXER				hMix,
	DWORD				dwComponentType)
{
	MIXERLINE			ml;
	MIXERLINECONTROLS	mlc;
	MIXERCONTROL		mctrl[4];
	int					i;

	if (hMix == INVALID_HANDLE_VALUE)
		return (DWORD)-1;

	// wR|[lg̃~LT[C𓾂 
	ml.cbStruct = sizeof(MIXERLINE);
	ml.dwComponentType = dwComponentType;
	if (mixerGetLineInfo((HMIXEROBJ)hMix, &ml, MIXER_OBJECTF_HMIXER|MIXER_GETLINEINFOF_COMPONENTTYPE) != MMSYSERR_NOERROR)
	{
		return (DWORD)-1;
	}

	for (i = 0; i < sizeof(mctrl)/sizeof(mctrl[0]); ++i)
	{
		mctrl[i].cbStruct = sizeof(MIXERCONTROL);
	}

	// ~LT[C̃Rg[𓾂 
	mlc.cbStruct		= sizeof(MIXERLINECONTROLS);
	mlc.dwLineID		= ml.dwLineID;
	mlc.dwControlType	= MIXERCONTROL_CONTROLTYPE_VOLUME;
	mlc.cControls		= ml.cControls;
	mlc.cbmxctrl		= sizeof(MIXERCONTROL);
	mlc.pamxctrl		= (LPMIXERCONTROL)&mctrl[0];

	if (mixerGetLineControls((HMIXEROBJ)hMix, &mlc, MIXER_OBJECTF_HMIXER|MIXER_GETLINECONTROLSF_ONEBYTYPE/*MIXER_GETLINECONTROLSF_ALL*/) != MMSYSERR_NOERROR)
	{
		return (DWORD)-1;
	}

	for (i = 0; i < ml.cControls; ++i)
	{
		if (mctrl[i].dwControlType == MIXERCONTROL_CONTROLTYPE_VOLUME)
		{
			return mctrl[i].dwControlID;
		}
	}

	return (DWORD)-1;
}


/* get max volume by component type */
static
Sint32
get_max_volume(
	HMIXER				hMix,
	DWORD				dwComponentType)
{
	MIXERLINE			ml;
	MIXERLINECONTROLS	mlc;
	MIXERCONTROL		mctrl[4];
	int					i;

	if (hMix == INVALID_HANDLE_VALUE)
		return -1;

	// wR|[lg̃~LT[C𓾂 
	ml.cbStruct = sizeof(MIXERLINE);
	ml.dwComponentType = dwComponentType;
	if (mixerGetLineInfo((HMIXEROBJ)hMix, &ml, MIXER_OBJECTF_HMIXER|MIXER_GETLINEINFOF_COMPONENTTYPE) != MMSYSERR_NOERROR)
	{
		return -1;
	}

	for (i = 0; i < sizeof(mctrl)/sizeof(mctrl[0]); ++i)
	{
		mctrl[i].cbStruct = sizeof(MIXERCONTROL);
	}

	// ~LT[C̃Rg[𓾂 
	mlc.cbStruct		= sizeof(MIXERLINECONTROLS);
	mlc.dwLineID		= ml.dwLineID;
	mlc.dwControlType	= MIXERCONTROL_CONTROLTYPE_VOLUME;
	mlc.cControls		= ml.cControls;
	mlc.cbmxctrl		= sizeof(MIXERCONTROL);
	mlc.pamxctrl		= (LPMIXERCONTROL)&mctrl[0];

	if (mixerGetLineControls((HMIXEROBJ)hMix, &mlc, MIXER_OBJECTF_HMIXER|MIXER_GETLINECONTROLSF_ONEBYTYPE) != MMSYSERR_NOERROR)
	{
		return -1;
	}

	for (i = 0; i < ml.cControls; ++i)
	{
		if (mctrl[i].dwControlType == MIXERCONTROL_CONTROLTYPE_VOLUME)
		{
			return (Sint32)mctrl[i].Bounds.lMaximum;
		}
	}

	return -1;
}


/*-----------------------------------------------------------------------------
	[Init]
		Mixer ܂B
		volumeXXX  0 - 65535 ̒lŎw肵܂B
-----------------------------------------------------------------------------*/
BOOL
CDFADER_Init(
	Uint32				volumeMaster,
	Uint32				volumeWave,
	Uint32				volumeCd)
{
	UINT				id;
	MIXERCAPS			mc					= {0, };

	if (_hMix != INVALID_HANDLE_VALUE)
		CDFADER_Deinit();

	id = mixerGetNumDevs();

	if (id < 1)
		return FALSE;

	// ŏ̃TEhJ[hw肷 
	id = 0;
	if (mixerOpen(&_hMix, id, (DWORD)WINMAIN_GetHwnd(), 0, MIXER_OBJECTF_MIXER|CALLBACK_WINDOW) != MMSYSERR_NOERROR)
		return FALSE;

	if (mixerGetDevCaps(id, &mc, sizeof(mc)) != MMSYSERR_NOERROR)
	{
		CDFADER_Deinit();
		return FALSE;
	}

	_dwIDControlMaster = get_control_id(_hMix, MIXERLINE_COMPONENTTYPE_DST_SPEAKERS);
	_dwIDControlCd     = get_control_id(_hMix, MIXERLINE_COMPONENTTYPE_SRC_COMPACTDISC);
	_dwIDControlWave   = get_control_id(_hMix, MIXERLINE_COMPONENTTYPE_SRC_WAVEOUT);

	/* preserve the initial state of the mixer device */
	_PreservedVolumeMaster = get_volume(_hMix, _dwIDControlMaster);
	_PreservedVolumeCd     = get_volume(_hMix, _dwIDControlCd);
	_PreservedVolumeWave   = get_volume(_hMix, _dwIDControlWave);

	if (_PreservedVolumeMaster < 0)	_PreservedVolumeMaster = 0;
	if (_PreservedVolumeCd < 0)		_PreservedVolumeCd     = 0;
	if (_PreservedVolumeWave < 0)		_PreservedVolumeWave   = 0;

	_MaxVolumeMaster = get_max_volume(_hMix, MIXERLINE_COMPONENTTYPE_DST_SPEAKERS);
	_MaxVolumeCd     = get_max_volume(_hMix, MIXERLINE_COMPONENTTYPE_SRC_COMPACTDISC);
	_MaxVolumeWave   = get_max_volume(_hMix, MIXERLINE_COMPONENTTYPE_SRC_WAVEOUT);

	/* apply initial value if valid */
	if (volumeMaster >= 0 && volumeMaster <= _MaxVolumeMaster)
	{
		double tempVol = _MaxVolumeMaster;
		tempVol *= (double)volumeMaster / 65535.0;
		set_volume(_hMix, _dwIDControlMaster, (Sint32)tempVol);
		PRINTF("CDFADER_Init: Setting the mixer master volume to %d (%d[%%])\n", (int)tempVol, (int)((tempVol/(double)_MaxVolumeMaster)*100.0+0.5));
	}

	if (volumeWave >= 0 && volumeWave <= _MaxVolumeWave)
	{
		double tempVol = _MaxVolumeWave;
		tempVol *= (double)volumeWave / 65535.0;
		set_volume(_hMix, _dwIDControlWave, (Sint32)tempVol);
		PRINTF("CDFADER_Init: Setting the mixer Wave/DirectSound volume to %d (%ld[%%])\n", (int)tempVol, (int)((tempVol/(double)_MaxVolumeWave)*100.0+0.5));
	}

	if (volumeCd >= 0 && volumeCd <= _MaxVolumeCd)
	{
		double tempVol = _MaxVolumeCd;
		tempVol *= (double)volumeCd / 65535.0;
		set_volume(_hMix, _dwIDControlCd, (Sint32)tempVol);
		_InitialCdVolume = (Sint32)tempVol;
		PRINTF("CDFADER_Init: Setting the mixer CD volume to %d (%ld[%%])\n", (int)tempVol, (int)((tempVol/(double)_MaxVolumeCd)*100.0+0.5));
	}
	else
	{
		_InitialCdVolume = _PreservedVolumeCd;
	}

	_CurrentCdVolume = _InitialCdVolume;
	_VolumeStep      = _InitialCdVolume / 100;

	if (_VolumeStep == 0)
		_VolumeStep = 1;

	_bFadeOut = FALSE;
	_bFadeIn  = FALSE;

	return TRUE;
}

void
CDFADER_SetMasterVolume(
	Uint32		volume)		/* 0 - 65535 */
{
	if (_hMix)
	{
		double tempVol = _MaxVolumeMaster;
		tempVol = tempVol * (double)volume / 65535.0;
		set_volume(_hMix, _dwIDControlMaster, (Sint32)tempVol);
		_PreservedVolumeMaster = (Sint32)tempVol;
	}
}


void
CDFADER_SetCdVolume(
	Uint32		volume)
{
	if (_hMix)
	{
		double tempVol = _MaxVolumeCd;
		tempVol = tempVol * (double)volume / 65535.0;
		set_volume(_hMix, _dwIDControlCd, (Sint32)tempVol);
		_InitialCdVolume = (Sint32)tempVol;
	}
}


void
CDFADER_SetWaveVolume(
	Uint32		volume)
{
	if (_hMix)
	{
		double tempVol = _MaxVolumeWave;
		tempVol = tempVol * (double)volume / 65535.0;
		set_volume(_hMix, _dwIDControlWave, (Sint32)tempVol);
		_PreservedVolumeWave = (Sint32)tempVol;
	}
}


Uint32
CDFADER_GetMasterVolume()
{
	if (_hMix)
	{
		double tempVol = get_volume(_hMix, _dwIDControlMaster); /*_PreservedVolumeMaster*/
		tempVol = tempVol * 65535.0 / (double)_MaxVolumeMaster;
		return (Uint32)tempVol;
	}

	return 0;
}


Uint32
CDFADER_GetCdVolume()
{
	if (_hMix)
	{
		double tempVol = get_volume(_hMix, _dwIDControlCd);
		tempVol = tempVol * 65535.0 / (double)_MaxVolumeCd;
		return (Uint32)tempVol;
	}

	return 0;
}

Uint32
CDFADER_GetWaveVolume()
{
	if (_hMix)
	{
		double tempVol = get_volume(_hMix, _dwIDControlWave); /*_PreservedVolumeWave*/
		tempVol = tempVol * 65535.0 / (double)_MaxVolumeWave;
		return (Uint32)tempVol;
	}

	return 0;
}

/*-----------------------------------------------------------------------------
	[Deinit]
		bctF[_[j܂B
-----------------------------------------------------------------------------*/
void
CDFADER_Deinit()
{
	if (_hMix)
	{
//		set_volume(_hMix, _dwIDControlMaster, _PreservedVolumeMaster);
//		set_volume(_hMix, _dwIDControlCd, _InitialCdVolume);
//		set_volume(_hMix, _dwIDControlWave, _PreservedVolumeWave);
		mixerClose(_hMix);
		_hMix = INVALID_HANDLE_VALUE;
	}
}


void
CDFADER_Synchronize(
	Uint32			mixerHandle,
	Uint32			controlID)
{
	if (controlID == _dwIDControlMaster)
	{
		_PreservedVolumeMaster = get_volume((HMIXER)mixerHandle, controlID);
//		printf("Master volume = %d\n", _PreservedVolumeMaster);
	}
	else if (controlID == _dwIDControlWave)
	{
		_PreservedVolumeWave = get_volume((HMIXER)mixerHandle, controlID);
//		printf("Wave volume = %d\n", _PreservedVolumeWave);
	}

#if 0
	else if (controlID == _dwIDControlCd)
	{
		_InitialCdVolume = get_volume((HMIXER)mixerHandle, controlID);
//		printf("cd volume = %d\n", _PreservedVolumeCd);

		/* {W[tF[ĥƂ InitialCdVolume ̂ݍXV */
		if (!_bFadeOut && !_bFadeIn)
		{
//			_InitialCdVolume = _PreservedVolumeCd;
//		}
//		else
//		{
			_bFadeOut = FALSE;
			_bFadeIn  = FALSE;
			_FadeCycle = 0;
//			_InitialCdVolume = _PreservedVolumeCd;
			_CurrentCdVolume = _InitialCdVolume;
			_VolumeStep      = _InitialCdVolume / 100;

			if (_VolumeStep == 0)
				_VolumeStep = 1;
		}
	}
#endif
}


/*
	[tF[hAEg]

	S bŃtF[hAEgƂA
	NbN C [Hz]ŗƂƁA

		S * C [cycles]

	ŊSɉʂ[ɂȂ΂悢B

	݂̉ʂ V ƂƁA

	S * C / V [cycles]

	Ƃ V PĂ䂯΂悢B

	Ȃ V PɁADꍇ

		S * C / (V / D) [cycles]

	Ƃ V B

	tF[hAEgɃ~LT[ݒ肪
	ύX\lB
*/
void
CDFADER_FadeOut(
	Sint32			ms)
{
	if (!_hMix)
		return;

	if (ms == 0)
	{
		_CurrentCdVolume = 0;
		set_volume(_hMix, _dwIDControlCd, _CurrentCdVolume);
		_bFadeOut = FALSE;
		_bFadeIn  = FALSE;
		_FadeCycle = 0;
	}
	else if (_CurrentCdVolume > 0)
	{
		_FadeCycle = ((7159090.0 / ((double)_CurrentCdVolume / (double)_VolumeStep)) * (double)ms) / 1000.0;
		_bFadeOut	= TRUE;
		_bFadeIn	= FALSE;
	}
	else
	{
		_bFadeOut = FALSE;
		_bFadeIn  = FALSE;
		_FadeCycle = 0;
	}
//	printf("fade-out ms = %d\n", ms);
//	printf("fade-out cycle = %d\n", _FadeCycle);
}


void
CDFADER_FadeIn(
	Sint32			ms)
{
	if (!_hMix)
		return;

	if (ms == 0)
	{
		_CurrentCdVolume = _InitialCdVolume;
		set_volume(_hMix, _dwIDControlCd, _CurrentCdVolume);
		_bFadeOut = FALSE;
		_bFadeIn  = FALSE;
		_FadeCycle = 0;
	}
	else if (_InitialCdVolume - _CurrentCdVolume > 0)
	{
		_FadeCycle = ((7159090.0 / (((double)_InitialCdVolume - (double)_CurrentCdVolume) / (double)_VolumeStep)) * (double)ms) / 1000.0;
		_bFadeOut = FALSE;
		_bFadeIn  = TRUE;
	}
	else
	{
		_bFadeOut = FALSE;
		_bFadeIn  = FALSE;
		_FadeCycle = 0;
	}
//	printf("fade-in ms = %d\n", ms);
//	printf("fade-in cycle = %d\n", _FadeCycle);
}


void
CDFADER_AdvanceClock(
	Sint32			cycles)	// 21477270 / 3 = 7159090 [Hz] ̎gŗB 
{
	if (_bFadeOut || _bFadeIn)
	{
		_ClockCount += cycles;

		while (_ClockCount >= _FadeCycle)
		{
			_ClockCount -= _FadeCycle;

			if (_bFadeOut)
			{
				if (_CurrentCdVolume > 0)
				{
					_CurrentCdVolume -= _VolumeStep;
					if (_CurrentCdVolume < 0)
					{
						_CurrentCdVolume = 0;
						_bFadeOut = FALSE;
					}
					set_volume(_hMix, _dwIDControlCd, _CurrentCdVolume);
				}
			}
			else if (_bFadeIn)
			{
				if (_CurrentCdVolume < _InitialCdVolume)
				{
					_CurrentCdVolume += _VolumeStep;
					if (_CurrentCdVolume > _InitialCdVolume)
					{
						_CurrentCdVolume = _InitialCdVolume;
						_bFadeIn = FALSE;
					}
					set_volume(_hMix, _dwIDControlCd, _CurrentCdVolume);
				}
			}
		}
	}
}

