/*-----------------------------------------------------------------------------
	[App.c]
		AvP[V̓Lq܂B

		Implement the application's behaviour.

	Copyright (C) 2004 Ki

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
-----------------------------------------------------------------------------*/
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

#include "AppInterface.h"
#include "MainBoard.h"
#include "ScreenInterface.h"
#include "InputInterface.h"
#include "CdFader.h"
#include "Config.h"
#include "WinMain.h"
#include "AppEvent.h"
#include "CriticalSection.h"
#include "Menu.h"
#include "Printf.h"
#include "PadConfig.h"
#include "AudioConfig.h"
#include "Mouse.h"
#include "MB128.h"


#ifdef DEBUGBUILD
	#include "DisasmWindow.h"
#endif


static char		_GameFilePathName[MAX_PATH];
static char		_AppDir[MAX_PATH];


// [APP DISPLAY] (default values)
static Sint32	_ScreenWidth			= 256;
static Sint32	_ScreenHeight			= 224;
static Sint32	_Magnification			= 1;
static float	_FPS					= 60.0;
static BOOL		_bFullScreen			= FALSE;
static BOOL		_bScanLine				= FALSE;
static BOOL		_bStretched				= FALSE;
static BOOL		_bSyncTo60HzFullscreen	= FALSE;
static BOOL		_bHardwareAcceleration	= FALSE;


// [APP SOUND] (default values)
static Sint32	_SoundBufferSize		= 2560;
static Sint32	_SoundSampleRate		= 44100;
static Sint32	_SoundOversampleRate	= 8;		/* currently not used (fixed to 8) */
static Uint32	_SoundApuMasterVolume	= 65535;	/* 0 - 65535 */
static Uint32	_SoundAdpcmVolume		= 65535;
static Uint32	_SoundMixerVolumeMaster = 65535/2;
static Uint32	_SoundMixerVolumeCd 	= 65535*3/4;
static Uint32	_SoundMixerVolumeWave	= 65535*3/4;


// [APP INPUT] (default values)
static JoyPad	_JoyPad[5] = 
{
	{
		INPUT_JOYSTICK_BUTTON2, INPUT_JOYSTICK_BUTTON1, INPUT_JOYSTICK_BUTTON5, INPUT_JOYSTICK_BUTTON6,
		INPUT_JOYSTICK_BUTTON7, INPUT_JOYSTICK_BUTTON8, INPUT_JOYSTICK_BUTTON3, INPUT_JOYSTICK_BUTTON4,
		INPUT_JOYSTICK_UP,		INPUT_JOYSTICK_RIGHT,	INPUT_JOYSTICK_DOWN,	INPUT_JOYSTICK_LEFT
	},

	{
		INPUT_JOYSTICK_BUTTON2, INPUT_JOYSTICK_BUTTON1, INPUT_JOYSTICK_BUTTON5, INPUT_JOYSTICK_BUTTON6,
		INPUT_JOYSTICK_BUTTON7, INPUT_JOYSTICK_BUTTON8, INPUT_JOYSTICK_BUTTON3, INPUT_JOYSTICK_BUTTON4,
		INPUT_JOYSTICK_UP,		INPUT_JOYSTICK_RIGHT,	INPUT_JOYSTICK_DOWN,	INPUT_JOYSTICK_LEFT
	},

	{
		INPUT_JOYSTICK_BUTTON2, INPUT_JOYSTICK_BUTTON1, INPUT_JOYSTICK_BUTTON5, INPUT_JOYSTICK_BUTTON6,
		INPUT_JOYSTICK_BUTTON7, INPUT_JOYSTICK_BUTTON8, INPUT_JOYSTICK_BUTTON3, INPUT_JOYSTICK_BUTTON4,
		INPUT_JOYSTICK_UP,		INPUT_JOYSTICK_RIGHT,	INPUT_JOYSTICK_DOWN,	INPUT_JOYSTICK_LEFT
	},

	{
		INPUT_JOYSTICK_BUTTON2, INPUT_JOYSTICK_BUTTON1, INPUT_JOYSTICK_BUTTON5, INPUT_JOYSTICK_BUTTON6,
		INPUT_JOYSTICK_BUTTON7, INPUT_JOYSTICK_BUTTON8, INPUT_JOYSTICK_BUTTON3, INPUT_JOYSTICK_BUTTON4,
		INPUT_JOYSTICK_UP,		INPUT_JOYSTICK_RIGHT,	INPUT_JOYSTICK_DOWN,	INPUT_JOYSTICK_LEFT
	},

	{
		INPUT_JOYSTICK_BUTTON2, INPUT_JOYSTICK_BUTTON1, INPUT_JOYSTICK_BUTTON5, INPUT_JOYSTICK_BUTTON6,
		INPUT_JOYSTICK_BUTTON7, INPUT_JOYSTICK_BUTTON8, INPUT_JOYSTICK_BUTTON3, INPUT_JOYSTICK_BUTTON4,
		INPUT_JOYSTICK_UP,		INPUT_JOYSTICK_RIGHT,	INPUT_JOYSTICK_DOWN,	INPUT_JOYSTICK_LEFT
	}
};


// [APP INPUT] (Keyboard)
/*
static Sint32	_JoyPadKey_I			= INPUT_KEYBOARD_F;
static Sint32	_JoyPadKey_II			= INPUT_KEYBOARD_D;
static Sint32	_JoyPadKey_SELECT		= INPUT_KEYBOARD_TAB;
static Sint32	_JoyPadKey_RUN			= INPUT_KEYBOARD_ENTER;
static Sint32	_JoyPadKey_UP			= INPUT_KEYBOARD_UP;
static Sint32	_JoyPadKey_RIGHT		= INPUT_KEYBOARD_RIGHT;
static Sint32	_JoyPadKey_DOWN			= INPUT_KEYBOARD_DOWN;
static Sint32	_JoyPadKey_LEFT			= INPUT_KEYBOARD_LEFT;
*/



static FILE*	_fpRecord;
static FILE*	_fpPlayback;


static HANDLE	_Menu;
static HANDLE	_FileMenu;
static HANDLE	_ConfigMenu;
static HANDLE	_ScreenMenu;
static HANDLE	_LogMenu;
static HANDLE	_CpuMenu;
static HANDLE	_InputMenu;
static HANDLE	_AudioMenu;
static HANDLE	_SampleRateMenu;
static HANDLE	_SoundBufMenu;

static BOOL		_bRunning = TRUE;


/*----------------------------------------------------------------------------
	t@C_CAO\AJt@C̐΃pX擾B
----------------------------------------------------------------------------*/
static
BOOL
file_dialog(
	HWND		hWnd,
	char*		filePathName)		// t@C̃pXƖO󂯎zւ̃|C^
{
	OPENFILENAME ofn;
	char fileName[MAX_PATH] = "*.*";
	BOOL ret;

	if (strlen(filePathName) > 0)
	{
		strcpy(fileName, filePathName);
		if (strrchr(fileName, '\\') != NULL)
		{
			strcat(strrchr(fileName, '\\')+1, "*.*");
		}
	}

	ZeroMemory(&ofn, sizeof(OPENFILENAME));

	/* OPENFILENAME \ ofn p[^ݒ肷 */
	ofn.lStructSize 	  = sizeof(OPENFILENAME);
	ofn.hwndOwner		  = hWnd;
	ofn.hInstance		  = (HINSTANCE)GetWindowLong(hWnd, GWL_HINSTANCE);
	ofn.lpstrFilter 	  = "Plain ROM images";
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter	  = 0;
	ofn.nFilterIndex	  = 0;
	ofn.lpstrFile		  = fileName;
	ofn.nMaxFile		  = sizeof(fileName) - 1;
	ofn.lpstrFileTitle	  = NULL;
	ofn.nMaxFileTitle	  = 0;
	ofn.lpstrInitialDir   = filePathName;
	ofn.lpstrTitle		  = "Open";
	ofn.nFileOffset 	  = 0;
	ofn.nFileExtension	  = 0;
	ofn.lpstrDefExt 	  = (LPSTR)NULL;
	ofn.lCustData		  = 0;
	ofn.lpfnHook 		  = (LPOFNHOOKPROC)NULL;
	ofn.lpTemplateName	  = (LPSTR)NULL;

	ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;
	ret = GetOpenFileName(&ofn);

	if (ret == TRUE)
	{
		strcpy(filePathName, fileName);
		return TRUE;
	}
	return FALSE;
}


/*----------------------------------------------------------------------------
	[run_emulator]
----------------------------------------------------------------------------*/
static
void
run_emulator(
	BOOL		bRun)
{
	_bRunning = bRun;

	if (_bRunning)
	{
//		MENU_ChangeItemText(_CpuMenu, WM_RUN_EMULATOR, "Pause\tF2");
		MENU_Show(NULL);

		if (_bFullScreen)
			WINMAIN_SetFullScreenWindow(SCREEN_GetWidth(), SCREEN_GetHeight());
		else
			WINMAIN_SetNormalWindow(SCREEN_GetWidth(), SCREEN_GetHeight());

		if (MOUSE_IsConnected() || SCREEN_IsFullScreen())
		{
			WINMAIN_ShowCursor(FALSE);
		}
		MAINBOARD_Pause(FALSE);
	}
	else
	{
//		MENU_ChangeItemText(_CpuMenu, WM_RUN_EMULATOR, "Run\tF2");
		MENU_Show(_Menu);

		if (_bFullScreen)
			WINMAIN_SetFullScreenWindow(SCREEN_GetWidth(), SCREEN_GetHeight());
		else
			WINMAIN_SetNormalWindow(SCREEN_GetWidth(), SCREEN_GetHeight());

		WINMAIN_ShowCursor(TRUE);
		MAINBOARD_Pause(TRUE);
	}
}


static
void
set_audio_sample_rate(
	Sint32		sampleRate)
{
	if (APU_SetSampleRate(sampleRate))
	{
		_SoundSampleRate = sampleRate;
		CONFIG_Set("[APP SOUND] Sound Sample Rate", &_SoundSampleRate, sizeof(_SoundSampleRate));
	}

	switch (sampleRate)
	{
		case 11025: MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR96000, WM_AUDIO_SR11025); break;
		case 22050: MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR96000, WM_AUDIO_SR22050); break;
		case 32000: MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR96000, WM_AUDIO_SR32000); break;
		case 44100: MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR96000, WM_AUDIO_SR44100); break;
		case 48000: MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR96000, WM_AUDIO_SR48000); break;
		case 64000: MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR96000, WM_AUDIO_SR64000); break;
		case 88200: MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR96000, WM_AUDIO_SR88200); break;
		case 96000: MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR96000, WM_AUDIO_SR96000); break;
	}
}


static
void
set_audio_buffer_size(
	Sint32		bufSize)
{
	if (APU_SetBufferSize(bufSize))
	{
		_SoundBufferSize = bufSize;
		CONFIG_Set("[APP SOUND] Sound Buffer Size", &_SoundBufferSize, sizeof(_SoundBufferSize));
	}

	switch (bufSize)
	{
		case 2048: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB2048); break;
		case 2304: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB2304); break;
		case 2560: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB2560); break;
		case 2816: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB2816); break;
		case 3072: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB3072); break;
		case 3328: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB3328); break;
		case 3584: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB3584); break;
		case 3840: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB3840); break;
		case 4096: MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB4096); break;
	}
}


/*----------------------------------------------------------------------------
	[save_config]
----------------------------------------------------------------------------*/
static
BOOL
save_config(
	const char*		pAppDir)
{
	char	filePathName[MAX_PATH];

	sprintf(filePathName, "%s" APP_CONFIG_FILENAME, pAppDir);
	return CONFIG_Save(filePathName);
}


/*----------------------------------------------------------------------------
	[load_config]
----------------------------------------------------------------------------*/
static
BOOL
load_config(
	const char*		pAppDir)
{
	char	filePathName[MAX_PATH];

	sprintf(filePathName, "%s" APP_CONFIG_FILENAME, pAppDir);
	return CONFIG_Load(filePathName);
}


/*----------------------------------------------------------------------------
	[save_bram]
----------------------------------------------------------------------------*/
static
BOOL
save_bram(
	const char*		pAppDir)
{
	char	filePathName[MAX_PATH];
	sprintf(filePathName, "%sBRAM.DAT", pAppDir);
	return MAINBOARD_SaveBRAM(filePathName);
}


/*----------------------------------------------------------------------------
	[load_bram]
----------------------------------------------------------------------------*/
static
BOOL
load_bram(
	const char*		pAppDir)
{
	char	filePathName[MAX_PATH];
	sprintf(filePathName, "%sBRAM.DAT", pAppDir);
	return MAINBOARD_LoadBRAM(filePathName);
}


/*----------------------------------------------------------------------------
	[set_mb128_path]
		MB128 t@C̕ۑw肵܂B
----------------------------------------------------------------------------*/
/*
static
void
set_mb128_path(
	const char*		pAppDir)
{
	char	filePathName[MAX_PATH];
	sprintf(filePathName, "%s"MB128_FILENAME, pAppDir);
	MB128_SetPath(filePathName);
}
*/

/*----------------------------------------------------------------------------
	[end_recording]
		End recording in progress.
----------------------------------------------------------------------------*/
static
void
end_recording()
{
	/* L^ꍇ͏I */
	if (_fpRecord != NULL)
	{
		PRINTF("APP: End recording.");
		INPUT_Record(FALSE);
		INPUT_WriteBuffer(_fpRecord);
		fclose(_fpRecord);
		_fpRecord = NULL;
		MENU_ChangeItemText(_FileMenu, WM_RECORD_GAMEPLAY, "Record\tF6");
	}
}


/*----------------------------------------------------------------------------
	[end_playback]
		End playback in progress.
----------------------------------------------------------------------------*/
static
void
end_playback()
{
	/* Đꍇ͏I */
	if (_fpPlayback != NULL)
	{
		PRINTF("APP: End playback.");
		INPUT_Playback(FALSE);
		fclose(_fpPlayback);
		_fpPlayback = NULL;
		MENU_ChangeItemText(_FileMenu, WM_PLAYBACK_GAMEPLAY, "Playback\tF8");
	}
}


/*----------------------------------------------------------------------------
	[save_state]
----------------------------------------------------------------------------*/
static
BOOL
save_state(
	const char*		pAppDir,
	const char*		pGameFilePathName)
{
	char	buf[MAX_PATH];
	FILE*	fp;

	// st@C̃fBNgRs[ 
	strcpy(buf, _AppDir);

	// Q[t@CRs[ 
	if (strrchr(pGameFilePathName, '\\') != NULL)
		strcat(buf, strrchr(pGameFilePathName, '\\')+1);

	//  _STATE.DAT ǉ
	strcat(buf, "_STATE.DAT");

	fp = fopen(buf, "wb");
	if (fp != NULL)
	{
		MAINBOARD_SaveState(fp);
		fclose(fp);
		PRINTF("APP: State saved to \"%s\".", buf);
		return TRUE;
	}
	PRINTF("APP: Could not save state to \"%s\".\n", buf);
	return FALSE;
}


/*----------------------------------------------------------------------------
	[load_state]
----------------------------------------------------------------------------*/
static
BOOL
load_state(
	const char*		pAppDir,
	const char*		pGameFilePathName)
{
	char	buf[MAX_PATH];
	FILE*	fp;

	/* L^EĐꍇ͏I */
	end_recording();
	end_playback();

	strcpy(buf, _AppDir);
	if (strrchr(pGameFilePathName, '\\') != NULL)
		strcat(buf, strrchr(pGameFilePathName, '\\')+1);
	strcat(buf, "_STATE.DAT");

	fp = fopen(buf, "rb");
	if (fp != NULL)
	{
		MAINBOARD_LoadState(fp);
		fclose(fp);
		PRINTF("APP: State loaded from \"%s\".", buf);
		return TRUE;
	}

	PRINTF("APP: Could not open state from \"%s\".", buf);
	return FALSE;
}


/*----------------------------------------------------------------------------
	[record_gameplay]
----------------------------------------------------------------------------*/
static
BOOL
record_gameplay(
	const char*		pAppDir,
	const char*		pGameFilePathName)
{
	if (_fpRecord == NULL)		// L^Jn 
	{
		char	buf[MAX_PATH];

		/* Đꍇ͏I */
		end_playback();

		strcpy(buf, _AppDir);
		if (strrchr(pGameFilePathName, '\\') != NULL)
			strcat(buf, strrchr(pGameFilePathName, '\\')+1);
		strcat(buf, "_RECORD.DAT");

		_fpRecord = fopen(buf, "wb");
		if (_fpRecord != NULL)
		{
			PRINTF("APP: Start recording game play to \"%s\".", buf);
			MAINBOARD_SaveState(_fpRecord);
			INPUT_Record(TRUE);

			MENU_ChangeItemText(_FileMenu, WM_RECORD_GAMEPLAY, "End recording\tF6");

			return TRUE;
		}
		PRINTF("APP: Failed to open \"%s\" for recording game play.", buf);
	}
	else	// L^I 
	{
		end_recording();
		return TRUE;
	}

	return FALSE;
}


/*----------------------------------------------------------------------------
	[playback_gameplay]
----------------------------------------------------------------------------*/
static
BOOL
playback_gameplay(
	const char*		pAppDir,
	const char*		pGameFilePathName)
{
	if (_fpPlayback == NULL)		// ĐJn 
	{
		char	buf[MAX_PATH];

		/* L^ꍇ͏I */
		end_recording();

		strcpy(buf, _AppDir);
		if (strrchr(pGameFilePathName, '\\') != NULL)
			strcat(buf, strrchr(pGameFilePathName, '\\')+1);
		strcat(buf, "_RECORD.DAT");

		_fpPlayback = fopen(buf, "rb");
		if (_fpPlayback != NULL)
		{
			PRINTF("APP: Start playback from file \"%s\".", buf);
			MAINBOARD_LoadState(_fpPlayback);
			INPUT_ReadBuffer(_fpPlayback);
			INPUT_Playback(TRUE);

			MENU_ChangeItemText(_FileMenu, WM_PLAYBACK_GAMEPLAY, "End playback\tF8");

			return TRUE;
		}
		PRINTF("APP: Can't open record file \"%s\".", buf);
	}
	else	/* ĐI (ʏ͂ɖ߂) */
	{
		end_playback();
		return TRUE;
	}

	return FALSE;
}


/*  */
static
BOOL
change_screen_mode(
	Sint32		width,
	Sint32		height,
	Sint32		magnification,
	BOOL		bFullScreen,
	BOOL		bScanLine,
	BOOL		bStretched,
	BOOL		bSync60HzScreen,
	BOOL		bHardwareAcceleration)
{
	if (bFullScreen || bStretched)
	{
		switch (magnification)
		{
			case 1:		width =  320; height = 240; break;
			case 2:		width =  640; height = 480; break;
			case 3:		width =  800; height = 600; break;
			case 4:		width = 1024; height = 768; break;
		}
	}
	else
	{
		width  *= magnification;
		height *= magnification;
	}

	MENU_CheckItem(_ScreenMenu, WM_SCREEN_STRETCHED,   _bStretched);
	MENU_CheckItem(_ScreenMenu, WM_SCREEN_SCANLINED,   _bScanLine);
	MENU_CheckItem(_ScreenMenu, WM_SCREEN_HWACCEL,	   _bHardwareAcceleration);
	MENU_CheckItem(_ScreenMenu, WM_SCREEN_SYNC_VBLANK, _bSyncTo60HzFullscreen);
	MENU_CheckItem(_ScreenMenu, WM_SCREEN_FULLSCREEN,  _bFullScreen);
	MENU_CheckRadioItem(_ScreenMenu, WM_SCREEN_X1, WM_SCREEN_X4, WM_SCREEN_X1+magnification-1);

	// disable input configuration menu items on fullscreen modes
	MENU_EnableItem(_InputMenu, WM_INPUT_CONFIGURE_PAD1, bFullScreen == FALSE);
	MENU_EnableItem(_InputMenu, WM_INPUT_CONFIGURE_PAD2, bFullScreen == FALSE);
	MENU_EnableItem(_InputMenu, WM_INPUT_CONFIGURE_PAD3, bFullScreen == FALSE);
	MENU_EnableItem(_InputMenu, WM_INPUT_CONFIGURE_PAD4, bFullScreen == FALSE);
	MENU_EnableItem(_InputMenu, WM_INPUT_CONFIGURE_PAD5, bFullScreen == FALSE);

	// disable audio volume configuration menu item on fullscreen modes
	MENU_EnableItem(_AudioMenu, WM_AUDIO_SETVOLUME, bFullScreen == FALSE);

	// change window style 
	// ɃX^CύXĂȂƁA
	// tXN[[hɂ SetCoorperativeLevel() 
	// DDSCL_EXCLUSIVE w肵uԈ؂̃bZ[W~܂Ă܂B
	if (bFullScreen)
		WINMAIN_SetFullScreenWindow(width, height);
	else
		WINMAIN_SetNormalWindow(width, height);

	//  
	if (MAINBOARD_ChangeScreenMode(width, height, magnification, 16, bFullScreen, bScanLine, bStretched, bSync60HzScreen, bHardwareAcceleration))
	{
		CONFIG_Set("[APP DISPLAY] Full Screen Mode", &_bFullScreen, sizeof(_bFullScreen));
		CONFIG_Set("[APP DISPLAY] Scanline Mode", &_bScanLine, sizeof(_bScanLine));
		CONFIG_Set("[APP DISPLAY] Stretched Mode", &_bStretched, sizeof(_bStretched));
		CONFIG_Set("[APP DISPLAY] Magnification", &magnification, sizeof(magnification));
		CONFIG_Set("[APP DISPLAY] Sync To 60Hz Fullscreen", &_bSyncTo60HzFullscreen, sizeof(_bSyncTo60HzFullscreen));
		CONFIG_Set("[APP DISPLAY] Use Hardware Acceleration", &_bHardwareAcceleration, sizeof(_bHardwareAcceleration));

		_Magnification			= magnification;
		_bFullScreen			= bFullScreen;
		_bScanLine				= bScanLine;
		_bStretched				= bStretched;
		_bSyncTo60HzFullscreen	= bSync60HzScreen;
		_bHardwareAcceleration	= bHardwareAcceleration;
	}
	else
	{
		PRINTF("APP: Change screen mode failed; recovering the default screen mode.");

		_ScreenWidth			= 256;
		_ScreenHeight			= 224;
		_Magnification			= 1;
		_bFullScreen			= FALSE;
		_bScanLine				= FALSE;
		_bStretched				= FALSE;
		_bSyncTo60HzFullscreen	= FALSE;
		_bHardwareAcceleration	= FALSE;

		WINMAIN_SetNormalWindow(_ScreenWidth, _ScreenHeight);
		if (!MAINBOARD_ChangeScreenMode(_ScreenWidth, _ScreenHeight, _Magnification, 16, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration))
		{
			// ł_炠߂B 
			PRINTF("APP: Error; can't set the default screen mode.");
			return FALSE;
		}
		CONFIG_Set("[APP DISPLAY] Screen Width", &_ScreenWidth, sizeof(_ScreenWidth));
		CONFIG_Set("[APP DISPLAY] Screen Height", &_ScreenHeight, sizeof(_ScreenHeight));
		CONFIG_Set("[APP DISPLAY] Full Screen Mode", &_bFullScreen, sizeof(_bFullScreen));
		CONFIG_Set("[APP DISPLAY] Scanline Mode", &_bScanLine, sizeof(_bScanLine));
		CONFIG_Set("[APP DISPLAY] Stretched Mode", &_bStretched, sizeof(_bStretched));
		CONFIG_Set("[APP DISPLAY] Magnification", &_Magnification, sizeof(_Magnification));
		CONFIG_Set("[APP DISPLAY] Sync To 60Hz Fullscreen", &_bSyncTo60HzFullscreen, sizeof(_bSyncTo60HzFullscreen));
		CONFIG_Set("[APP DISPLAY] Use Hardware Acceleration", &_bHardwareAcceleration, sizeof(_bHardwareAcceleration));
	}

	return TRUE;
}


/*----------------------------------------------------------------------------
	[toggle_fullscreen]
----------------------------------------------------------------------------*/
static
BOOL
toggle_fullscreen()
{
#ifndef DEBUGBUILD
	_bFullScreen ^= 1;
	CONFIG_Set("[APP DISPLAY] Full Screen Mode", &_bFullScreen, sizeof(_bFullScreen));

	if (_bFullScreen)
	{
		_ScreenWidth  = 320;
		_ScreenHeight = 240;
	}
	else
	{
		_ScreenWidth  = VDC_GetDisplayWidth();
		_ScreenHeight = VDC_GetDisplayHeight();

		if (_ScreenWidth < 256)	_ScreenWidth = 256;
		_ScreenHeight = 224;

		if (_ScreenWidth >= 512)	// hires mode
			_ScreenWidth /= 2;
	}

	return change_screen_mode(_ScreenWidth, _ScreenHeight, _Magnification, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
#else
	PRINTF("DEBUGBUILD version does not support fullscreen modes.");
	return FALSE;
#endif
}


/*----------------------------------------------------------------------------
	[load_memory_values]
----------------------------------------------------------------------------*/
static
BOOL
load_memory_values(
	const char*		pAppDir,
	const char*		pGameFilePathName)
{
	FILE*	p;
	char	buf[MAX_PATH];
	char	linebuf[256];
	char	value[256];
	char*	c;
	Uint32	addr;
	Uint32	data;

	strcpy(buf, _AppDir);
	if (strrchr(pGameFilePathName, '\\') != NULL)
		strcat(buf, strrchr(pGameFilePathName, '\\')+1);
	strcat(buf, "_PM.TXT");

	p = fopen(buf, "rt");
	if (p == NULL)
	{
		PRINTF("APP: Failed loading partial memory data from file \"%s\".", buf);
		return FALSE;
	}
		
	while (fgets(linebuf, sizeof(linebuf), p) != NULL)
	{
		if (strstr(linebuf, "MAINRAM") == NULL && 
			strstr(linebuf, "BUFFERRAM") == NULL &&
			strstr(linebuf, "ARCADERAM") == NULL) continue;

		c = strchr(linebuf, ' ');
		if (c == NULL)	continue;
		while (*c == ' ') ++c;

		sprintf(value, "0x%s", c);
		sscanf(value, "%lx %lx", &addr, &data);

		if (strstr(linebuf, "MAINRAM"))
		{
			MAINBOARD_ChangeMemoryValue(MAINBOARD_MAINRAM, addr, (Uint8)data);
			PRINTF("APP: $%02lX is written to offset $%04lX of MAIN RAM.", data, addr);
		}
		else if (strstr(linebuf, "BUFFERRAM"))
		{
			MAINBOARD_ChangeMemoryValue(MAINBOARD_BUFFERRAM, addr, (Uint8)data);
			PRINTF("APP: $%02lX is written to offset $%05lX of BUFFER RAM.", data, addr);
		}
		else if (strstr(linebuf, "ARCADERAM"))
		{
			MAINBOARD_ChangeMemoryValue(MAINBOARD_BUFFERRAM, addr, (Uint8)data);
			PRINTF("APP: $%02lX is written to offset $%06lX of ARCADE RAM.", data, addr);
		}
	}

	fclose(p);
	return TRUE;
}


static
void
set_input_as_configured()
{
	int		i;

	for (i = 0; i < 5; ++i)
	{
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_I,		_JoyPad[i].i);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_II,		_JoyPad[i].ii);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_III,		_JoyPad[i].iii);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_IV,		_JoyPad[i].iv);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_V,		_JoyPad[i].v);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_VI,		_JoyPad[i].vi);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_SELECT,	_JoyPad[i].select);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_RUN,		_JoyPad[i].run);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_UP,		_JoyPad[i].up);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_RIGHT,	_JoyPad[i].right);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_DOWN,		_JoyPad[i].down);
		MAINBOARD_ChangeButtonConnection(i, JOYPAD_BUTTON_LEFT,		_JoyPad[i].left);
	}
}


static
BOOL
init_menu()
{
	_Menu			= NULL;
	_FileMenu		= NULL;
	_ConfigMenu		= NULL;
	_ScreenMenu		= NULL;
	_LogMenu		= NULL;
	_CpuMenu		= NULL;
	_InputMenu		= NULL;
	_AudioMenu		= NULL;
	_SampleRateMenu	= NULL;
	_SoundBufMenu	= NULL;

	_Menu			= MENU_Init();
	_FileMenu		= MENU_CreateSubMenu();
	_ConfigMenu		= MENU_CreateSubMenu();
	_CpuMenu		= MENU_CreateSubMenu();
	_ScreenMenu		= MENU_CreateSubMenu();
	_InputMenu		= MENU_CreateSubMenu();
	_AudioMenu		= MENU_CreateSubMenu();
	_SampleRateMenu	= MENU_CreateSubMenu();
	_SoundBufMenu	= MENU_CreateSubMenu();

	// add sub menu
	MENU_AddItem(_Menu, _ConfigMenu, "&Settings", 0);
	MENU_AddItem(_Menu, _CpuMenu,  "&CPU", 0);
	MENU_AddItem(_Menu, _FileMenu,	 "&File", 0);

	// add sub^2 menu
	MENU_AddItem(_ConfigMenu, _AudioMenu,  "&Audio", 0);
	MENU_AddItem(_ConfigMenu, _InputMenu,  "&Input", 0);
	MENU_AddItem(_ConfigMenu, _ScreenMenu, "&Screen", 0);

	// add sub^3 menu
	MENU_AddItem(_AudioMenu, NULL, "&Volume Settings...", WM_AUDIO_SETVOLUME);
	MENU_AddItem(_AudioMenu, NULL, NULL, 0);	// separator 
	MENU_AddItem(_AudioMenu, _SampleRateMenu, "&Sample Rate", 0);
	MENU_AddItem(_AudioMenu, _SoundBufMenu,   "Sound &Buffer", 0);


#ifdef DEBUGBUILD
	_LogMenu		= MENU_CreateSubMenu();

	MENU_AddItem(_CpuMenu, NULL, "Advance 10000 instructions", WM_TRACE_10000_STEP);
	MENU_AddItem(_CpuMenu, NULL, "Advance  1000 instructions", WM_TRACE_1000_STEP);
	MENU_AddItem(_CpuMenu, NULL, "Advance   100 instructions", WM_TRACE_100_STEP);
	MENU_AddItem(_CpuMenu, NULL, "Advance    10 instructions", WM_TRACE_10_STEP);
	MENU_AddItem(_CpuMenu, NULL, "Advance     1 instruction", WM_TRACE_1_STEP);
	MENU_AddItem(_CpuMenu, NULL, NULL, 0);	// separator 

	MENU_AddItem(_LogMenu, NULL, "BIOS", WM_CDROM_LOG);
	MENU_AddItem(_LogMenu, NULL, "Hardware", WM_HDWR_LOG);

	MENU_CheckRadioItem(_LogMenu, WM_CDROM_LOG, WM_HDWR_LOG, WM_HDWR_LOG);

	MENU_AddItem(_CpuMenu, _LogMenu, "Log-message Modes", 0);
	MENU_AddItem(_CpuMenu, NULL, NULL, 0);	// separator 
#endif

	/* 88200[Hz] and 96000[Hz] are not supported by the APU */
//	MENU_AddItem(_SampleRateMenu, NULL, "96000 [Hz]", WM_AUDIO_SR96000);
//	MENU_AddItem(_SampleRateMenu, NULL, "88200 [Hz]", WM_AUDIO_SR88200);
	MENU_AddItem(_SampleRateMenu, NULL, "64000 [Hz]", WM_AUDIO_SR64000);
	MENU_AddItem(_SampleRateMenu, NULL, "48000 [Hz]", WM_AUDIO_SR48000);
	MENU_AddItem(_SampleRateMenu, NULL, "44100 [Hz]", WM_AUDIO_SR44100);
	MENU_AddItem(_SampleRateMenu, NULL, "32000 [Hz]", WM_AUDIO_SR32000);
	MENU_AddItem(_SampleRateMenu, NULL, "22050 [Hz]", WM_AUDIO_SR22050);
	MENU_AddItem(_SampleRateMenu, NULL, "11025 [Hz]", WM_AUDIO_SR11025);

	MENU_CheckRadioItem(_SampleRateMenu, WM_AUDIO_SR11025, WM_AUDIO_SR48000, WM_AUDIO_SR44100);

	MENU_AddItem(_SoundBufMenu, NULL, "4096 [sample]", WM_AUDIO_SB4096);
	MENU_AddItem(_SoundBufMenu, NULL, "3840 [sample]", WM_AUDIO_SB3840);
	MENU_AddItem(_SoundBufMenu, NULL, "3584 [sample]", WM_AUDIO_SB3584);
	MENU_AddItem(_SoundBufMenu, NULL, "3328 [sample]", WM_AUDIO_SB3328);
	MENU_AddItem(_SoundBufMenu, NULL, "3072 [sample]", WM_AUDIO_SB3072);
	MENU_AddItem(_SoundBufMenu, NULL, "2816 [sample]", WM_AUDIO_SB2816);
	MENU_AddItem(_SoundBufMenu, NULL, "2560 [sample]", WM_AUDIO_SB2560);
	MENU_AddItem(_SoundBufMenu, NULL, "2304 [sample]", WM_AUDIO_SB2304);
	MENU_AddItem(_SoundBufMenu, NULL, "2048 [sample]", WM_AUDIO_SB2048);

	MENU_CheckRadioItem(_SoundBufMenu, WM_AUDIO_SB2048, WM_AUDIO_SB4096, WM_AUDIO_SB2560);

	MENU_AddItem(_CpuMenu, NULL, "Advance 100 [frames]", WM_TRACE_100_FRAME);
	MENU_AddItem(_CpuMenu, NULL, "Advance  10 [frames]", WM_TRACE_10_FRAME);
	MENU_AddItem(_CpuMenu, NULL, "Advance   1 [frame]", WM_TRACE_1_FRAME);
	MENU_AddItem(_CpuMenu, NULL, NULL, 0);	// separator 
	MENU_AddItem(_CpuMenu, NULL, "Run\tF2", WM_RUN_EMULATOR);
	MENU_AddItem(_CpuMenu, NULL, "Reset\tF1", WM_RESET_EMULATOR);

	MENU_AddItem(_ScreenMenu, NULL, "Sync to v-blank (60Hz fullscreen)", WM_SCREEN_SYNC_VBLANK);
	MENU_AddItem(_ScreenMenu, NULL, "Use hardware-acceleration (fullscreen)", WM_SCREEN_HWACCEL);
	MENU_AddItem(_ScreenMenu, NULL, NULL, 0);	// separator 
	MENU_AddItem(_ScreenMenu, NULL, "Scanlined", WM_SCREEN_SCANLINED);
	MENU_AddItem(_ScreenMenu, NULL, "Stretched", WM_SCREEN_STRETCHED);
	MENU_AddItem(_ScreenMenu, NULL, "Fullscreen\tF12", WM_SCREEN_FULLSCREEN);
	MENU_AddItem(_ScreenMenu, NULL, NULL, 0);	// separator 
	MENU_AddItem(_ScreenMenu, NULL, "x4", WM_SCREEN_X4);
	MENU_AddItem(_ScreenMenu, NULL, "x3", WM_SCREEN_X3);
	MENU_AddItem(_ScreenMenu, NULL, "x2", WM_SCREEN_X2);
	MENU_AddItem(_ScreenMenu, NULL, "x1", WM_SCREEN_X1);

	MENU_AddItem(_InputMenu, NULL, "Configure pad #5...", WM_INPUT_CONFIGURE_PAD5);
	MENU_AddItem(_InputMenu, NULL, "Configure pad #4...", WM_INPUT_CONFIGURE_PAD4);
	MENU_AddItem(_InputMenu, NULL, "Configure pad #3...", WM_INPUT_CONFIGURE_PAD3);
	MENU_AddItem(_InputMenu, NULL, "Configure pad #2...", WM_INPUT_CONFIGURE_PAD2);
	MENU_AddItem(_InputMenu, NULL, "Configure pad #1...", WM_INPUT_CONFIGURE_PAD1);
	MENU_AddItem(_InputMenu, NULL, NULL, 0);	// separator 
	MENU_AddItem(_InputMenu, NULL, "MB128", WM_INPUT_MB128);
	MENU_AddItem(_InputMenu, NULL, "Multi-tap", WM_INPUT_MULTI_TAP);
	MENU_AddItem(_InputMenu, NULL, NULL, 0);	// separator 
	MENU_AddItem(_InputMenu, NULL, "Mouse", WM_INPUT_MOUSE);
	MENU_AddItem(_InputMenu, NULL, "6-button pad", WM_INPUT_SIX_BUTTON_PAD);
	MENU_AddItem(_InputMenu, NULL, "2-button pad", WM_INPUT_TWO_BUTTON_PAD);

	MENU_CheckRadioItem(_InputMenu, WM_INPUT_TWO_BUTTON_PAD, WM_INPUT_MOUSE, WM_INPUT_TWO_BUTTON_PAD);
	MENU_CheckItem(_InputMenu, WM_INPUT_MB128, TRUE);
	MENU_CheckItem(_InputMenu, WM_INPUT_MULTI_TAP, TRUE);

	MENU_AddItem(_FileMenu, NULL, "E&xit", WM_EXIT);
//	MENU_AddItem(_FileMenu, NULL, NULL, 0);	// separator 
//	MENU_AddItem(_FileMenu, NULL, "Save &BMP", WM_SAVE_BMP);
	MENU_AddItem(_FileMenu, NULL, NULL, 0);	// separator 
	MENU_AddItem(_FileMenu, NULL, "&Playback\tF8", WM_PLAYBACK_GAMEPLAY);
	MENU_AddItem(_FileMenu, NULL, "&Record\tF6", WM_RECORD_GAMEPLAY);
	MENU_AddItem(_FileMenu, NULL, NULL, 0);	// separator 

	MENU_AddItem(_FileMenu, NULL, "&Save State\tF5", WM_SAVE_STATE);
	MENU_AddItem(_FileMenu, NULL, "&Load State\tF7", WM_LOAD_STATE);
	MENU_AddItem(_FileMenu, NULL, NULL, 0);	// separator 

	MENU_AddItem(_FileMenu, NULL, "&Open\tSPACE", WM_OPEN_FILE);

	return TRUE;
}


static
void
deinit_menu()
{
	MENU_Deinit(_Menu);
}


/*-----------------------------------------------------------------------------
	[Init]
		AvP[V܂B
-----------------------------------------------------------------------------*/
BOOL
APP_Init(
	int			argc,
	char**		argv)
{
#ifdef DEBUGBUILD
	puts("APP_Init: PROJECT PC2E DEBUG BUILD " __DATE__ " beta.");
#else
	puts("APP_Init: PROJECT PC2E " __DATE__ " beta.");
#endif

#if defined(__GNUC__)
	puts("          Compiled with GCC version " __VERSION__);
#endif

#ifndef DEBUGBUILD
	// NEVER REMOVE THIS LINE OR THE PROJECT IS DEAD!!
	if (MessageBox(	WINMAIN_GetHwnd(),
					"Before using this program, please accept the following terms:\r\n"
					"\r\n"
					"1. This program must not be distributed with any copyrighted ROM images.\r\n"
					"\r\n"
					"2. This program must not be used with illegally obtained ROM images.\r\n"
					"   Use this program only with ROM images dumped by yourself.\r\n"
					"\r\n"
					"If you accept the terms above, click on the \"Yes\" button.\r\n"
					"If you do not, click on the \"No\" button.\r\n"
					"Do you accept the terms above?",
					"THIS PROGRAM IS NOT FOR PIRACY!!",
					MB_YESNO) != IDYES)	return FALSE;
#endif

	// AvP[ṼfBNgۑĂ 
	strcpy(_AppDir, argv[0]);
	if (strrchr(_AppDir, '\\') != NULL)
	{
		*(strrchr(_AppDir, '\\')+1) = '\0';
	}

	{
		// init & set MB128 file path
		char	filePathName[MAX_PATH];
		sprintf(filePathName, "%s"MB128_FILENAME, _AppDir);
		MB128_Init(filePathName);
	}

	// NeBJZNV 
	CS_Init();
	PRINTF_Init();

	// XN[ 
	if (SCREEN_Init(_ScreenWidth, _ScreenHeight, 16, SCREEN_FDEFAULT))
	{
		PRINTF("APP_Init: screen init OK.");
	}

	if (argc > 2)
	{
		PRINTF("APP_Init: too many arguments.");
		PRINTF("ex.)");
		PRINTF("	pc2e hucard.rom");
		PRINTF("	pc2e \"filename with spaces.rom\"");
		return FALSE;
	}

	if (argc < 2)
	{
		// No game specified.
		if (!file_dialog(WINMAIN_GetHwnd(), _GameFilePathName))
		{
			PRINTF("No game specified.");
			return FALSE;
		}
		else
		{
			// cartridge mode
			if (!MAINBOARD_Init(_GameFilePathName))
			{
				PRINTF("MAINBOARD_Init() returned error.");
				return FALSE;
			}
			PRINTF("APP_Init: MAINBOARD_Init() successfully done.");
		}
	}
	else if (argc == 2)
	{
		if (!MAINBOARD_Init(argv[1]))
		{
			PRINTF("MAINBOARD_Init() returned error.");
			return FALSE;
		}
		strcpy(_GameFilePathName, argv[1]);
		PRINTF("APP_Init: MAINBOARD_Init() successfully done.");
	}

	if (load_bram(_AppDir))
		PRINTF("APP_Init: Restored BRAM from \"BRAM.DAT\".");

	// RtBOB 
	if (!CONFIG_Init())
		PRINTF("APP_Init: CONFIG_Init() failed.");

	// ݒt@CǂݍށB 
	if (!load_config(_AppDir))
	{
		// ݒt@Cǂݍ߂Ȃꍇ̓ftHgݒ肷B 
		PRINTF("APP_Init: Failed opening \"" APP_CONFIG_FILENAME "\"; applying default settings.");

		// fBXvC֘A 
		CONFIG_Set("[APP DISPLAY] Screen Width", &_ScreenWidth, sizeof(_ScreenWidth));
		CONFIG_Set("[APP DISPLAY] Screen Height", &_ScreenHeight, sizeof(_ScreenHeight));
		CONFIG_Set("[APP DISPLAY] Frames Per Second (FPS)", &_FPS, sizeof(_FPS));
		CONFIG_Set("[APP DISPLAY] Full Screen Mode", &_bFullScreen, sizeof(_bFullScreen));
		CONFIG_Set("[APP DISPLAY] Scanline Mode", &_bScanLine, sizeof(_bScanLine));
		CONFIG_Set("[APP DISPLAY] Stretched Mode", &_bStretched, sizeof(_bStretched));
		CONFIG_Set("[APP DISPLAY] Magnification", &_Magnification, sizeof(_Magnification));
		CONFIG_Set("[APP DISPLAY] Sync To 60Hz Fullscreen", &_bSyncTo60HzFullscreen, sizeof(_bSyncTo60HzFullscreen));
		CONFIG_Set("[APP DISPLAY] Use Hardware Acceleration", &_bHardwareAcceleration, sizeof(_bHardwareAcceleration));

		// TEh֘A 
		CONFIG_Set("[APP SOUND] Sound Buffer Size", &_SoundBufferSize, sizeof(_SoundBufferSize));
		CONFIG_Set("[APP SOUND] Sound Sample Rate", &_SoundSampleRate, sizeof(_SoundSampleRate));
		CONFIG_Set("[APP SOUND] Sound Oversample Rate", &_SoundOversampleRate, sizeof(_SoundOversampleRate));
		CONFIG_Set("[APP SOUND] APU Master Volume", &_SoundApuMasterVolume, sizeof(_SoundApuMasterVolume));
		CONFIG_Set("[APP SOUND] ADPCM Volume", &_SoundAdpcmVolume, sizeof(_SoundAdpcmVolume));
		CONFIG_Set("[APP SOUND] Mixer Master Volume", &_SoundMixerVolumeMaster, sizeof(_SoundMixerVolumeMaster));
		CONFIG_Set("[APP SOUND] Mixer CD Volume", &_SoundMixerVolumeCd, sizeof(_SoundMixerVolumeCd));
		CONFIG_Set("[APP SOUND] Mixer Wave Volume", &_SoundMixerVolumeWave, sizeof(_SoundMixerVolumeWave));

		// ͊֘A 
		CONFIG_Set("[APP INPUT] Button Connections", _JoyPad, sizeof(_JoyPad));
	}
	else
	{
		// ǂݍ񂾐ݒ𕜌B 
		CONFIG_Get("[APP DISPLAY] Screen Width", &_ScreenWidth, sizeof(_ScreenWidth));
		CONFIG_Get("[APP DISPLAY] Screen Height", &_ScreenHeight, sizeof(_ScreenHeight));
		CONFIG_Get("[APP DISPLAY] Frames Per Second (FPS)", &_FPS, sizeof(_FPS));
		CONFIG_Get("[APP DISPLAY] Full Screen Mode", &_bFullScreen, sizeof(_bFullScreen));
		CONFIG_Get("[APP DISPLAY] Scanline Mode", &_bScanLine, sizeof(_bScanLine));
		CONFIG_Get("[APP DISPLAY] Stretched Mode", &_bStretched, sizeof(_bStretched));
		CONFIG_Get("[APP DISPLAY] Magnification", &_Magnification, sizeof(_Magnification));
		CONFIG_Get("[APP DISPLAY] Sync To 60Hz Fullscreen", &_bSyncTo60HzFullscreen, sizeof(_bSyncTo60HzFullscreen));
		CONFIG_Get("[APP DISPLAY] Use Hardware Acceleration", &_bHardwareAcceleration, sizeof(_bHardwareAcceleration));

		// TEh֘A 
		CONFIG_Get("[APP SOUND] Sound Buffer Size", &_SoundBufferSize, sizeof(_SoundBufferSize));
		CONFIG_Get("[APP SOUND] Sound Sample Rate", &_SoundSampleRate, sizeof(_SoundSampleRate));
		CONFIG_Get("[APP SOUND] Sound Oversample Rate", &_SoundOversampleRate, sizeof(_SoundOversampleRate));
		CONFIG_Get("[APP SOUND] APU Master Volume", &_SoundApuMasterVolume, sizeof(_SoundApuMasterVolume));
		CONFIG_Get("[APP SOUND] ADPCM Volume", &_SoundAdpcmVolume, sizeof(_SoundAdpcmVolume));
		CONFIG_Get("[APP SOUND] Mixer Master Volume", &_SoundMixerVolumeMaster, sizeof(_SoundMixerVolumeMaster));
		CONFIG_Get("[APP SOUND] Mixer CD Volume", &_SoundMixerVolumeCd, sizeof(_SoundMixerVolumeCd));
		CONFIG_Get("[APP SOUND] Mixer Wave Volume", &_SoundMixerVolumeWave, sizeof(_SoundMixerVolumeWave));

		// ͊֘A 
		CONFIG_Get("[APP INPUT] Button Connections", _JoyPad, sizeof(_JoyPad));

		PRINTF("APP_Init: Loaded configurations from \"" APP_CONFIG_FILENAME "\".");
	}

	// init pop-up menu 
	init_menu();

	// CxgnB 
	APPEVENT_Init();

	// XN[ݒ̃[hɕύXB 
	if (!change_screen_mode(_ScreenWidth, _ScreenHeight, _Magnification, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration))
	{
		PRINTF("APP_Init: error setting screen mode from configuration file %s.  Please delete the file and restart.", APP_CONFIG_FILENAME);
		return FALSE;
	}

	// TEhݒ̃[hɕύX 
	if (!MAINBOARD_ChangeSoundMode(_SoundBufferSize, _SoundSampleRate, _SoundOversampleRate, _SoundApuMasterVolume, FALSE))
	{
		// sftHgɖ߂B
		_SoundBufferSize		= 2560;
		_SoundSampleRate		= 44100;
		_SoundOversampleRate	= 8;
		_SoundApuMasterVolume	= 65535;
		_SoundAdpcmVolume		= 65535;
		_SoundMixerVolumeMaster = 65535/2;
		_SoundMixerVolumeCd		= 65535*3/4;
		_SoundMixerVolumeWave	= 65535*3/4;

		if (!MAINBOARD_ChangeSoundMode(_SoundBufferSize, _SoundSampleRate, _SoundOversampleRate, _SoundApuMasterVolume, FALSE))
		{
			// ł_炠߂B 
			PRINTF("APP_Init: error setting sound mode from configuration file %s.\n", APP_CONFIG_FILENAME);
			PRINTF("APP_Init: there will be no sound during emulation.");
		}
		else
		{
			CONFIG_Set("[APP SOUND] Sound Buffer Size", &_SoundBufferSize, sizeof(_SoundBufferSize));
			CONFIG_Set("[APP SOUND] Sound Sample Rate", &_SoundSampleRate, sizeof(_SoundSampleRate));
			CONFIG_Set("[APP SOUND] Sound Oversample Rate", &_SoundOversampleRate, sizeof(_SoundOversampleRate));
			CONFIG_Set("[APP SOUND] APU Master Volume", &_SoundApuMasterVolume, sizeof(_SoundApuMasterVolume));
			CONFIG_Set("[APP SOUND] ADPCM Volume", &_SoundAdpcmVolume, sizeof(_SoundAdpcmVolume));
			CONFIG_Set("[APP SOUND] Mixer Master Volume", &_SoundMixerVolumeMaster, sizeof(_SoundMixerVolumeMaster));
			CONFIG_Set("[APP SOUND] Mixer CD Volume", &_SoundMixerVolumeCd, sizeof(_SoundMixerVolumeCd));
			CONFIG_Set("[APP SOUND] Mixer Wave Volume", &_SoundMixerVolumeWave, sizeof(_SoundMixerVolumeWave));
		}
	}

	set_audio_sample_rate(_SoundSampleRate);
	set_audio_buffer_size(_SoundBufferSize);

	CDFADER_Init(_SoundMixerVolumeMaster, _SoundMixerVolumeWave, _SoundMixerVolumeCd);

	// ͂ݒǂɂ 
	set_input_as_configured();

	// t[[gݒ肷 
	MAINBOARD_ChangeFPS(_FPS);

#ifdef DEBUGBUILD
	run_emulator(FALSE);
#else
	run_emulator(TRUE);
#endif

	PRINTF("APP_Init: done.");

	return TRUE;
}


static
void
trace_instructions(
	Uint32				nTrace)
{
#ifdef DEBUGBUILD
	Sint32				flag;

	/* This enables sound output during trace mode */
	run_emulator(TRUE);

	flag = MAINBOARD_AdvanceInstruction(nTrace);
	if (flag == MAINBOARD_BREAKPOINT)
	{
		PRINTF("Stopped at breakpoint.");
	}
	if (flag == MAINBOARD_WATCHPOINT)
	{
		PRINTF("Stopped at watchpoint.");
	}

	run_emulator(FALSE);
	MAINBOARD_Update();
#endif
}


static
void
advance_frames(
	Uint32				nFrame)
{
	run_emulator(TRUE);
	while (nFrame--)
	{
		MAINBOARD_AdvanceFrame();
	}
	run_emulator(FALSE);
}


static
void
configure_pad(
	Uint32				joyID,
	JoyPad*				pJoy)
{
	if (!PADCONFIG_Init(WINMAIN_GetHInstance()))
	{
		PRINTF("PADCONFIG_Init() failed.\n");
		return;
	}

	if (PADCONFIG_Configure(joyID, pJoy))
	{
		CONFIG_Set("[APP INPUT] Button Connections", _JoyPad, sizeof(_JoyPad));
	}
	else
	{
		PRINTF("PADCONFIG_Configure() failed.\n");
	}

//	puts("padconfig_configure done");

	PADCONFIG_Deinit();

	INPUT_Deinit();
	INPUT_Init(0);
	set_input_as_configured();
}


#if 0
static
void
do_save_bmp_24bpp(
	const char*		pFilePathName,
	const Uint8*	pData,
	const Sint32	width,
	const Sint32	height)
{
	struct
	{
		Uint16		type;
		Uint32		size;
		Uint16		res1;
		Uint16		res2;
		Uint32		offBits;
	} header;

	struct
	{
		Uint32		size;
		Sint32		width;
		Sint32		height;
		Uint16		planes;
		Uint16		bitCount;
		Uint32		compression;
		Uint32		sizeImage;
		Sint32		xPixPerMeter;
		Sint32		yPixPerMeter;
		Uint32		clrUsed;
		Uint32		clrImportant;
	} info;

	FILE*			fp;
	Sint32			bmpW;
	Uint16			buf[1024];

	// ̃oCgvZ(S̔{ɐ) 
	if ((width * 3) & 3 == 0)	bmpW = width * 3;
	else						bmpW = (width*3 + 3) & ~3;

	memset(&header, 0, sizeof(header));
	header.type    = 'M'*256+'B';
	header.offBits = sizeof(header)+sizeof(info);
	header.size    = header.offBits+bmpW*height;

	memset(&info, 0, sizeof(info));
	info.size = sizeof(info);
	info.width = width;
	info.height = height;
	info.planes = 1;
	info.bitCount = 24;

	fp = fopen(pFilePathName);
	fwrite(&header, sizeof(header), 1, fp);
	fwrite(&info, sizeof(info), 1, fp);


	fclose(fp);
}


static
void
save_bmp(
	const char*		pAppDir,
	const char*		pGameFilePathName)
{
	char	buf[MAX_PATH];
	FILE*	fp;

	// st@C̃fBNgRs[ 
	strcpy(buf, pAppDir);

	// Q[t@CRs[ 
	if (strrchr(pGameFilePathName, '\\') != NULL)
		strcat(buf, strrchr(pGameFilePathName, '\\')+1);

	// .  '\0' ɒu 
	if (strrchr(buf, '.') != NULL)
		*strrchr(buf, '.') = '\0';

	//  .BMP ǉ
	strcat(buf, ".BMP");

//	do_save_bmp_24bpp(buf, 
}
#endif


/*----------------------------------------------------------------------------
	[handle_event]
		AvP[Vɑ΂ĔCxgɑ΂鏈sȂ܂B
----------------------------------------------------------------------------*/
static
void
handle_event(
	Sint32		event)
{
	BOOL			bRunNormalState;

#ifdef DEBUGBUILD
	bRunNormalState = FALSE;
#else
	bRunNormalState = TRUE;
#endif

	/* Cxg */
	switch (event)
	{
		case APPEVENT_REDRAWSCREEN:
			MAINBOARD_Update();
			SCREEN_Update(0, 0, 0, 0);
			break;

		case APPEVENT_SHOWMENU:
			run_emulator(FALSE);
			break;

		case APPEVENT_HIDEMENU:
			run_emulator(TRUE);
			break;

		case APPEVENT_RUN:
			run_emulator(_bRunning ^ TRUE);
			break;

		case APPEVENT_TRACE_1:			trace_instructions(1);		break;
		case APPEVENT_TRACE_10:			trace_instructions(10);		break;
		case APPEVENT_TRACE_100:		trace_instructions(100);	break;
		case APPEVENT_TRACE_1000:		trace_instructions(1000);	break;
		case APPEVENT_TRACE_10000:		trace_instructions(10000);	break;
		case APPEVENT_ADVANCEFRAME_1:	advance_frames(1);			break;
		case APPEVENT_ADVANCEFRAME_10:	advance_frames(10);			break;
		case APPEVENT_ADVANCEFRAME_100:	advance_frames(100);		break;

		case APPEVENT_FILEDIALOG:
			//run_emulator(FALSE); //  fullscreen ̂Ƃ͂܂炵is)
			MAINBOARD_Pause(TRUE);
			WINMAIN_ShowCursor(TRUE);
			_GameFilePathName[0] = '\0'; 
			if (file_dialog(WINMAIN_GetHwnd(), _GameFilePathName))
			{
				/* L^EĐI */
				end_playback();
				end_recording();

				save_bram(_AppDir);
				MAINBOARD_Deinit();
				MAINBOARD_Init(_GameFilePathName);

				// ͂ݒǂɂ 
				set_input_as_configured();

				load_bram(_AppDir);
				SCREEN_Update(0, 0, 0, 0);
			}
			run_emulator(bRunNormalState);	//   fullscreen ̂Ƃłnjis)
			break;

		case APPEVENT_OPENGAME:
			/* L^EĐI */
			end_playback();
			end_recording();

			save_bram(_AppDir);
			MAINBOARD_Deinit();
			MAINBOARD_Init(APPEVENT_GetParam(event));

			// ͂ݒǂɂ 
			set_input_as_configured();

			strcpy(_GameFilePathName, APPEVENT_GetParam(event));
			load_bram(_AppDir);
			SCREEN_Update(0, 0, 0, 0);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_RESET:
			/* L^EĐI */
			end_playback();
			end_recording();

			MAINBOARD_Reset();
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_TOGGLEFULLSCREEN:
			run_emulator(FALSE);
			toggle_fullscreen();
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_SAVESTATE:
			save_state(_AppDir, _GameFilePathName);
			MAINBOARD_Update();
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_LOADSTATE:
			load_state(_AppDir, _GameFilePathName);
			MAINBOARD_Update();
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_PARTIALMEMORYLOAD:
			load_memory_values(_AppDir, _GameFilePathName);
			break;

		case APPEVENT_RECORD:
			record_gameplay(_AppDir, _GameFilePathName);
			run_emulator(TRUE);
			break;

		case APPEVENT_PLAYBACK:
			playback_gameplay(_AppDir, _GameFilePathName);
			run_emulator(TRUE);
			break;

		case APPEVENT_SCREEN_X1:	change_screen_mode(VDC_GetDisplayWidth(), _ScreenHeight, 1, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
									run_emulator(bRunNormalState); break;
		case APPEVENT_SCREEN_X2:	change_screen_mode(VDC_GetDisplayWidth(), _ScreenHeight, 2, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
									run_emulator(bRunNormalState); break;
		case APPEVENT_SCREEN_X3:	change_screen_mode(VDC_GetDisplayWidth(), _ScreenHeight, 3, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
									run_emulator(bRunNormalState); break;
		case APPEVENT_SCREEN_X4:	change_screen_mode(VDC_GetDisplayWidth(), _ScreenHeight, 4, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
									run_emulator(bRunNormalState); break;

#ifdef DEBUGBUILD
		case APPEVENT_SWITCH_MEMORYEDITOR:
			MAINBOARD_SwitchMemoryEditor();
			break;

		case APPEVENT_CDROM_LOG:
			DISASMWND_SetCDROMLog(TRUE);
			MENU_CheckRadioItem(_LogMenu, WM_CDROM_LOG, WM_HDWR_LOG, WM_CDROM_LOG);
			break;

		case APPEVENT_HDWR_LOG:
			DISASMWND_SetCDROMLog(FALSE);
			MENU_CheckRadioItem(_LogMenu, WM_CDROM_LOG, WM_HDWR_LOG, WM_HDWR_LOG);
			break;
#endif

		case APPEVENT_INPUT_SIX_BUTTON_PAD:
			JOYPAD_ConnectMouse(FALSE);
			JOYPAD_UseSixButton(TRUE);
			PRINTF("Connected 6-button pad.");
			MENU_CheckRadioItem(_InputMenu, WM_INPUT_TWO_BUTTON_PAD, WM_INPUT_MOUSE, WM_INPUT_SIX_BUTTON_PAD);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_INPUT_TWO_BUTTON_PAD:
			JOYPAD_ConnectMouse(FALSE);
			JOYPAD_UseSixButton(FALSE);
			PRINTF("Connected 2-button pad.");
			MENU_CheckRadioItem(_InputMenu, WM_INPUT_TWO_BUTTON_PAD, WM_INPUT_MOUSE, WM_INPUT_TWO_BUTTON_PAD);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_INPUT_MOUSE:
			JOYPAD_ConnectMouse(TRUE);
			JOYPAD_ConnectMultiTap(FALSE);
			PRINTF("Connected mouse.");
			MENU_CheckRadioItem(_InputMenu, WM_INPUT_TWO_BUTTON_PAD, WM_INPUT_MOUSE, WM_INPUT_MOUSE);
			MENU_CheckItem(_InputMenu, WM_INPUT_MULTI_TAP, FALSE);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_INPUT_MULTI_TAP:
			if (JOYPAD_ConnectMultiTap(TRUE))
			{
				JOYPAD_ConnectMultiTap(FALSE);
				PRINTF("Disconnected multi-tap.");
				MENU_CheckItem(_InputMenu, WM_INPUT_MULTI_TAP, FALSE);
			}
			else
			{
				JOYPAD_ConnectMultiTap(TRUE);
				PRINTF("Connected multi-tap.");
				MENU_CheckItem(_InputMenu, WM_INPUT_MULTI_TAP, TRUE);
			}
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_INPUT_MB128:
			if (JOYPAD_ConnectMB128(TRUE))
			{
				JOYPAD_ConnectMB128(FALSE);
				PRINTF("Disconnected MB128.");
				MENU_CheckItem(_InputMenu, WM_INPUT_MB128, FALSE);
			}
			else
			{
				PRINTF("Connected MB128.");
				MENU_CheckItem(_InputMenu, WM_INPUT_MB128, TRUE);
			}
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_SCREEN_STRETCHED:
			_bStretched ^= TRUE;
			change_screen_mode(_ScreenWidth, _ScreenHeight, _Magnification, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_SCREEN_SCANLINED:
			_bScanLine ^= TRUE;
			change_screen_mode(_ScreenWidth, _ScreenHeight, _Magnification, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_SCREEN_HWACCEL:
			_bHardwareAcceleration ^= TRUE;
			change_screen_mode(_ScreenWidth, _ScreenHeight, _Magnification, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_SCREEN_SYNC_VBLANK:
			_bSyncTo60HzFullscreen ^= TRUE;
			change_screen_mode(_ScreenWidth, _ScreenHeight, _Magnification, _bFullScreen, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_INPUT_CONFIGURE_PAD1:	configure_pad(0, &_JoyPad[0]);	break;
		case APPEVENT_INPUT_CONFIGURE_PAD2:	configure_pad(1, &_JoyPad[1]);	break;
		case APPEVENT_INPUT_CONFIGURE_PAD3:	configure_pad(2, &_JoyPad[2]);	break;
		case APPEVENT_INPUT_CONFIGURE_PAD4:	configure_pad(3, &_JoyPad[3]);	break;
		case APPEVENT_INPUT_CONFIGURE_PAD5:	configure_pad(4, &_JoyPad[4]);	break;

		case APPEVENT_AUDIO_SR96000: set_audio_sample_rate(96000); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SR88200: set_audio_sample_rate(88200); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SR64000: set_audio_sample_rate(64000); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SR48000: set_audio_sample_rate(48000); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SR44100: set_audio_sample_rate(44100); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SR32000: set_audio_sample_rate(32000); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SR22050: set_audio_sample_rate(22050); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SR11025: set_audio_sample_rate(11025); run_emulator(bRunNormalState); break;

		case APPEVENT_AUDIO_SB4096: set_audio_buffer_size(4096); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SB3840: set_audio_buffer_size(3840); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SB3584: set_audio_buffer_size(3584); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SB3328: set_audio_buffer_size(3328); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SB3072: set_audio_buffer_size(3072); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SB2816: set_audio_buffer_size(2816); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SB2560: set_audio_buffer_size(2560); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SB2304: set_audio_buffer_size(2304); run_emulator(bRunNormalState); break;
		case APPEVENT_AUDIO_SB2048: set_audio_buffer_size(2048); run_emulator(bRunNormalState); break;

		case APPEVENT_AUDIO_SETVOLUME: 
			AUDIOCONFIG_Deinit();
			AUDIOCONFIG_Init(	WINMAIN_GetHInstance(),
								&_SoundApuMasterVolume,
								&_SoundAdpcmVolume,
								&_SoundMixerVolumeMaster,
								&_SoundMixerVolumeCd,
								&_SoundMixerVolumeWave);
			run_emulator(bRunNormalState);
			break;

		case APPEVENT_SAVE_BMP:
//			save_bmp();
			break;

		default:
			break;
	}

	APPEVENT_Ack();
}


/*-----------------------------------------------------------------------------
	[ProcessEvents]
		AvP[Vɑ΂͂⃁bZ[WȂǂ܂B 
-----------------------------------------------------------------------------*/
Sint32
APP_ProcessEvents()
{
	static Uint32	prevW = 256;
	Uint32			width = VDC_GetDisplayWidth();
	Sint32			event;

	/*	EBhE[hŕ\̈̃WIgύXꂽA 
		u 256 ȏ 512 v
		𖞂ꍇɌEChETCYύX */
	if (!_bFullScreen && !_bStretched)
	{
		if (prevW != width && width >= 256 && width < 512)
		{
			prevW = width;

			if (!change_screen_mode(width, _ScreenHeight, _Magnification, FALSE, _bScanLine, _bStretched, _bSyncTo60HzFullscreen, _bHardwareAcceleration))
			{
				return APP_QUIT;
			}
		}
	}

	if ((event = APPEVENT_GetEventID()) == APPEVENT_EXIT)
		return APP_QUIT;
	else
		handle_event(event);

	/* ͍XV */
	INPUT_UpdateState();

	MAINBOARD_AdvanceFrame();

	PRINTF_Update();

	return APP_IDLE;
}


/*-----------------------------------------------------------------------------
	[APP_Deinit]
		AvP[V̏IȂ܂B
-----------------------------------------------------------------------------*/
BOOL
APP_Deinit(
	int			argc,
	char**		argv)
{
	PRINTF("Entered APP_Deinit().");

	/* L^EĐI */
	end_playback();
	end_recording();

	if (save_bram(_AppDir))
		PRINTF("APP_Deinit: saved BRAM to \"BRAM.DAT\".");

	MB128_Deinit();

	AUDIOCONFIG_Deinit();
	CONFIG_Set("[APP SOUND] APU Master Volume", &_SoundApuMasterVolume, sizeof(_SoundApuMasterVolume));
	CONFIG_Set("[APP SOUND] ADPCM Volume", &_SoundAdpcmVolume, sizeof(_SoundAdpcmVolume));
	CONFIG_Set("[APP SOUND] Mixer Master Volume", &_SoundMixerVolumeMaster, sizeof(_SoundMixerVolumeMaster));
	CONFIG_Set("[APP SOUND] Mixer CD Volume", &_SoundMixerVolumeCd, sizeof(_SoundMixerVolumeCd));
	CONFIG_Set("[APP SOUND] Mixer Wave Volume", &_SoundMixerVolumeWave, sizeof(_SoundMixerVolumeWave));

	MAINBOARD_Deinit();

	SCREEN_Deinit();

	/* ݒۑB */
	if (save_config(_AppDir))
		PRINTF("APP_Deinit: configurations saved to \"" APP_CONFIG_FILENAME "\".");

	CONFIG_Deinit();

	/* NeBJZNVSĔj */
	CS_Deinit();

	deinit_menu();

	/* Cxgn̔j͂΂Ō */
	APPEVENT_Deinit();

	PRINTF("Leaving APP_Deinit().");

	PRINTF_Deinit();

	return TRUE;
}

