/*-----------------------------------------------------------------------------
	[CDROM.c]
		CD-ROM^2 hCuLq܂B
		Implements the CD-ROM^2 drive hardware.

	Copyright (C) 2004 Ki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
-----------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "CDROM.h"
#include "TocDB.h"
#include "IntCtrl.h"
#include "CdromInterface.h"
#include "ADPCM.h"
#include "CdFader.h"
#include "Printf.h"

#ifdef DEBUGBUILD
	#include "DisasmWindow.h"
#endif


#define BCD(A)	((A / 10) * 16 + (A % 10))		// convert INT --> BCD
#define INT(B)	((B / 16) * 10 + (B % 16))		// convert BCD --> INT


typedef struct
{
	Uint8	minStart;
	Uint8	secStart;
	Uint8	frameStart;

	Uint8	minEnd;
	Uint8	secEnd;
	Uint8	frameEnd;

	Uint8	subqMinTrack;
	Uint8	subqSecTrack;
	Uint8	subqFrmTrack;
	Uint8	subqMinTotal;
	Uint8	subqSecTotal;
	Uint8	subqFrmTotal;
	Uint8	subqTrackNum;

	Sint32	elapsedSec;
	Sint32	playEndSec;
	Sint32	searchMode;
	Sint32	playMode;

	BOOL	bRepeat;
	BOOL	bPlaying;
	BOOL	bPaused;
	BOOL	bSeeking;
	BOOL	bSeekDone;
	BOOL	bInterrupt;
} AudioTrack;


static BOOL			_bBRAMEnabled;


static Uint8		_Port[15];

static Uint8		_ReadBuffer[2048*256];	// 512KB
static Sint32		_ReadBufferIndex;		// ǂݏoobt@̃CfbNX 
static Sint32		_ReadByteCount;		// ǂݏooCg  F_EJE^ 
static Sint32		_CheckCountAfterRead;
static Uint32		_ResetDelayCount = 10;

// R}hobt@ 
static Sint32		_Command;
static Sint32		_ArgsLeft;
static Uint8		_CmdArgBuffer[10];
static Sint32		_CmdArgBufferIndex;

static BOOL			_bCommandReset = TRUE;
static BOOL			_bCommandReceived = FALSE;
static BOOL			_bCommandDone = FALSE;
static BOOL			_bDriveBusy = FALSE;
static BOOL			_bError = FALSE;		// set when command execution fail

static AudioTrack	_AudioTrack;
static Sint32		_ClockCount = 0;
static BOOL			_bCdromInit = FALSE;


static
BOOL
check_disc_toc()
{
	int			i;
	int			lastTrack = CDIF_GetLastTrack();
	Uint32		msft;
	Uint32		lba;
	DISCINFO	di;
	const DISCINFO*	pDI;

	memset(&di, 0, sizeof(DISCINFO));

	// OɂPgbNڂ̃f[^ 
	// lastTrack+1̓[hAEg 
	for (i = 0; i <= lastTrack; i++)
	{
		msft = CDIF_GetTrackStartPositionMSF(i+1);
		lba  = CDIF_GetTrackStartPositionLBA(i+1);

		if (i != lastTrack)
			di.TOC[i].trackNum = i+1;

		if (i != lastTrack)
			di.TOC[i].isAudio = (msft & 0xff) == 0;

		di.TOC[i].min   = (msft >> 24) & 0xff;
		di.TOC[i].sec   = (msft >> 16) & 0xff;
		di.TOC[i].frame = (msft >>  8) & 0xff;

		di.TOC[i].lbaH  = (lba >> 16) & 0xff;
		di.TOC[i].lbaM  = (lba >>  8) & 0xff;
		di.TOC[i].lbaL  = (lba      ) & 0xff;
	}

	if ((pDI = TOCDB_IsMatch(&di)) != NULL)
	{
		PRINTF("CDROM_Init: found \"%s\"\n", pDI->pTitle);
		return TRUE;
	}

	return FALSE;
}


static
BOOL
check_cdrom2_disc()
{
	int		nDevices = CDIF_GetNumDevices();
	int		i;

	PRINTF("CDROM_Init: %d drive(s) found.\n", nDevices);

	for (i = 0; i < nDevices; i++)
	{
		PRINTF("CDROM_Init: checking drive #%d...\n", i);

		if (CDIF_SelectDevice(i))
		{
			if (!check_disc_toc())
				continue;

			PRINTF("CDROM_Init: selected drive #%d.\n", i);
			return TRUE;
		}
	}

	PRINTF("CDROM_Init: no drive seems to have a CD-ROM^2 CD.");
	return FALSE;
}


static
void
lba2msf(
	Uint32		lba,
	Uint8*		m,
	Uint8*		s,
	Uint8*		f)
{
	*m = lba / 75 / 60;
	*s = (lba - *m * 75 * 60) / 75;
	*f = lba - (*m * 75 * 60) - (*s * 75);
}


static
Uint8
get_first_track(void)
{
	if (!_bCdromInit)
		return 0;

	return CDIF_GetFirstTrack();
}


static
Uint8
get_last_track(void)
{
	if (!_bCdromInit)
		return 0;

	return CDIF_GetLastTrack();
}


static
void
get_track_start_position(
	Uint8		track,
	Uint8*		min,
	Uint8*		sec,
	Uint8*		frame,
	Uint8*		type)		// f[^gbN(4)I[fBIgbN(0)
{
	Uint32		msft = CDIF_GetTrackStartPositionMSF(track);

	*min   = (Uint8)(msft >> 24);
	*sec   = (Uint8)(msft >> 16);
	*frame = (Uint8)(msft >> 8);
	*type  = (Uint8)(msft);
}


static
Uint8
get_track_number_by_msf(
	Uint8		m,
	Uint8		s,
	Uint8		f)
{
	int			track = get_first_track();
	int			lastTrack = get_last_track();
	int			msf;

	while (track <= lastTrack)
	{
		msf = CDIF_GetTrackStartPositionMSF(track) >> 8;

		if ((m << 16) + (s << 8) + f < msf)
			return track - 1;
		++track;
	}

	return 0;
}


void
load_sectors(
	Sint32		sectorAddr,
	Uint8*		pBuffer,
	Sint32		nSector)
{
	if (_bCdromInit)
	{
		CDIF_ReadSector(pBuffer, sectorAddr, nSector, TRUE);
	}
}


static
void
seek_track(
	AudioTrack*		p)
{
	if (_bCdromInit)
		CDIF_Seek(p->minStart, p->secStart, p->frameStart, TRUE);
}


static
void
play_track(
	AudioTrack*		p)
{
	if (!_bCdromInit)
		return;

	// ignore frame
	// +1 ͍ĐIOɃs[gĂ܂̂mɖh߂ɂ 
	p->playEndSec = (p->minEnd - p->minStart) * 60 + (p->secEnd - p->secStart) + 1;
	p->elapsedSec = 0;
	CDIF_PlayAudioMSF(p->minStart, p->secStart, p->frameStart, p->minEnd, p->secEnd, p->frameEnd, TRUE);
}


static
void
stop_play_track()
{
	if (_bCdromInit)
		CDIF_StopAudioTrack(FALSE);
}


static
void
pause_track(
	BOOL		bPause)
{
	if (_bCdromInit)
	{
		if (!CDIF_PauseAudioTrack(bPause, TRUE))
		{
			/* play ĂȂƂ pause ꂽꍇ(Ƃ肠)Ώ */
			/* {̓G[Ԃ悤ɂׂ */
			//_bError = TRUE;
			_Port[0] = 0xd8;
			_ReadByteCount = 0;
			_CheckCountAfterRead = 2;
			_bCommandDone = TRUE;
		}
	}
}


static
void
update_irq_state()
{
	Uint8		irq = _Port[2] & _Port[3] & (0x4|0x8|/*0x10|*/0x20|0x40);

	if (irq != 0)
		INTCTRL_Request(INTCTRL_IRQ2);
	else
		INTCTRL_Cancel(INTCTRL_IRQ2);
}


void
adpcm_state_notification_callback_function(
	Uint32		adpcmState)
{
	switch (adpcmState)
	{
		case ADPCM_STATE_NORMAL:
			_Port[3] &= ~(4 | 8);
			break;

		case ADPCM_STATE_HALF_PLAYED:
			_Port[3] |= 4;
			_Port[3] &= ~8;
			break;

		case ADPCM_STATE_FULL_PLAYED:
			_Port[3] &= ~4;
			_Port[3] |= 8;
			break;

		case ADPCM_STATE_STOPPED:
			_Port[3] &= ~4;
			_Port[3] |= 8;
			break;
	}

	update_irq_state();
}


static
void
simulate_subchannel_q()
{
	if (!_bCdromInit)
		return;

	if (_AudioTrack.bPlaying && !_AudioTrack.bPaused)
	{
		if (++_AudioTrack.subqSecTrack >= 60)
		{
			_AudioTrack.subqSecTrack = 0;
			++_AudioTrack.subqMinTrack;
		}

		if (++_AudioTrack.subqSecTotal >= 60)
		{
			_AudioTrack.subqSecTotal = 0;
			++_AudioTrack.subqMinTotal;
		}
	}
}


/*-----------------------------------------------------------------------------
	[read_1801]
		CD-ROM ̃f[^ǂݏoB
-----------------------------------------------------------------------------*/
static
Uint8
read_1801(void)
{
	if (_ReadByteCount > 0)
	{
		Uint8	ret = _ReadBuffer[_ReadBufferIndex++];

		if (--_ReadByteCount > 0)
		{
			_Port[0] = 0xc8;	// data still exist in buffer
		}
		else
		{
			_Port[3] &= ~0x40;	// "data transfer ready" I 
			_Port[3] |= 0x20;	// data transfer done 
			update_irq_state();

			_Port[0] = 0xd8;	// no more data left in buffer
			_ReadBufferIndex = 0;

			// ǂݏoI
			// ɏIǂmF邽߂
			// ǂݏoQsȂB
			_CheckCountAfterRead = 2;
		}
		return ret;
	}
	else	// obt@̃f[^SēǂݏoꂽɊmF̂߂ $1801 ǂݏoQsȂB 
	{
		if (_CheckCountAfterRead == 2)
		{
			--_CheckCountAfterRead;
			_Port[0] = 0xf8;

			if (_bError)
			{
				_bError = FALSE;
				return 1;
			}
		}
		else if (_CheckCountAfterRead == 1)
		{
			--_CheckCountAfterRead;
			_Port[0] &= ~0x80;
		}
	}

	return 0;
}


/*-----------------------------------------------------------------------------
	[read_1808]
		CD-ROM ̃f[^ǂݏoB

	[]
		read_1801 Ƃ̈Ⴂ́Abc|qnlobt@̒lǂݏI
		Q̓ǂݏo̒lԂȂ_B 
-----------------------------------------------------------------------------*/
static
Uint8
read_1808(void)
{
	if (_ReadByteCount > 0)
	{
		Uint8	ret = _ReadBuffer[_ReadBufferIndex++];

		if (--_ReadByteCount > 0)
		{
			_Port[0] = 0xc8;	// data still exist in buffer
		}
		else
		{
			_Port[3] &= ~0x40;		// "data transfer ready" I 
			_Port[3] |= 0x20;		// data transfer done 
			update_irq_state();

			_Port[0] = 0xd8;	// no more data left in buffer
			_ReadBufferIndex = 0;

			// ǂݏoI
			// ɏIǂmF邽߂
			// ǂݏoQsȂB
			_CheckCountAfterRead = 2;
		}

		return ret;
	}
	else
	{
		_Port[0] = 0xd8;	// no more data left in buffer
	}

	return 0;
}


void
show_command_string(
	const char*		pCmdName,
	const Uint8*	pCmdString)
{
#ifdef DEBUGBUILD
	// This function only used for hardware-level logging
	//
	if (DISASMWND_GetCDROMLog() == FALSE)
		PRINTF("%s %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n",
			pCmdName,
			pCmdString[0],pCmdString[1],pCmdString[2],pCmdString[3],pCmdString[4],
			pCmdString[5],pCmdString[6],pCmdString[7],pCmdString[8],pCmdString[9]);
#endif
}


/*
	A callback function for notifying that command is successfully completed.
*/
static
void
cd_callback(
	Uint32		flag)
{
	_bCommandDone = TRUE;

	switch (flag)
	{
		case CDIF_SEEK:
			_Port[0] = 0xd8;
			_ReadByteCount = 0;
			_CheckCountAfterRead = 2;

			_AudioTrack.bSeekDone = TRUE;
			_AudioTrack.bPaused   = FALSE;
			_AudioTrack.bPlaying  = FALSE;

			_Port[3] |= 0x20;
			update_irq_state();

			_bError = FALSE;
//			T[`[hP̂Ƃ́AȂōĐJn 
//			if (_AudioTrack.searchMode)
//			{
//				
//			}
			break;

		case CDIF_PLAY:
			_Port[0] = 0xd8;
			_ReadByteCount = 0;
			_CheckCountAfterRead = 2;

			_AudioTrack.bSeekDone = FALSE;
			_AudioTrack.bPaused   = FALSE;
			_AudioTrack.bPlaying  = TRUE;
			if (_AudioTrack.playMode == 1 || _AudioTrack.playMode == 2)
			{
				_bDriveBusy = TRUE;
			}

			/* initialize sub Q channel for simulation */
			_AudioTrack.subqMinTrack = 0;
			_AudioTrack.subqSecTrack = 0;
			_AudioTrack.subqFrmTrack = 0;

			_AudioTrack.subqMinTotal = _AudioTrack.minStart;
			_AudioTrack.subqSecTotal = _AudioTrack.secStart;
			_AudioTrack.subqFrmTotal = _AudioTrack.frameStart;

			_AudioTrack.subqTrackNum = get_track_number_by_msf(_AudioTrack.minStart, _AudioTrack.secStart, _AudioTrack.frameStart);

			_bError = FALSE;
			break;

		case CDIF_READ:
			_Port[0] = 0xc8;
			_Port[3] |= 0x40;
			update_irq_state();
			_bError = FALSE;
			break;

		case CDIF_PAUSE:
			_Port[0] = 0xd8;
			_ReadByteCount = 0;
			_CheckCountAfterRead = 2;

			_AudioTrack.bSeekDone = FALSE;
			_AudioTrack.bPaused   = TRUE;
			_AudioTrack.bPlaying  = FALSE;
			_bError = FALSE;
			break;

		case CDIF_STOP:
			_Port[0] = 0xd8;
			_ReadByteCount = 0;
			_CheckCountAfterRead = 2;

			_AudioTrack.bSeekDone = FALSE;
			_AudioTrack.bPaused   = FALSE;
			_AudioTrack.bPlaying  = FALSE;
			_bError = FALSE;
			break;
	}
}


/*-----------------------------------------------------------------------------
	[Init]
-----------------------------------------------------------------------------*/
Sint32
CDROM_Init()
{
	_bCdromInit = FALSE;

	if (!CDIF_Init(cd_callback))
		return -1;

	if (!check_cdrom2_disc())
		return -1;

	if (!CDFADER_Init(-1, -1, -1))
		return -1;

	ADPCM_SetNotificationFunction(adpcm_state_notification_callback_function);

	_bCdromInit = TRUE;
	_bError = FALSE;

	return 0;
}


/*-----------------------------------------------------------------------------
	[Deinit]
-----------------------------------------------------------------------------*/
void
CDROM_Deinit(void)
{
	INTCTRL_Cancel(INTCTRL_IRQ2);

	CDIF_Deinit();
	CDFADER_Deinit();

	_bBRAMEnabled = FALSE;
	memset(_Port, 0, sizeof(_Port));

	_ReadBufferIndex = 0;
	_ReadByteCount = 0;

	_Command = 0;
	_ArgsLeft = 0;
	_CmdArgBufferIndex = 0;

	_bCommandReset = TRUE;
	_bCommandReceived = FALSE;
	_CheckCountAfterRead = 0;
	memset(&_AudioTrack, 0, sizeof(_AudioTrack));
	_ClockCount = 0;

	_bDriveBusy = FALSE;
	_bError = FALSE;

	_bCdromInit = FALSE;
}


/*-----------------------------------------------------------------------------
	[Reset]
-----------------------------------------------------------------------------*/
BOOL
CDROM_Reset()
{
	CDROM_Deinit();
	return CDROM_Init() == 0;
}


/*
	[Pause]
		Đ̂bcgbNꎞ~B
		G~[VŜ~ꍇɂ̂ݎgpB
		͎gpȂB
*/
void
CDROM_Pause(
	BOOL		bPause)
{
	if (_AudioTrack.bPlaying)
	{
		CDIF_PauseAudioTrack(bPause, FALSE);
	}
}


BOOL
CDROM_IsCDROMEnabled()
{
	return _bCdromInit;
}


BOOL
CDROM_IsBRAMEnabled()
{
	return _bBRAMEnabled;
}


Uint8
CDROM_Read(
	Uint32		physAddr)
{
	if ((physAddr & 0x18c0) == 0x18c0)
	{
		switch (physAddr & 0x18cf)
		{
			case 0x18c1:
				return 0xaa;
			case 0x18c2:
				return 0x55;
			case 0x18c3:
				return 0;
			case 0x18c5:
				return 0xaa;
			case 0x18c6:
				return 0x55;
			case 0x18c7:
				return 0x03;
		}
	}

	switch (physAddr & 0xf)
	{
		case 0x0:
//			PRINTF("$1800 = %02x\n", _Port[0]);
			if (_Port[2] & 0x80)
			{
				if (_CheckCountAfterRead == 0)
				{
					_Port[3] &= ~0x20;
					_bDriveBusy = FALSE;
					update_irq_state();
				}
				return _Port[0] & ~0x40;
			}
			else if (_bCommandReceived && !_bCommandDone)
			{
				return _Port[0] & ~0x40;
			}
			else if (_bDriveBusy)
			{
				return _Port[0] | 0x80;
			}
			else if (_ResetDelayCount > 0)
			{
				--_ResetDelayCount;
				return _Port[0] & ~0x40;
			}

			return _Port[0] | 0x40;

		case 0x1:
			return read_1801();

		case 0x2: // read/write port (control port)
			return _Port[2];

		case 0x3:	// obNAbv֎~B (read only)
					// status-read port
			_bBRAMEnabled = FALSE;
			/* switch left/right of digitized cd playback */
			_Port[3] ^= 2;
			return _Port[3] | 0x10;

		case 0x4:
			return _Port[4];

		case 0x5:
			return ++_Port[5];

		case 0x6:
			return ++_Port[6];

		case 0x7:
			// CD subchannel read
			return _Port[7];

		case 0x8:	// CD-ROM ZN^ǂݏoB
			return read_1808();

		case 0xa:
			return ADPCM_ReadBuffer();

		case 0xb:
			return _Port[0xb] & ~1;

		case 0xc:
			// D0: _ ADPCM ̍ĐI܂͒~Ăꍇ͂P 
			// D2: CD --> DMA ]̓[H (1: busy prepareing ADPCM data)
			// D3: ADPCM Đ͂P
			// D7: O $180A ̓ǂݏołatrx̏ꍇ
			// D7 = 1 ƂȂB
			if (!ADPCM_IsPlaying())
			{
				_Port[0xc] |= 1;
				_Port[0xc] &= ~8;
			}
			else
			{
				_Port[0xc] &= ~1;
				_Port[0xc] |= 8;
			}
			return _Port[0xc];

		case 0xd:
//			return _Port[0xd];
			return 0;
	}

	return 0;
}



static
void
execute_read_sector()
{
	Uint32		sectorAddr;

	show_command_string("[READ SECTOR]", _CmdArgBuffer);

	sectorAddr = _CmdArgBuffer[1]*65536 + _CmdArgBuffer[2]*256 + _CmdArgBuffer[3];
	load_sectors(sectorAddr, _ReadBuffer, _CmdArgBuffer[4]);
	_ReadByteCount = 2048 * _CmdArgBuffer[4];
}


static
void
execute_cd_playback_start_position()
{
	show_command_string("[PLAY AUDIO1]", _CmdArgBuffer);
	memset(&_AudioTrack, 0, sizeof(AudioTrack));

	/* Să[̂Ƃ͒~ */
	if ((_CmdArgBuffer[2] | _CmdArgBuffer[3] | _CmdArgBuffer[4]) == 0)
	{
		stop_play_track();
		_Port[0] = 0xd8;
		_ReadByteCount = 0;
		_CheckCountAfterRead = 2;

		_AudioTrack.bSeekDone = FALSE;
		_AudioTrack.bPaused   = FALSE;
		_AudioTrack.bPlaying  = FALSE;

		_Port[3] |= 0x20;
		update_irq_state();

		_bError = FALSE;
		return;
	}

	switch (_CmdArgBuffer[9] & 0xc0)
	{
		case 0x00:	// ka`w胂[h
			lba2msf((_CmdArgBuffer[2]<<16)|(_CmdArgBuffer[3]<<8)|_CmdArgBuffer[4],
					&_AudioTrack.minStart,
					&_AudioTrack.secStart,
					&_AudioTrack.frameStart);
			seek_track(&_AudioTrack);
			break;

		case 0x40:	// lrew胂[h
			_AudioTrack.minStart   = INT(_CmdArgBuffer[2]);
			_AudioTrack.secStart   = INT(_CmdArgBuffer[3]);
			_AudioTrack.frameStart = INT(_CmdArgBuffer[4]);
			seek_track(&_AudioTrack);
			break;

		case 0x80:	// gbNԍw胂[h
		{
			Uint8		not_used;

			// gbNԍw胂[h
			get_track_start_position(	INT(_CmdArgBuffer[2]),
										&_AudioTrack.minStart,
										&_AudioTrack.secStart,
										&_AudioTrack.frameStart,
										&not_used);
			seek_track(&_AudioTrack);
			break;
		}

		case 0xc0:	// ???
			//PRINTF("unknown mode");
			break;
	}

	_AudioTrack.searchMode = _CmdArgBuffer[1];
}


static
void
execute_cd_playback_end_position()
{
	show_command_string("[PLAY AUDIO2]", _CmdArgBuffer);
	switch (_CmdArgBuffer[9] & 0xc0)
	{
		case 0x00:	// ka`w胂[h
			lba2msf((_CmdArgBuffer[2]<<16)|(_CmdArgBuffer[3]<<8)|_CmdArgBuffer[4],
					&_AudioTrack.minEnd,
					&_AudioTrack.secEnd,
					&_AudioTrack.frameEnd);
			break;

		case 0x40:	// lrew胂[h
			_AudioTrack.minEnd   = INT(_CmdArgBuffer[2]);
			_AudioTrack.secEnd   = INT(_CmdArgBuffer[3]);
			_AudioTrack.frameEnd = INT(_CmdArgBuffer[4]);
			break;

		case 0x80:	// gbNԍw胂[h
		{
			Uint8		not_used;

			// gbNԍw胂[h
			get_track_start_position(	INT(_CmdArgBuffer[2]),
										&_AudioTrack.minEnd,
										&_AudioTrack.secEnd,
										&_AudioTrack.frameEnd,
										&not_used);
			break;
		}

		case 0xc0:	// gbNIʒu?? 
			break;
	}

	switch (_CmdArgBuffer[1])
	{
		case 0x00:		// no operation ??
			_AudioTrack.playMode = 0;
			stop_play_track();
			_Port[0] = 0xd8;
			_ReadByteCount = 0;
			_CheckCountAfterRead = 2;
			_bCommandDone = TRUE;
			return;

		case 0x01:		// repeat play
			_AudioTrack.bRepeat = TRUE;
			_AudioTrack.playMode = 1;
			play_track(&_AudioTrack);
			break;

		case 0x02:		// play, IRQ2 when finished ??
			_AudioTrack.bInterrupt = TRUE;
			_AudioTrack.playMode = 2;
			play_track(&_AudioTrack);
			break;

		case 0x03:		// play without repeat
			_AudioTrack.playMode = 3;
			if (_AudioTrack.bPaused)
				pause_track(FALSE);
			else
				play_track(&_AudioTrack);
			break;
	}
}


static
void
execute_pause_cd_playback()
{
	show_command_string("[PAUSE AUDIO]", _CmdArgBuffer);
	pause_track(TRUE);
}


static
void
execute_read_subchannel_q()
{
	_ReadBuffer[2] = BCD(_AudioTrack.subqTrackNum);
	_ReadBuffer[4] = BCD(_AudioTrack.subqMinTrack);
	_ReadBuffer[5] = BCD(_AudioTrack.subqSecTrack);
	_ReadBuffer[6] = BCD(_AudioTrack.subqFrmTrack);
	_ReadBuffer[7] = BCD(_AudioTrack.subqMinTotal);
	_ReadBuffer[8] = BCD(_AudioTrack.subqSecTotal);
	_ReadBuffer[9] = BCD(_AudioTrack.subqFrmTotal);

	if (_AudioTrack.bPlaying)
		_ReadBuffer[0] = 0;
	else if (_AudioTrack.bPaused || _AudioTrack.bSeekDone)
		_ReadBuffer[0] = 2;
	else
		_ReadBuffer[0] = 3;
/*
	PRINTF("[Q SUB CHANN] %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n",
			_ReadBuffer[0],_ReadBuffer[1],_ReadBuffer[2],_ReadBuffer[3],_ReadBuffer[4],
			_ReadBuffer[5],_ReadBuffer[6],_ReadBuffer[7],_ReadBuffer[8],_ReadBuffer[9]);
*/
	_ReadByteCount = 10;
	_Port[0] = 0xc8;
	_bCommandDone = TRUE;
}


static
void
execute_get_dir_info()
{
	switch (_CmdArgBuffer[1])
	{
		case 0:	// get first and last track number
			_ReadBuffer[0] = BCD(get_first_track());
			_ReadBuffer[1] = BCD(get_last_track());
			_ReadByteCount = 2;
			break;

		case 1:	// get total running time of disc
			_ReadBuffer[0] = BCD(72);
			_ReadBuffer[1] = BCD(26);
			_ReadBuffer[2] = BCD(29);
			_ReadByteCount = 3;
			break;

		case 2:	// get track starting position and mode
		{
			Uint8		min;
			Uint8		sec;
			Uint8		frame;
			Uint8		type;

			get_track_start_position(_CmdArgBuffer[2], &min, &sec, &frame, &type);

			_ReadBuffer[0] = BCD(min);
			_ReadBuffer[1] = BCD(sec);
			_ReadBuffer[2] = BCD(frame);
			_ReadBuffer[3] = type;
			_ReadByteCount = 4;
			break;
		}
	}
	_Port[0] = 0xc8;
	_bCommandDone = TRUE;
}


static
void
execute_command(void)
{
	switch (_CmdArgBuffer[0])
	{
		case 0x08:	// read sector
			execute_read_sector();
			break;

		case 0xd8:	// set audio playback start position
			execute_cd_playback_start_position();
			break;

		case 0xd9:	// set audio playback end position and start playing
			execute_cd_playback_end_position();
			break;

		case 0xda:	// pause audio
			execute_pause_cd_playback();
			break;

		case 0xdd:	// read Q sub-channel
			execute_read_subchannel_q();
			break;

		case 0xde:	// get dir info
			execute_get_dir_info();
			break;
	}
}


static
void
receive_command(
	Uint8		data)
{
	if (_bCommandReset)
	{
		_bCommandDone = FALSE;
		_Command = data;

		// R}hnϐZbgB 
		_bCommandReset = FALSE;
		_bCommandReceived = FALSE;
		_CmdArgBufferIndex = 0;
		_ArgsLeft = 0;
		_ReadBufferIndex = 0;
		_ReadByteCount = 0;

		// R}h󂯕tB 
		switch (_Command)
		{
			case 0x00:	// TEST UNIT READY 
				_Port[0] = 0xd8;	// no more data needed
				_ArgsLeft = 0;
				_ReadByteCount = 0;
				_CheckCountAfterRead = 2;
				break;

			case 0x03:	// REQUEST SENSE
				break;

			case 0x08:	// read sector
				_CmdArgBuffer[_CmdArgBufferIndex++] = 0x08;
				_ArgsLeft = 5;
				break;

			case 0xd8:	// play audio (start position)
				_CmdArgBuffer[_CmdArgBufferIndex++] = 0xd8;
				_ArgsLeft = 9;
				break;

			case 0xd9:	// play audio (end position)
				_CmdArgBuffer[_CmdArgBufferIndex++] = 0xd9;
				_ArgsLeft = 9;
				break;

			case 0xda:	// pause audio 
				_CmdArgBuffer[_CmdArgBufferIndex++] = 0xda;
				_ArgsLeft = 9;
				_ReadByteCount = 0;
				_CheckCountAfterRead = 2;
				break;

			case 0xdd:
				_CmdArgBuffer[_CmdArgBufferIndex++] = 0xdd;
				_ArgsLeft = 9;
				_ReadByteCount = 10;
				break;

			case 0xde:	// get CD directory info
				_CmdArgBuffer[_CmdArgBufferIndex++] = 0xde;
				_ArgsLeft = 9;
				_ReadByteCount = 4;
				break;

			default:
				break;
		}
	}
	else
	{
		// 󂯕t 
		_CmdArgBuffer[_CmdArgBufferIndex++] = data;

		if (--_ArgsLeft > 0)
		{
			_Port[0] = 0xd0;		// 0xd0: need more data
		}
		else
		{
			execute_command();
			_bCommandReceived = TRUE;
		}
	}
}


void
CDROM_Write(
	Uint32		physAddr,
	Uint8		data)
{
	switch (physAddr & 0xf)
	{
		case 0x0:		// $1800 write: resets the command input
			_Port[0] = 0xd0;	// = need more data
			_bCommandReset = TRUE;
			_bCommandReceived = FALSE;
			_bCommandDone = FALSE;
			_bDriveBusy = FALSE;
			_ResetDelayCount = 10;
//			PRINTF("$1800 <-- $81: command reset?? $1800 = 0xd0");

			/* reset irq status */
			_Port[3] = 0;
			update_irq_state();
			return;

		case 0x1:		// $1801
			_Port[1] = data;
			if (data == 0x81)
			{
				// _ArgsLeft > 0 ܂R}ht̂Ƃ
				// ZbgȂB
				if (_ArgsLeft == 0)
				{
//					PRINTF("$1801 <-- $81: cd reset?? $1800 = 0x00");
					_bCommandReset = TRUE;
					_bCommandReceived = FALSE;
					_bCommandDone = FALSE;
					_Port[0] = 0x00;

					/* reset irq status */
					_Port[3] = 0;
					update_irq_state();
					return;
				}
			}
			receive_command(data);
			return;

		case 0x2:		// $1802
			_Port[2] = data;
			update_irq_state();
			return;

//		case 0x3:		// read only
//			return;

		case 0x4:
			if (data & 2)
			{
				// cd reset
				_bCommandReset = TRUE;
				_bCommandReceived = FALSE;
				_bCommandDone = FALSE;
				_bDriveBusy = FALSE;
				_ResetDelayCount = 10;
				stop_play_track();

				/* reset irq status */
				_Port[3] = 0;
				update_irq_state();
			}
			_Port[4] = data;
			return;

		case 0x7:	// $1807: D7=1 enables backup ram 
			if (data & 0x80)
			{
				_bBRAMEnabled = TRUE;
			}
			return;
	
		case 0x8:
			ADPCM_SetAddrLo(data);
			return;

		case 0x9:
			ADPCM_SetAddrHi(data);
			return;

		case 0xa:
			ADPCM_WriteBuffer(data);
			return;

		case 0xb:	// adpcm dma
			if (data & 3)
			{
				while (_ReadByteCount > 0)
					ADPCM_WriteBuffer(read_1801());
			}

			/*	$180C  D2  ADPCM obt@ɏ݂sȂĂ 
				΂炭̊ԃZbg悤B */
			if (_ReadByteCount == 0)
			{
				_Port[0xc] &= ~4;		// busy writing data
			}
			else
			{
				_Port[0xc] |= 4;
			}

			_Port[0xb] = data;
			return;

//		case 0xc:		// read-only
//			return;

		case 0xd:
			if (data & 0x80)
			{
				ADPCM_Reset();
			}

			// D5  D6 ZbgƍĐJnH 
			ADPCM_Repeat((data & 0x20) == 0x20);
			ADPCM_Play((data & 0x60) == 0x60);

			if (data & 0x10)
			{
				ADPCM_SetLength();
			}

			if (data & 0x08)
			{
				ADPCM_SetReadAddr();
			}

			if ((data & 0x03) == 0x03)
			{
				ADPCM_SetWriteAddr();
			}
			_Port[0xd] = data;
			return;

		case 0xe:		// Set ADPCM playback rate
			ADPCM_SetFreq(32 * 1000 / (16 - (data & 15)));
			return;

		case 0xf:
			switch (data & 0xf)
			{
				case 0:	// tF[hAEg 
					CDFADER_FadeIn(0);
					ADPCM_FadeIn(0);
					break;

				case 8:	// fade out CD (6[s])
				case 9:
					CDFADER_FadeOut(6000);
					break;

				case 0xa: // fade out ADPCM (6[s])
					//PRINTF("ADPCM fade (6[s])");
					ADPCM_FadeOut(6000);
					break;

				case 0xc:
				case 0xd:
					CDFADER_FadeOut(2500);
					break;

				case 0xe: // fade out ADPCM (2.5[s])
					//PRINTF("ADPCM fade (2.5[s])");
					ADPCM_FadeOut(2500);
					break;
			}
			return;
	}
}


Uint32
CDROM_AdvanceClock(
	Sint32		clock)		// 21477270/3 [Hz]
{
	Uint32		ret = 0;

	_ClockCount += clock;

	if (_ClockCount >= 21477270 / 3)
	{
		_ClockCount -= 21477270 / 3;

		if (_AudioTrack.bPlaying && !_AudioTrack.bPaused)
		{
			if (++_AudioTrack.elapsedSec >= _AudioTrack.playEndSec)
			{
				_AudioTrack.bPlaying = FALSE;
				if (_AudioTrack.bRepeat)
				{
					_AudioTrack.elapsedSec = 0;
					_AudioTrack.bPlaying = CDIF_PlayAudioMSF(	_AudioTrack.minStart,
																_AudioTrack.secStart,
																_AudioTrack.frameStart,
																_AudioTrack.minEnd,
																_AudioTrack.secEnd,
																_AudioTrack.frameEnd,
																FALSE);
				}
				else if (_AudioTrack.bInterrupt)
				{
					_AudioTrack.bInterrupt = FALSE;
					if (_AudioTrack.playMode == 2)
					{
						_Port[3] |= 0x20;
						update_irq_state();
					}
				}
			}
		}
		simulate_subchannel_q();
	}

	CDFADER_AdvanceClock(clock);

	return ret;
}


// save variable
#define SAVE_V(V)	if (fwrite(&V, sizeof(V), 1, p) != 1)	return FALSE
#define LOAD_V(V)	if (fread(&V, sizeof(V), 1, p) != 1)	return FALSE
// save array
#define SAVE_A(A)	if (fwrite(A, sizeof(A), 1, p) != 1)	return FALSE
#define LOAD_A(A)	if (fread(A, sizeof(A), 1, p) != 1)		return FALSE
/*-----------------------------------------------------------------------------
	[SaveState]
		Ԃt@Cɕۑ܂B 
-----------------------------------------------------------------------------*/
BOOL
CDROM_SaveState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	SAVE_V(_bBRAMEnabled);
	SAVE_V(_ReadBufferIndex);
	SAVE_V(_ReadByteCount);
	SAVE_V(_Command);
	SAVE_V(_ArgsLeft);
	SAVE_V(_CmdArgBufferIndex);
	SAVE_V(_bCommandReset);
	SAVE_V(_bCommandDone);
	SAVE_V(_bDriveBusy);
	SAVE_V(_bError);
	SAVE_V(_ResetDelayCount);
	SAVE_V(_CheckCountAfterRead);
	SAVE_V(_AudioTrack);
	SAVE_V(_ClockCount);
	SAVE_V(_bCdromInit);

	SAVE_A(_Port);
	SAVE_A(_ReadBuffer);
	SAVE_A(_CmdArgBuffer);
	
	return TRUE;
}


/*-----------------------------------------------------------------------------
	[LoadState]
		Ԃt@Cǂݍ݂܂B 
-----------------------------------------------------------------------------*/
BOOL
CDROM_LoadState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	LOAD_V(_bBRAMEnabled);
	LOAD_V(_ReadBufferIndex);
	LOAD_V(_ReadByteCount);
	LOAD_V(_Command);
	LOAD_V(_ArgsLeft);
	LOAD_V(_CmdArgBufferIndex);
	LOAD_V(_bCommandReset);
	LOAD_V(_bCommandDone);
	LOAD_V(_bDriveBusy);
	LOAD_V(_bError);
	LOAD_V(_ResetDelayCount);
	LOAD_V(_CheckCountAfterRead);
	LOAD_V(_AudioTrack);
	LOAD_V(_ClockCount);
	LOAD_V(_bCdromInit);

	LOAD_A(_Port);
	LOAD_A(_ReadBuffer);
	LOAD_A(_CmdArgBuffer);
	
	return TRUE;
}

#undef SAVE_V
#undef SAVE_A
#undef LOAD_V
#undef LOAD_A
