/******************************************************************************
	[CPU.c]

		Implements a HuC6280 Emulator.

	Copyright (C) 2004-2005 Ki

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
******************************************************************************/
#include "CPU.h"
#include "Timer.h"

#ifdef DEBUGBUILD
	#include <stdio.h>				// debug
	#include "CDROM.h"				// debug
	#include "DisAsm.h"				// debug
	#include "DisasmWindow.h"		// debug

	//	oOB
	#define			OUTPUT_START		0
	static Uint32	CPUDEBUG_STARTCOUNTER = 0;
#endif


//#define VERBOSE

static Uint8		_A;
static Uint8		_X;
static Uint8		_Y;
static Uint8		_S;
static Uint8		_P;
static Uint16		_PC;

static Uint32		_MPR[8];

static Sint32		_ClockElapsed;

static Sint32		_ClockCount;

static Uint32		_ClockUpCount;

static Uint8		_CF;
static Uint8		_ZF;
static Uint8		_IF;
static Uint8		_DF;
static Uint8		_BF;
static Uint8		_TF;
static Uint8		_VF;
static Uint8		_NF;

static Uint8*		_pZpRam;


// ݓ 
static BOOL			_bRDY;		// DMA v 
static BOOL			_bIRQ1;
static BOOL			_bIRQ2;
static BOOL			_bTIRQ;
static BOOL			_bNMI;


// ǂݏo֐ 
static Uint8 (*Read)(Uint32 addr);

// ݊֐ 
static void (*Write)(Uint32 addr, Uint8 data);


static const Sint32	_CycleTable[256] =
{
	8,7,3, 4,6,4,6,7,3,2,2,2,7,5,7,6,
	2,7,7, 4,6,4,6,7,2,5,2,2,7,5,7,6,
	7,7,3, 4,4,4,6,7,3,2,2,2,5,5,7,6,
	2,7,7, 2,4,4,6,7,2,5,2,2,5,5,7,6,
	7,7,3, 4,8,4,6,7,3,2,2,2,4,5,7,6,
	2,7,7, 5,2,4,6,7,2,5,3,2,2,5,7,6,
	7,7,2, 2,4,4,6,7,3,2,2,2,7,5,7,6,
	2,7,7,17,4,4,6,7,2,5,3,2,7,5,7,6,
	4,7,2, 7,4,4,4,7,2,2,2,2,5,5,5,6,
	2,7,7, 8,4,4,4,7,2,5,2,2,5,5,5,6,
	2,7,2, 7,4,4,4,7,2,2,2,2,5,5,5,6,
	2,7,7, 8,4,4,4,7,2,5,2,2,5,5,5,6,
	2,7,2,17,4,4,6,7,2,2,2,2,5,5,7,6,
	2,7,7,17,2,4,6,7,2,5,3,2,2,5,7,6,
	2,7,2,17,4,4,6,7,2,2,2,2,5,5,7,6,
	2,7,7,17,2,4,6,7,2,5,3,2,2,5,7,2
};


static Uint16		_BP16;
static Uint32		_BP21;
static BOOL			_bBP16Valid;
static BOOL			_bBP21Valid;
static BOOL			_bAtBreakPoint;

static Uint16		_WP16;
static Uint32		_WP21;
static BOOL			_bWP16Valid;
static BOOL			_bWP21Valid;
static BOOL			_bAtWatchPoint;


#ifdef DEBUGBUILD
// fobOp 
/*-----------------------------------------------------------------------------
** Implement memory read/write stages
**---------------------------------------------------------------------------*/
static inline Uint8		read(const Uint16 address)
{
	Uint32	addr21 = _MPR[address >> 13] + (address & 0x1fff);

	if (_bWP16Valid)
	{
		if (address == _WP16)	_bAtWatchPoint = TRUE;
	}
	if (_bWP21Valid)
	{
		if (addr21 == _WP21)	_bAtWatchPoint = TRUE;
	}
	return Read(addr21);
}

static inline Uint8		readZp(const Uint8 addressZp)
{
	if (_bWP16Valid)
	{
		if (addressZp == _WP16)			_bAtWatchPoint = TRUE;
	}
	if (_bWP21Valid)
	{
		if (0x1f0000 + addressZp == _WP21)	_bAtWatchPoint = TRUE;
	}
	return Read(0x1f0000 + addressZp);
}

static inline Uint8		readStack(const Uint8 stackAddress)
{
	if (_bWP16Valid)
	{
		if (stackAddress == _WP16)				_bAtWatchPoint = TRUE;
	}
	if (_bWP21Valid)
	{
		if (0x1f0100 + stackAddress == _WP21)	_bAtWatchPoint = TRUE;
	}
	return Read(0x1f0100 + stackAddress);
}


static inline void		write(const Uint16 address, const Uint8 data)
{
	Uint32		addr21 = _MPR[address >> 13] + (address & 0x1fff);

	if (_bWP16Valid)
	{
		if (address == _WP16)	_bAtWatchPoint = TRUE;
	}
	if (_bWP21Valid)
	{
		if (addr21 == _WP21)	_bAtWatchPoint = TRUE;
	}

	Write(addr21, data);
}

static inline void		writeZp(const Uint8 addressZp, const Uint8 data)
{
	if (_bWP16Valid)
	{
		if (addressZp == _WP16)			_bAtWatchPoint = TRUE;
	}
	if (_bWP21Valid)
	{
		if (0x1f0000 + addressZp == _WP21)	_bAtWatchPoint = TRUE;
	}
	Write(0x1f0000 + addressZp, data);
}

static inline void		writeStack(const Uint8 stackAddress, const Uint8 data)
{
	if (_bWP16Valid)
	{
		if (stackAddress == _WP16)				_bAtWatchPoint = TRUE;
	}
	if (_bWP21Valid)
	{
		if (0x1f0100 + stackAddress == _WP21)	_bAtWatchPoint = TRUE;
	}
	Write(0x1f0100 + stackAddress, data);
}
#else
/*-----------------------------------------------------------------------------
** Implement memory read/write stages
**---------------------------------------------------------------------------*/
static inline Uint8		read(const Uint16 address)
{
	return Read(_MPR[address >> 13] + (address & 0x1fff));
}

static inline Uint8		readZp(const Uint8 addressZp)
{
	return _pZpRam[addressZp];
}

static inline Uint8		readStack(const Uint8 stackAddress)
{
	return _pZpRam[0x100+stackAddress];
}


static inline void		write(const Uint16 address, const Uint8 data)
{
	Write(_MPR[address >> 13] + (address & 0x1fff), data);
}

static inline void		writeZp(const Uint8 addressZp, const Uint8 data)
{
	_pZpRam[addressZp] = data;
}

static inline void		writeStack(const Uint8 stackAddress, const Uint8 data)
{
	_pZpRam[0x100+stackAddress] = data;
}
#endif


/*-----------------------------------------------------------------------------
** Implement op-code fetch and addressing modes
**---------------------------------------------------------------------------*/
static inline Uint8		fetchOpcode()
{
	return read(_PC++);
}

static inline Uint8		fetchNextOperand()
{
	return read(_PC++);
}

static inline Sint8		fetchRelative()
{
	return (Sint8)read(_PC++);
}


/*-----------------------------------------------------------------------------
** IMM
*/
static inline Uint8		fetchImmediate()
{
	return fetchNextOperand();
}


/*-----------------------------------------------------------------------------
** $ZP
*/
static inline Uint8		fetchZp()
{
	return fetchNextOperand();
}


/*-----------------------------------------------------------------------------
** $ZP,X
*/
static inline Uint8		fetchZpX()
{
	return fetchNextOperand() + _X;
}


/*-----------------------------------------------------------------------------
** ($ZP,Y)
*/
static inline Uint8		fetchZpY()
{
	return fetchNextOperand() + _Y;
}


/*-----------------------------------------------------------------------------
** ($ZP)
*/
static inline Uint16	fetchZpIndirect()
{
	Uint8		zpAddress = fetchNextOperand();
	Uint16		effectiveAddress = readZp(zpAddress++);
	effectiveAddress |= readZp(zpAddress) << 8;

	return effectiveAddress;
}


/*-----------------------------------------------------------------------------
** ($ZP,X)
*/
static inline Uint16	fetchZpIndexIndirect()
{
	Uint8		zpAddress = fetchNextOperand() + _X;
	Uint16		effectiveAddress = readZp(zpAddress++);
	effectiveAddress |= readZp(zpAddress) << 8;

	return effectiveAddress;
}


/*-----------------------------------------------------------------------------
** ($ZP),Y
*/
static inline Uint16	fetchZpIndirectIndex()
{
	Uint8		zpAddress = fetchNextOperand();
	Uint16		effectiveAddress = readZp(zpAddress++);
	effectiveAddress |= readZp(zpAddress) << 8;

	return effectiveAddress + _Y;
}


/*-----------------------------------------------------------------------------
** $ABS
*/
static inline Uint16	fetchAbs()
{
	Uint16		effectiveAddress = fetchNextOperand();
	effectiveAddress |= fetchNextOperand() << 8;
	return effectiveAddress;
}


/*-----------------------------------------------------------------------------
** $ABS,X
*/
static inline Uint16	fetchAbsX()
{
	Uint16		effectiveAddress = fetchNextOperand();
	effectiveAddress |= fetchNextOperand() << 8;
	return effectiveAddress + _X;
}


/*-----------------------------------------------------------------------------
** $ABS,Y
*/
static inline Uint16	fetchAbsY()
{
	Uint16		effectiveAddress = fetchNextOperand();
	effectiveAddress |= fetchNextOperand() << 8;
	return effectiveAddress + _Y;
}


/*-----------------------------------------------------------------------------
** ($ABS)
*/
static inline Uint16	fetchAbsIndirect()
{
	Uint16 absAddr = fetchAbs();

	return read(absAddr) | (read(absAddr + 1) << 8);
}


/*-----------------------------------------------------------------------------
** ($ABS,X)
*/
static inline Uint16	fetchAbsIndirectX()
{
	Uint16 absAddr = fetchAbs() + _X;

	return (read(absAddr) | (read(absAddr + 1) << 8));
}


/*-----------------------------------------------------------------------------
** Implement stack operations
**---------------------------------------------------------------------------*/
static inline void		push(Uint8		reg8)
{
	writeStack(_S--, reg8);
}


static inline Uint8 pull()
{
	return readStack(++_S);
}


/*-----------------------------------------------------------------------------
** Implement flag operations
** The idea is taken from nes6502.c by Matthew Conte.
**
** CF --- use CPU_CF (D0)
** ZF --- use 8-bit value
** IF --- use CPU_IF (D2)
** DF --- use CPU_DF (D3)
** BF --- use CPU_BF (D4)
** TF --- use CPU_TF (D5)
** VF --- use CPU_VF (D6)
** NF --- use 8-bit value
**---------------------------------------------------------------------------*/
static inline void 		updateFlagZN(Uint8	val)
{
	_NF = _ZF = val;
}

static inline void		separateFlags(Uint8 p)
{
	_CF = p & CPU_CF;
	_ZF = ~(p >> 1) & 1;
	_IF = p & CPU_IF;
	_DF = p & CPU_DF;
	_BF = p & CPU_BF;
	_TF = p & CPU_TF;
	_VF = p;
	_NF = p;
}


static inline Uint8		gatherFlags()
{
	return _CF | ((_ZF==0)<<1) | _IF | _DF | _BF | _TF | (_VF&CPU_VF) | (_NF&CPU_NF);
}


/*-----------------------------------------------------------------------------
** Implement relaive branch
**---------------------------------------------------------------------------*/
static inline void relativeBranch(Sint8		rel)
{
	_PC += (Sint16)rel;
}


/*-----------------------------------------------------------------------------
** Implement instruction execute stages
**---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
** DEC
*/
static inline Uint8		dec(Uint8		val)
{
	updateFlagZN(--val);
	return val;
}


/*-----------------------------------------------------------------------------
** INC
*/
static inline Uint8 		inc(Uint8		val)
{
	updateFlagZN(++val);
	return val;
}


/*-----------------------------------------------------------------------------
** BIT
*/
static inline void		bit(Uint8		val)
{
	_ZF = _A & val;
	_VF = _NF = val;
}


/*-----------------------------------------------------------------------------
** ADC
*/
static inline void		adc(Uint8		val)
{
	if (_DF)
	{
		int lo = (_A & 0x0f) + (val & 0x0f) + _CF;
		int hi = (_A & 0xf0) + (val & 0xf0);
		if (lo > 0x09)
		{
			hi += 0x10;
			lo += 0x06;
		}
		_VF = (~(_A^val) & (_A^hi) & CPU_NF) >> 1;

		if (hi > 0x90)
			hi += 0x60;

		_CF = (hi & 0x100) >> 8;
		_A = (lo & 0x0f) + (hi & 0xf0);
		++_ClockElapsed;
	}
	else
	{
		Uint32 sum = _A + val + _CF;
		_VF = (~(_A^val) & (_A^sum) & CPU_NF) >> 1;
		_CF = (sum & 0x100) >> 8;
		updateFlagZN(_A = (Uint8)sum);
	}
}


/*-----------------------------------------------------------------------------
** ADC with T-flag set
*/
static inline void		adc_t(Uint8		val)
{
	Uint8 M			= readZp(_X);

	if (_DF)
	{
		int lo = (M & 0x0f) + (val & 0x0f) + _CF;
		int hi = (M & 0xf0) + (val & 0xf0);
		if (lo > 0x09)
		{
			hi += 0x10;
			lo += 0x06;
		}
		_VF = (~(M^val) & (M^hi) & CPU_NF) >> 1;
		if (hi > 0x90)
			hi += 0x60;
		_CF = (hi & 0x100) >> 8;
		M = (lo & 0x0f) + (hi & 0xf0);
		_ClockElapsed += 4;
	}
	else
	{
		Uint32 sum = M + val + _CF;

		_VF = (~(M^val) & (M^sum) & CPU_NF) >> 1;
		_CF = (sum & 0x100) >> 8;
		updateFlagZN(M = (Uint8)sum);
		_ClockElapsed += 3;
	}
	writeZp(_X, M);
}


/*-----------------------------------------------------------------------------
** SBC
*/
static inline void		sbc(Uint8		val)
{
	Uint8 _flagC	= (~_CF) & CPU_CF;		// D0
	Uint32 temp		= _A - val - _flagC;

	if (_DF)
	{
		Uint32 lo = (_A & 0x0F) - (val & 0x0F) - _flagC;
		Uint32 hi = (_A >> 4) - (val >> 4) - ((lo & 0x10) == 0x10);
		if (lo & 0x10)
			lo -= 6;
		if (hi & 0x10)
			hi -= 6;
		_CF = (~hi & 0x10) >> 4;
		_VF = ((_A ^ temp) & (_A ^ val) & CPU_NF) >> 1;
		updateFlagZN(_A = (Uint8)((lo & 0x0F) | (hi << 4)));
		++_ClockElapsed;
	}
	else
	{
		_CF = (~temp & 0x100) >> 8;
		_VF = (Uint8)(((_A ^ temp) & (_A ^ val) & CPU_NF) >> 1);
		updateFlagZN(_A = (Uint8)temp);
	}
}


/*-----------------------------------------------------------------------------
** AND
*/
static inline void		and(Uint8		val)
{
	_A &= val;
	updateFlagZN(_A);
}


/*-----------------------------------------------------------------------------
** AND with T-flag set
*/
static inline void		and_t(Uint8		val)
{
	Uint8 M = readZp(_X) & val;
	updateFlagZN(M);
	writeZp(_X, M);
	_ClockElapsed += 3;
}


/*-----------------------------------------------------------------------------
** ASL
*/
static inline Uint8		asl(Uint8		val)
{
	_CF = (val & CPU_NF) >> 7;
	updateFlagZN(val <<= 1);
	return val;
}


/*-----------------------------------------------------------------------------
** LSR
*/
static inline Uint8		lsr(Uint8		val)
{
	_CF = val & CPU_CF;		// bit 0
	updateFlagZN(val >>= 1);
	return val;
}


/*-----------------------------------------------------------------------------
** ROL
*/
static inline Uint8		rol(Uint8		val)
{
	Uint8 oldFlagC = _CF;			// bit 0
	_CF = (val & CPU_NF) >> 7;
	updateFlagZN(val = (val << 1) | oldFlagC);
	return val;
}


/*-----------------------------------------------------------------------------
** ROR
*/
static inline Uint8		ror(Uint8		val)
{
	Uint8 oldFlagC = _CF << 7;
	_CF = val & CPU_CF;
	updateFlagZN(val = (val >> 1) | oldFlagC);
	return val;
}


/*-----------------------------------------------------------------------------
** CMP
*/
static inline void		cmp(Uint8		val)
{
	Uint32 temp = _A - val;

	updateFlagZN((Uint8)temp);
	_CF = (Uint8)((~temp & 0x100) >> 8);
}


/*-----------------------------------------------------------------------------
** CPX
*/
static inline void		cpx(Uint8		val)
{
	Uint32 temp = _X - val;

	updateFlagZN((Uint8)temp);
	_CF = (Uint8)((~temp & 0x100) >> 8);
}


/*-----------------------------------------------------------------------------
** CPY
*/
static inline void		cpy(Uint8		val)
{
	Uint32 temp = _Y - val;

	updateFlagZN((Uint8)temp);
	_CF = (Uint8)((~temp & 0x100) >> 8);
}


/*-----------------------------------------------------------------------------
** EOR
*/
static inline void		eor(Uint8		val)
{
	_A ^= val;
	updateFlagZN(_A);
}


/*-----------------------------------------------------------------------------
** EOR with T-flag set
*/
static inline void		eor_t(Uint8		val)
{
	Uint8 M = readZp(_X) ^ val;
	updateFlagZN(M);
	writeZp(_X, M);
	_ClockElapsed += 3;
}


/*-----------------------------------------------------------------------------
** ORA
*/
static inline void		ora(Uint8		val)
{
	_A |= val;
	updateFlagZN(_A);
}


/*-----------------------------------------------------------------------------
** ORA with T-flag set
*/
static inline void		ora_t(Uint8		val)
{
	Uint8 M = readZp(_X) | val;
	updateFlagZN(M);
	writeZp(_X, M);
	_ClockElapsed += 3;
}


/*-----------------------------------------------------------------------------
** LDA
*/
static inline void		lda(Uint8		val)
{
	_A = val;
	updateFlagZN(_A);
}


/*-----------------------------------------------------------------------------
** LDX
*/
static inline void		ldx(Uint8		val)
{
	_X = val;
	updateFlagZN(_X);
}


/*-----------------------------------------------------------------------------
** LDY
*/
static inline void		ldy(Uint8		val)
{
	_Y = val;
	updateFlagZN(_Y);
}


/*-----------------------------------------------------------------------------
** TAX
*/
static inline void		tax()
{
	_X = _A;
	updateFlagZN(_X);
}


/*-----------------------------------------------------------------------------
** TAY
*/
static inline void		tay()
{
	_Y = _A;
	updateFlagZN(_Y);
}


/*-----------------------------------------------------------------------------
** TXA
*/
static inline void		txa()
{
	_A = _X;
	updateFlagZN(_A);
}


/*-----------------------------------------------------------------------------
** TYA
*/
static inline void		tya()
{
	_A = _Y;
	updateFlagZN(_A);
}


/*-----------------------------------------------------------------------------
** TSX
*/
static inline void		tsx()
{
	_X = _S;
	updateFlagZN(_X);
}


/*-----------------------------------------------------------------------------
** BRK
*/
static inline void		brk()
{
	++_PC;
	push(_PC >> 8);
	push(_PC & 0xff);

	_BF = CPU_BF;
	push(gatherFlags());

	_IF = CPU_IF;
	_TF = _DF = 0;
	_PC  = read(CPU_IRQ2VECTOR);
	_PC |= read(CPU_IRQ2VECTOR+1) << 8;
}


/*-----------------------------------------------------------------------------
** BBRi (Branch on Bit Reset)
*/
static inline void		BBRi(const Uint8 bit)
{
	Uint8 addr8 = fetchZp();
	Sint8 rel8	= fetchRelative();
	Uint8 ureg8 = readZp(addr8);
	if ((ureg8 & bit) == 0)
	{
		relativeBranch(rel8);
		_ClockElapsed += 2;
	}
}


/*-----------------------------------------------------------------------------
** BBSi (Branch on Bit Set)
*/
static inline void		BBSi(const Uint8 bit)
{
	Uint8 addr8 = fetchZp();
	Sint8 rel8	= fetchRelative();
	Uint8 ureg8 = readZp(addr8);
	if (ureg8 & bit)
	{
		relativeBranch(rel8);
		_ClockElapsed += 2;
	}
}


/*-----------------------------------------------------------------------------
** TRB
*/
static inline Uint8		trb(Uint8		val)
{
	Uint8 M 	 = ~_A & val;
	_VF = _NF = val;
	_ZF = M;
	return M;
}


/*-----------------------------------------------------------------------------
** TSB
*/
static inline Uint8		tsb(Uint8		val)
{
	Uint8 M 	 = _A | val;
	_VF = _NF = val;
	_ZF = M;
	return M;
}


/*-----------------------------------------------------------------------------
** TST
*/
static inline void		tst(Uint8 imm, Uint8 M)
{
	_VF = _NF = M;
	_ZF = M & imm;
}

/*-----------------------------------------------------------------------------
** RMBi (Reset Memory Bit)
*/
static inline void		RMBi(Uint8 zp, Uint8 bit)
{
	writeZp(zp, readZp(zp) & (~bit));
}


/*-----------------------------------------------------------------------------
** SMBi (Set Memory Bit)
*/
static inline void		SMBi(Uint8 zp, Uint8 bit)
{
	writeZp(zp, readZp(zp) | bit);
}


/*-----------------------------------------------------------------------------
** Check for pending IRQ1 / IRQ2 / TIMER / NMI
*/
static inline void		fetchInterrupt()
{
	if (_bNMI)
	{
		push(_PC >> 8);
		push(_PC & 0xff);

		// B tO̓NAꂽɃX^bN֑ޔB
		_BF = 0;
		push(gatherFlags());

		_IF = CPU_IF;
		_TF = _DF = 0;
		_PC  = read(CPU_NMIVECTOR);
		_PC |= read(CPU_NMIVECTOR + 1) << 8;
		_ClockElapsed += 7;
	}
	else if (_IF == 0)
	{
		if (_bTIRQ)
		{
			push(_PC >> 8);
			push(_PC & 0xff);

			_BF = 0;
			push(gatherFlags());

			_IF = CPU_IF;
			_TF = _DF = 0;
			_PC  = read(CPU_TIMERVECTOR);
			_PC |= read(CPU_TIMERVECTOR + 1) << 8;
			_ClockElapsed += 7;

#ifdef VERBOSE
			puts("-------- TIMER INTERRUPT -----------");
#endif
		}
		else if (_bIRQ1)
		{
			push(_PC >> 8);
			push(_PC & 0xff);

			_BF = 0;
			push(gatherFlags());

			_IF = CPU_IF;
			_TF = _DF = 0;
			_PC  = read(CPU_IRQ1VECTOR);
			_PC |= read(CPU_IRQ1VECTOR + 1) << 8;
			_ClockElapsed += 7;

#ifdef VERBOSE
			puts("-------- IRQ1 INTERRUPT -----------");
#endif
		}
		else if (_bIRQ2)
		{
			push(_PC >> 8);
			push(_PC & 0xff);

			_BF = 0;
			push(gatherFlags());

			_IF = CPU_IF;
			_TF = _DF = 0;
			_PC  = read(CPU_IRQ2VECTOR);
			_PC |= read(CPU_IRQ2VECTOR + 1) << 8;
			_ClockElapsed += 7;

#ifdef VERBOSE
			puts("-------- IRQ2 INTERRUPT -----------");
#endif
		}
	}
}


static
inline
void
check_breakpoint()
{
#ifdef DEBUGBUILD
	if (_bBP16Valid)
	{
		if (_PC == _BP16)								_bAtBreakPoint = TRUE;
	}
	if (_bBP21Valid)
	{
		if (_MPR[_PC >> 13] + (_PC & 0x1fff) == _BP21)	_bAtBreakPoint = TRUE;
	}
#endif
}


static
void
check_messages()
{
#ifdef DEBUGBUILD
	char msgbuf[256];
	char msg1[256];
	char msg2[256];
	char msg3[256];
	int byte;
	int sector, min, sec, frame;
	int startaddr, len, endaddr, destaddr, samplerate;

	// If not CDROM, or messaging not enabled, return here
	if (CDROM_IsCDROMEnabled() == FALSE)
		return;

	if (DISASMWND_GetCDROMLog() == FALSE)
		return;


	// Print out information about CDROM system card calls
	//
	// CD_READ:
	if (_PC == 0xE009)
	{
		memset(msgbuf, 0, sizeof(msgbuf));

		sector = (readZp(0xfc) * 65536) + (readZp(0xfd) * 256) + readZp(0xfe);
		min = sector / 4500;
		sec = (sector - (min * 4500)) / 75;
		frame = sector % 75;

		sprintf(msg1, "CD_READ: Sector %6.6X (%2.2d:%2.2d:%2.2d) to ", sector, min, sec, frame);

		switch(readZp(0xff))
		{
			case 1:
				startaddr = (readZp(0xfb) * 256) + readZp(0xfa);
				len = 0x800 * readZp(0xf8);
				endaddr   = startaddr + len - 1;
				sprintf(msg2, "RAM %4.4X-%4.4X (%d sec = %dKB)  LOCAL", startaddr, endaddr, readZp(0xf8), readZp(0xf8)*2);
				break;

			case 0xfe:
				startaddr = (readZp(0xfb) * 256) + readZp(0xfa);
				len = (readZp(0xf9)* 256) + readZp(0xf8);
				endaddr   = startaddr + len - 1;
				sprintf(msg2, "VRAM %4.4X-%4.4X, (%4.4X bytes)", startaddr, endaddr, len);
				break;

			case 0xff:
				startaddr = (readZp(0xfb) * 256) + readZp(0xfa);
				len = 0x800 * readZp(0xf8);
				endaddr   = startaddr + len - 1;
				sprintf(msg2, "VRAM %4.4X-%4.4X (%d sec = %dKB)", startaddr, endaddr, readZp(0xf8), readZp(0xf8)*2);
				break;

			default:
				len = 0x800 * readZp(0xf8) - 1;
				sprintf(msg2, "%2.2X:%4.4X-%2.2X:%4.4X (%d sec = %dKB)  MPR%d",
						readZp(0xfa), 0,
						readZp(0xfa)+(len/0x2000), (len&0x1fff),
						readZp(0xf8), readZp(0xf8)*2, readZp(0xff));
				break;

		}
		strcpy(msgbuf, msg1);
		strcat(msgbuf, msg2);

		puts(msgbuf);
	}
	else if (_PC == 0xE030)
	{
		puts("AD_RESET");
	}
	else if (_PC == 0xE033)
	{
		memset(msgbuf, 0, sizeof(msgbuf));

		sector = (readZp(0xfc) * 65536) + (readZp(0xfd) * 256) + readZp(0xfe);
		min = sector / 4500;
		sec = (sector - (min * 4500)) / 75;
		frame = sector % 75;

		sprintf(msg1, "AD_TRANS: Sector %6.6X (%2.2d:%2.2d:%2.2d) to ADPCM:", sector, min, sec, frame);
		if (readZp(0xff) == 0)
			sprintf(msg2, "%2.2X%2.2X ", readZp(0xfb), readZp(0xfa));
		else
			sprintf(msg2, "<current> ");

		sprintf(msg3, "(%d sec = %dKB)", readZp(0xf8), readZp(0xf8)*2);

		strcpy(msgbuf, msg1);
		strcat(msgbuf, msg2);
		strcat(msgbuf, msg3);

		puts(msgbuf);
	}

	else if (_PC == 0xE036)
	{
		memset(msgbuf, 0, sizeof(msgbuf));

		startaddr = (readZp(0xfd) * 256) + readZp(0xfc);
		destaddr = (readZp(0xfb)* 256) + readZp(0xfa);
		byte = readZp(0xff);

		sprintf(msg1, "AD_TRANS: xfer from ADPCM:%4.4X to ", startaddr);
		
		if (byte == 0)
			sprintf(msg2, "LOCAL:%4.4X", destaddr);
		else if (byte == 0xff)
			sprintf(msg2, "VRAM:%4.4X", destaddr);
		else
			sprintf(msg2, "%2.2X:0000 (MPR%d)", readZp(0xfa), byte);

		strcpy(msgbuf, msg1);
		strcat(msgbuf, msg2);

		puts(msgbuf);
	}

	else if (_PC == 0xE039)
	{
		memset(msgbuf, 0, sizeof(msgbuf));

		startaddr = (readZp(0xfb)* 256) + readZp(0xfa);
		destaddr = (readZp(0xfd) * 256) + readZp(0xfc);
		byte = readZp(0xff);

		sprintf(msg1, "AD_TRANS: xfer from ");
		
		if (byte == 0)
			sprintf(msg2, "LOCAL:%4.4X", startaddr);
		else if (byte == 0xff)
			sprintf(msg2, "VRAM:%4.4X", startaddr);
		else
			sprintf(msg2, "%2.2X:0000 (MPR%d)", readZp(0xfa), byte);

		sprintf(msg1, " to ADPCM:%4.4X", destaddr);
		
		strcpy(msgbuf, msg1);
		strcat(msgbuf, msg2);

		puts(msgbuf);
	}

	else if (_PC == 0xE03C)
	{
		memset(msgbuf, 0, sizeof(msgbuf));

		startaddr = (readZp(0xfb)* 256) + readZp(0xfa);
		len = (readZp(0xf9) * 256) + readZp(0xf8);
		byte = readZp(0xff);
		if (byte > 15) 
			samplerate = 0;
		else
			samplerate = 32000 / (16-byte);

		sprintf(msg1, "AD_PLAY: ADPCM:%4.4X %4.4X bytes @ %dHz (%d seconds) (mode %2.2X)",
				startaddr, len, samplerate,
				(samplerate == 0) ? 0 : (len*2)/samplerate,
				readZp(0xfe));

		strcpy(msgbuf, msg1);

		puts(msgbuf);
	}

	else if (_PC == 0xE03F)
	{
		memset(msgbuf, 0, sizeof(msgbuf));

		sector = (readZp(0xfc) * 65536) + (readZp(0xfd) * 256) + readZp(0xfe);
		min = sector / 4500;
		sec = (sector - (min * 4500)) / 75;
		frame = sector % 75;

		sprintf(msg1, "AD_CPLAY: Sector %6.6X (%2.2d:%2.2d:%2.2d); ", sector, min, sec, frame);

		sector = (readZp(0xf8) * 65536) + (readZp(0xf9) * 256) + readZp(0xfa);
		min = sector / 4500;
		sec = (sector - (min * 4500)) / 75;
		frame = sector % 75;

		byte = readZp(0xff);
		if (byte > 15) 
			samplerate = 0;
		else
			samplerate = 32000 / (16-byte);

		len = sector * 0x800;

		sprintf(msg2, " %6.6X sectors (%2.2d:%2.2d:%2.2d) ", sector, min, sec, frame);
		sprintf(msg3, " @ %dHz (%d seconds)", samplerate, (samplerate == 0) ? 0 : (len*2)/samplerate);

		strcpy(msgbuf, msg1);
		strcat(msgbuf, msg2);
		strcat(msgbuf, msg3);

		puts(msgbuf);
	}

	else if (_PC == 0xE042)
	{
		puts("AD_STOP");
	}

	else if (_PC == 0xE045)
	{
		puts("AD_STAT");
	}
#endif
}


/******************************************************************************
**							   OJ֐
******************************************************************************/


/*-----------------------------------------------------------------------------
** [CPU_SetReadFunction]
**	 oNƂ̃[h֐o^܂B(K{)
**---------------------------------------------------------------------------*/
void CPU_SetReadFunction(Uint8 (*RdFunc)(Uint32))
{
	Read = RdFunc;
#ifdef DEBUGBUILD
	DISASM_SetReadFunction(RdFunc);
#endif
}

/*-----------------------------------------------------------------------------
** [CPU_SetWriteFunction]
**	 oNƂ̃Cg֐o^܂B(K{)
**---------------------------------------------------------------------------*/
void CPU_SetWriteFunction(void (*WrFunc)(Uint32, Uint8))
{
	Write = WrFunc;
}

/*-----------------------------------------------------------------------------
** [CPU_SetZeroPageMemory]
**	 [y[Wւ̃|C^o^܂B(K{)
**---------------------------------------------------------------------------*/
void CPU_SetZeroPageMemory(Uint8* pZpMemory)
{
	_pZpRam = pZpMemory;
}


/*-----------------------------------------------------------------------------
** [Reset]
** botRAZbg܂BNbNԂ܂B
**---------------------------------------------------------------------------*/
Sint32
CPU_Reset()
{
	_IF = CPU_IF;
	_BF = 0;

	_MPR[7] = 0;

	_PC = read(CPU_RESETVECTOR);
	_PC |= read(CPU_RESETVECTOR + 1) << 8;

	_ClockCount = 0;
	_ClockUpCount = 0;

	_ClockElapsed = 6;

	return 6;
}


/*-----------------------------------------------------------------------------
** [CPU_DelayClock]
**	 w̃TCNx܂B 
**---------------------------------------------------------------------------*/
void
CPU_DelayClock(
	Sint32		cycles)
{
	_ClockElapsed += cycles;
}


/*-----------------------------------------------------------------------------
** [GetClockCount]
**	 Zbǧo߃TCNԂ܂B 
**	 RQrbĝ߃I[o[t[܂B
**---------------------------------------------------------------------------*/
Uint32
CPU_GetClockCount()
{
	return _ClockUpCount;
}

/*-----------------------------------------------------------------------------
** [CPU_ActivateRDY]
**	 cl`vqcxANeBuɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_ActivateRDY()
{
	_bRDY = TRUE;
}


/*-----------------------------------------------------------------------------
** [CPU_ActivateIRQ1]
**	 ݓ͐hqpPANeBuɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_ActivateIRQ1()
{
	_bIRQ1 = TRUE;
}


/*-----------------------------------------------------------------------------
** [CPU_ActivateIRQ2]
**	 ݓ͐hqpQANeBuɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_ActivateIRQ2()
{
	_bIRQ2 = TRUE;
}


/*-----------------------------------------------------------------------------
** [CPU_ActivateTIMER]
**	 ^C}[ݗvshldqANeBuɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_ActivateTIMER()
{
	_bTIRQ = TRUE;
}


/*-----------------------------------------------------------------------------
** [CPU_ActivateNMI]
**	 ݓ͐mlhANeBuɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_ActivateNMI()
{
	_bNMI = TRUE;
}


/*-----------------------------------------------------------------------------
** [CPU_InactivateRDY]
**	 cl`vqcx𖳌ɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_InactivateRDY()
{
	_bRDY = FALSE;
}


/*-----------------------------------------------------------------------------
** [CPU_InactivateIRQ1]
**	 ݓ͐hqpP𖳌ɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_InactivateIRQ1()
{
	_bIRQ1 = FALSE;
}


/*-----------------------------------------------------------------------------
** [CPU_InactivateIRQ2]
**	 ݓ͐hqpQ𖳌ɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_InactivateIRQ2()
{
	_bIRQ2 = FALSE;
}


/*-----------------------------------------------------------------------------
** [CPU_InactivateTIMER]
**	 ^C}[ݗvshldq𖳌ɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_InactivateTIMER()
{
	_bTIRQ = FALSE;
}


/*-----------------------------------------------------------------------------
** [CPU_InactivateNMI]
**	 ݓ͐mlh𖳌ɂ܂B
**---------------------------------------------------------------------------*/
void
CPU_InactivateNMI()
{
	_bNMI = FALSE;
}


/*-----------------------------------------------------------------------------
** [CPU_Setxx]
**	 WX^̒lݒ肵܂B 
**---------------------------------------------------------------------------*/
void CPU_SetA(Uint8 A) { _A = A; }
void CPU_SetX(Uint8 X) { _X = X; }
void CPU_SetY(Uint8 Y) { _Y = Y; }
void CPU_SetS(Uint8 S) { _S = S; }
void CPU_SetP(Uint8 P) { _P = P; separateFlags(P); }
void CPU_SetPC(Uint16 PC) { _PC = PC; }

void CPU_SetMPR(Sint32 i, Uint32 mpr) { _MPR[i] = mpr; }


/*-----------------------------------------------------------------------------
** [CPU_Getxx]
**	 WX^̒l擾܂B 
**---------------------------------------------------------------------------*/
Uint8 CPU_GetA() { return _A; }
Uint8 CPU_GetX() { return _X; }
Uint8 CPU_GetY() { return _Y; }
Uint8 CPU_GetS() { return _S; }
Uint8 CPU_GetP() { _P = gatherFlags(); return _P; }
Uint16 CPU_GetPC() { return _PC; }

Uint32 CPU_GetMPR(Sint32 i) { return _MPR[i]; }

// Set / Reset Break Point
void CPU_SetBreakPoint16(Uint16 BP) { _BP16 = BP; }
void CPU_SetBreakPoint21(Uint32 BP) { _BP21 = BP; }
void CPU_ValidateBreakPoint16(BOOL b) { _bBP16Valid = b; }
void CPU_ValidateBreakPoint21(BOOL b) { _bBP21Valid = b; }
BOOL CPU_IsBreakPoint16Valid() { return _bBP16Valid; }
BOOL CPU_IsBreakPoint21Valid() { return _bBP21Valid; }
BOOL CPU_IsBreakPoint()
{
	BOOL	b = _bAtBreakPoint;

	_bAtBreakPoint = FALSE;
	return b;
}

// Get Break Point
Uint16 CPU_GetBreakPoint16() { return _BP16; }
Uint32 CPU_GetBreakPoint21() { return _BP21; }


// Set / Reset Watch Point
void CPU_SetWatchPoint16(Uint16 WP) { _WP16 = WP; }
void CPU_SetWatchPoint21(Uint32 WP) { _WP21 = WP; }
void CPU_ValidateWatchPoint16(BOOL b) { _bWP16Valid = b; }
void CPU_ValidateWatchPoint21(BOOL b) { _bWP21Valid = b; }
BOOL CPU_IsWatchPoint16Valid() { return _bWP16Valid; }
BOOL CPU_IsWatchPoint21Valid() { return _bWP21Valid; }
BOOL CPU_IsWatchPoint()
{
	BOOL	b = _bAtWatchPoint;

	_bAtWatchPoint = FALSE;
	return b;
}

// Get Watch Point
Uint32 CPU_GetWatchPoint21() { return _WP21; }
Uint16 CPU_GetWatchPoint16() { return _WP16; }


#ifdef DEBUGBUILD
static		Uint8	_debug[256];

void
showdasm()
{
	if (CPUDEBUG_STARTCOUNTER++ >= OUTPUT_START)
	{
		DISASM_DisassembleSingleInstruction(_debug, _MPR, _PC);
		printf("%s\t\t\tA=%02X X=%02X Y=%02X S=%02X IF=%02X\n", _debug, _A, _X, _Y, _S, _IF);
	}
}
#endif


#ifdef DEBUGBUILD
/*-----------------------------------------------------------------------------
** [CPU_ExecuteSingleInstruction]
** P߂s܂BNbNԂ܂B
**---------------------------------------------------------------------------*/
Sint32
CPU_ExecuteSingleInstruction()
{
	Uint8	ureg8;
	Uint8	opcode;
	Uint8	addr8;
	Sint8	rel8;
	Uint16	addr16;

	_bAtBreakPoint = 0;
	_bAtWatchPoint = 0;

	opcode = fetchOpcode();

	_ClockElapsed = _CycleTable[opcode];

	switch (opcode)
	{
		/*-------------------------------------------------------------------**
		** ALU instructions
		**-------------------------------------------------------------------*/

		/*---- ADC ----------------------------------------------------------*/
		case CPU_INST_ADC_IMM:
			ureg8 = fetchImmediate();
			if (_TF)	adc_t(ureg8);
			else		adc(ureg8);
			break;

		case CPU_INST_ADC_ZP:
			ureg8 = fetchZp();
			ureg8 = readZp(ureg8);
			if (_TF)
				adc_t(ureg8);
			else
				adc(ureg8);
			break;

		case CPU_INST_ADC_ZP_X:
			ureg8 = fetchZpX();
			ureg8 = readZp(ureg8);
			if (_TF)
				adc_t(ureg8);
			else
				adc(ureg8);
			break;

		case CPU_INST_ADC_IND:
			addr16 = fetchZpIndirect();
			ureg8 = read(addr16);
			if (_TF)
				adc_t(ureg8);
			else
				adc(ureg8);
			break;

		case CPU_INST_ADC_IND_X:
			addr16 = fetchZpIndexIndirect();
			ureg8 = read(addr16);
			if (_TF)
				adc_t(ureg8);
			else
				adc(ureg8);
			break;

		case CPU_INST_ADC_IND_Y:
			addr16 = fetchZpIndirectIndex();
			ureg8 = read(addr16);
			if (_TF)
				adc_t(ureg8);
			else
				adc(ureg8);
			break;

		case CPU_INST_ADC_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			if (_TF)
				adc_t(ureg8);
			else
				adc(ureg8);
			break;

		case CPU_INST_ADC_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			if (_TF)
				adc_t(ureg8);
			else
				adc(ureg8);
			break;

		case CPU_INST_ADC_ABS_Y:
			addr16 = fetchAbsY();
			ureg8 = read(addr16);
			if (_TF)
				adc_t(ureg8);
			else
				adc(ureg8);
			break;

		/*---- SBC ----------------------------------------------------------*/
		case CPU_INST_SBC_IMM:
			ureg8 = fetchImmediate();
			sbc(ureg8);
			break;

		case CPU_INST_SBC_ZP:
			ureg8 = fetchZp();
			ureg8 = readZp(ureg8);
			sbc(ureg8);
			break;

		case CPU_INST_SBC_ZP_X:
			ureg8 = fetchZpX();
			ureg8 = readZp(ureg8);
			sbc(ureg8);
			break;

		case CPU_INST_SBC_IND:
			addr16 = fetchZpIndirect();
			ureg8 = read(addr16);
			sbc(ureg8);
			break;

		case CPU_INST_SBC_IND_X:
			addr16 = fetchZpIndexIndirect();
			ureg8 = read(addr16);
			sbc(ureg8);
			break;

		case CPU_INST_SBC_IND_Y:
			addr16 = fetchZpIndirectIndex();
			ureg8 = read(addr16);
			sbc(ureg8);
			break;

		case CPU_INST_SBC_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			sbc(ureg8);
			break;

		case CPU_INST_SBC_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			sbc(ureg8);
			break;

		case CPU_INST_SBC_ABS_Y:
			addr16 = fetchAbsY();
			ureg8 = read(addr16);
			sbc(ureg8);
			break;

		/*---- AND ----------------------------------------------------------*/
		case CPU_INST_AND_IMM:
			ureg8 = fetchImmediate();
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		case CPU_INST_AND_ZP:
			ureg8 = fetchZp();
			ureg8 = readZp(ureg8);
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		case CPU_INST_AND_ZP_X:
			ureg8 = fetchZpX();
			ureg8 = readZp(ureg8);
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		case CPU_INST_AND_IND:
			addr16 = fetchZpIndirect();
			ureg8 = read(addr16);
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		case CPU_INST_AND_IND_X:
			addr16 = fetchZpIndexIndirect();
			ureg8 = read(addr16);
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		case CPU_INST_AND_IND_Y:
			addr16 = fetchZpIndirectIndex();
			ureg8 = read(addr16);
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		case CPU_INST_AND_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		case CPU_INST_AND_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		case CPU_INST_AND_ABS_Y:
			addr16 = fetchAbsY();
			ureg8 = read(addr16);
			if (_TF)
				and_t(ureg8);
			else
				and(ureg8);
			break;

		/*---- EOR ----------------------------------------------------------*/
		case CPU_INST_EOR_IMM:
			ureg8 = fetchImmediate();
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		case CPU_INST_EOR_ZP:
			ureg8 = fetchZp();
			ureg8 = readZp(ureg8);
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		case CPU_INST_EOR_ZP_X:
			ureg8 = fetchZpX();
			ureg8 = readZp(ureg8);
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		case CPU_INST_EOR_IND:
			addr16 = fetchZpIndirect();
			ureg8 = read(addr16);
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		case CPU_INST_EOR_IND_X:
			addr16 = fetchZpIndexIndirect();
			ureg8 = read(addr16);
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		case CPU_INST_EOR_IND_Y:
			addr16 = fetchZpIndirectIndex();
			ureg8 = read(addr16);
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		case CPU_INST_EOR_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		case CPU_INST_EOR_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		case CPU_INST_EOR_ABS_Y:
			addr16 = fetchAbsY();
			ureg8 = read(addr16);
			if (_TF)
				eor_t(ureg8);
			else
				eor(ureg8);
			break;

		/*---- ORA ----------------------------------------------------------*/
		case CPU_INST_ORA_IMM:
			ureg8 = fetchImmediate();
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		case CPU_INST_ORA_ZP:
			ureg8 = fetchZp();
			ureg8 = readZp(ureg8);
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		case CPU_INST_ORA_ZP_X:
			ureg8 = fetchZpX();
			ureg8 = readZp(ureg8);
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		case CPU_INST_ORA_IND:
			addr16 = fetchZpIndirect();
			ureg8 = read(addr16);
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		case CPU_INST_ORA_IND_X:
			addr16 = fetchZpIndexIndirect();
			ureg8 = read(addr16);
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		case CPU_INST_ORA_IND_Y:
			addr16 = fetchZpIndirectIndex();
			ureg8 = read(addr16);
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		case CPU_INST_ORA_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		case CPU_INST_ORA_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		case CPU_INST_ORA_ABS_Y:
			addr16 = fetchAbsY();
			ureg8 = read(addr16);
			if (_TF)
				ora_t(ureg8);
			else
				ora(ureg8);
			break;

		/*---- ASL ----------------------------------------------------------*/
		case CPU_INST_ASL_ACCUM:
			_A = asl(_A);
			break;

		case CPU_INST_ASL_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			ureg8 = asl(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_ASL_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			ureg8 = asl(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_ASL_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			ureg8 = asl(ureg8);
			write(addr16, ureg8);
			break;
		
		case CPU_INST_ASL_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			ureg8 = asl(ureg8);
			write(addr16, ureg8);
			break;

		/*---- LSR ----------------------------------------------------------*/
		case CPU_INST_LSR_ACCUM:
			_A = lsr(_A);
			break;

		case CPU_INST_LSR_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			ureg8 = lsr(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_LSR_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			ureg8 = lsr(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_LSR_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			ureg8 = lsr(ureg8);
			write(addr16, ureg8);
			break;
		
		case CPU_INST_LSR_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			ureg8 = lsr(ureg8);
			write(addr16, ureg8);
			break;

		/*---- CMP ----------------------------------------------------------*/
		case CPU_INST_CMP_IMM:
			ureg8 = fetchImmediate();
			cmp(ureg8);
			break;

		case CPU_INST_CMP_ZP:
			ureg8 = fetchZp();
			ureg8 = readZp(ureg8);
			cmp(ureg8);
			break;

		case CPU_INST_CMP_ZP_X:
			ureg8 = fetchZpX();
			ureg8 = readZp(ureg8);
			cmp(ureg8);
			break;

		case CPU_INST_CMP_IND:
			addr16 = fetchZpIndirect();
			ureg8 = read(addr16);
			cmp(ureg8);
			break;

		case CPU_INST_CMP_IND_X:
			addr16 = fetchZpIndexIndirect();
			ureg8 = read(addr16);
			cmp(ureg8);
			break;

		case CPU_INST_CMP_IND_Y:
			addr16 = fetchZpIndirectIndex();
			ureg8 = read(addr16);
			cmp(ureg8);
			break;

		case CPU_INST_CMP_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			cmp(ureg8);
			break;

		case CPU_INST_CMP_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			cmp(ureg8);
			break;

		case CPU_INST_CMP_ABS_Y:
			addr16 = fetchAbsY();
			ureg8 = read(addr16);
			cmp(ureg8);
			break;

		/*---- CPX ----------------------------------------------------------*/
		case CPU_INST_CPX_IMM:
			ureg8 = fetchImmediate();
			cpx(ureg8);
			break;

		case CPU_INST_CPX_ZP:
			ureg8 = fetchZp();
			ureg8 = readZp(ureg8);
			cpx(ureg8);
			break;
		
		case CPU_INST_CPX_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			cpx(ureg8);
			break;

		/*---- CPY ----------------------------------------------------------*/
		case CPU_INST_CPY_IMM:
			ureg8 = fetchImmediate();
			cpy(ureg8);
			break;

		case CPU_INST_CPY_ZP:
			ureg8 = fetchZp();
			ureg8 = readZp(ureg8);
			cpy(ureg8);
			break;
		
		case CPU_INST_CPY_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			cpy(ureg8);
			break;

		/*---- DEC ----------------------------------------------------------*/
		case CPU_INST_DEC_ACCUM:
			_A = dec(_A);
			break;

		case CPU_INST_DEC_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			ureg8 = dec(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_DEC_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			ureg8 = dec(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_DEC_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			ureg8 = dec(ureg8);
			write(addr16, ureg8);
			break;

		case CPU_INST_DEC_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			ureg8 = dec(ureg8);
			write(addr16, ureg8);
			break;

		/*---- DEX ----------------------------------------------------------*/
		case CPU_INST_DEX:
			_X = dec(_X);
			break;
				
		/*---- DEY ----------------------------------------------------------*/
		case CPU_INST_DEY:
			_Y = dec(_Y);
			break;
				
		/*---- INC ----------------------------------------------------------*/
		case CPU_INST_INC_ACCUM:
			_A = inc(_A);
			break;

		case CPU_INST_INC_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			ureg8 = inc(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_INC_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			ureg8 = inc(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_INC_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			ureg8 = inc(ureg8);
			write(addr16, ureg8);
			break;

		case CPU_INST_INC_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			ureg8 = inc(ureg8);
			write(addr16, ureg8);
			break;

		/*---- INX ----------------------------------------------------------*/
		case CPU_INST_INX:
			_X = inc(_X);
			break;
				
		/*---- INY ----------------------------------------------------------*/
		case CPU_INST_INY:
			_Y = inc(_Y);
			break;
				
		/*---- ROL ----------------------------------------------------------*/
		case CPU_INST_ROL_ACCUM:
			_A = rol(_A);
			break;

		case CPU_INST_ROL_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			ureg8 = rol(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_ROL_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			ureg8 = rol(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_ROL_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			ureg8 = rol(ureg8);
			write(addr16, ureg8);
			break;
		
		case CPU_INST_ROL_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			ureg8 = rol(ureg8);
			write(addr16, ureg8);
			break;

		/*---- ROR ----------------------------------------------------------*/
		case CPU_INST_ROR_ACCUM:
			_A = ror(_A);
			break;

		case CPU_INST_ROR_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			ureg8 = ror(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_ROR_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			ureg8 = ror(ureg8);
			writeZp(addr8, ureg8);
			break;

		case CPU_INST_ROR_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			ureg8 = ror(ureg8);
			write(addr16, ureg8);
			break;
		
		case CPU_INST_ROR_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			ureg8 = ror(ureg8);
			write(addr16, ureg8);
			break;

		/*---- CLA ----------------------------------------------------------*/
		case CPU_INST_CLA:
			_A = 0;
			break;

		/*---- CLX ----------------------------------------------------------*/
		case CPU_INST_CLX:
			_X = 0;
			break;

		/*---- CLY ----------------------------------------------------------*/
		case CPU_INST_CLY:
			_Y = 0;
			break;

		/*-------------------------------------------------------------------**
		** flag instructions
		**-------------------------------------------------------------------*/
		/*---- CLC ----------------------------------------------------------*/
		case CPU_INST_CLC:
			_CF = 0;
			break;

		/*---- CLD ----------------------------------------------------------*/
		case CPU_INST_CLD:
			_DF = 0;
			break;

		/*---- CLI ----------------------------------------------------------*/
		case CPU_INST_CLI:
			// I tONA IRQ ̃tFb`sȂȂ 
			_IF = _TF = 0;
			TIMER_AdvanceClock(_ClockElapsed);
			_ClockUpCount += _ClockElapsed;
			check_breakpoint();
			return _ClockElapsed;

		/*---- CLV ----------------------------------------------------------*/
		case CPU_INST_CLV:
			_VF = 0;
			break;

		/*---- SEC ----------------------------------------------------------*/
		case CPU_INST_SEC:
			_CF = CPU_CF;
			break;

		/*---- SED ----------------------------------------------------------*/
		case CPU_INST_SED:
			_DF = CPU_DF;
			break;

		/*---- SEI ----------------------------------------------------------*/
		case CPU_INST_SEI:
			//if (_IF == 0)
			//{
			//	fetchInterrupt();	// Ɋ荞݂ȂƂȂ 
			//	_IF = CPU_IF;
			//}
			//TIMER_AdvanceClock(_ClockElapsed);
			//_ClockUpCount += _ClockElapsed;
			//check_breakpoint();
			//return _ClockElapsed;
			_IF = CPU_IF;
			break;

		/*---- SET ----------------------------------------------------------*/
		case CPU_INST_SET:
			_TF = CPU_TF;
			return _ClockElapsed;

		/*-------------------------------------------------------------------**
		** data transfer instructions
		**-------------------------------------------------------------------*/
		/*---- LDA ----------------------------------------------------------*/
		case CPU_INST_LDA_IMM:
			ureg8 = fetchImmediate();
			lda(ureg8);
			break;

		case CPU_INST_LDA_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			lda(ureg8);
			break;

		case CPU_INST_LDA_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			lda(ureg8);
			break;

		case CPU_INST_LDA_IND:
			addr16 = fetchZpIndirect();
			ureg8 = read(addr16);
			lda(ureg8);
			break;

		case CPU_INST_LDA_IND_X:
			addr16 = fetchZpIndexIndirect();
			ureg8 = read(addr16);
			lda(ureg8);
			break;

		case CPU_INST_LDA_IND_Y:
			addr16 = fetchZpIndirectIndex();
			ureg8 = read(addr16);
			lda(ureg8);
			break;

		case CPU_INST_LDA_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			lda(ureg8);
			break;

		case CPU_INST_LDA_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			lda(ureg8);
			break;

		case CPU_INST_LDA_ABS_Y:
			addr16 = fetchAbsY();
			ureg8 = read(addr16);
			lda(ureg8);
			break;

		/*---- LDX ----------------------------------------------------------*/
		case CPU_INST_LDX_IMM:
			ureg8 = fetchImmediate();
			ldx(ureg8);
			break;

		case CPU_INST_LDX_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			ldx(ureg8);
			break;

		case CPU_INST_LDX_ZP_Y:
			addr8 = fetchZpY();
			ureg8 = readZp(addr8);
			ldx(ureg8);
			break;

		case CPU_INST_LDX_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			ldx(ureg8);
			break;

		case CPU_INST_LDX_ABS_Y:
			addr16 = fetchAbsY();
			ureg8 = read(addr16);
			ldx(ureg8);
			break;

		/*---- LDY ----------------------------------------------------------*/
		case CPU_INST_LDY_IMM:
			ureg8 = fetchImmediate();
			ldy(ureg8);
			break;

		case CPU_INST_LDY_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			ldy(ureg8);
			break;

		case CPU_INST_LDY_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			ldy(ureg8);
			break;

		case CPU_INST_LDY_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			ldy(ureg8);
			break;

		case CPU_INST_LDY_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			ldy(ureg8);
			break;

		/*---- STA ----------------------------------------------------------*/
		case CPU_INST_STA_ZP:
			addr8 = fetchZp();
			writeZp(addr8, _A);
			break;

		case CPU_INST_STA_ZP_X:
			addr8 = fetchZpX();
			writeZp(addr8, _A);
			break;

		case CPU_INST_STA_IND:
			addr16 = fetchZpIndirect();
			write(addr16, _A);
			break;

		case CPU_INST_STA_IND_X:
			addr16 = fetchZpIndexIndirect();
			write(addr16, _A);
			break;

		case CPU_INST_STA_IND_Y:
			addr16 = fetchZpIndirectIndex();
			write(addr16, _A);
			break;

		case CPU_INST_STA_ABS:
			addr16 = fetchAbs();
			write(addr16, _A);
			break;

		case CPU_INST_STA_ABS_X:
			addr16 = fetchAbsX();
			write(addr16, _A);
			break;

		case CPU_INST_STA_ABS_Y:
			addr16 = fetchAbsY();
			write(addr16, _A);
			break;

		/*---- STX ----------------------------------------------------------*/
		case CPU_INST_STX_ZP:
			addr8 = fetchZp();
			writeZp(addr8, _X);
			break;

		case CPU_INST_STX_ZP_Y:
			addr8 = fetchZpY();
			writeZp(addr8, _X);
			break;

		case CPU_INST_STX_ABS:
			addr16 = fetchAbs();
			write(addr16, _X);
			break;

		/*---- STY ----------------------------------------------------------*/
		case CPU_INST_STY_ZP:
			addr8 = fetchZp();
			writeZp(addr8, _Y);
			break;

		case CPU_INST_STY_ZP_X:
			addr8 = fetchZpX();
			writeZp(addr8, _Y);
			break;

		case CPU_INST_STY_ABS:
			addr16 = fetchAbs();
			write(addr16, _Y);
			break;

		/*---- SAX ----------------------------------------------------------*/
		case CPU_INST_SAX:
			ureg8 = _A;
			_A = _X;
			_X = ureg8;
			break;

		/*---- SAY ----------------------------------------------------------*/
		case CPU_INST_SAY:
			ureg8 = _A;
			_A = _Y;
			_Y = ureg8;
			break;

		/*---- SXY ----------------------------------------------------------*/
		case CPU_INST_SXY:
			ureg8 = _X;
			_X = _Y;
			_Y = ureg8;
			break;

		/*---- ST0 ----------------------------------------------------------*/
		case CPU_INST_ST0_IMM:
			ureg8 = fetchImmediate();
			Write(0x1fe000, ureg8);
			break;

		/*---- ST1 ----------------------------------------------------------*/
		case CPU_INST_ST1_IMM:
			ureg8 = fetchImmediate();
			Write(0x1fe002, ureg8);
			break;

		/*---- ST2 ----------------------------------------------------------*/
		case CPU_INST_ST2_IMM:
			ureg8 = fetchImmediate();
			Write(0x1fe003, ureg8);
			break;

		/*---- STZ ----------------------------------------------------------*/
		case CPU_INST_STZ_ZP:
			addr8 = fetchZp();
			writeZp(addr8, 0);
			break;

		case CPU_INST_STZ_ZP_X:
			addr8 = fetchZpX();
			writeZp(addr8, 0);
			break;

		case CPU_INST_STZ_ABS:
			addr16 = fetchAbs();
			write(addr16, 0);
			break;

		case CPU_INST_STZ_ABS_X:
			addr16 = fetchAbsX();
			write(addr16, 0);
			break;

		/*---- TAI ----------------------------------------------------------*/
		case CPU_INST_TAI:
		{
			Uint16	srcAddr = fetchAbs();
			Uint16	dstAddr = fetchAbs();
			Uint16	length	= fetchAbs();

			push(_Y);
			push(_A);
			push(_X);
			do
			{
				write(dstAddr++, read(srcAddr++));
				_ClockElapsed += 6;
				if (--length == 0)
				{
					break;
				}
				write(dstAddr++, read(srcAddr--));
				_ClockElapsed += 6;
			} while (--length);
			_X = pull();
			_A = pull();
			_Y = pull();
		}
		break;

		/*---- TDD ----------------------------------------------------------*/
		case CPU_INST_TDD:
		{
			Uint16	srcAddr = fetchAbs();
			Uint16	dstAddr = fetchAbs();
			Uint16	length	= fetchAbs();

			push(_Y);
			push(_A);
			push(_X);
			do
			{
				write(dstAddr--, read(srcAddr--));
				_ClockElapsed += 6;
			} while (--length);
			_X = pull();
			_A = pull();
			_Y = pull();
		}
		break;

		/*---- TIA ----------------------------------------------------------*/
		case CPU_INST_TIA:
		{
			Uint16	srcAddr = fetchAbs();
			Uint16	dstAddr = fetchAbs();
			Uint16	length	= fetchAbs();

			push(_Y);
			push(_A);
			push(_X);
			do
			{
				write(dstAddr++, read(srcAddr++));
				_ClockElapsed += 6;
				if (--length == 0)
				{
					break;
				}
				write(dstAddr--, read(srcAddr++));
				_ClockElapsed += 6;
			} while (--length);
			_X = pull();
			_A = pull();
			_Y = pull();
		}
		break;

		/*---- TII ----------------------------------------------------------*/
		case CPU_INST_TII:
		{
			Uint16	srcAddr = fetchAbs();
			Uint16	dstAddr = fetchAbs();
			Uint16	length	= fetchAbs();

			push(_Y);
			push(_A);
			push(_X);
			do
			{
				write(dstAddr++, read(srcAddr++));
				_ClockElapsed += 6;
			} while (--length);
			_X = pull();
			_A = pull();
			_Y = pull();
		}
		break;

		/*---- TIN ----------------------------------------------------------*/
		case CPU_INST_TIN:
		{
			Uint16	srcAddr = fetchAbs();
			Uint16	dstAddr = fetchAbs();
			Uint16	length	= fetchAbs();

			push(_Y);
			push(_A);
			push(_X);
			do
			{
				write(dstAddr, read(srcAddr++));
				_ClockElapsed += 6;
			} while (--length);
			_X = pull();
			_A = pull();
			_Y = pull();
		}
		break;

		/*---- TAMi ---------------------------------------------------------*/
		case CPU_INST_TAM:
			ureg8 = fetchImmediate();
			if (ureg8 & 0x01) _MPR[0] = _A << 13;
			if (ureg8 & 0x02) _MPR[1] = _A << 13;
			if (ureg8 & 0x04) _MPR[2] = _A << 13;
			if (ureg8 & 0x08) _MPR[3] = _A << 13;
			if (ureg8 & 0x10) _MPR[4] = _A << 13;
			if (ureg8 & 0x20) _MPR[5] = _A << 13;
			if (ureg8 & 0x40) _MPR[6] = _A << 13;
			if (ureg8 & 0x80) _MPR[7] = _A << 13;
			break;

		/*---- TMAi ---------------------------------------------------------*/
		case CPU_INST_TMA:
			ureg8 = fetchImmediate();
			if (ureg8 & 0x01) { _A = _MPR[0] >> 13; break; }
			if (ureg8 & 0x02) { _A = _MPR[1] >> 13; break; }
			if (ureg8 & 0x04) { _A = _MPR[2] >> 13; break; }
			if (ureg8 & 0x08) { _A = _MPR[3] >> 13; break; }
			if (ureg8 & 0x10) { _A = _MPR[4] >> 13; break; }
			if (ureg8 & 0x20) { _A = _MPR[5] >> 13; break; }
			if (ureg8 & 0x40) { _A = _MPR[6] >> 13; break; }
			if (ureg8 & 0x80) { _A = _MPR[7] >> 13; break; }
			break;

		/*---- TAX ----------------------------------------------------------*/
		case CPU_INST_TAX:
			tax();
			break;

		/*---- TAY ----------------------------------------------------------*/
		case CPU_INST_TAY:
			tay();
			break;

		/*---- TSX ----------------------------------------------------------*/
		case CPU_INST_TSX:
			tsx();
			break;

		/*---- TXA ----------------------------------------------------------*/
		case CPU_INST_TXA:
			txa();
			break;

		/*---- TXS ----------------------------------------------------------*/
		case CPU_INST_TXS:
			_S = _X;
			break;

		/*---- TYA ----------------------------------------------------------*/
		case CPU_INST_TYA:
			tya();
			break;

		/*-------------------------------------------------------------------**
		** branch / jump instructions
		**-------------------------------------------------------------------*/
		/*---- BBRi ---------------------------------------------------------*/
		case CPU_INST_BBR0_ZP_REL:
			BBRi(0x01);
			break;

		case CPU_INST_BBR1_ZP_REL:
			BBRi(0x02);
			break;

		case CPU_INST_BBR2_ZP_REL:
			BBRi(0x04);
			break;

		case CPU_INST_BBR3_ZP_REL:
			BBRi(0x08);
			break;

		case CPU_INST_BBR4_ZP_REL:
			BBRi(0x10);
			break;

		case CPU_INST_BBR5_ZP_REL:
			BBRi(0x20);
			break;

		case CPU_INST_BBR6_ZP_REL:
			BBRi(0x40);
			break;

		case CPU_INST_BBR7_ZP_REL:
			BBRi(0x80);
			break;

		/*---- BBSi ---------------------------------------------------------*/
		case CPU_INST_BBS0_ZP_REL:
			BBSi(0x01);
			break;

		case CPU_INST_BBS1_ZP_REL:
			BBSi(0x02);
			break;

		case CPU_INST_BBS2_ZP_REL:
			BBSi(0x04);
			break;

		case CPU_INST_BBS3_ZP_REL:
			BBSi(0x08);
			break;

		case CPU_INST_BBS4_ZP_REL:
			BBSi(0x10);
			break;

		case CPU_INST_BBS5_ZP_REL:
			BBSi(0x20);
			break;

		case CPU_INST_BBS6_ZP_REL:
			BBSi(0x40);
			break;

		case CPU_INST_BBS7_ZP_REL:
			BBSi(0x80);
			break;

		/*---- BCC ----------------------------------------------------------*/
		case CPU_INST_BCC_REL:
			rel8 = fetchRelative();
			if (_CF == 0)
			{
				relativeBranch(rel8);
				_ClockElapsed += 2;
			}
			break;

		/*---- BNE ----------------------------------------------------------*/
		case CPU_INST_BNE_REL:
			rel8 = fetchRelative();
			if (_ZF)
			{
				relativeBranch(rel8);
				_ClockElapsed += 2;
			}
			break;

		/*---- BVC ----------------------------------------------------------*/
		case CPU_INST_BVC_REL:
			rel8 = fetchRelative();
			if ((_VF & CPU_VF) == 0)
			{
				relativeBranch(rel8);
				_ClockElapsed += 2;
			}
			break;

		/*---- BPL ----------------------------------------------------------*/
		case CPU_INST_BPL_REL:
			rel8 = fetchRelative();
			if ((_NF & CPU_NF) == 0)
			{
				relativeBranch(rel8);
				_ClockElapsed += 2;
			}
			break;

		/*---- BCS ----------------------------------------------------------*/
		case CPU_INST_BCS_REL:
			rel8 = fetchRelative();
			if (_CF)
			{
				relativeBranch(rel8);
				_ClockElapsed += 2;
			}
			break;

		/*---- BEQ ----------------------------------------------------------*/
		case CPU_INST_BEQ_REL:
			rel8 = fetchRelative();
			if (_ZF == 0)
			{
				relativeBranch(rel8);
				_ClockElapsed += 2;
			}
			break;

		/*---- BVS ----------------------------------------------------------*/
		case CPU_INST_BVS_REL:
			rel8 = fetchRelative();
			if (_VF & CPU_VF)
			{
				relativeBranch(rel8);
				_ClockElapsed += 2;
			}
			break;

		/*---- BMI ----------------------------------------------------------*/
		case CPU_INST_BMI_REL:
			rel8 = fetchRelative();
			if (_NF & CPU_NF)
			{
				relativeBranch(rel8);
				_ClockElapsed += 2;
			}
			break;

		/*---- BRA ----------------------------------------------------------*/
		case CPU_INST_BRA_REL:
			relativeBranch(fetchRelative());
			break;

		/*---- JMP ----------------------------------------------------------*/
		case CPU_INST_JMP_ABS:
			_PC = fetchAbs();
			break;

		case CPU_INST_JMP_INDIR:
			_PC = fetchAbsIndirect();
			break;

		case CPU_INST_JMP_INDIRX:
			_PC = fetchAbsIndirectX();
			break;

		/*-------------------------------------------------------------------**
		** subroutine instructions
		**-------------------------------------------------------------------*/
		/*---- BSR ----------------------------------------------------------*/
		case CPU_INST_BSR_REL:
			rel8 = fetchRelative();
			--_PC;
			push(_PC >> 8);
			push(_PC & 0xff);
			++_PC;
			relativeBranch(rel8);
			break;

		/*---- JSR ----------------------------------------------------------*/
		case CPU_INST_JSR_ABS:
			addr16 = fetchAbs();
			--_PC;
			push(_PC >> 8);
			push(_PC & 0xff);
			_PC = addr16;
			break;

		/*---- PHA ----------------------------------------------------------*/
		case CPU_INST_PHA:
			push(_A);
			break;

		/*---- PHP ----------------------------------------------------------*/
		case CPU_INST_PHP:
			_BF = _TF = 0;
			push(gatherFlags());
			break;

		/*---- PHX ----------------------------------------------------------*/
		case CPU_INST_PHX:
			push(_X);
			break;

		/*---- PHY ----------------------------------------------------------*/
		case CPU_INST_PHY:
			push(_Y);
			break;

		/*---- PLA ----------------------------------------------------------*/
		case CPU_INST_PLA:
			_A = pull();
			updateFlagZN(_A);
			break;

		/*---- PLP ----------------------------------------------------------*/
		// PLP ߎśF
		// - T tOZbgȂ 
		// - 荞݃tFb`sȂȂ 
		case CPU_INST_PLP:
			separateFlags(pull());
			TIMER_AdvanceClock(_ClockElapsed);
			_ClockUpCount += _ClockElapsed;
			check_breakpoint();
			return _ClockElapsed;

		/*---- PLX ----------------------------------------------------------*/
		case CPU_INST_PLX:
			_X = pull();
			updateFlagZN(_X);
			break;

		/*---- PLY ----------------------------------------------------------*/
		case CPU_INST_PLY:
			_Y = pull();
			updateFlagZN(_Y);
			break;

		/*---- RTI ----------------------------------------------------------*/
		case CPU_INST_RTI:
			// RTI ߎs
			// - T tOZbgȂ 
			// - 荞݃tFb`ȂȂ 
			separateFlags(pull());
			_PC = pull();
			_PC |= pull() << 8;
			TIMER_AdvanceClock(_ClockElapsed);
			_ClockUpCount += _ClockElapsed;
			check_breakpoint();
			return _ClockElapsed;
		
		/*---- RTS ----------------------------------------------------------*/
		case CPU_INST_RTS:
			_PC = pull();
			_PC |= pull() << 8;
			++_PC;
			break;

		/*-------------------------------------------------------------------**
		** test instructions
		**-------------------------------------------------------------------*/
		/*---- BIT ----------------------------------------------------------*/
		case CPU_INST_BIT_IMM:
			ureg8 = fetchImmediate();
			bit(ureg8);
			break;

		case CPU_INST_BIT_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			bit(ureg8);
			break;

		case CPU_INST_BIT_ZP_X:
			addr8 = fetchZpX();
			ureg8 = readZp(addr8);
			bit(ureg8);
			break;

		case CPU_INST_BIT_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			bit(ureg8);
			break;

		case CPU_INST_BIT_ABS_X:
			addr16 = fetchAbsX();
			ureg8 = read(addr16);
			bit(ureg8);
			break;

		/*---- TRB ----------------------------------------------------------*/
		case CPU_INST_TRB_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			writeZp(addr8, trb(ureg8));
			break;

		case CPU_INST_TRB_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			write(addr16, trb(ureg8));
			break;

		/*---- TSB ----------------------------------------------------------*/
		case CPU_INST_TSB_ZP:
			addr8 = fetchZp();
			ureg8 = readZp(addr8);
			writeZp(addr8, tsb(ureg8));
			break;

		case CPU_INST_TSB_ABS:
			addr16 = fetchAbs();
			ureg8 = read(addr16);
			write(addr16, tsb(ureg8));
			break;

		/*---- TST ----------------------------------------------------------*/
		case CPU_INST_TST_IMM_ZP:
			ureg8 = fetchImmediate();
			addr8 = fetchZp();
			tst(ureg8, readZp(addr8));
			break;

		case CPU_INST_TST_IMM_ZP_X:
			ureg8 = fetchImmediate();
			addr8 = fetchZpX();
			tst(ureg8, readZp(addr8));
			break;

		case CPU_INST_TST_IMM_ABS:
			ureg8 = fetchImmediate();
			addr16 = fetchAbs();
			tst(ureg8, read(addr16));
			break;

		case CPU_INST_TST_IMM_ABS_X:
			ureg8 = fetchImmediate();
			addr16 = fetchAbsX();
			tst(ureg8, read(addr16));
			break;

		/*-------------------------------------------------------------------**
		** control instructions
		**-------------------------------------------------------------------*/
		/*---- CSL ----------------------------------------------------------*/
		case CPU_INST_CSL:
			break;

		/*---- CSH ----------------------------------------------------------*/
		case CPU_INST_CSH:
			break;

		/*---- BRK ----------------------------------------------------------*/
		case CPU_INST_BRK:
			brk();
			break;

		/*---- NOP ----------------------------------------------------------*/
		case CPU_INST_NOP:
			break;

		/*---- RMBi ---------------------------------------------------------*/
		case CPU_INST_RMB0_ZP:
			RMBi(fetchZp(), 0x01);
			break;

		case CPU_INST_RMB1_ZP:
			RMBi(fetchZp(), 0x02);
			break;

		case CPU_INST_RMB2_ZP:
			RMBi(fetchZp(), 0x04);
			break;

		case CPU_INST_RMB3_ZP:
			RMBi(fetchZp(), 0x08);
			break;

		case CPU_INST_RMB4_ZP:
			RMBi(fetchZp(), 0x10);
			break;

		case CPU_INST_RMB5_ZP:
			RMBi(fetchZp(), 0x20);
			break;

		case CPU_INST_RMB6_ZP:
			RMBi(fetchZp(), 0x40);
			break;

		case CPU_INST_RMB7_ZP:
			RMBi(fetchZp(), 0x80);
			break;

		/*---- SMBi ---------------------------------------------------------*/
		case CPU_INST_SMB0_ZP:
			SMBi(fetchZp(), 0x01);
			break;

		case CPU_INST_SMB1_ZP:
			SMBi(fetchZp(), 0x02);
			break;

		case CPU_INST_SMB2_ZP:
			SMBi(fetchZp(), 0x04);
			break;

		case CPU_INST_SMB3_ZP:
			SMBi(fetchZp(), 0x08);
			break;

		case CPU_INST_SMB4_ZP:
			SMBi(fetchZp(), 0x10);
			break;

		case CPU_INST_SMB5_ZP:
			SMBi(fetchZp(), 0x20);
			break;

		case CPU_INST_SMB6_ZP:
			SMBi(fetchZp(), 0x40);
			break;

		case CPU_INST_SMB7_ZP:
			SMBi(fetchZp(), 0x80);
			break;

		/*------------------------------------------------------------------**
		** invalid instructions
		**------------------------------------------------------------------*/
		default:
			break;
	}

	_TF = 0;

	TIMER_AdvanceClock(_ClockElapsed);
	_ClockUpCount += _ClockElapsed;

	fetchInterrupt();
	check_breakpoint();
	check_messages();

	return _ClockElapsed;
}
#endif

/*-----------------------------------------------------------------------------
** [CPU_AdvanceClock]
**	w̃NbNi߂܂B
**---------------------------------------------------------------------------*/
Sint32
CPU_AdvanceClock(
	Sint32	clock)
{
	Uint8	ureg8;
	Uint8	opcode;
	Uint8	addr8;
	Sint8	rel8;
	Uint16	addr16;

	TIMER_AdvanceClock(_ClockElapsed);
	_ClockCount += clock - _ClockElapsed;
	_ClockUpCount += _ClockElapsed;

	while (_ClockCount > 0)
	{

#ifdef VERBOSE
		showdasm();
#endif

		opcode = fetchOpcode();

		_ClockElapsed = _CycleTable[opcode];

		switch (opcode)
		{
			/*-------------------------------------------------------------------**
			** ALU instructions
			**-------------------------------------------------------------------*/

			/*---- ADC ----------------------------------------------------------*/
			case CPU_INST_ADC_IMM:
				ureg8 = fetchImmediate();
				if (_TF)	adc_t(ureg8);
				else		adc(ureg8);
				break;

			case CPU_INST_ADC_ZP:
				ureg8 = fetchZp();
				ureg8 = readZp(ureg8);
				if (_TF)
				{
					adc_t(ureg8);
				}
				else
				{
					adc(ureg8);
				}
				break;

			case CPU_INST_ADC_ZP_X:
				ureg8 = fetchZpX();
				ureg8 = readZp(ureg8);
				if (_TF)
				{
					adc_t(ureg8);
				}
				else
				{
					adc(ureg8);
				}
				break;

			case CPU_INST_ADC_IND:
				addr16 = fetchZpIndirect();
				ureg8 = read(addr16);
				if (_TF)
				{
					adc_t(ureg8);
				}
				else
				{
					adc(ureg8);
				}
				break;

			case CPU_INST_ADC_IND_X:
				addr16 = fetchZpIndexIndirect();
				ureg8 = read(addr16);
				if (_TF)
				{
					adc_t(ureg8);
				}
				else
				{
					adc(ureg8);
				}
				break;

			case CPU_INST_ADC_IND_Y:
				addr16 = fetchZpIndirectIndex();
				ureg8 = read(addr16);
				if (_TF)
				{
					adc_t(ureg8);
				}
				else
				{
					adc(ureg8);
				}
				break;

			case CPU_INST_ADC_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				if (_TF)
				{
					adc_t(ureg8);
				}
				else
				{
					adc(ureg8);
				}
				break;

			case CPU_INST_ADC_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				if (_TF)
				{
					adc_t(ureg8);
				}
				else
				{
					adc(ureg8);
				}
				break;

			case CPU_INST_ADC_ABS_Y:
				addr16 = fetchAbsY();
				ureg8 = read(addr16);
				if (_TF)
				{
					adc_t(ureg8);
				}
				else
				{
					adc(ureg8);
				}
				break;

			/*---- SBC ----------------------------------------------------------*/
			case CPU_INST_SBC_IMM:
				ureg8 = fetchImmediate();
				sbc(ureg8);
				break;

			case CPU_INST_SBC_ZP:
				ureg8 = fetchZp();
				ureg8 = readZp(ureg8);
				sbc(ureg8);
				break;

			case CPU_INST_SBC_ZP_X:
				ureg8 = fetchZpX();
				ureg8 = readZp(ureg8);
				sbc(ureg8);
				break;

			case CPU_INST_SBC_IND:
				addr16 = fetchZpIndirect();
				ureg8 = read(addr16);
				sbc(ureg8);
				break;

			case CPU_INST_SBC_IND_X:
				addr16 = fetchZpIndexIndirect();
				ureg8 = read(addr16);
				sbc(ureg8);
				break;

			case CPU_INST_SBC_IND_Y:
				addr16 = fetchZpIndirectIndex();
				ureg8 = read(addr16);
				sbc(ureg8);
				break;

			case CPU_INST_SBC_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				sbc(ureg8);
				break;

			case CPU_INST_SBC_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				sbc(ureg8);
				break;

			case CPU_INST_SBC_ABS_Y:
				addr16 = fetchAbsY();
				ureg8 = read(addr16);
				sbc(ureg8);
				break;

			/*---- AND ----------------------------------------------------------*/
			case CPU_INST_AND_IMM:
				ureg8 = fetchImmediate();
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			case CPU_INST_AND_ZP:
				ureg8 = fetchZp();
				ureg8 = readZp(ureg8);
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			case CPU_INST_AND_ZP_X:
				ureg8 = fetchZpX();
				ureg8 = readZp(ureg8);
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			case CPU_INST_AND_IND:
				addr16 = fetchZpIndirect();
				ureg8 = read(addr16);
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			case CPU_INST_AND_IND_X:
				addr16 = fetchZpIndexIndirect();
				ureg8 = read(addr16);
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			case CPU_INST_AND_IND_Y:
				addr16 = fetchZpIndirectIndex();
				ureg8 = read(addr16);
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			case CPU_INST_AND_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			case CPU_INST_AND_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			case CPU_INST_AND_ABS_Y:
				addr16 = fetchAbsY();
				ureg8 = read(addr16);
				if (_TF)
				{
					and_t(ureg8);
				}
				else
				{
					and(ureg8);
				}
				break;

			/*---- EOR ----------------------------------------------------------*/
			case CPU_INST_EOR_IMM:
				ureg8 = fetchImmediate();
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			case CPU_INST_EOR_ZP:
				ureg8 = fetchZp();
				ureg8 = readZp(ureg8);
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			case CPU_INST_EOR_ZP_X:
				ureg8 = fetchZpX();
				ureg8 = readZp(ureg8);
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			case CPU_INST_EOR_IND:
				addr16 = fetchZpIndirect();
				ureg8 = read(addr16);
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			case CPU_INST_EOR_IND_X:
				addr16 = fetchZpIndexIndirect();
				ureg8 = read(addr16);
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			case CPU_INST_EOR_IND_Y:
				addr16 = fetchZpIndirectIndex();
				ureg8 = read(addr16);
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			case CPU_INST_EOR_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			case CPU_INST_EOR_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			case CPU_INST_EOR_ABS_Y:
				addr16 = fetchAbsY();
				ureg8 = read(addr16);
				if (_TF)
				{
					eor_t(ureg8);
				}
				else
				{
					eor(ureg8);
				}
				break;

			/*---- ORA ----------------------------------------------------------*/
			case CPU_INST_ORA_IMM:
				ureg8 = fetchImmediate();
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			case CPU_INST_ORA_ZP:
				ureg8 = fetchZp();
				ureg8 = readZp(ureg8);
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			case CPU_INST_ORA_ZP_X:
				ureg8 = fetchZpX();
				ureg8 = readZp(ureg8);
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			case CPU_INST_ORA_IND:
				addr16 = fetchZpIndirect();
				ureg8 = read(addr16);
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			case CPU_INST_ORA_IND_X:
				addr16 = fetchZpIndexIndirect();
				ureg8 = read(addr16);
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			case CPU_INST_ORA_IND_Y:
				addr16 = fetchZpIndirectIndex();
				ureg8 = read(addr16);
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			case CPU_INST_ORA_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			case CPU_INST_ORA_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			case CPU_INST_ORA_ABS_Y:
				addr16 = fetchAbsY();
				ureg8 = read(addr16);
				if (_TF)
				{
					ora_t(ureg8);
				}
				else
				{
					ora(ureg8);
				}
				break;

			/*---- ASL ----------------------------------------------------------*/
			case CPU_INST_ASL_ACCUM:
				_A = asl(_A);
				break;

			case CPU_INST_ASL_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				ureg8 = asl(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_ASL_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				ureg8 = asl(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_ASL_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				ureg8 = asl(ureg8);
				write(addr16, ureg8);
				break;
			
			case CPU_INST_ASL_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				ureg8 = asl(ureg8);
				write(addr16, ureg8);
				break;

			/*---- LSR ----------------------------------------------------------*/
			case CPU_INST_LSR_ACCUM:
				_A = lsr(_A);
				break;

			case CPU_INST_LSR_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				ureg8 = lsr(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_LSR_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				ureg8 = lsr(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_LSR_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				ureg8 = lsr(ureg8);
				write(addr16, ureg8);
				break;
			
			case CPU_INST_LSR_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				ureg8 = lsr(ureg8);
				write(addr16, ureg8);
				break;

			/*---- CMP ----------------------------------------------------------*/
			case CPU_INST_CMP_IMM:
				ureg8 = fetchImmediate();
				cmp(ureg8);
				break;

			case CPU_INST_CMP_ZP:
				ureg8 = fetchZp();
				ureg8 = readZp(ureg8);
				cmp(ureg8);
				break;

			case CPU_INST_CMP_ZP_X:
				ureg8 = fetchZpX();
				ureg8 = readZp(ureg8);
				cmp(ureg8);
				break;

			case CPU_INST_CMP_IND:
				addr16 = fetchZpIndirect();
				ureg8 = read(addr16);
				cmp(ureg8);
				break;

			case CPU_INST_CMP_IND_X:
				addr16 = fetchZpIndexIndirect();
				ureg8 = read(addr16);
				cmp(ureg8);
				break;

			case CPU_INST_CMP_IND_Y:
				addr16 = fetchZpIndirectIndex();
				ureg8 = read(addr16);
				cmp(ureg8);
				break;

			case CPU_INST_CMP_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				cmp(ureg8);
				break;

			case CPU_INST_CMP_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				cmp(ureg8);
				break;

			case CPU_INST_CMP_ABS_Y:
				addr16 = fetchAbsY();
				ureg8 = read(addr16);
				cmp(ureg8);
				break;

			/*---- CPX ----------------------------------------------------------*/
			case CPU_INST_CPX_IMM:
				ureg8 = fetchImmediate();
				cpx(ureg8);
				break;

			case CPU_INST_CPX_ZP:
				ureg8 = fetchZp();
				ureg8 = readZp(ureg8);
				cpx(ureg8);
				break;
			
			case CPU_INST_CPX_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				cpx(ureg8);
				break;

			/*---- CPY ----------------------------------------------------------*/
			case CPU_INST_CPY_IMM:
				ureg8 = fetchImmediate();
				cpy(ureg8);
				break;

			case CPU_INST_CPY_ZP:
				ureg8 = fetchZp();
				ureg8 = readZp(ureg8);
				cpy(ureg8);
				break;
			
			case CPU_INST_CPY_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				cpy(ureg8);
				break;

			/*---- DEC ----------------------------------------------------------*/
			case CPU_INST_DEC_ACCUM:
				_A = dec(_A);
				break;

			case CPU_INST_DEC_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				ureg8 = dec(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_DEC_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				ureg8 = dec(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_DEC_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				ureg8 = dec(ureg8);
				write(addr16, ureg8);
				break;

			case CPU_INST_DEC_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				ureg8 = dec(ureg8);
				write(addr16, ureg8);
				break;

			/*---- DEX ----------------------------------------------------------*/
			case CPU_INST_DEX:
				_X = dec(_X);
				break;
					
			/*---- DEY ----------------------------------------------------------*/
			case CPU_INST_DEY:
				_Y = dec(_Y);
				break;
					
			/*---- INC ----------------------------------------------------------*/
			case CPU_INST_INC_ACCUM:
				_A = inc(_A);
				break;

			case CPU_INST_INC_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				ureg8 = inc(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_INC_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				ureg8 = inc(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_INC_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				ureg8 = inc(ureg8);
				write(addr16, ureg8);
				break;

			case CPU_INST_INC_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				ureg8 = inc(ureg8);
				write(addr16, ureg8);
				break;

			/*---- INX ----------------------------------------------------------*/
			case CPU_INST_INX:
				_X = inc(_X);
				break;
					
			/*---- INY ----------------------------------------------------------*/
			case CPU_INST_INY:
				_Y = inc(_Y);
				break;
					
			/*---- ROL ----------------------------------------------------------*/
			case CPU_INST_ROL_ACCUM:
				_A = rol(_A);
				break;

			case CPU_INST_ROL_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				ureg8 = rol(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_ROL_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				ureg8 = rol(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_ROL_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				ureg8 = rol(ureg8);
				write(addr16, ureg8);
				break;
			
			case CPU_INST_ROL_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				ureg8 = rol(ureg8);
				write(addr16, ureg8);
				break;

			/*---- ROR ----------------------------------------------------------*/
			case CPU_INST_ROR_ACCUM:
				_A = ror(_A);
				break;

			case CPU_INST_ROR_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				ureg8 = ror(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_ROR_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				ureg8 = ror(ureg8);
				writeZp(addr8, ureg8);
				break;

			case CPU_INST_ROR_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				ureg8 = ror(ureg8);
				write(addr16, ureg8);
				break;
			
			case CPU_INST_ROR_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				ureg8 = ror(ureg8);
				write(addr16, ureg8);
				break;

			/*---- CLA ----------------------------------------------------------*/
			case CPU_INST_CLA:
				_A = 0;
				break;

			/*---- CLX ----------------------------------------------------------*/
			case CPU_INST_CLX:
				_X = 0;
				break;

			/*---- CLY ----------------------------------------------------------*/
			case CPU_INST_CLY:
				_Y = 0;
				break;

			/*-------------------------------------------------------------------**
			** flag instructions
			**-------------------------------------------------------------------*/
			/*---- CLC ----------------------------------------------------------*/
			case CPU_INST_CLC:
				_CF = 0;
				break;

			/*---- CLD ----------------------------------------------------------*/
			case CPU_INST_CLD:
				_DF = 0;
				break;

			/*---- CLI ----------------------------------------------------------*/
			case CPU_INST_CLI:
				// I tONA IRQ ̃tFb`sȂȂ 
				_IF = _TF = 0;
				_ClockCount -= _ClockElapsed;
				TIMER_AdvanceClock(_ClockElapsed);
				_ClockUpCount += _ClockElapsed;
				continue;

			/*---- CLV ----------------------------------------------------------*/
			case CPU_INST_CLV:
				_VF = 0;
				break;

			/*---- SEC ----------------------------------------------------------*/
			case CPU_INST_SEC:
				_CF = CPU_CF;
				break;

			/*---- SED ----------------------------------------------------------*/
			case CPU_INST_SED:
				_DF = CPU_DF;
				break;

			/*---- SEI ----------------------------------------------------------*/
			case CPU_INST_SEI:
				//if (_IF == 0)
				//{
				//	fetchInterrupt();	// Ɋ荞݂ȂƂȂ 
				//	_IF = CPU_IF;
				//}
				//_ClockCount -= _ClockElapsed;
				//TIMER_AdvanceClock(_ClockElapsed);
				//_ClockUpCount += _ClockElapsed;
				//continue;
				_IF = CPU_IF;
				break;

			/*---- SET ----------------------------------------------------------*/
			case CPU_INST_SET:
				_TF = CPU_TF;
				_ClockCount -= _ClockElapsed;
				TIMER_AdvanceClock(_ClockElapsed);
				_ClockUpCount += _ClockElapsed;
				continue;

			/*-------------------------------------------------------------------**
			** data transfer instructions
			**-------------------------------------------------------------------*/
			/*---- LDA ----------------------------------------------------------*/
			case CPU_INST_LDA_IMM:
				ureg8 = fetchImmediate();
				lda(ureg8);
				break;

			case CPU_INST_LDA_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				lda(ureg8);
				break;

			case CPU_INST_LDA_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				lda(ureg8);
				break;

			case CPU_INST_LDA_IND:
				addr16 = fetchZpIndirect();
				ureg8 = read(addr16);
				lda(ureg8);
				break;

			case CPU_INST_LDA_IND_X:
				addr16 = fetchZpIndexIndirect();
				ureg8 = read(addr16);
				lda(ureg8);
				break;

			case CPU_INST_LDA_IND_Y:
				addr16 = fetchZpIndirectIndex();
				ureg8 = read(addr16);
				lda(ureg8);
				break;

			case CPU_INST_LDA_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				lda(ureg8);
				break;

			case CPU_INST_LDA_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				lda(ureg8);
				break;

			case CPU_INST_LDA_ABS_Y:
				addr16 = fetchAbsY();
				ureg8 = read(addr16);
				lda(ureg8);
				break;

			/*---- LDX ----------------------------------------------------------*/
			case CPU_INST_LDX_IMM:
				ureg8 = fetchImmediate();
				ldx(ureg8);
				break;

			case CPU_INST_LDX_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				ldx(ureg8);
				break;

			case CPU_INST_LDX_ZP_Y:
				addr8 = fetchZpY();
				ureg8 = readZp(addr8);
				ldx(ureg8);
				break;

			case CPU_INST_LDX_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				ldx(ureg8);
				break;

			case CPU_INST_LDX_ABS_Y:
				addr16 = fetchAbsY();
				ureg8 = read(addr16);
				ldx(ureg8);
				break;

			/*---- LDY ----------------------------------------------------------*/
			case CPU_INST_LDY_IMM:
				ureg8 = fetchImmediate();
				ldy(ureg8);
				break;

			case CPU_INST_LDY_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				ldy(ureg8);
				break;

			case CPU_INST_LDY_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				ldy(ureg8);
				break;

			case CPU_INST_LDY_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				ldy(ureg8);
				break;

			case CPU_INST_LDY_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				ldy(ureg8);
				break;

			/*---- STA ----------------------------------------------------------*/
			case CPU_INST_STA_ZP:
				addr8 = fetchZp();
				writeZp(addr8, _A);
				break;

			case CPU_INST_STA_ZP_X:
				addr8 = fetchZpX();
				writeZp(addr8, _A);
				break;

			case CPU_INST_STA_IND:
				addr16 = fetchZpIndirect();
				write(addr16, _A);
				break;

			case CPU_INST_STA_IND_X:
				addr16 = fetchZpIndexIndirect();
				write(addr16, _A);
				break;

			case CPU_INST_STA_IND_Y:
				addr16 = fetchZpIndirectIndex();
				write(addr16, _A);
				break;

			case CPU_INST_STA_ABS:
				addr16 = fetchAbs();
				write(addr16, _A);
				break;

			case CPU_INST_STA_ABS_X:
				addr16 = fetchAbsX();
				write(addr16, _A);
				break;

			case CPU_INST_STA_ABS_Y:
				addr16 = fetchAbsY();
				write(addr16, _A);
				break;

			/*---- STX ----------------------------------------------------------*/
			case CPU_INST_STX_ZP:
				addr8 = fetchZp();
				writeZp(addr8, _X);
				break;

			case CPU_INST_STX_ZP_Y:
				addr8 = fetchZpY();
				writeZp(addr8, _X);
				break;

			case CPU_INST_STX_ABS:
				addr16 = fetchAbs();
				write(addr16, _X);
				break;

			/*---- STY ----------------------------------------------------------*/
			case CPU_INST_STY_ZP:
				addr8 = fetchZp();
				writeZp(addr8, _Y);
				break;

			case CPU_INST_STY_ZP_X:
				addr8 = fetchZpX();
				writeZp(addr8, _Y);
				break;

			case CPU_INST_STY_ABS:
				addr16 = fetchAbs();
				write(addr16, _Y);
				break;

			/*---- SAX ----------------------------------------------------------*/
			case CPU_INST_SAX:
				ureg8 = _A;
				_A = _X;
				_X = ureg8;
				break;

			/*---- SAY ----------------------------------------------------------*/
			case CPU_INST_SAY:
				ureg8 = _A;
				_A = _Y;
				_Y = ureg8;
				break;

			/*---- SXY ----------------------------------------------------------*/
			case CPU_INST_SXY:
				ureg8 = _X;
				_X = _Y;
				_Y = ureg8;
				break;

			/*---- ST0 ----------------------------------------------------------*/
			case CPU_INST_ST0_IMM:
				ureg8 = fetchImmediate();
				Write(0x1fe000, ureg8);
				break;

			/*---- ST1 ----------------------------------------------------------*/
			case CPU_INST_ST1_IMM:
				ureg8 = fetchImmediate();
				Write(0x1fe002, ureg8);
				break;

			/*---- ST2 ----------------------------------------------------------*/
			case CPU_INST_ST2_IMM:
				ureg8 = fetchImmediate();
				Write(0x1fe003, ureg8);
				break;

			/*---- STZ ----------------------------------------------------------*/
			case CPU_INST_STZ_ZP:
				addr8 = fetchZp();
				writeZp(addr8, 0);
				break;

			case CPU_INST_STZ_ZP_X:
				addr8 = fetchZpX();
				writeZp(addr8, 0);
				break;

			case CPU_INST_STZ_ABS:
				addr16 = fetchAbs();
				write(addr16, 0);
				break;

			case CPU_INST_STZ_ABS_X:
				addr16 = fetchAbsX();
				write(addr16, 0);
				break;

			/*---- TAI ----------------------------------------------------------*/
			case CPU_INST_TAI:
			{
				Uint16	srcAddr = fetchAbs();
				Uint16	dstAddr = fetchAbs();
				Uint16	length	= fetchAbs();

				push(_Y);
				push(_A);
				push(_X);
				do
				{
					write(dstAddr++, read(srcAddr++));
					_ClockElapsed += 6;
					if (--length == 0)
					{
						break;
					}
					write(dstAddr++, read(srcAddr--));
					_ClockElapsed += 6;
				} while (--length);
				_X = pull();
				_A = pull();
				_Y = pull();
			}
			break;

			/*---- TDD ----------------------------------------------------------*/
			case CPU_INST_TDD:
			{
				Uint16	srcAddr = fetchAbs();
				Uint16	dstAddr = fetchAbs();
				Uint16	length	= fetchAbs();

				push(_Y);
				push(_A);
				push(_X);
				do
				{
					write(dstAddr--, read(srcAddr--));
					_ClockElapsed += 6;
				} while (--length);
				_X = pull();
				_A = pull();
				_Y = pull();
			}
			break;

			/*---- TIA ----------------------------------------------------------*/
			case CPU_INST_TIA:
			{
				Uint16	srcAddr = fetchAbs();
				Uint16	dstAddr = fetchAbs();
				Uint16	length	= fetchAbs();

				push(_Y);
				push(_A);
				push(_X);
				do
				{
					write(dstAddr++, read(srcAddr++));
					_ClockElapsed += 6;
					if (--length == 0)
					{
						break;
					}
					write(dstAddr--, read(srcAddr++));
					_ClockElapsed += 6;
				} while (--length);
				_X = pull();
				_A = pull();
				_Y = pull();
			}
			break;

			/*---- TII ----------------------------------------------------------*/
			case CPU_INST_TII:
			{
				Uint16	srcAddr = fetchAbs();
				Uint16	dstAddr = fetchAbs();
				Uint16	length	= fetchAbs();

				push(_Y);
				push(_A);
				push(_X);
				do
				{
					write(dstAddr++, read(srcAddr++));
					_ClockElapsed += 6;
				} while (--length);
				_X = pull();
				_A = pull();
				_Y = pull();
			}
			break;

			/*---- TIN ----------------------------------------------------------*/
			case CPU_INST_TIN:
			{
				Uint16	srcAddr = fetchAbs();
				Uint16	dstAddr = fetchAbs();
				Uint16	length	= fetchAbs();

				push(_Y);
				push(_A);
				push(_X);
				do
				{
					write(dstAddr, read(srcAddr++));
					_ClockElapsed += 6;
				} while (--length);
				_X = pull();
				_A = pull();
				_Y = pull();
			}
			break;

			/*---- TAMi ---------------------------------------------------------*/
			case CPU_INST_TAM:
				ureg8 = fetchImmediate();
				if (ureg8 & 0x01) _MPR[0] = _A << 13;
				if (ureg8 & 0x02) _MPR[1] = _A << 13;
				if (ureg8 & 0x04) _MPR[2] = _A << 13;
				if (ureg8 & 0x08) _MPR[3] = _A << 13;
				if (ureg8 & 0x10) _MPR[4] = _A << 13;
				if (ureg8 & 0x20) _MPR[5] = _A << 13;
				if (ureg8 & 0x40) _MPR[6] = _A << 13;
				if (ureg8 & 0x80) _MPR[7] = _A << 13;
				break;

			/*---- TMAi ---------------------------------------------------------*/
			case CPU_INST_TMA:
				ureg8 = fetchImmediate();
				if (ureg8 & 0x01) { _A = _MPR[0] >> 13; break; }
				if (ureg8 & 0x02) { _A = _MPR[1] >> 13; break; }
				if (ureg8 & 0x04) { _A = _MPR[2] >> 13; break; }
				if (ureg8 & 0x08) { _A = _MPR[3] >> 13; break; }
				if (ureg8 & 0x10) { _A = _MPR[4] >> 13; break; }
				if (ureg8 & 0x20) { _A = _MPR[5] >> 13; break; }
				if (ureg8 & 0x40) { _A = _MPR[6] >> 13; break; }
				if (ureg8 & 0x80) { _A = _MPR[7] >> 13; break; }
				break;

			/*---- TAX ----------------------------------------------------------*/
			case CPU_INST_TAX:
				tax();
				break;

			/*---- TAY ----------------------------------------------------------*/
			case CPU_INST_TAY:
				tay();
				break;

			/*---- TSX ----------------------------------------------------------*/
			case CPU_INST_TSX:
				tsx();
				break;

			/*---- TXA ----------------------------------------------------------*/
			case CPU_INST_TXA:
				txa();
				break;

			/*---- TXS ----------------------------------------------------------*/
			case CPU_INST_TXS:
				_S = _X;
				break;

			/*---- TYA ----------------------------------------------------------*/
			case CPU_INST_TYA:
				tya();
				break;

			/*-------------------------------------------------------------------**
			** branch / jump instructions
			**-------------------------------------------------------------------*/
			/*---- BBRi ---------------------------------------------------------*/
			case CPU_INST_BBR0_ZP_REL:
				BBRi(0x01);
				break;

			case CPU_INST_BBR1_ZP_REL:
				BBRi(0x02);
				break;

			case CPU_INST_BBR2_ZP_REL:
				BBRi(0x04);
				break;

			case CPU_INST_BBR3_ZP_REL:
				BBRi(0x08);
				break;

			case CPU_INST_BBR4_ZP_REL:
				BBRi(0x10);
				break;

			case CPU_INST_BBR5_ZP_REL:
				BBRi(0x20);
				break;

			case CPU_INST_BBR6_ZP_REL:
				BBRi(0x40);
				break;

			case CPU_INST_BBR7_ZP_REL:
				BBRi(0x80);
				break;

			/*---- BBSi ---------------------------------------------------------*/
			case CPU_INST_BBS0_ZP_REL:
				BBSi(0x01);
				break;

			case CPU_INST_BBS1_ZP_REL:
				BBSi(0x02);
				break;

			case CPU_INST_BBS2_ZP_REL:
				BBSi(0x04);
				break;

			case CPU_INST_BBS3_ZP_REL:
				BBSi(0x08);
				break;

			case CPU_INST_BBS4_ZP_REL:
				BBSi(0x10);
				break;

			case CPU_INST_BBS5_ZP_REL:
				BBSi(0x20);
				break;

			case CPU_INST_BBS6_ZP_REL:
				BBSi(0x40);
				break;

			case CPU_INST_BBS7_ZP_REL:
				BBSi(0x80);
				break;

			/*---- BCC ----------------------------------------------------------*/
			case CPU_INST_BCC_REL:
				rel8 = fetchRelative();
				if (_CF == 0)
				{
					relativeBranch(rel8);
					_ClockElapsed += 2;
				}
				break;

			/*---- BNE ----------------------------------------------------------*/
			case CPU_INST_BNE_REL:
				rel8 = fetchRelative();
				if (_ZF)
				{
					relativeBranch(rel8);
					_ClockElapsed += 2;
				}
				break;

			/*---- BVC ----------------------------------------------------------*/
			case CPU_INST_BVC_REL:
				rel8 = fetchRelative();
				if ((_VF & CPU_VF) == 0)
				{
					relativeBranch(rel8);
					_ClockElapsed += 2;
				}
				break;

			/*---- BPL ----------------------------------------------------------*/
			case CPU_INST_BPL_REL:
				rel8 = fetchRelative();
				if ((_NF & CPU_NF) == 0)
				{
					relativeBranch(rel8);
					_ClockElapsed += 2;
				}
				break;

			/*---- BCS ----------------------------------------------------------*/
			case CPU_INST_BCS_REL:
				rel8 = fetchRelative();
				if (_CF)
				{
					relativeBranch(rel8);
					_ClockElapsed += 2;
				}
				break;

			/*---- BEQ ----------------------------------------------------------*/
			case CPU_INST_BEQ_REL:
				rel8 = fetchRelative();
				if (_ZF == 0)
				{
					relativeBranch(rel8);
					_ClockElapsed += 2;
				}
				break;

			/*---- BVS ----------------------------------------------------------*/
			case CPU_INST_BVS_REL:
				rel8 = fetchRelative();
				if (_VF & CPU_VF)
				{
					relativeBranch(rel8);
					_ClockElapsed += 2;
				}
				break;

			/*---- BMI ----------------------------------------------------------*/
			case CPU_INST_BMI_REL:
				rel8 = fetchRelative();
				if (_NF & CPU_NF)
				{
					relativeBranch(rel8);
					_ClockElapsed += 2;
				}
				break;

			/*---- BRA ----------------------------------------------------------*/
			case CPU_INST_BRA_REL:
	//			rel8 = fetchRelative();
				relativeBranch(fetchRelative());
				break;

			/*---- JMP ----------------------------------------------------------*/
			case CPU_INST_JMP_ABS:
				_PC = fetchAbs();
				break;

			case CPU_INST_JMP_INDIR:
				_PC = fetchAbsIndirect();
				break;

			case CPU_INST_JMP_INDIRX:
				_PC = fetchAbsIndirectX();
				break;

			/*-------------------------------------------------------------------**
			** subroutine instructions
			**-------------------------------------------------------------------*/
			/*---- BSR ----------------------------------------------------------*/
			case CPU_INST_BSR_REL:
				rel8 = fetchRelative();
				--_PC;
				push(_PC >> 8);
				push(_PC & 0xff);
				++_PC;
				relativeBranch(rel8);
				break;

			/*---- JSR ----------------------------------------------------------*/
			case CPU_INST_JSR_ABS:
				addr16 = fetchAbs();
				--_PC;
				push(_PC >> 8);
				push(_PC & 0xff);
				_PC = addr16;
				break;

			/*---- PHA ----------------------------------------------------------*/
			case CPU_INST_PHA:
				push(_A);
				break;

			/*---- PHP ----------------------------------------------------------*/
			case CPU_INST_PHP:
				_BF = _TF = 0;
				push(gatherFlags());
				break;

			/*---- PHX ----------------------------------------------------------*/
			case CPU_INST_PHX:
				push(_X);
				break;

			/*---- PHY ----------------------------------------------------------*/
			case CPU_INST_PHY:
				push(_Y);
				break;

			/*---- PLA ----------------------------------------------------------*/
			case CPU_INST_PLA:
				_A = pull();
				updateFlagZN(_A);
				break;

			/*---- PLP ----------------------------------------------------------*/
			// PLP ߎśF
			// - T tOZbgȂ 
			// - 荞݃tFb`sȂȂ 
			case CPU_INST_PLP:
				separateFlags(pull());
				TIMER_AdvanceClock(_ClockElapsed);
				_ClockCount -= _ClockElapsed;
				_ClockUpCount += _ClockElapsed;
				continue;

			/*---- PLX ----------------------------------------------------------*/
			case CPU_INST_PLX:
				_X = pull();
				updateFlagZN(_X);
				break;

			/*---- PLY ----------------------------------------------------------*/
			case CPU_INST_PLY:
				_Y = pull();
				updateFlagZN(_Y);
				break;

			/*---- RTI ----------------------------------------------------------*/
			case CPU_INST_RTI:
				// RTI ߎs
				// - T tOZbgȂ 
				// - 荞݃tFb`ȂȂ 
	//			if (_IF)
	//			{
					separateFlags(pull());
					_PC = pull();
					_PC |= pull() << 8;
	//			}
	//			else
	//			{
	//				separateFlags(pull());
	//				_PC = pull();
	//				_PC |= pull() << 8;
	//			}

				TIMER_AdvanceClock(_ClockElapsed);
				_ClockCount -= _ClockElapsed;
				_ClockUpCount += _ClockElapsed;
				continue;
			
			/*---- RTS ----------------------------------------------------------*/
			case CPU_INST_RTS:
				_PC = pull();
				_PC |= pull() << 8;
				++_PC;
				break;

			/*-------------------------------------------------------------------**
			** test instructions
			**-------------------------------------------------------------------*/
			/*---- BIT ----------------------------------------------------------*/
			case CPU_INST_BIT_IMM:
				ureg8 = fetchImmediate();
				bit(ureg8);
				break;

			case CPU_INST_BIT_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				bit(ureg8);
				break;

			case CPU_INST_BIT_ZP_X:
				addr8 = fetchZpX();
				ureg8 = readZp(addr8);
				bit(ureg8);
				break;

			case CPU_INST_BIT_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				bit(ureg8);
				break;

			case CPU_INST_BIT_ABS_X:
				addr16 = fetchAbsX();
				ureg8 = read(addr16);
				bit(ureg8);
				break;

			/*---- TRB ----------------------------------------------------------*/
			case CPU_INST_TRB_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				writeZp(addr8, trb(ureg8));
				break;

			case CPU_INST_TRB_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				write(addr16, trb(ureg8));
				break;

			/*---- TSB ----------------------------------------------------------*/
			case CPU_INST_TSB_ZP:
				addr8 = fetchZp();
				ureg8 = readZp(addr8);
				writeZp(addr8, tsb(ureg8));
				break;

			case CPU_INST_TSB_ABS:
				addr16 = fetchAbs();
				ureg8 = read(addr16);
				write(addr16, tsb(ureg8));
				break;

			/*---- TST ----------------------------------------------------------*/
			case CPU_INST_TST_IMM_ZP:
				ureg8 = fetchImmediate();
				addr8 = fetchZp();
				tst(ureg8, readZp(addr8));
				break;

			case CPU_INST_TST_IMM_ZP_X:
				ureg8 = fetchImmediate();
				addr8 = fetchZpX();
				tst(ureg8, readZp(addr8));
				break;

			case CPU_INST_TST_IMM_ABS:
				ureg8 = fetchImmediate();
				addr16 = fetchAbs();
				tst(ureg8, read(addr16));
				break;

			case CPU_INST_TST_IMM_ABS_X:
				ureg8 = fetchImmediate();
				addr16 = fetchAbsX();
				tst(ureg8, read(addr16));
				break;

			/*-------------------------------------------------------------------**
			** control instructions
			**-------------------------------------------------------------------*/
			/*---- CSL ----------------------------------------------------------*/
			case CPU_INST_CSL:
				break;

			/*---- CSH ----------------------------------------------------------*/
			case CPU_INST_CSH:
				break;

			/*---- BRK ----------------------------------------------------------*/
			case CPU_INST_BRK:
				brk();
				break;

			/*---- NOP ----------------------------------------------------------*/
			case CPU_INST_NOP:
				break;

			/*---- RMBi ---------------------------------------------------------*/
			case CPU_INST_RMB0_ZP:
				RMBi(fetchZp(), 0x01);
				break;

			case CPU_INST_RMB1_ZP:
				RMBi(fetchZp(), 0x02);
				break;

			case CPU_INST_RMB2_ZP:
				RMBi(fetchZp(), 0x04);
				break;

			case CPU_INST_RMB3_ZP:
				RMBi(fetchZp(), 0x08);
				break;

			case CPU_INST_RMB4_ZP:
				RMBi(fetchZp(), 0x10);
				break;

			case CPU_INST_RMB5_ZP:
				RMBi(fetchZp(), 0x20);
				break;

			case CPU_INST_RMB6_ZP:
				RMBi(fetchZp(), 0x40);
				break;

			case CPU_INST_RMB7_ZP:
				RMBi(fetchZp(), 0x80);
				break;

			/*---- SMBi ---------------------------------------------------------*/
			case CPU_INST_SMB0_ZP:
				SMBi(fetchZp(), 0x01);
				break;

			case CPU_INST_SMB1_ZP:
				SMBi(fetchZp(), 0x02);
				break;

			case CPU_INST_SMB2_ZP:
				SMBi(fetchZp(), 0x04);
				break;

			case CPU_INST_SMB3_ZP:
				SMBi(fetchZp(), 0x08);
				break;

			case CPU_INST_SMB4_ZP:
				SMBi(fetchZp(), 0x10);
				break;

			case CPU_INST_SMB5_ZP:
				SMBi(fetchZp(), 0x20);
				break;

			case CPU_INST_SMB6_ZP:
				SMBi(fetchZp(), 0x40);
				break;

			case CPU_INST_SMB7_ZP:
				SMBi(fetchZp(), 0x80);
				break;

			/*------------------------------------------------------------------**
			** invalid instructions
			**------------------------------------------------------------------*/
			default:
				break;
		}

		_TF = 0;

		TIMER_AdvanceClock(_ClockElapsed);

		fetchInterrupt();

		_ClockCount -= _ClockElapsed;
		_ClockUpCount += _ClockElapsed;
	}

	_ClockElapsed = 0;
	return _ClockCount;
}


/*-----------------------------------------------------------------------------
	[SaveState]
		bot̏Ԃt@Cɕۑ܂B 
-----------------------------------------------------------------------------*/
BOOL
CPU_SaveState(
	FILE*		p)
{
	if (fwrite(&_A, sizeof(_A), 1, p) != 1)	return FALSE;
	if (fwrite(&_X, sizeof(_X), 1, p) != 1)	return FALSE;
	if (fwrite(&_Y, sizeof(_Y), 1, p) != 1)	return FALSE;
	if (fwrite(&_S, sizeof(_S), 1, p) != 1)	return FALSE;
	if (fwrite(&_P, sizeof(_P), 1, p) != 1)	return FALSE;
	if (fwrite(&_CF, sizeof(_CF), 1, p) != 1)	return FALSE;
	if (fwrite(&_ZF, sizeof(_ZF), 1, p) != 1)	return FALSE;
	if (fwrite(&_IF, sizeof(_IF), 1, p) != 1)	return FALSE;
	if (fwrite(&_DF, sizeof(_DF), 1, p) != 1)	return FALSE;
	if (fwrite(&_BF, sizeof(_BF), 1, p) != 1)	return FALSE;
	if (fwrite(&_TF, sizeof(_TF), 1, p) != 1)	return FALSE;
	if (fwrite(&_VF, sizeof(_VF), 1, p) != 1)	return FALSE;
	if (fwrite(&_NF, sizeof(_NF), 1, p) != 1)	return FALSE;
	if (fwrite(&_PC, sizeof(_PC), 1, p) != 1)	return FALSE;
	if (fwrite(_MPR, sizeof(_MPR), 1, p) != 1)	return FALSE;

	if (fwrite(&_ClockUpCount, sizeof(_ClockUpCount), 1, p) != 1)	return FALSE;

	if (fwrite(&_bRDY, sizeof(_bRDY), 1, p) != 1)		return FALSE;
	if (fwrite(&_bIRQ1, sizeof(_bIRQ1), 1, p) != 1)	return FALSE;
	if (fwrite(&_bIRQ2, sizeof(_bIRQ2), 1, p) != 1)	return FALSE;
	if (fwrite(&_bNMI, sizeof(_bNMI), 1, p) != 1)		return FALSE;
	if (fwrite(&_bTIRQ, sizeof(_bTIRQ), 1, p) != 1)	return FALSE;
	
	return TRUE;
}


/*-----------------------------------------------------------------------------
	[LoadState]
		bot̏Ԃt@Cǂݍ݂܂B 
-----------------------------------------------------------------------------*/
BOOL
CPU_LoadState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	if (fread(&_A, sizeof(_A), 1, p) != 1)	return FALSE;
	if (fread(&_X, sizeof(_X), 1, p) != 1)	return FALSE;
	if (fread(&_Y, sizeof(_Y), 1, p) != 1)	return FALSE;
	if (fread(&_S, sizeof(_S), 1, p) != 1)	return FALSE;
	if (fread(&_P, sizeof(_P), 1, p) != 1)	return FALSE;
	if (fread(&_CF, sizeof(_CF), 1, p) != 1)	return FALSE;
	if (fread(&_ZF, sizeof(_ZF), 1, p) != 1)	return FALSE;
	if (fread(&_IF, sizeof(_IF), 1, p) != 1)	return FALSE;
	if (fread(&_DF, sizeof(_DF), 1, p) != 1)	return FALSE;
	if (fread(&_BF, sizeof(_BF), 1, p) != 1)	return FALSE;
	if (fread(&_TF, sizeof(_TF), 1, p) != 1)	return FALSE;
	if (fread(&_VF, sizeof(_VF), 1, p) != 1)	return FALSE;
	if (fread(&_NF, sizeof(_NF), 1, p) != 1)	return FALSE;
	if (fread(&_PC, sizeof(_PC), 1, p) != 1)	return FALSE;
	if (fread(_MPR, sizeof(_MPR), 1, p) != 1)	return FALSE;

	if (fread(&_ClockUpCount, sizeof(_ClockUpCount), 1, p) != 1)	return FALSE;

	if (fread(&_bRDY, sizeof(_bRDY), 1, p) != 1)		return FALSE;
	if (fread(&_bIRQ1, sizeof(_bIRQ1), 1, p) != 1)	return FALSE;
	if (fread(&_bIRQ2, sizeof(_bIRQ2), 1, p) != 1)	return FALSE;
	if (fread(&_bNMI, sizeof(_bNMI), 1, p) != 1)		return FALSE;
	if (fread(&_bTIRQ, sizeof(_bTIRQ), 1, p) != 1)	return FALSE;

	return TRUE;
}

