/*-----------------------------------------------------------------------------
	[PadConfig.c]

		Implements a pad configuration window.

	Copyright (C) 2005 Ki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
-----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#include "WinMain.h"
#include "PadConfig.h"
#include "InputInterface.h"


#define CAPTION		"PAD CONFIGURATION"


#define N_LINES				20
#define LINE_LEN			80

#define N_MAXJOYSTICK		5


typedef struct
{
	Sint32	userButtonID;
	Sint32	inputButtonID;
} ButtonConnector;

typedef struct
{
	JOYINFOEX					joyState;
	ButtonConnector				buttonConnector[INPUT_NUM_BUTTON];
	Uint16						nConnectedButtons;
	Uint16						buttonState;
	Sint32						xpos;
	Sint32						ypos;
	Sint32						xmin;
	Sint32						xmax;
	Sint32						ymin;
	Sint32						ymax;
} JOYSTICK;


static Uint32		_FontWidth;
static Uint32		_FontHeight;
static const char*	_pCaption = CAPTION;
static HINSTANCE	_hInstance = NULL;
static HWND			_hWnd = NULL;

static char*		_pText[N_LINES];
static char			_Text[N_LINES][LINE_LEN];
static Uint32		_Line = 0;


static Sint32		_nJoySticks;
static JOYSTICK		_Joystick[N_MAXJOYSTICK];
static Sint32		_JoyID2SystemID[N_MAXJOYSTICK];



/* tHg̍擾 */
static
Uint32
get_font_height(
	HWND			hWnd)
{
	HDC				hDC;
	HFONT			hFont;
	HFONT			hFontOld;
	TEXTMETRIC		tm;

	hDC      = GetDC(hWnd);
	hFont    = GetStockObject(OEM_FIXED_FONT);   /* Œsb`tHg */
	hFontOld = SelectObject(hDC, hFont);

	GetTextMetrics(hDC, &tm);

	SelectObject(hDC, hFontOld);
	DeleteObject(hFont);
	ReleaseDC(hWnd, hDC);

	return (Uint32)(tm.tmHeight);
}

/* tHg̉擾 */
static
Uint32
get_font_width(
	HWND			hWnd)
{
	HDC				hDC;
	HFONT			hFont;
	HFONT			hFontOld;
	TEXTMETRIC		tm;

	hDC      = GetDC(hWnd);
	hFont    = GetStockObject(OEM_FIXED_FONT);   /* Œsb`tHg */
	hFontOld = SelectObject(hDC, hFont);

	GetTextMetrics(hDC, &tm);

	SelectObject(hDC, hFontOld);
	DeleteObject(hFont);
	ReleaseDC(hWnd, hDC);

	return (Uint32)tm.tmAveCharWidth;
}


static
void
set_window_size(
	HWND			hWnd)
{
	RECT			rc;
	Uint32			wndW = _FontWidth  * LINE_LEN;
	Uint32			wndH = _FontHeight * N_LINES;

	SetRect(&rc, 0, 0, wndW, wndH);
	AdjustWindowRectEx(&rc, GetWindowLong(hWnd, GWL_STYLE),
						GetMenu(hWnd) != NULL, GetWindowLong(hWnd, GWL_EXSTYLE));
	wndW = rc.right - rc.left;
	wndH = rc.bottom - rc.top;
	MoveWindow(hWnd, 50, 50, wndW, wndH, TRUE);
}


static
void
update_window(
	HWND			hWnd)
{
	HDC				hDC;
	HFONT			hFont;
	HFONT			hFontOld;
	PAINTSTRUCT		ps;
	Uint32			i;

	/* `揀 */
	hDC      = BeginPaint(hWnd, &ps);
	hFont    = GetStockObject(OEM_FIXED_FONT);
	hFontOld = SelectObject(hDC, hFont);
	SetBkColor(hDC, RGB(0,0,0));
	SetTextColor(hDC, RGB(224, 224, 224));

	/* ̔wihԂ */
	SetBkMode(hDC, OPAQUE);

	for (i = 0; i < _Line; ++i)
	{
		TextOut(hDC, 0, i*_FontHeight, _pText[i], strlen(_pText[i]));
	}

	/* I */
	EndPaint(hWnd, &ps);
	SelectObject(hDC, hFontOld);
	DeleteObject(hFont);
	ReleaseDC(hWnd, hDC);
}


static
void
add_text(
	const char*		pText, ...)
{
	Uint32			i;
	va_list			ap;
	char*			p;

	va_start(ap, pText);
	vsprintf(_pText[_Line++], pText, ap);
	va_end(ap);

	// scroll a line
	if (_Line == N_LINES)
	{
		p = _pText[0];
		for (i = 1; i < N_LINES; ++i)
		{
			_pText[i-1] = _pText[i];
		}
		_pText[N_LINES-1] = p;
		*p = '\0';

		--_Line;
	}
	PADCONFIG_Update(_hWnd);
}



inline static BOOL is_up(JOYSTICK* pJoy)    { return pJoy->ypos <= (pJoy->ymax-pJoy->ymin)/4; }
inline static BOOL is_down(JOYSTICK* pJoy)  { return pJoy->ypos >= (pJoy->ymax-pJoy->ymin)*3/4; }
inline static BOOL is_left(JOYSTICK* pJoy)  { return pJoy->xpos <= (pJoy->xmax-pJoy->xmin)/4; }
inline static BOOL is_right(JOYSTICK* pJoy) { return pJoy->xpos >= (pJoy->xmax-pJoy->xmin)*3/4; }


static
BOOL
pad_init()
{
	int			i;
	int			nDev;
	JOYCAPS		jc;

	ZeroMemory(_Joystick, sizeof(_Joystick));
	ZeroMemory(_JoyID2SystemID, sizeof(_JoyID2SystemID));

	if ((nDev = joyGetNumDevs()) == 0)
	{
		add_text("PADCONFIG: No supported joystick found.");
		return FALSE;
	}

	_nJoySticks = 0;
	for (i = 0; i < nDev; i++)
	{
		_Joystick[_nJoySticks].joyState.dwSize = sizeof(JOYINFOEX);
		_Joystick[_nJoySticks].joyState.dwFlags = JOY_RETURNBUTTONS | JOY_RETURNX | JOY_RETURNY;

		if (joyGetPosEx(i, &_Joystick[_nJoySticks].joyState) == JOYERR_UNPLUGGED)
			add_text("PADCONFIG: Joystick #%d is not plugged.", i);
		else
		{
			// capability 𒲂ׂ 
			if (joyGetDevCaps(i, &jc, sizeof(jc)) == JOYERR_NOERROR)
			{
				_JoyID2SystemID[_nJoySticks] = i;
				add_text("PADCONFIG: Selected joystick = \"%s\"", jc.szPname);
				add_text("PADCONFIG: Min X coordinate  = %d", jc.wXmin);
				add_text("PADCONFIG: Max X coordinate  = %d", jc.wXmax);
				add_text("PADCONFIG: Min Y coordinate  = %d", jc.wYmin);
				add_text("PADCONFIG: Max Y coordinate  = %d", jc.wYmax);
				add_text("PADCONFIG: Number of buttons = %d", jc.wNumButtons);

				_Joystick[_nJoySticks].xmin = (Sint32)jc.wXmin;
				_Joystick[_nJoySticks].xmax = (Sint32)jc.wXmax;
				_Joystick[_nJoySticks].ymin = (Sint32)jc.wYmin;
				_Joystick[_nJoySticks].ymax = (Sint32)jc.wYmax;
				if (++_nJoySticks == N_MAXJOYSTICK)
					break;
			}
		}
	}

	if (_nJoySticks == 0)
	{
		add_text("PADCONFIG: No joystick seems to be available.");
		return FALSE;
	}

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[ConnectButton]
		[U[`̃{^Ɠ̓{^ڑ܂B
-----------------------------------------------------------------------------*/
static
BOOL
pad_connect_button(
	Sint32		joyID,
	Sint32		userButtonID,
	Sint32		inputButtonID)
{
	if (joyID < 0 || joyID >= _nJoySticks)
		return FALSE;

	// ܂o^ł邩H 
	if (_Joystick[joyID].nConnectedButtons == INPUT_NUM_BUTTON)
		return FALSE;

	// o^ 
	_Joystick[joyID].buttonConnector[_Joystick[joyID].nConnectedButtons].userButtonID = userButtonID;
	_Joystick[joyID].buttonConnector[_Joystick[joyID].nConnectedButtons].inputButtonID = inputButtonID;

	++_Joystick[joyID].nConnectedButtons;

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[pad_update_state]
		͏󋵂XV܂B
-----------------------------------------------------------------------------*/
static
void
pad_update_state()
{
	int		i;
	int		sysID;

	for (i = 0; i < _nJoySticks; ++i)
	{
		sysID = _JoyID2SystemID[i];

		// WCXeBbÑXe[^Xǂ 
		_Joystick[i].joyState.dwSize = sizeof(JOYINFOEX);
		_Joystick[i].joyState.dwFlags = JOY_RETURNBUTTONS | JOY_RETURNX | JOY_RETURNY;
		joyGetPosEx(sysID, &_Joystick[i].joyState);

		_Joystick[i].buttonState = _Joystick[i].joyState.dwButtons;
		_Joystick[i].xpos = _Joystick[i].joyState.dwXpos;
		_Joystick[i].ypos = _Joystick[i].joyState.dwYpos;

		if (is_up(&_Joystick[i]))		_Joystick[i].buttonState |= INPUT_JOYSTICK_UP;
		if (is_down(&_Joystick[i]))		_Joystick[i].buttonState |= INPUT_JOYSTICK_DOWN;
		if (is_left(&_Joystick[i]))		_Joystick[i].buttonState |= INPUT_JOYSTICK_LEFT;
		if (is_right(&_Joystick[i]))	_Joystick[i].buttonState |= INPUT_JOYSTICK_RIGHT;
	}
}


/*-----------------------------------------------------------------------------
	[pad_is_pressed]
		w̃{^ꂽǂԂ܂D
-----------------------------------------------------------------------------*/
static
BOOL
pad_is_pressed(
	Sint32	joyID,
	Sint32	userButtonID)
{
	int					i;
	ButtonConnector*	pBC;
	int					nConnectedButtons;

	if (joyID < 0 || joyID >= _nJoySticks)
		return FALSE;

	pBC = _Joystick[joyID].buttonConnector;
	nConnectedButtons = _Joystick[joyID].nConnectedButtons;

	for (i = 0; i < nConnectedButtons; i++)
	{
		if (pBC[i].userButtonID == userButtonID)
		{
			return (_Joystick[joyID].buttonState & pBC[i].inputButtonID) != 0;
		}
	}

	return FALSE;
}


static
BOOL
pump_message()
{
	MSG			msg;

	if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	if (_hInstance == NULL)
		return FALSE;

	return TRUE;
}


static
BOOL
get_button(
	Sint32		joyID,
	Sint32*		pButton)
{
	const Sint32	button[INPUT_NUM_BUTTON] =
	{
		INPUT_JOYSTICK_BUTTON1,  INPUT_JOYSTICK_BUTTON2,  INPUT_JOYSTICK_BUTTON3,  INPUT_JOYSTICK_BUTTON4,
		INPUT_JOYSTICK_BUTTON5,  INPUT_JOYSTICK_BUTTON6,  INPUT_JOYSTICK_BUTTON7,  INPUT_JOYSTICK_BUTTON8,
		INPUT_JOYSTICK_BUTTON9,  INPUT_JOYSTICK_BUTTON10, INPUT_JOYSTICK_BUTTON11, INPUT_JOYSTICK_BUTTON12,
		INPUT_JOYSTICK_UP,       INPUT_JOYSTICK_DOWN,     INPUT_JOYSTICK_LEFT,     INPUT_JOYSTICK_RIGHT
	};

	int			i;
	Sint32		theButton = -1;

	while (theButton < 0)
	{
		Sleep(20);

		if (!pump_message())
			return FALSE;

		pad_update_state();

		for (i = 0; i < INPUT_NUM_BUTTON; i++)
		{
			if (pad_is_pressed(joyID, button[i]))
			{
				theButton = button[i];
				break;
			}
		}
	}

	while (pad_is_pressed(joyID, theButton))
	{
		Sleep(20);

		if (!pump_message())
			return FALSE;

		pad_update_state();
	}

	*pButton = theButton;

	return TRUE;
}


static
BOOL
configure(
	Sint32			joyID,
	JoyPad*			pJoy)
{
	const Sint32	button[INPUT_NUM_BUTTON] =
	{
		INPUT_JOYSTICK_BUTTON1,  INPUT_JOYSTICK_BUTTON2,  INPUT_JOYSTICK_BUTTON3,  INPUT_JOYSTICK_BUTTON4,
		INPUT_JOYSTICK_BUTTON5,  INPUT_JOYSTICK_BUTTON6,  INPUT_JOYSTICK_BUTTON7,  INPUT_JOYSTICK_BUTTON8,
		INPUT_JOYSTICK_BUTTON9,  INPUT_JOYSTICK_BUTTON10, INPUT_JOYSTICK_BUTTON11, INPUT_JOYSTICK_BUTTON12,
		INPUT_JOYSTICK_UP,       INPUT_JOYSTICK_DOWN,     INPUT_JOYSTICK_LEFT,     INPUT_JOYSTICK_RIGHT
	};

	Uint32			numPad;
	Uint32			i;
	Sint32			buttonI   = -1;
	Sint32			buttonII  = -1;
	Sint32			buttonIII = -1;
	Sint32			buttonIV  = -1;
	Sint32			buttonV   = -1;
	Sint32			buttonVI  = -1;
	Sint32			select    = -1;
	Sint32			run       = -1;
	Sint32			up        = -1;
	Sint32			right     = -1;
	Sint32			down      = -1;
	Sint32			left      = -1;

	if (!pad_init())
	{
		add_text("Failed initializing input.");
		return FALSE;
	}

	numPad = _nJoySticks;

	add_text("Initialized Input: %ld joystick(s) found.", numPad);

	if (numPad == 0 || joyID >= numPad)
	{
		add_text("Input device for pad #%ld not found.", joyID+1);
		MessageBox(_hWnd, "Input device not found.", "Exiting configuration.", MB_OK);
		return FALSE;
	}

	for (i = 0; i < INPUT_NUM_BUTTON; i++)
	{
		pad_connect_button(joyID, button[i], button[i]);
	}

	add_text("\n");
	add_text("Press ESC to abort.");
	add_text("Press a button for \"button I\"...");   if (!get_button(joyID, &buttonI))   return FALSE;
	add_text("Press a button for \"button II\"...");  if (!get_button(joyID, &buttonII))  return FALSE;
	add_text("Press a button for \"button III\"..."); if (!get_button(joyID, &buttonIII)) return FALSE;
	add_text("Press a button for \"button IV\"...");  if (!get_button(joyID, &buttonIV))  return FALSE;
	add_text("Press a button for \"button V\"...");   if (!get_button(joyID, &buttonV))   return FALSE;
	add_text("Press a button for \"button VI\"...");  if (!get_button(joyID, &buttonVI))  return FALSE;
	add_text("Press a button for \"SELECT\"...");     if (!get_button(joyID, &select))    return FALSE;
	add_text("Press a button for \"RUN\"...");        if (!get_button(joyID, &run))       return FALSE;
	add_text("Press a button for \"UP\"...");         if (!get_button(joyID, &up))        return FALSE;
	add_text("Press a button for \"RIGHT\"...");      if (!get_button(joyID, &right))     return FALSE;
	add_text("Press a button for \"DOWN\"...");       if (!get_button(joyID, &down))      return FALSE;
	add_text("Press a button for \"LEFT\"...");       if (!get_button(joyID, &left))      return FALSE;

	pJoy->i      = buttonI;
	pJoy->ii     = buttonII;
	pJoy->iii    = buttonIII;
	pJoy->iv     = buttonIV;
	pJoy->v      = buttonV;
	pJoy->vi     = buttonVI;
	pJoy->select = select;
	pJoy->run    = run;
	pJoy->up     = up;
	pJoy->right  = right;
	pJoy->down   = down;
	pJoy->left   = left;

	return TRUE;
}


static
LRESULT
CALLBACK
padconfig_wnd_proc(
	HWND		hWnd,
	UINT		uMsg,
	WPARAM		wParam,
	LPARAM		lParam)
{
	switch(uMsg)
	{
	case WM_CREATE:
		_FontWidth  = get_font_width(hWnd);
		_FontHeight = get_font_height(hWnd);
		set_window_size(hWnd);
		break;

	case WM_PAINT:
		update_window(hWnd);
		break;

	case WM_KEYDOWN:
		if (wParam == VK_ESCAPE)
			PostMessage(hWnd, WM_CLOSE, 0, 0);
		break;

	case WM_CLOSE:
		DestroyWindow(hWnd);
		UnregisterClass(_pCaption, _hInstance);
		_hInstance = NULL;
		break;
    }

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}

static
BOOL
pad_config_init()
{
	WNDCLASS		wc;
	HWND			hWnd;

	ZeroMemory(&wc, sizeof(wc));
	wc.style		 = 0;
	wc.lpfnWndProc	 = padconfig_wnd_proc;
	wc.cbClsExtra	 = 0;
	wc.cbWndExtra	 = 0;
	wc.hInstance	 = _hInstance;
	wc.hIcon		 = LoadIcon(NULL, IDI_APPLICATION);
	wc.hCursor		 = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
	wc.lpszMenuName	 = "";
	wc.lpszClassName = _pCaption;

	if (RegisterClass(&wc) == 0)
		return FALSE;

	hWnd = CreateWindow(
		_pCaption,
		_pCaption,
		WS_MINIMIZEBOX | WS_SYSMENU | WS_CAPTION,
		CW_USEDEFAULT,
		CW_USEDEFAULT,
		0,
		0,
		NULL,
		NULL,
		_hInstance,
		NULL
	);

	if (hWnd == NULL)
		return FALSE;

	_hWnd      = hWnd;

	/* uv𖳌ɂ */
//	hMenu = GetSystemMenu(hWnd, FALSE);
//	RemoveMenu(hMenu, SC_CLOSE, MF_BYCOMMAND);

	ShowWindow(hWnd, SW_SHOWNORMAL);
	UpdateWindow(hWnd);

	return TRUE;
}


BOOL
PADCONFIG_Init(
	HINSTANCE		hInstance)
{
	int				i;

	if (_hInstance != NULL)
		return FALSE;

	_hInstance = hInstance;
	_Line      = 0;

	for (i = 0; i < N_LINES; ++i)
	{
		_pText[i] = _Text[i];
	}

	return pad_config_init();
}


BOOL
PADCONFIG_Configure(
	Sint32			joyID,
	JoyPad*			pJoy)
{
	if (_hInstance != NULL)
	{
		return configure(joyID, pJoy);
	}

	return FALSE;
}


void
PADCONFIG_Deinit()
{
	if (_hInstance)
	{
		DestroyWindow(_hWnd);
		UnregisterClass(_pCaption, _hInstance);
		_hInstance = NULL;
	}
}


void
PADCONFIG_Update()
{
	if (_hInstance != NULL)
	{
		update_window(_hWnd);
		InvalidateRect(_hWnd, NULL, TRUE);
	}
}


