#ifndef PSX_MEMORY_HPP
#define PSX_MEMORY_HPP

#include "EFramework.hpp"
#include "PSXHardwareManager.hpp"

namespace NeoPSX
{
	/// Allocates playstation memory and manages access to it.
	class PSXMemory
	{
	public:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Constructors

		/** The default constructor. Allocates memory and checks for errors.
		  */
		PSXMemory();

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Destructors

		/** The default destructor. 
		  */
		~PSXMemory();

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Basic Interface Functions

		/** Resets the memory areas.
		  */
		void Reset();

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Memory Writing Functions

		/** Writes 8 bits of data to Playstation memory.
		  */
		void WriteByte(uint32_t address , uint8_t data);
		/** Writes 16 bits of data to Playstation memory.
		  */
		void WriteHword(uint32_t address, uint16_t data);
		/** Writes 32 bits of data to Playstation memory.
		  */
		void WriteWord(uint32_t address , uint32_t data);

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Memory Reading Functions

		/** Reads 8 bits of data from Playstation memory.
		  */
		uint8_t ReadByte  (uint32_t address);
		/** Reads 16 bits of data from Playstation memory.
		  */
		uint16_t ReadHword(uint32_t address);
		/** Reads 32 bits of data from Playstation memory.
		  */
		uint32_t ReadWord (uint32_t address);

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Pointer Retrieval

		/** Retrieves a pointer at RAM, ROM, or SPAD memory.
		  */
		uint8_t* GetPointer(uint32_t address);

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Addressing Macros

		#define Group1(A1,A2)             ADDRESSRANGE<A1,A2>
		#define Group2(A1,A2,A3,A4)       ADDRESSRANGEGROUP<Group1(A1,A2), Group1(A3,A4) >
		#define Group3(A1,A2,A3,A4,A5,A6) ADDRESSRANGEGROUP<Group1(A1,A2), Group2(A3,A4,A5,A6) >
	protected:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Protected Enumerations

		enum MemorySize
		{
			MS_RAM         = (2 * 1024 * 1024),
			MS_ROM         = (512 * 1024),
			MS_SCRATCH_PAD = 1024
		};

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Protected Variables

		uint8_t* mRAM;  ///< The playstation random access memory area.
		uint8_t* mROM;  ///< The playstation read only memory area.
		uint8_t* mSPAD; ///< The playstation scratchpad memory area.

		PSXHardwareManager mHardware; ///< The hardware device manager.

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Memory Range Variables

		typedef Group1(0x1F800000UL, 0x1F8003FFUL)                             SPAD; ///< The scratchpad memory range.
		typedef Group1(0x1F801000UL, 0x1F802FFFUL)                             HW;   ///< The hardware memory range.
		typedef Group2(0xBFC00000UL, 0xBFC7FFFFUL, 0x1FC00000UL, 0x1FC7FFFFUL) ROM;  ///< The rom memory range.
		typedef Group3(0x00000000UL, 0x001FFFFFUL, 0x80000000UL, 0x801FFFFFUL, 
												   0xA0000000UL, 0xA01FFFFFUL) RAM;  ///< The ram memory range.
	};
} // Namespace NeoPSX

#endif // PSX_MEMORY_HPP