#include "PSXInterpreter.hpp"
#include "PSXUtility.hpp"

namespace NeoPSX
{

	PSXInterpreter::PSXInterpreter()
	: mMemory( NULL )
	{
		//===============================================================
		// Create a new instance of the Playstation memory class
		//===============================================================
		mMemory.reset( new PSXMemory() );

		//===============================================================
		// Reset the Playstation CPU and its components.
		//===============================================================
		Reset();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void PSXInterpreter::Execute()
	{
		try
		{
			//===============================================================
			// Read the next opcode from memory, increment PC and cycles
			//===============================================================
			mContext.Opcode = mMemory->ReadWord(mContext.Pc);
			mContext.Pc += 4; mContext.Cycle++;

			//===============================================================
			// Execute opcode through function pointer table.
			//===============================================================
			(this->*mTableBASIC[ _Opcode_ ])();
		} catch(EFException error) {
			EFLogManager::GetSingleton().LogMessage(LML_CRITICAL, "%s @ 0x%08X", error.GetDetailedInformation().c_str(), mContext.Pc);

			//if(error.GetExceptionCode() == EFException::EC_FATAL_ERROR)
			//	std::terminate();
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void PSXInterpreter::Reset()
	{
		//===============================================================
		// Clear the processor context data.
		//===============================================================
		PSXClearMemory(&mContext, sizeof(TR3000Context));

		//===============================================================
		// Set the CoProcessor0 register bits.
		//===============================================================
		mContext.Cp0[RD_COP0_STATUS] = 0x10900000;
		mContext.Cp0[RD_COP0_PRID  ] = 0x00000002;

		//===============================================================
		// Clear the CoProcessor0 status register stack.
		//===============================================================
		mCp0Stack.clear();

		//===============================================================
		// Reset the memory interface.
		//===============================================================
		mMemory->Reset();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	TR3000Context& PSXInterpreter::GetContext() const
	{
		return *(TR3000Context*)&mContext;
	}

	PSXMemory& PSXInterpreter::GetMemory() const
	{
		return *(mMemory.get());
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	const void PSXInterpreter::SetRegisterGPR(uint32_t reg, uint32_t value)
	{
		if(reg > 31) return;
		mContext.Reg[reg] = value;
	}

	const uint32_t PSXInterpreter::GetRegisterGPR(uint32_t reg) const
	{
		if(reg > 31) return 0;
		return mContext.Reg[reg];
	}

	const void PSXInterpreter::SetPC(uint32_t value)
	{
		mContext.Pc = value;
	}

	const uint32_t PSXInterpreter::GetPC() const
	{
		return mContext.Pc;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //                                                Opcode Functions                                                     //
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void PSXInterpreter::PushStatusCP0()
	{
		//===============================================================
		// Pop an item off if the stack is too large.
		//===============================================================
		if(mCp0Stack.size() >= 3)
		{
			mCp0Stack.pop_front();
		}

		//===============================================================
		// Push the next item on the stack.
		//===============================================================
		mCp0Stack.push_back( mContext.Cp0[RD_COP0_STATUS] );
	}

	void PSXInterpreter::PopStatusCP0()
	{
		//===============================================================
		// Ensure that the status register stack isn't empty.
		//===============================================================
		if(mCp0Stack.empty()) return;
		
		//===============================================================
		// Put the most recent item into the status register and pop.
		//===============================================================
		mContext.Cp0[RD_COP0_STATUS] = mCp0Stack.back();
		mCp0Stack.pop_back();
	}

	void PSXInterpreter::PerformBranchOrJump(uint32_t address, bool link, uint32_t linkRegister)
	{
		//===============================================================
		// Execute the delay slot instruction.
		//===============================================================
		Execute();

		//===============================================================
		// Set the link register if requested.
		//===============================================================
		if(link && linkRegister > 0 && linkRegister < 32)
		{
			mContext.Reg[linkRegister] = mContext.Pc;
		}

		//===============================================================
		// Update the program counter.
		//===============================================================
		mContext.Pc = address;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Jump Instructions ==//

	void PSXInterpreter::cpuJ()
	{
		PerformBranchOrJump(CALCULATE_JUMP_ADDRESS());
	}

	void PSXInterpreter::cpuJAL()
	{
		PerformBranchOrJump(CALCULATE_JUMP_ADDRESS(), true, 31);
	}

	void PSXInterpreter::cpuJR()
	{
		PerformBranchOrJump(_Rs_);
	}

	void PSXInterpreter::cpuJALR()
	{
		PerformBranchOrJump(_Rs_, true, _Rd_);
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Branch Instructions ==//

	void PSXInterpreter::cpuBEQ()
	{
		if(_Rs_ == _Rt_)
		{
			PerformBranchOrJump(CALCULATE_BRANCH_ADDRESS());
		}
	}

	void PSXInterpreter::cpuBNE()
	{
		if(_Rs_ != _Rt_)
		{
			PerformBranchOrJump(CALCULATE_BRANCH_ADDRESS());
		}
	}

	void PSXInterpreter::cpuBLEZ()
	{
		if(_Rs_ <= 0)
		{
			PerformBranchOrJump(CALCULATE_BRANCH_ADDRESS());
		}
	}

	void PSXInterpreter::cpuBGTZ()
	{
		if(_Rs_ > 0)
		{
			PerformBranchOrJump(CALCULATE_BRANCH_ADDRESS());
		}
	}

	void PSXInterpreter::cpuBLTZ()
	{
		if(_Rs_ < 0)
		{
			PerformBranchOrJump(CALCULATE_BRANCH_ADDRESS());
		}
	}

	void PSXInterpreter::cpuBGEZ()
	{
		if(_Rs_ >= 0)
		{
			PerformBranchOrJump(CALCULATE_BRANCH_ADDRESS());
		}
	}

	void PSXInterpreter::cpuBLTZAL()
	{
		if(_Rs_ < 0)
		{
			PerformBranchOrJump(CALCULATE_BRANCH_ADDRESS(), true, 31);
		}
	}

	void PSXInterpreter::cpuBGEZAL()
	{
		if(_Rs_ >= 0)
		{
			PerformBranchOrJump(CALCULATE_BRANCH_ADDRESS(), true, 31);
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Immediate Instructions ==//
	
	void PSXInterpreter::cpuADDI()
	{
		if(!Rt) return;
		_Rt_ = (_Rs_ + _SImmediate_);
	}

	void PSXInterpreter::cpuADDIU()
	{
		if(!Rt) return;
		_Rt_ = (_Rs_ + _UImmediate_);
	}

	void PSXInterpreter::cpuSLTI()
	{
		if(!Rt) return;
		_Rt_ = ((signed long)_Rs_ < _SImmediate_);
	}

	void PSXInterpreter::cpuSLTIU()
	{
		if(!Rt) return;
		_Rt_ = (_Rs_ < _UImmediate_);
	}

	void PSXInterpreter::cpuANDI()
	{
		if(!Rt) return;
		_Rt_ = (_Rs_ & _UImmediate_);
	}

	void PSXInterpreter::cpuORI()
	{
		if(!Rt) return;
		_Rt_ = (_Rs_ | _UImmediate_);
	}

	void PSXInterpreter::cpuXORI()
	{
		if(!Rt) return;
		_Rt_ = (_Rs_ ^ _UImmediate_);
	}

	void PSXInterpreter::cpuLUI()
	{
		if(!Rt) return;
		_Rt_ = (_UImmediate_<<16);
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Load Instructions ==//

	void PSXInterpreter::cpuLB()
	{
		if(!Rt) return;
		_Rt_ = mMemory->ReadByte( ((_SImmediate_) + _Rs_) );
	}

	void PSXInterpreter::cpuLH()
	{
		if(!Rt) return;
		_Rt_ = mMemory->ReadHword( ((_SImmediate_) + _Rs_) );
	}

	void PSXInterpreter::cpuLWL()
	{
		if(!Rt) return;
		uint32_t address = ((_SImmediate_) + _Rs_);
		uint32_t word    = mMemory->ReadWord( address&~3 );

		switch(address&3)
		{
		case 3:
			{
				_Rt_ = word;
			}
			break;
		case 2:
			{
				_Rt_ = (_Rt_&0x000000FF) | (word<<8);
			}
			break;
		case 1:
			{
				_Rt_ = (_Rt_&0x0000FFFF) | (word<<16);
			}
			break;
		case 0:
			{
				_Rt_ = (_Rt_&0x00FFFFFF) | (word<<24);
			}
			break;
		}
	}

	void PSXInterpreter::cpuLW()
	{
		if(!Rt) return;
		_Rt_ = mMemory->ReadWord( ((_SImmediate_) + _Rs_) );
	}

	void PSXInterpreter::cpuLBU()
	{
		if(!Rt) return;
		_Rt_ = mMemory->ReadByte( ((_SImmediate_) + _Rs_) );
	}

	void PSXInterpreter::cpuLHU()
	{
		if(!Rt) return;
		_Rt_ = mMemory->ReadHword( ((_SImmediate_) + _Rs_) );
	}

	void PSXInterpreter::cpuLWR()
	{
		if(!Rt) return;
		uint32_t address = ((_SImmediate_) + _Rs_);
		uint32_t word    = mMemory->ReadWord( address&~3 );

		switch(address&3)
		{
		case 3:
			{
				_Rt_ = word;
			}
			break;
		case 2:
			{
				_Rt_ = (_Rt_&0xFF000000) | (word>>8);
			}
			break;
		case 1:
			{
				_Rt_ = (_Rt_&0xFFFF0000) | (word>>16);
			}
			break;
		case 0:
			{
				_Rt_ = (_Rt_&0xFFFFFF00) | (word>>24);
			}
			break;
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Store Instructions ==//

	void PSXInterpreter::cpuSB()
	{
		mMemory->WriteByte(((_SImmediate_) + _Rs_), (uchar)_Rt_);
	}

	void PSXInterpreter::cpuSH()
	{
		mMemory->WriteHword(((_SImmediate_) + _Rs_), (uchar)_Rt_);
	}

	void PSXInterpreter::cpuSWL()
	{
		uint32_t address = ((_SImmediate_) + _Rs_);
		uint32_t word   = NULL;

		switch(address&3)
		{
		case 3:
			{
				word = _Rt_;
			}
			break;
		case 2:
			{
				word = (_Rt_&0x000000FF) | (word<<8);
			}
			break;
		case 1:
			{
				word = (_Rt_&0x0000FFFF) | (word<<16);
			}
			break;
		case 0:
			{
				word = (_Rt_&0x00FFFFFF) | (word<<24);
			}
			break;
		}

		mMemory->WriteWord(address&~3, word);
	}

	void PSXInterpreter::cpuSW()
	{
		mMemory->WriteWord(((_SImmediate_) + _Rs_), _Rt_);
	}

	void PSXInterpreter::cpuSWR()
	{
		uint32_t address = ((_SImmediate_) + _Rs_);
		uint32_t word   = NULL;

		switch(address&3)
		{
		case 3:
			{
				word = _Rt_;
			}
			break;
		case 2:
			{
				word = (_Rt_&0xFF000000) | (word>>8);
			}
			break;
		case 1:
			{
				word = (_Rt_&0xFFFF0000) | (word>>16);
			}
			break;
		case 0:
			{
				word = (_Rt_&0xFFFFFF00) | (word>>24);
			}
			break;
		}

		mMemory->WriteWord(address&~3, word);
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Shift Instructions ==//

	void PSXInterpreter::cpuSLL()
	{
		if(!Rd) return;
		_Rd_ = (_Rt_ << _Shift_);
	}

	void PSXInterpreter::cpuSRL()
	{
		if(!Rd) return;
		_Rd_ = (_Rt_ >> _Shift_);
	}

	void PSXInterpreter::cpuSRA()
	{
		if(!Rd) return;
		_Rd_ = (_Rt_ >> _Shift_);
	}

	void PSXInterpreter::cpuSLLV()
	{
		if(!Rd) return;
		_Rd_ = (_Rt_ << (_Rs_&0x1F));
	}

	void PSXInterpreter::cpuSRLV()
	{
		if(!Rd) return;
		_Rd_ = (_Rt_ >> (_Rs_&0x1F));
	}

	void PSXInterpreter::cpuSRAV()
	{
		if(!Rd) return;
		_Rd_ = (_Rt_ >> (_Rs_&0x1F));
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Special Instructions ==//

	void PSXInterpreter::cpuSYSCALL()
	{
		EFLogManager::GetSingleton().LogMessage(LML_CRITICAL, "PSXInterpreter: SYSCALL instruction.\n");

		// Push the current status register value.
		PushStatusCP0();

		// Set exception code and error PC.
		mContext.Cp0[RD_COP0_CAUSE] = (0x08<<2);
		mContext.Cp0[RD_COP0_EPC  ] = mContext.Pc;
	}

	void PSXInterpreter::cpuBREAK()
	{
		EFLogManager::GetSingleton().LogMessage(LML_CRITICAL, "PSXInterpreter: BREAK instruction.\n");

		// Push the current status register value.
		PushStatusCP0();

		// Set exception code and error PC.
		mContext.Cp0[RD_COP0_CAUSE] = (0x09<<2);
		mContext.Cp0[RD_COP0_EPC  ] = mContext.Pc;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Multiply and Divide Instructions ==//

	void PSXInterpreter::cpuMULT()
	{
		sint64_t value = ((signed)_Rs_ * (signed)_Rt_);
		mContext.Lo   = (uint32_t)(value&0x00000000FFFFFFFF);
		mContext.Hi   = (uint32_t)(value&0xFFFFFFFF00000000)>>32;
	}

	void PSXInterpreter::cpuMULTU()
	{
		uint64_t value = (_Rs_ * _Rt_);
		mContext.Lo    = (uint32_t)(value&0x00000000FFFFFFFF);
		mContext.Hi    = (uint32_t)(value&0xFFFFFFFF00000000)>>32;
	}

	void PSXInterpreter::cpuDIV()
	{
		if(_Rt_ == 0) 
		{
			mContext.Lo = mContext.Hi = 0;
			return;
		}
		mContext.Lo = (signed)_Rs_ / (signed)_Rt_;
		mContext.Hi = (signed)_Rs_ % (signed)_Rt_;
	}

	void PSXInterpreter::cpuDIVU()
	{
		if(_Rt_ == 0)
		{
			mContext.Lo = mContext.Hi = 0;
			return;
		}
		mContext.Lo = _Rs_ / _Rt_;
		mContext.Hi = _Rs_ % _Rt_;
	}

	void PSXInterpreter::cpuMFHI()
	{
		if(!Rd) return;
		_Rd_ = mContext.Hi;
	}

	void PSXInterpreter::cpuMTHI()
	{
		mContext.Hi = _Rd_;
	}

	void PSXInterpreter::cpuMFLO()
	{
		if(!Rd) return;
		_Rd_ = mContext.Lo;
	}

	void PSXInterpreter::cpuMTLO()
	{
		mContext.Lo = _Rd_;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== Three Operand Register Type Instructions ==//

	void PSXInterpreter::cpuADD()
	{
		if(!Rd) return;
		_Rd_ = (signed)_Rt_ + (signed)_Rs_;
	}

	void PSXInterpreter::cpuADDU()
	{
		if(!Rd) return;
		_Rd_ = _Rt_ + _Rs_;
	}

	void PSXInterpreter::cpuSUB()
	{
		if(!Rd) return;
		_Rd_ = (signed)_Rt_ + (signed)_Rs_;
	}

	void PSXInterpreter::cpuSUBU()
	{
		if(!Rd) return;
		_Rd_ = _Rt_ + _Rs_;
	}

	void PSXInterpreter::cpuAND()
	{
		if(!Rd) return;
		_Rd_ = (_Rs_ & _Rt_);
	}

	void PSXInterpreter::cpuOR()
	{
		if(!Rd) return;
		_Rd_ = (_Rs_ | _Rt_);
	}

	void PSXInterpreter::cpuXOR()
	{
		if(!Rd) return;
		_Rd_ = (_Rs_ ^ _Rt_);
	}

	void PSXInterpreter::cpuNOR()
	{
		if(!Rd) return;
		_Rd_ = ~(_Rs_ | _Rt_);
	}

	void PSXInterpreter::cpuSLT()
	{
		if(!Rd) return;
		_Rd_ = ((signed)_Rs_ < (signed)_Rt_);
	}

	void PSXInterpreter::cpuSLTU()
	{
		if(!Rd) return;
		_Rd_ = (_Rs_ < _Rt_);
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//== CoProcessor0 Instructions ==//

	void PSXInterpreter::cpuMFC0()
	{
		if(!Rt) return;
		_Rt_ = mContext.Cp0[Rd];
	}

	void PSXInterpreter::cpuMTC0()
	{
		mContext.Cp0[Rd] = _Rt_;
	}

	void PSXInterpreter::cpuRFE()
	{
		PopStatusCP0();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void PSXInterpreter::cpuUNK()
	{
		EF_THROW(EFException::EC_ILLEGAL_OPCODE, "Unknown opcode execution attempted.");
	}

	void PSXInterpreter::cpuSPECIAL()
	{
		(this->*mTableSPECIAL[ (mContext.Opcode&0x3F) ])();
	}

	void PSXInterpreter::cpuBCOND()
	{
		(this->*mTableBCOND[ Rt ])();
	}

	void PSXInterpreter::cpuCOP0()
	{
		(this->*mTableCOP0[ Rs ])();
	}

	void PSXInterpreter::cpuCOP2()
	{
		(this->*mTableCOP2Basic[ (mContext.Opcode&0x3F) ])();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	const PSXInterpreter::FUNCPTR PSXInterpreter::mTableBASIC[64] = 
	{
		cpuSPECIAL, cpuBCOND, cpuJ   , cpuJAL  , cpuBEQ , cpuBNE, cpuBLEZ, cpuBGTZ,
		cpuADDI   , cpuADDIU, cpuSLTI, cpuSLTIU, cpuANDI, cpuORI, cpuXORI, cpuLUI ,
		cpuCOP0   , cpuUNK  , cpuCOP2, cpuUNK  , cpuUNK , cpuUNK, cpuUNK , cpuUNK ,
		cpuUNK    , cpuUNK  , cpuUNK , cpuUNK  , cpuUNK , cpuUNK, cpuUNK , cpuUNK ,
		cpuLB     , cpuLH   , cpuLWL , cpuLW   , cpuLBU , cpuLHU, cpuLWR , cpuUNK ,
		cpuSB     , cpuSH   , cpuSWL , cpuSW   , cpuUNK , cpuUNK, cpuSWR , cpuUNK ,
		cpuUNK    , cpuUNK  , cpuUNK , cpuUNK  , cpuUNK , cpuUNK, cpuUNK , cpuUNK ,
		cpuUNK    , cpuUNK  , cpuUNK , cpuUNK  , cpuUNK , cpuUNK, cpuUNK , cpuUNK
	};

	const PSXInterpreter::FUNCPTR PSXInterpreter::mTableSPECIAL[64] = 
	{
		cpuSLL , cpuUNK  , cpuSRL , cpuSRA , cpuSLLV   , cpuUNK  , cpuSRLV, cpuSRAV,
		cpuJR  , cpuJALR , cpuUNK , cpuUNK , cpuSYSCALL, cpuBREAK, cpuUNK , cpuUNK ,
		cpuMFHI, cpuMTHI , cpuMFLO, cpuMTLO, cpuUNK    , cpuUNK  , cpuUNK , cpuUNK ,
		cpuMULT, cpuMULTU, cpuDIV , cpuDIVU, cpuUNK    , cpuUNK  , cpuUNK , cpuUNK ,
		cpuADD , cpuADDU , cpuSUB , cpuSUBU, cpuAND    , cpuOR   , cpuXOR , cpuNOR ,
		cpuUNK , cpuUNK  , cpuSLT , cpuSLTU, cpuUNK    , cpuUNK  , cpuUNK , cpuUNK ,
		cpuUNK , cpuUNK  , cpuUNK , cpuUNK , cpuUNK    , cpuUNK  , cpuUNK , cpuUNK ,
		cpuUNK , cpuUNK  , cpuUNK , cpuUNK , cpuUNK    , cpuUNK  , cpuUNK , cpuUNK
	};

	const PSXInterpreter::FUNCPTR PSXInterpreter::mTableBCOND[32] = 
	{
		cpuBLTZ  , cpuBGEZ  , cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK   , cpuUNK   , cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuBLTZAL, cpuBGEZAL, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK   , cpuUNK   , cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK
	};

	const PSXInterpreter::FUNCPTR PSXInterpreter::mTableCOP0[32] = 
	{
		cpuMFC0, cpuUNK, cpuUNK, cpuUNK, cpuMTC0, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK , cpuUNK, cpuUNK, cpuUNK, cpuUNK , cpuUNK, cpuUNK, cpuUNK,
		cpuRFE , cpuUNK, cpuUNK, cpuUNK, cpuUNK , cpuUNK, cpuUNK, cpuUNK,
		cpuUNK , cpuUNK, cpuUNK, cpuUNK, cpuUNK , cpuUNK, cpuUNK, cpuUNK
	};

	const PSXInterpreter::FUNCPTR PSXInterpreter::mTableCOP2[64] = 
	{
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK
	};

	const PSXInterpreter::FUNCPTR PSXInterpreter::mTableCOP2Basic[32] = 
	{
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK,
		cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK, cpuUNK
	};
} // Namespace NeoPSX