#ifndef EF_EXCEPTION_HPP
#define EF_EXCEPTION_HPP

#include "EFPlatform.hpp"

namespace EFramework
{
	/// Macro which simplifies the throwing of an exception.
	#define EF_THROW(code, message) { \
		throw( EFException(code, message, __FILE__, __FUNCTION__, __LINE__) ); \
	}

	/// Custom assertion function
	#define EF_ASSERT(expression) {                                                              \
		if(!(expression)) {                                                                      \
			EF_THROW(EFException::EC_FATAL_ERROR, "Assertion Failed: " + EFString(#expression)); \
		}                                                                                        \
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/// The definition of an a generic exception containing detailed information about an error.
	class EF_EXPORT EFException
	{
	public:
		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Enumerations

		/// An enumeration with individual exception error codes.
		enum ExceptionCode
		{
			EC_FATAL_ERROR,
			EC_GENERAL_ERROR,
			EC_INVALID_PARAMETERS,
			EC_ILLEGAL_ADDRESS,
			EC_ILLEGAL_OPCODE,
			EC_ILLEGAL_PACKET,
			EC_PLUGIN_ERROR,
			EC_STREAM_ERROR,
			EC_GUI_ERROR
		};

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Constructors

		/** The basic constructor.
		  */
		EFException(int code, const EFString& message, const EFString& file, const EFString& function, long line);
		/** The assignment operator.
		  */
		EFException operator = (EFException& right);

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Exception Information Functions

		/** Returns the exception code for this message.
		  */
		int GetExceptionCode() throw();
		/** Returns detailed information about the error.
		  */
		EFString GetDetailedInformation() throw();
	protected:
		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Exception Code Function

		/** Returns the name of the exception code for this exception.
		  */
		EFString GetExceptionCodeStr() throw();

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Variables

		long      mLine;     ///< The line in the file at which the error occurred.
		int       mCode;     ///< The error code for this exception.
		EFString  mFile;     ///< The name of the file which threw the exception.
		EFString  mMessage;  ///< The message describing the error which triggered this exception.
		EFString  mFunction; ///< The name of the function in which the error was thrown.
	};
} // Namespace EFramework

#endif // EF_EXCEPTION_HPP