#ifndef PSX_ROOT_COUNTER_HPP
#define PSX_ROOT_COUNTER_HPP

#include "PSXUtility.hpp"

namespace NeoPSX
{
	/// Emulates the programmable timers inside the Playstation.
	class PSXRootCounter : public ADDRESSRANGE<0x1F801100, 0x1F801128>
	{
	public:
		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Constructors

		/** The default constructor.
		  */
		PSXRootCounter();

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Emulation Functions

		/** Resets all of the Playstation root counters.
		  */
		void Reset();
		/** Updates a single Playstation root counter.
		  */
		void Update(uint16_t counter);
		/** Updates all Playstation root counters.
		  */
		void Update();

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Root Counter Register Writes

		/** Writes a byte sized value to the root counters.
		  */
		void WriteByte(uint32_t address, uint8_t value);
		/** Writes a halfword sized value to the root counters.
		  */
		void WriteHword(uint32_t address, uint16_t value);
		/** Writes a word sized value to the root counters.
		  */
		void WriteWord(uint32_t address, uint32_t value);

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Root Counter Register Reads

		/** Reads a byte sized value from the root counters.
		  */
		uint8_t ReadByte(uint32_t address) const;
		/** Reads a halfword sized value from the root counters.
		  */
		uint16_t ReadHword(uint32_t address) const;
		/** Reads a word sized value from the root counters.
		  */
		uint32_t ReadWord(uint32_t address) const;
	private:
		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Structures

		/// Defines a Playstation root counter device.
		struct TRootCounter
		{
			uint32_t Count;
			uint32_t Mode;
			uint32_t Target;
		};

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Enumerations

		/// Defines the root counter mode register masks.
		enum CounterModeMasks
		{
			CMM_TARGET_MASK = 0x00000008
		};

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Defines

		#define COUNTER_INDEX(address)    (((address)&0x00000030)>>4)
		#define COUNTER_REGISTER(address) (((address)&0x0000000F)%3)
		#define _Counter_ (mRootCounter[(COUNTER_INDEX(address))])

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Variables

		TRootCounter mRootCounter[4]; ///< The four Playstation root counters.
	};
} // Namespace NeoPSX

#endif // PSX_ROOT_COUNTER_HPP