#include "PSXMainWindow.hpp"
#include <atldlgs.h>
#include <atlmisc.h>
#include <atlcom.h>
#include <atlctl.h>

#include "PSXUtility.hpp"
#include "PSXAboutDialog.hpp"
#include "PSXPluginDialog.hpp"
#include "PSXEmulationDialog.hpp"


namespace NeoPSX
{
	RECT PSXMainWindow::mMainWindowRect = {0, 0, 460, 360};

	PSXIdleHandler PSXMainWindow::mIdleHandler;

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	#define STATUSBAR_REGKEY "StatusBarState"
	#define RECENTDOC_REGKEY "Software\\NeoPSX"

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	PSXMainWindow::PSXMainWindow()
	: mWindowTitle("NeoPSX - Playstation Emulator v")
	{
		//===============================================================
		// Construct our window title string.
		//===============================================================
		mVersion.SetVersionData(0, 4, 0);

		mWindowTitle += EFString( mVersion.GetString() );
		#if defined (_DEBUG_)
				mWindowTitle += " [DEBUG]";
		#endif

		//===============================================================
		// Attempt to create our main window.
		//===============================================================
		Create(NULL, mMainWindowRect, _T( mWindowTitle.c_str() ), WS_OVERLAPPEDWINDOW);
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	LRESULT PSXMainWindow::OnCreate(UINT, WPARAM, LPARAM, BOOL)
	{
		//===============================================================
		// Show our window and center it on the screen.
		//===============================================================
		ShowWindow(SW_SHOW);
		CenterWindow();
		UpdateWindow();

		//===============================================================
		// Add any desired window controls.
		//===============================================================
		EFString statusBarString = "NeoPSX v" + EFString( mVersion.GetString() ) + " :: EFramework v";
		statusBarString         += EFString( EFPlatformManager::GetFrameworkVersion().GetString() ) + " :: Initialized";
		CreateSimpleStatusBar(_T(statusBarString.c_str()));

		//===============================================================
		// Set the menu and icon for this window.
		//===============================================================
		mMenu.LoadMenu(IDR_NEOPSX_MENU);
		SetMenu(mMenu.m_hMenu);
		SetIcon( AtlLoadIcon(MAKEINTRESOURCE(IDR_NEOPSX_ICON)) );

		//===============================================================
		// Create the recent file menu.
		//===============================================================
		CMenuHandle menu = mMenu.m_hMenu;
		CMenuHandle file = menu.GetSubMenu(0);
		CMenuHandle rcnt = file.GetSubMenu(5);

		mDocList.SetMenuHandle(rcnt);
		mDocList.SetMaxEntries(12);
		mDocList.ReadFromRegistry(RECENTDOC_REGKEY);

		//===============================================================
		// Load and set the status bar visibility state.
		//===============================================================
		bool IsVisible = true;

		try
		{
			EFRegistryKey key("NeoPSX");
			IsVisible = (key.ReadInteger(STATUSBAR_REGKEY, 1) == 1) ? true : false;
		} catch(EFException& exception) {
			AtlMessageBox(NULL, exception.GetDetailedInformation().c_str(), "Exception", MB_OK|MB_ICONERROR);
		}

		UISetCheck(ID_VIEW_STATUSBAR,IsVisible);
		::ShowWindow(m_hWndStatusBar, IsVisible ? SW_SHOWNOACTIVATE : SW_HIDE);

		//===============================================================
		// Now we initialize the emulator components.
		//===============================================================
		InitializeLogFiles();

		return 0;
	}

	LRESULT PSXMainWindow::OnClose(UINT, WPARAM, LPARAM, BOOL)
	{
		//===============================================================
		// Ensure that they really want to quit.
		//===============================================================
		if(!NeoPSX::Ask("Are you sure?", "Quit")) return 1;

		//===============================================================
		// Save the status bar visibility state.
		//===============================================================
		try
		{
			EFRegistryKey key("NeoPSX");
			key.WriteInteger(STATUSBAR_REGKEY, ::IsWindowVisible(m_hWndStatusBar));
		} catch(EFException& exception) {
			AtlMessageBox(NULL, exception.GetDetailedInformation().c_str(), "Exception", MB_OK|MB_ICONERROR);
		}

		DestroyWindow();
		return 0;
	}

	LRESULT PSXMainWindow::OnPaint(UINT, WPARAM, LPARAM, BOOL)
	{
		CPaintDC PaintDC( m_hWnd );
		return 0;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	LRESULT PSXMainWindow::OnFileExit(WORD, WORD, HWND, BOOL)
	{
		OnClose(WM_CLOSE, NULL, NULL, TRUE);
		return 0;
	}

	LRESULT PSXMainWindow::OnFileOpenExe(WORD, WORD, HWND, BOOL)
	{
		//===============================================================
		// Show the open file dialog box.
		//===============================================================
		CFileDialog FileDialog(TRUE, _T("exe"), NULL, OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT, _T("Playstation Executable Files (*.exe, *.psx)\0*.exe;*.psx\0All Files (*.*)\0*.*\0"), m_hWnd);
		
		//===============================================================
		// If the dialog wasn't cancelled the load the executable file.
		//===============================================================
		if(FileDialog.DoModal() == IDOK)
		{
			if(mIdleHandler.LoadExecutable((char*)FileDialog.m_szFileName))
			{
				mDocList.AddToList(FileDialog.m_ofn.lpstrFile);
				mDocList.WriteToRegistry(RECENTDOC_REGKEY);
			}
			else
			{
				AtlMessageBox(NULL, "Could not load the specified file or is not valid executable.", "Error");
			}
		}
		else
		{
			AtlMessageBox(NULL, "No PSXExe File Selected.", "Error", MB_OK|MB_ICONERROR);
		}
		return 0;
	}

	LRESULT PSXMainWindow::OnFileOpenIso(WORD, WORD, HWND, BOOL)
	{
		//===============================================================
		// Show the open file dialog box.
		//===============================================================
		CFileDialog FileDialog(TRUE, _T("iso"), NULL, OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT, _T("ISO CD-ROM Image (*.iso)\0*.iso\0All Files (*.*)\0*.*\0"), m_hWnd);
		
		//===============================================================
		// If the dialog wasn't cancelled the load the executable file.
		//===============================================================
		if(FileDialog.DoModal() == IDOK)
		{
		}
		else
		{
			AtlMessageBox(NULL, "No ISO CD Image File Selected.", "Error", MB_OK|MB_ICONERROR);
		}
		return 0;
	}

	void PSXMainWindow::OnFileRecent(UINT, int nID, CWindow)
	{
		//===============================================================
		// Create our file name character array.
		//===============================================================
		TCHAR szFile[MAX_PATH];

		//===============================================================
		// Attempt to grab the specified item from the list.
		//===============================================================
		if(mDocList.GetFromList(nID, szFile))
		{
			// Attempt to load the requested file.
			if(mIdleHandler.LoadExecutable((char*)szFile))
			{
				mDocList.MoveToTop(nID);
				SetStatusBarText  (0, "Executable file loaded succesfully...");
			}
			// Remove it from the recent files list if the loading failed.
			else
			{
				mDocList.RemoveFromList(nID);
				SetStatusBarText  (0, "Executable file load failed!");
			}
			// Write the new list to the registry.
			mDocList.WriteToRegistry(RECENTDOC_REGKEY);
		}
		else
		{
			::MessageBeep(MB_ICONERROR);
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	LRESULT PSXMainWindow::OnConfigureVideo(WORD, WORD, HWND, BOOL)
	{
		mPlugin.reset( new PSXPluginDialog( PSXPluginDialog::LS_VIDEO ) );
		mPlugin->Create(m_hWnd);

		return 0;
	}

	LRESULT PSXMainWindow::OnConfigureSound(WORD, WORD, HWND, BOOL)
	{
		mPlugin.reset( new PSXPluginDialog( PSXPluginDialog::LS_SOUND ) );
		mPlugin->Create(m_hWnd);

		return 0;
	}

	LRESULT PSXMainWindow::OnConfigureCdrom(WORD, WORD, HWND, BOOL)
	{
		mPlugin.reset( new PSXPluginDialog( PSXPluginDialog::LS_CDROM ) );
		mPlugin->Create(m_hWnd);

		return 0;
	}

	LRESULT PSXMainWindow::OnConfigurePad1(WORD, WORD, HWND, BOOL)
	{
		mPlugin.reset( new PSXPluginDialog( PSXPluginDialog::LS_PAD1 ) );
		mPlugin->Create(m_hWnd);

		return 0;
	}

	LRESULT PSXMainWindow::OnConfigurePad2(WORD, WORD, HWND, BOOL)
	{
		mPlugin.reset( new PSXPluginDialog( PSXPluginDialog::LS_PAD2 ) );
		mPlugin->Create(m_hWnd);

		return 0;
	}

	LRESULT PSXMainWindow::OnConfigureEmulation(WORD, WORD, HWND, BOOL)
	{
		//===============================================================
		// Display the emulation configuration window.
		//===============================================================
		PSXEmulationDialog EmulationDialog;
		EmulationDialog.DoModal();

		return 0;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void PSXMainWindow::OnViewStatusbar(WORD, int, CWindow)
	{
		//===============================================================
		// Show or hide the status bar window.
		//===============================================================
		BOOL IsVisible = !(::IsWindowVisible( m_hWndStatusBar ));
		::ShowWindow(m_hWndStatusBar, IsVisible ? SW_SHOWNOACTIVATE : SW_HIDE);

		//===============================================================
		// Update our user interface and layout.
		//===============================================================
		UISetCheck(ID_VIEW_STATUSBAR, IsVisible);
		UpdateLayout();
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	LRESULT PSXMainWindow::OnHelpAbout(WORD, WORD, HWND, BOOL)
	{
		//===============================================================
		// Display the NeoPSX about window.
		//===============================================================
		PSXAboutDialog AboutDialog(mVersion, EFPlatformManager::GetFrameworkVersion());
		AboutDialog.DoModal();

		return 0;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	LRESULT PSXMainWindow::OnDebugCpu(WORD, WORD, HWND, BOOL)
	{
		mIdleHandler.DisplayProcessorDebug();
		return 0;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void PSXMainWindow::InitializeLogFiles() const
	{
		//===============================================================
		// Initialize the main NeoPSX log file.
		//===============================================================
		try
		{
			EFLogManager::GetSingleton().CreateLog("Logs\\NeoPSX - MainLog.txt", true);
			EFLogManager::GetSingleton().LogMessage(LML_CRITICAL, "NeoPSX v%s Initialized...", mVersion.GetString(), EFPlatformManager::GetFrameworkVersion().GetString());
		} catch(EFException& error) {
			AtlMessageBox(NULL, (LPCTSTR)error.GetDetailedInformation().c_str());
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	CIdleHandler* PSXMainWindow::GetIdleHandler()
	{
		return &mIdleHandler;
	}

	void PSXMainWindow::SetStatusBarText(int pane, const EFString& message)
	{
		CStatusBarCtrlT<CWindow> mStatus = m_hWndStatusBar;
		mStatus.SetText(pane, _T(message.c_str()));
	}
} // Namespace NeoPSX