#include "PSXMemory.hpp"
#include "PSXUtility.hpp"

namespace NeoPSX
{
	#define ILLEGAL_WRITE() EFLogManager::GetSingleton().LogMessage(LML_CRITICAL, "%s: Unknown memory address 0x%08X - 0x%08X", __FUNCTION__, address, data)
	#define ILLEGAL_READ() EFLogManager::GetSingleton().LogMessage(LML_CRITICAL, "%s: Unknown memory address 0x%08X", __FUNCTION__, address)
	#define HARDWARE_WRITE() EFLogManager::GetSingleton().LogMessage(LML_CRITICAL, "%s: Hardware write 0x%08X - 0x%08X", __FUNCTION__, address, data)
	#define HARDWARE_READ() EFLogManager::GetSingleton().LogMessage(LML_CRITICAL, "%s: Hardware write 0x%08X", __FUNCTION__, address)

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	PSXMemory::PSXMemory()
	: mRAM ( NULL ),
	  mROM ( NULL ),
	  mSPAD( NULL )
	{
		//========================================================
		// Attempt to allocate the scratch pad.
		//========================================================
		mSPAD = new uint8_t[MS_SCRATCH_PAD];
		PSXClearMemory(mSPAD, MS_SCRATCH_PAD, "PSXMemory: Could not allocate scratchpad.");

		//========================================================
		// Attempt to allocate the RAM.
		//========================================================
		mRAM = new uint8_t[MS_RAM];
		PSXClearMemory(mRAM, MS_RAM, "PSXMemory: Could not allocate ram");

		//========================================================
		// Attempt to allocate the ROM.
		//========================================================
		mROM = new uint8_t[MS_ROM];
		PSXClearMemory(mROM, MS_ROM, "PSXMemory: Could not allocate rom");
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	PSXMemory::~PSXMemory()
	{
		EF_UTIL_DELETE_ARRAY( mRAM  );
		EF_UTIL_DELETE_ARRAY( mROM  );
		EF_UTIL_DELETE_ARRAY( mSPAD );
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void PSXMemory::Reset()
	{
		PSXClearMemory(mSPAD, MS_SCRATCH_PAD, "PSXMemory: Could not allocate scratchpad.");
		PSXClearMemory(mRAM , MS_RAM        , "PSXMemory: Could not allocate ram");
		PSXClearMemory(mROM , MS_ROM        , "PSXMemory: Could not allocate rom");

		mHardware.Reset();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void PSXMemory::WriteByte(uint32_t address, uint8_t data)
	{
		//========================================================
		// Check if address is in playstation RAM.
		//========================================================
		if( RAM::InRange(address) )
		{
			mRAM[ (address&0x001FFFFFF) ] = data;
		}

		//========================================================
		// Check if address is in playstation ROM.
		//========================================================
		else if( ROM::InRange(address) )
		{
			mROM[ (address&0x0007FFFF) ] = data;
		}

		//========================================================
		// Check if address is in playstation Scratchpad.
		//========================================================
		else if( SPAD::InRange(address) )
		{
			mSPAD[ (address&0x000003FF) ] = data;
		}

		//========================================================
		// Check if address is in Playstation hardware.
		//========================================================
		else if( HW::InRange(address) )
		{
			HARDWARE_WRITE();
			mHardware.WriteByte(address, data);
		}

		//========================================================
		// Log an illegal address write.
		//========================================================
		else
		{
			ILLEGAL_WRITE();
		}
	}

	void PSXMemory::WriteHword(uint32_t address, uint16_t data)
	{
		//========================================================
		// Check if address is in playstation RAM.
		//========================================================
		if( RAM::InRange(address) )
		{
			CAST_SIZE(uint16_t, mRAM[(address&0x001FFFFF)]) = data;
		}

		//========================================================
		// Check if address is in playstation ROM.
		//========================================================
		else if( ROM::InRange(address) )
		{
			CAST_SIZE(uint16_t, mROM[(address&0x0007FFFF)]) = data;
		}

		//========================================================
		// Check if address is in playstation Scratchpad.
		//========================================================
		else if( SPAD::InRange(address) )
		{
			CAST_SIZE(uint16_t, mSPAD[(address&0x000003FF)]) = data;
		}

		//========================================================
		// Check if address is in Playstation hardware.
		//========================================================
		else if( HW::InRange(address) )
		{
			HARDWARE_WRITE();
			mHardware.WriteHword(address, data);
		}

		//========================================================
		// Log an illegal address write.
		//========================================================
		else
		{
			ILLEGAL_WRITE();
		}
	}

	void PSXMemory::WriteWord(uint32_t address, uint32_t data)
	{
		//========================================================
		// Check if address is in playstation RAM.
		//========================================================
		if( RAM::InRange(address) )
		{
			CAST_SIZE(uint32_t, mRAM[(address&0x001FFFFF)]) = data;
		}

		//========================================================
		// Check if address is in playstation ROM.
		//========================================================
		else if( ROM::InRange(address) )
		{
			CAST_SIZE(uint32_t, mROM[(address&0x0007FFFF)]) = data;
		}

		//========================================================
		// Check if address is in playstation Scratchpad.
		//========================================================
		else if( SPAD::InRange(address) )
		{
			CAST_SIZE(uint32_t, mSPAD[(address&0x000003FF)]) = data;
		}

		//========================================================
		// Check if address is in Playstation hardware.
		//========================================================
		else if( HW::InRange(address) )
		{
			HARDWARE_WRITE();
			mHardware.WriteWord(address, data);
		}

		//========================================================
		// Log an illegal address write.
		//========================================================
		else
		{
			ILLEGAL_WRITE();
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	uint8_t PSXMemory::ReadByte(uint32_t address)
	{
		//========================================================
		// Check if address is in playstation RAM.
		//========================================================
		if( RAM::InRange(address) )
		{
			return mRAM[ (address&0x001FFFFFF) ];
		}

		//========================================================
		// Check if address is in playstation ROM.
		//========================================================
		else if( ROM::InRange(address) )
		{
			return mROM[ (address&0x0007FFFF) ];
		}

		//========================================================
		// Check if address is in playstation Scratchpad.
		//========================================================
		else if( SPAD::InRange(address) )
		{
			return mSPAD[ (address&0x000003FF) ];
		}

		//========================================================
		// Check if address is in Playstation hardware.
		//========================================================
		else if( HW::InRange(address) )
		{
			HARDWARE_READ();
			return mHardware.ReadByte(address);
		}

		ILLEGAL_READ();
		return NULL;
	}

	uint16_t PSXMemory::ReadHword(uint32_t address)
	{
		//========================================================
		// Check if address is in playstation RAM.
		//========================================================
		if( RAM::InRange(address) )
		{
			return CAST_SIZE(uint16_t, mRAM[(address&0x001FFFFFF)]);
		}

		//========================================================
		// Check if address is in playstation ROM.
		//========================================================
		else if( ROM::InRange(address) )
		{
			return CAST_SIZE(uint16_t, mROM[(address&0x0007FFFF)]);
		}

		//========================================================
		// Check if address is in playstation Scratchpad.
		//========================================================
		else if( SPAD::InRange(address) )
		{
			return CAST_SIZE(uint16_t, mSPAD[(address&0x000003FF)]);
		}

		//========================================================
		// Check if address is in Playstation hardware.
		//========================================================
		else if( HW::InRange(address) )
		{
			HARDWARE_READ();
			return mHardware.ReadHword(address);
		}

		ILLEGAL_READ();
		return NULL;
	}

	uint32_t PSXMemory::ReadWord(uint32_t address)
	{
		//========================================================
		// Check if address is in playstation RAM.
		//========================================================
		if( RAM::InRange(address) )
		{
			return CAST_SIZE(uint32_t, mRAM[(address&0x001FFFFFF)]);
		}

		//========================================================
		// Check if address is in playstation ROM.
		//========================================================
		else if( ROM::InRange(address) )
		{
			return CAST_SIZE(uint32_t, mROM[(address&0x0007FFFF)]);
		}

		//========================================================
		// Check if address is in playstation Scratchpad.
		//========================================================
		else if( SPAD::InRange(address) )
		{
			return CAST_SIZE(uint32_t, mSPAD[(address&0x000003FF)]);
		}

		//========================================================
		// Check if address is in Playstation hardware.
		//========================================================
		else if( HW::InRange(address) )
		{
			HARDWARE_READ();
			return mHardware.ReadWord(address);
		}

		ILLEGAL_READ();
		return NULL;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	uint8_t* PSXMemory::GetPointer(uint32_t address)
	{
		//========================================================
		// Check if address is in playstation RAM.
		//========================================================
		if( RAM::InRange(address) )
		{
			return &mRAM[ (address&0x001FFFFFF) ];
		}

		//========================================================
		// Check if address is in playstation ROM.
		//========================================================
		else if( ROM::InRange(address) )
		{
			return &mROM[ (address&0x0007FFFF) ];
		}

		//========================================================
		// Check if address is in playstation Scratchpad.
		//========================================================
		else if( SPAD::InRange(address) )
		{
			return &mSPAD[ (address&0x000003FF) ];
		}

		ILLEGAL_READ();
		return NULL;
	}
} // Namespace NeoPSX