#include "EFException.hpp"
#include "EFLogManager.hpp"

namespace EFramework
{
	EFLogManager* EFSingleton<EFLogManager>::mSingleton = NULL;

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	EFLogManager::EFLogManager()
	: mDefault( NULL )
	{ }

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	EFLogManager::~EFLogManager()
	{ 
		//==================================================
		// Run through and delete all of our managed logs.
		//==================================================
		LogList::iterator iter;

		for(iter = mLogs.begin(); iter != mLogs.end(); iter++)
		{
			EF_UTIL_DELETE( iter->second );
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	EFLog* EFLogManager::CreateLog(const EFString& name, bool defaultLog)
	{
		//==================================================
		// Create a new log file with the specified name.
		//==================================================
		EFLog* newLog = new EFLog(name);

		//==================================================
		// Set the default log if it is not set or desired.
		//==================================================
		if(!mDefault || defaultLog)
		{
			mDefault = newLog;
		}

		//==================================================
		// Insert the new log into our map.
		//==================================================
		mLogs.insert( LogList::value_type(name, newLog) );
		return newLog;
	}

	EFLog* EFLogManager::GetLog(const EFString& name)
	{
		//==================================================
		// Attempt to find the specified log file.
		//==================================================
		LogList::iterator iter = mLogs.find( name );

		//==================================================
		// If the log is found then return its pointer.
		//==================================================
		if(iter != mLogs.end())
			return iter->second;
		else
			return NULL;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	EFLog* EFLogManager::GetDefaultLog()
	{
		//==================================================
		// Attempt to retrieve the default log if it exists.
		//==================================================
		if(mDefault == NULL) return NULL;
		return mDefault;
	}

	void EFLogManager::SetDefaultLog(EFLog* log)
	{
		mDefault = log;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void EFLogManager::LogMessage(const EFString& message, LogMessageLevel level)
	{
		GetDefaultLog()->LogMessage(message, level);
	}

	void EFLogManager::LogMessage(LogMessageLevel level, const char* format, ...)
	{
		//==================================================
		// Log a C-Style formatted message.
		//==================================================
		static char logBuffer[4097];
		va_list list;
		va_start(list, format);
		vsprintf(logBuffer, format, list);

		GetDefaultLog()->LogMessage(EFString( logBuffer ), level);

		va_end(list);
	}
} // Namespace EFramework