#ifndef EF_TYPES_HPP
#define EF_TYPES_HPP

#include <string>

namespace EFramework
{
	//////////////////////////////////////////////////////////////////////////////////////
	// Used for terminating type lists
	//////////////////////////////////////////////////////////////////////////////////////
	class NullType { };

	//////////////////////////////////////////////////////////////////////////////////////
	// Used to recursively create lists of types
	//////////////////////////////////////////////////////////////////////////////////////
	template<class Type1, class Type2>
	struct TypeList
	{
		typedef Type1 HeadType;
		typedef Type2 TailType;
	};

	//////////////////////////////////////////////////////////////////////////////////////
	// Type list helper macros
	//////////////////////////////////////////////////////////////////////////////////////
	#define TypeList_1(T1)             TypeList<T1, NullType>
	#define TypeList_2(T1,T2)          TypeList<T1, TypeList_1(T2) >
	#define TypeList_3(T1,T2,T3)       TypeList<T1, TypeList_2(T2, T3) >
	#define TypeList_4(T1,T2,T3,T4)    TypeList<T1, TypeList_3(T2, T3, T4) >
	#define TypeList_5(T1,T2,T3,T4,T5) TypeList<T1, TypeList_4(T2,T3,T4,T5) >

	//////////////////////////////////////////////////////////////////////////////////////
	// Chooses between two types at compile time
	//////////////////////////////////////////////////////////////////////////////////////
	template<bool Expression, class tType, class fType>
	struct TypeSelector
	{
		typedef tType ResultType;
	};

	//////////////////////////////////////////////////////////////////////////////////////
	// Template specialization for false expressions
	//////////////////////////////////////////////////////////////////////////////////////
	template<class tType, class fType>
	struct TypeSelector<false, tType, fType>
	{
		typedef fType ResultType;
	};

	//////////////////////////////////////////////////////////////////////////////////////
	// Chooses a type based on the selected size
	//////////////////////////////////////////////////////////////////////////////////////
	template<unsigned int Size, class TList>
	struct SizeTypeSelector;

	template<unsigned int Size, class Type1, class Type2>
	struct SizeTypeSelector<Size, TypeList<Type1, Type2> >
	{
		typedef typename TypeSelector
				< Size == sizeof(Type1),
				  Type1,
				  typename SizeTypeSelector<Size, Type2>::ResultType
				>::ResultType ResultType;
	};

	//////////////////////////////////////////////////////////////////////////////////////
	// Specialization if the type is not found
	//////////////////////////////////////////////////////////////////////////////////////
	template<unsigned int Size>
	struct SizeTypeSelector<Size, NullType>
	{
		typedef NullType ResultType;
	};

	//////////////////////////////////////////////////////////////////////////////////////
	// Define our result type lists for signed integers
	//////////////////////////////////////////////////////////////////////////////////////
	#if defined (_MSC_VER) || defined (_WINDOWS) || defined (WIN32)
		typedef TypeList_5(signed char, short int, int, long int, signed   __int64) SignedIntegers;
	#else
		typedef TypeList_5(signed char, short int, int, long int, signed long long) SignedIntegers;
	#endif
	
	//////////////////////////////////////////////////////////////////////////////////////
	// Define our result type lists for unsigned integers
	//////////////////////////////////////////////////////////////////////////////////////
	#if defined (_MSC_VER) || defined (_WINDOWS) || defined (WIN32)
		typedef TypeList_5(unsigned char, unsigned short, unsigned int, unsigned long, unsigned   __int64) UnsignedIntegers;
	#else
		typedef TypeList_5(unsigned char, unsigned short, unsigned int, unsigned long, unsigned long long) UnsignedIntegers;
	#endif

	//////////////////////////////////////////////////////////////////////////////////////
	// Define our basic signed types as cross-platform size guaranteed
	//////////////////////////////////////////////////////////////////////////////////////
	typedef SizeTypeSelector<1, SignedIntegers>::ResultType sint8_t;
	typedef SizeTypeSelector<2, SignedIntegers>::ResultType sint16_t;
	typedef SizeTypeSelector<4, SignedIntegers>::ResultType sint32_t;
	typedef SizeTypeSelector<8, SignedIntegers>::ResultType sint64_t;

	//////////////////////////////////////////////////////////////////////////////////////
	// Define our basic unsigned types as cross-platform size guaranteed
	//////////////////////////////////////////////////////////////////////////////////////
	typedef SizeTypeSelector<1, UnsignedIntegers>::ResultType uint8_t;
	typedef SizeTypeSelector<2, UnsignedIntegers>::ResultType uint16_t;
	typedef SizeTypeSelector<4, UnsignedIntegers>::ResultType uint32_t;
	typedef SizeTypeSelector<8, UnsignedIntegers>::ResultType uint64_t;

	//////////////////////////////////////////////////////////////////////////////////////
	// Other type definitions
	//////////////////////////////////////////////////////////////////////////////////////
	typedef std::string EFString;
	typedef unsigned char  uchar;
	typedef unsigned short ushort;
	typedef unsigned int   uint;
	typedef unsigned long  ulong;

	//////////////////////////////////////////////////////////////////////////////////////
	// Defines a void class function pointer
	//////////////////////////////////////////////////////////////////////////////////////
	template<class Type>
	struct ClassFunctionPointer
	{
		typedef void (Type::*ResultType)();
	};
} // Namespace EFramework

#endif // EF_TYPES_HPP