#ifndef PSX_PLUGIN_DIALOG_HPP
#define PSX_PLUGIN_DIALOG_HPP

#include <atlbase.h>
#include <atlwin.h>
#include <atlapp.h>
#include <atlctrls.h>
#include <atlctrlx.h>
#include <atlddx.h>
#include <atldlgs.h>
#include <atlcrack.h>
#include <atlmisc.h>
#include <atlstr.h>
#include <ToolTipDialog.h>
#include "PSXPluginTypes.hpp"
#include "../resource.h"

namespace NeoPSX
{
	#define GET_PLUGIN_FUNC(iter, name) (EFPluginManager::GetSingleton().GetProcedure(EFPluginManager::GetSingleton().GetPluginFileName((iter)), EFString((name))))

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// List View Control Class Definition
	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	class CListViewCtrlEx : public CWindowImpl<CListViewCtrlEx, WTL::CListViewCtrl>
	{
	public:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Message Maps

		BEGIN_MSG_MAP(CListViewCtrlEx)
			MSG_WM_CONTEXTMENU(OnContextMenu)
		END_MSG_MAP()

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Message Map Functions

		/** Displays the context menu and process selected option.
		  */
		void OnContextMenu(CWindow wnd, CPoint point)
		{
			//===============================================================
			// If the message is for this window and something is selected.
			//===============================================================
			if(wnd.m_hWnd == m_hWnd && GetSelectedIndex() != -1)
			{
				//===============================================================
				// Create the context menu object.
				//===============================================================
				CMenu ContextMenu;
				ContextMenu.CreatePopupMenu();
				ContextMenu.AppendMenu(MF_STRING, 1, _T("Configure..."));
				ContextMenu.AppendMenu(MF_STRING, 2, _T("About..."));
				ContextMenu.AppendMenu(MF_STRING, 3, _T("Test..."));
				ContextMenu.AppendMenu(MF_STRING, 4, _T("Select..."));

				//===============================================================
				// Display and track the popup menu.
				//===============================================================
				UINT mCommand = (UINT)::TrackPopupMenu(ContextMenu, TPM_RETURNCMD | TPM_RIGHTBUTTON, point.x, point.y, 0, wnd, NULL);

				// Get the selected menu option.
				switch(mCommand)
				{
				//===============================================================
				// The user cancelled.
				//===============================================================
				case 0: 
					break;
				//===============================================================
				// The user chose the configure the plugin.
				//===============================================================
				case 1:
					{
						// Nullify Pointer
						PSXConfigure = NULL;

						// Replace plugin null function pointer with pointer to plugin function.
						switch(GetItemData(GetSelectedIndex()))
						{
						case PSE_LT_GPU: PSXConfigure = (PSEconfigure)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "GPUconfigure"); break;
						case PSE_LT_SPU: PSXConfigure = (PSEconfigure)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "SPUconfigure"); break;
						case PSE_LT_CDR: PSXConfigure = (PSEconfigure)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "CDRconfigure"); break;
						case PSE_LT_PAD: PSXConfigure = (PSEconfigure)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "PADconfigure"); break;
						default: break;
						}

						// If pointer retrieval succeeded, then execute function.
						if(PSXConfigure != NULL)
						{
							if(PSXConfigure() == PSE_ERR_FAILURE)
							{
								AtlMessageBox(NULL, "Plugin Configuration Failed.");
							}
						}
					}
					break;
				//===============================================================
				// The user chose to display plugin about window.
				//===============================================================
				case 2:
					{
						// Nullify Pointer
						PSXAbout = NULL;

						// Replace plugin null function pointer with pointer to plugin function.
						switch(GetItemData(GetSelectedIndex()))
						{
						case PSE_LT_GPU: PSXAbout = (PSEabout)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "GPUabout"); break;
						case PSE_LT_SPU: PSXAbout = (PSEabout)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "SPUabout"); break;
						case PSE_LT_CDR: PSXAbout = (PSEabout)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "CDRabout"); break;
						case PSE_LT_PAD: PSXAbout = (PSEabout)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "PADabout"); break;
						default: break;
						}

						// If pointer retrieval succeeded, then execute function.
						if(PSXAbout != NULL)
						{
							PSXAbout();
						}
					}
					break;
				//===============================================================
				// The user chose to test the plugin.
				//===============================================================
				case 3:
					{
						// Nullify Pointer
						PSXTest = NULL;

						// Replace plugin null function pointer with pointer to plugin function.
						switch(GetItemData(GetSelectedIndex()))
						{
						case PSE_LT_GPU: PSXTest = (PSEtest)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "GPUtest"); break;
						case PSE_LT_SPU: PSXTest = (PSEtest)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "SPUtest"); break;
						case PSE_LT_CDR: PSXTest = (PSEtest)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "CDRtest"); break;
						case PSE_LT_PAD: PSXTest = (PSEtest)GET_PLUGIN_FUNC(mGroupMap[GetSelectedIndex()], "PADtest"); break;
						default: break;
						}

						// If pointer retrieval succeeded, then execute function.
						if(PSXTest != NULL)
						{
							if(PSXTest() == PSE_ERR_FAILURE)
							{
								AtlMessageBox(NULL, "Plugin is not working correctly.", "Plugin Test [Failed]", MB_OK|MB_ICONERROR);
							}
							else
							{
								AtlMessageBox(NULL, "Plugin is working correctly", "Plugin Test [Working]");
							}
						}
					}
					break;
				//===============================================================
				// An uknown menu option was selected.
				//===============================================================
				default: break;
				}
			}
			else
			{
				//===============================================================
				// The message has not been handled.
				//===============================================================
				SetMsgHandled(FALSE);
			}
		}

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// STL Map Control Functions

		/** Clears the ListView message map.
		  */
		void ClearMap()
		{
			//===============================================================
			// Clear the list view group.
			//===============================================================
			mGroupMap.clear();
		}

		/** Add an item to the list view message map.
		  */
		void AddMapItem(ulong ListIndex, ulong PluginIndex)
		{
			//===============================================================
			// Add an item to the list view group.
			//===============================================================
			mGroupMap[ListIndex] = PluginIndex;
		}
	private:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Variables

		std::map<ulong, ulong> mGroupMap;
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Tree View Control Class Definition
	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	class CTreeViewCtrlExImpl : public CWindowImpl<CTreeViewCtrlExImpl, CTreeViewCtrlEx>
	{
	private:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Variables

		CProgressBarCtrl mProgress;
	public:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Message Maps

		BEGIN_MSG_MAP(CTreeViewCtrlExImpl)
			MSG_WM_CONTEXTMENU (OnContextMenu)
		END_MSG_MAP()

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Progress Bar Setter

		/** Sets the progress bar display used for display plugin loading progress.
		  */
		void SetProgressControl(CProgressBarCtrl& Progress)
		{
			mProgress = Progress;
		}

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Message Map Functions

		/** Displays the context menu and processes the selected option.
		  */
		void OnContextMenu(CWindow wnd, CPoint point)
		{
			//===============================================================
			// If the message is for this window.
			//===============================================================
			if(wnd.m_hWnd == m_hWnd)
			{
				//===============================================================
				// Create the context menu object.
				//===============================================================
				CMenu ContextMenu;
				ContextMenu.CreatePopupMenu();
				ContextMenu.AppendMenu(MF_STRING, 1, _T("Change BIOS Directory..."));
				ContextMenu.AppendMenu(MF_STRING, 2, _T("Change Plugin Directory..."));

				//===============================================================
				// Display and track the popup menu.
				//===============================================================
				UINT mCommand = (UINT)::TrackPopupMenu(ContextMenu, TPM_RETURNCMD | TPM_RIGHTBUTTON, point.x, point.y, 0, wnd, NULL);

				// Get the selected menu option.
				switch(mCommand)
				{
				//===============================================================
				// The user cancelled.
				//===============================================================
				case 0: break;
				//===============================================================
				// The user wants to change the plugin directory.
				//===============================================================
				case 2:
					{
						//Set Progress Bar To 0% Progress.
						mProgress.SetPos(0);

						// Browse For The Next Plugins Folder
						EFString FolderName = BrowseForFolder();

						// If a folder could be opened
						if(FolderName != "")
						{
							// Step the progress bar
							mProgress.StepIt();

							try
							{
								// Save the directory to the registry and step progress.
								EFRegistryKey RegKey("NeoPSX");
								RegKey.WriteString("PluginDirectory", FolderName);
								mProgress.StepIt();

								// Unload all current plugins and load new directory plugins, step progress twice.
								EFPluginManager::GetSingleton().UnloadPlugins();
								mProgress.StepIt();
								EFPluginManager::GetSingleton().LoadPlugins(FolderName);
								mProgress.StepIt();

								// Select root item and step the progress bar.
								SelectItem(GetRootItem());
								mProgress.StepIt();
							} catch(EFException& error) {
								AtlMessageBox(NULL, error.GetDetailedInformation().c_str());
							}
						}
						else
						{
							AtlMessageBox(NULL, "No Folder Was Selected.");
						}

						// Return The Progress Bar To 0% Progress.
						mProgress.StepIt();
						mProgress.SetPos(0);
					}
					break;
				//===============================================================
				// An uknown menu option was selected.
				//===============================================================
				default: break;
				}
			}
		}

		/** The tree item selected has changed.
		  */
		LRESULT OnTreeSelChanged(LPNMHDR TreeHeader, CListViewCtrlEx& ListView)
		{
			//===============================================================
			// Ensure that we are given a valid tree header.
			//===============================================================
			if(TreeHeader != NULL)
			{
				//===============================================================
				// Get The Selected Item.
				//===============================================================
				HTREEITEM Item = GetSelectedItem();

				// Switch selected item data.
				switch(GetItemData(Item))
				{
				//===============================================================
				// The plain plugin branch, so display them all.
				//===============================================================
				case 18:
					{
						DisplayAllPlugins(ListView);
					}
					break;
				//===============================================================
				// The GPU plugin leaf, display only GPU plugins.
				//===============================================================
				case PSE_LT_GPU:
					{
						DisplayOnlyPluginType(ListView, PSE_LT_GPU);
					}
					break;
				//===============================================================
				// The SPU plugin leaf, display only SPU plugins.
				//===============================================================
				case PSE_LT_SPU:
					{
						DisplayOnlyPluginType(ListView, PSE_LT_SPU);
					}
					break;
				//===============================================================
				// The CDR plugin leaf, display only CDR plugins.
				//===============================================================
				case PSE_LT_CDR:
					{
						DisplayOnlyPluginType(ListView, PSE_LT_CDR);
					}
					break;
				//===============================================================
				// The PAD plugin leaf, display only PAD plugins.
				//===============================================================
				case PSE_LT_PAD:
					{
						DisplayOnlyPluginType(ListView, PSE_LT_PAD);
					}
					break;
				//===============================================================
				// Some other branch, so just delete everything.
				//===============================================================
				default: ListView.DeleteAllItems(); break;
				}
			}
			return 0;
		}
	private:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Helper Functions

		/** Display all the plugins in the loaded plugin list.
		  */
		VOID DisplayAllPlugins(CListViewCtrlEx& ListView)
		{
			//===============================================================
			// Clear the list view and the list view map.
			//===============================================================
			ListView.DeleteAllItems();
			ListView.ClearMap();

			//===============================================================
			// Set our plugin list index.
			//===============================================================
			ulong ListViewIndex = 0;

			//===============================================================
			// Iterate through and display all of the plugins.
			//===============================================================
			for(ulong i = 0; i < EFPluginManager::GetSingleton().GetNumberOfPlugins(); i++)
			{
				// Retrieve all the plugin functions.
				PSXLibraryName    = (PSEgetLibName)GET_PLUGIN_FUNC(i, "PSEgetLibName");
				PSXLibraryVersion = (PSEgetLibVersion)GET_PLUGIN_FUNC(i, "PSEgetLibVersion");
				PSXLibraryType    = (PSEgetLibType)GET_PLUGIN_FUNC(i, "PSEgetLibType");

				// Add the item list index and plugin list index.
				ListView.AddMapItem(ListViewIndex, i);

				// Add the item to the list view.
				if(AddItem(ListViewIndex, ListView))
				{
					// Increment the item list index if add succeeded.
					ListViewIndex++;
				}
			}
		}
		/** Display only a specific plugin type in the loaded plugin list.
		  */
		VOID DisplayOnlyPluginType(CListViewCtrlEx& ListView, const ulong PluginType)
		{
			//===============================================================
			// Clear the list view and the list view map.
			//===============================================================
			ListView.DeleteAllItems();
			ListView.ClearMap();

			//===============================================================
			// Set our plugin list index.
			//===============================================================
			ulong ListViewIndex = 0;

			//===============================================================
			// Iterate through the plugins and display only specific type.
			//===============================================================
			for(ulong i = 0; i < EFPluginManager::GetSingleton().GetNumberOfPlugins(); i++)
			{
				// Retrieve all the plugin functions.
				PSXLibraryName    = (PSEgetLibName)GET_PLUGIN_FUNC(i, "PSEgetLibName");
				PSXLibraryVersion = (PSEgetLibVersion)GET_PLUGIN_FUNC(i, "PSEgetLibVersion");
				PSXLibraryType    = (PSEgetLibType)GET_PLUGIN_FUNC(i, "PSEgetLibType");

				// Ensure that the library type function exists.
				if(PSXLibraryType == NULL) 
				{
					continue;
				}

				// If plugin is the desired type then add it to the list view and plugin list.
				if(PSXLibraryType() == PluginType) 
				{
					// Add the item list index and plugin list index.
					ListView.AddMapItem(ListViewIndex, i);

					// Add the item to the list view.
					if(AddItem(ListViewIndex, ListView))
					{
						// Increment the item list index if add succeeded.
						ListViewIndex++;
					}
				}
			}
		}

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Miscellaneous Functions

		/** Adds an item to the list, doing proper error checking first.
		  */
		BOOL AddItem(ulong Index, CListViewCtrlEx& ListView)
		{
			//===============================================================
			// Ensure that none of our functions are null pointers.
			//===============================================================
			if(PSXLibraryName    == NULL ||
			   PSXLibraryType    == NULL ||
			   PSXLibraryVersion == NULL)
			{
				return FALSE;
			}

			//===============================================================
			// Ensure library is correct PSEmuPro revision.
			//===============================================================
			if(((PSXLibraryVersion()>>16)&0xFF) != PSE_LIB_VERSION)
				return FALSE;

			//===============================================================
			// Create our version string using WTL::CString.
			//===============================================================
			WTL::CString LibVersion = "";
			LibVersion.Format("v%d.%d", (PSXLibraryVersion()>>8)&0xFF, (PSXLibraryVersion())&0xFF);

			//===============================================================
			// Add the plugin name to the first list view pane.
			//===============================================================
			switch(PSXLibraryType())
			{
			case PSE_LT_GPU:
				ListView.InsertItem(Index, (LPCTSTR)PSXLibraryName(), 3);
				break;
			case PSE_LT_SPU:
				ListView.InsertItem(Index, (LPCTSTR)PSXLibraryName(), 4);
				break;
			case PSE_LT_CDR:
				ListView.InsertItem(Index, (LPCTSTR)PSXLibraryName(), 5);
				break;
			case PSE_LT_PAD:
				ListView.InsertItem(Index, (LPCTSTR)PSXLibraryName(), 6);
				break;
			default:
				ListView.InsertItem(Index, (LPCTSTR)PSXLibraryName(), 7);
				break;
			}

			//===============================================================
			// Set the item data to the plugin type.
			//===============================================================
			if(!ListView.SetItemData(Index, (DWORD_PTR)PSXLibraryType()))
				return FALSE;

			//===============================================================
			// Add the plugin version to the second list view pane.
			//===============================================================
			if(ListView.AddItem(Index, 1, (LPCTSTR)LibVersion, I_IMAGECALLBACK) == -1)
				return FALSE;

			return TRUE;
		}

		/** Displays the folder browsing window and returns the name of the selected foler.
		  */
		EFString BrowseForFolder()
		{
			//===============================================================
			// Setup our temporary variables.
			//===============================================================
			char Buffer[256];
			char Path[MAX_PATH];

			//===============================================================
			// Create and initialize our browse info structure.
			//===============================================================
			BROWSEINFO BrowseInfo;
			BrowseInfo.hwndOwner      = m_hWnd;
			BrowseInfo.pidlRoot       = NULL;
			BrowseInfo.pszDisplayName = Buffer;
			BrowseInfo.lpszTitle      = "Search For Directory";
			BrowseInfo.ulFlags        = BIF_RETURNFSANCESTORS | BIF_RETURNONLYFSDIRS;
			BrowseInfo.lpfn           = NULL;
			BrowseInfo.lParam         = 0;

			//===============================================================
			// Display the browse for folder dialog box.
			//===============================================================
			LPITEMIDLIST ItemList = ::SHBrowseForFolder(&BrowseInfo);

			//===============================================================
			// If the item list isn't null then return the path selected.
			//===============================================================
			if(ItemList != NULL)
			{
				if(::SHGetPathFromIDList(ItemList, (LPSTR)Path))
					return EFString(Path);
			}

			//===============================================================
			// The user cancelled or an error occurred.
			//===============================================================
			return "";
		}
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	class PSXPluginDialog : public CDialogImpl<PSXPluginDialog>, public CWinDataExchange<PSXPluginDialog>, public CToolTipDialog<PSXPluginDialog>
	{
	public:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Message Maps

		BEGIN_MSG_MAP(PSXPluginDialog)
			// Message Chaining
			CHAIN_MSG_MAP(CToolTipDialog<PSXPluginDialog>)
			// Message Crackers
			MSG_WM_INITDIALOG    (OnInitDialog)
			// Control Messages
			COMMAND_ID_HANDLER_EX(ID_OK, OnOK)
			COMMAND_ID_HANDLER_EX(ID_CANCEL, OnCancel)
			// Miscellaneous Control Messages
			NOTIFY_CODE_HANDLER(TVN_SELCHANGED, OnTreeSelChanged)
			NOTIFY_CODE_HANDLER(TVN_GETINFOTIP, OnTreeToolTip)
			// Message Reflectors
			REFLECT_NOTIFICATIONS()
		END_MSG_MAP()

		BEGIN_DDX_MAP(PSXPluginDialog)
			DDX_CONTROL(IDC_CONFIGURE_TREE, mConfigureTree)
			DDX_CONTROL(IDC_CONFIGURE_LIST, mListView)
		END_DDX_MAP()

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Enumerations

		enum { IDD = IDD_CONFIGURE };

		enum LeafSelection
		{
			LS_VIDEO = 1,
			LS_SOUND = 2,
			LS_CDROM = 3,
			LS_PAD1  = 4,
			LS_PAD2  = 5
		};

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Constructors

		/** Takes the selected leaf as a parameter.
		  */
		PSXPluginDialog(UINT SelectedLeaf);

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Message Functions

		/** Handles dialog control initialization.
		  */
		LRESULT OnInitDialog(HWND, LPARAM);
		/** Saves settings and closes dialog.
		  */
		LRESULT OnOK(UINT, INT, HWND);
		/** Doesn't save settings, but closes dialog.
		  */
		LRESULT OnCancel(UINT, INT, HWND);
		/** Forward selection change to tree view class.
		  */
		LRESULT OnTreeSelChanged(int, LPNMHDR pnmh, BOOL&)
		{
			mConfigureTree.OnTreeSelChanged(pnmh, mListView);
			return 0;
		}
		/** Forward tooltip getter to tree view.
		  */
		LRESULT OnTreeToolTip(int, LPNMHDR pnmh, BOOL&)
		{
			LPNMTVGETINFOTIP ToolTipData = (LPNMTVGETINFOTIP)pnmh;
			WTL::CString ToolTipMessage = "Right click twice for menu.";
			strcpy(ToolTipData->pszText, ToolTipMessage);

			return 0;
		}
	private:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Variables

		CTreeViewCtrlExImpl mConfigureTree;
		CListViewCtrlEx     mListView;
		CImageList          mImageList;
		CProgressBarCtrl    mProgress;
		UINT                mSelectedLeaf;

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Tooltip Stuff

		DWORD mInitialTextColor;
		DWORD mInitialBkndColor;
	};
} // Namespace NeoPSX

#endif // PSX_PLUGIN_DIALOG_HPP