; Zoft80 - A Z80 emulator
; /Mic, 2004


.486
.model flat,stdcall
option casemap:none


include \masm32\include\windows.inc
include \masm32\include\kernel32.inc
include .\mmu.inc

includelib kernel32.lib
includelib .\mmu.lib


.data

	C_FLAG 	equ 01h		; carry
	N_FLAG	equ 02h		; add/subtract
	V_FLAG	equ 04h		; overflow 
	H_FLAG	equ 10h		; half carry
	Z_FLAG	equ 40h		; zero
	S_FLAG	equ 80h		; sign

start_regs label byte
	_A	db 0	; 0
	_C	db 0
	_B	db 0
	_E	db 0
	_D	db 0
	_F	db 0
	_L	db 0
	_H	db 0
	_IX	dw 0	; 8
	_IY	dw 0
	_I	db 0	; 12
	_R	db 0
	_SP	dw 0	; 14
	_PC	dw 0
	IFF1	db 0	; 18
	IFF2	db 0
	IMF	db 0
	
	HalfCarry db 0
	AddSub	db 0
	Zero	db 0
	Carry	db 0
	Sign	db 0
	Overflow db 0

	
	cycle	dd 0

	_Ap	db 0
	_Cp	db 0
	_Bp	db 0
	_Ep	db 0
	_Dp	db 0
	_Fp	db 0
	_Lp	db 0
	_Hp	db 0

	lastPC	dw 0
	breakpoint dd 80000000h
	halted	dd 0
end_regs label byte

.data?
	hInst	dd ?
	RAM	dd ?
	
	
.code


ADD_CYCLES MACRO num
	add ecx,num*400h
ENDM



BITR8 MACRO reg,bitn
	movzx ax,reg
	mov bx,bitn
	bt ax,bx
	setc Zero
	mov HalfCarry,1
	mov AddSub,0
	ADD_CYCLES 8
	add _PC,2
	ret
ENDM


CPR8 MACRO op
	mov al,_A
	cmp al,op
	sets Sign
	setz Zero
	seto Overflow
	setc Carry
	mov AddSub,1
	ADD_CYCLES 4
	inc _PC
	ret
ENDM


DEC8_TAIL MACRO
	sets Sign
	setz Zero
	; half carry
	seto Overflow
	mov AddSub,1
	ADD_CYCLES 4
	inc _PC
	ret
ENDM


INC8_TAIL MACRO
	sets Sign
	setz Zero
	; half carry
	seto Overflow
	mov AddSub,0
	ADD_CYCLES 4
	inc _PC
	ret
ENDM


LDIMM8 MACRO reg
	mov al,[esi]
	mov reg,al
	ADD_CYCLES 7
	add _PC,2
	ret
ENDM


LDIMM16 MACRO hireg,loreg
	mov ax,[esi]
	mov loreg,al
	mov hireg,ah
	ADD_CYCLES 10
	add _PC,3
	ret
ENDM


LDM16 MACRO dest,hisrc,losrc
	movzx edx,word ptr [losrc]
	invoke mmu_read_byte,edx
	mov dest,al
	ADD_CYCLES 7
	inc _PC
	ret
ENDM


LDIXN MACRO ixreg,dest
	movzx edx,byte ptr [esi]
	add dx,ixreg
	invoke mmu_read_byte,edx
	mov dest,al
	ADD_CYCLES 19
	add _PC,3
	ret
ENDM


STRIXN MACRO ixreg,src
	movzx edx,byte ptr [esi]
	add dx,ixreg
	movzx eax,src
	invoke mmu_write_byte,edx,eax
	ADD_CYCLES 19
	add _PC,3
	ret
ENDM


MOVETO8 MACRO dest,src
	mov al,src
	mov dest,al
	ADD_CYCLES 4
	inc _PC
	ret
ENDM


POPR16 MACRO hireg,loreg
	movzx ebx,_SP
	add ebx,RAM
	mov ax,[ebx]
	add _SP,2
	mov word ptr [loreg],ax
	ADD_CYCLES 10
	inc _PC
	ret
ENDM


PUSHR16 MACRO hireg,loreg
	movzx ebx,_SP
	movzx eax,word ptr [loreg]
	add ebx,RAM
	mov [ebx-2],ax
	sub _SP,2
	ADD_CYCLES 11
	inc _PC
	ret
ENDM


RESR8 MACRO dest,bitn
	movzx eax,dest
	btr eax,bitn
	mov dest,al
	ADD_CYCLES 8
	add _PC,2
	ret
ENDM


SETR8 MACRO dest,bitn
	movzx eax,dest
	bts eax,bitn
	mov dest,al
	ADD_CYCLES 8
	add _PC,2
	ret
ENDM


ABSJUMP MACRO flag,jump
	LOCAL label
	cmp flag,0
	jump label
		mov bx,[esi]
		add _PC,bx
		ADD_CYCLES 10
		ret
	label:	
		ADD_CYCLES 1
		add _PC,2
		ret
ENDM


RELJUMP MACRO flag,jump
	LOCAL label
	cmp flag,0
	jump label
		mov al,[esi]
		add al,2
		movsx ebx,al
		add _PC,bx
		ADD_CYCLES 12
		ret
	label:	
		ADD_CYCLES 7
		add _PC,2
		ret
ENDM


ROTR8 MACRO reg,instr
	;mov al,0
	;cmp al,Carry
	cmp Carry,1
	cmc
	instr reg,1
	setc Carry
	setz Zero
	sets Sign
	setpe Overflow
	mov HalfCarry,0
	mov AddSub,0
	ADD_CYCLES 8
	add _PC,2
	ret
ENDM


RST MACRO address
	movzx ebx,_SP
	add ebx,RAM
	mov ax,_PC
	sub _SP,2
	add ax,1
	mov [ebx-2],ax
	mov _PC,address
	ADD_CYCLES 11
	ret
ENDM


STRR8 MACRO hidest,lodest,srcreg
	movzx edx,word ptr [lodest]
	movzx eax,srcreg
	invoke mmu_write_byte,edx,eax
	ADD_CYCLES 7
	inc _PC
	ret
ENDM

	
LibMain PROC hInstDLL:DWORD, reason:DWORD, unused:DWORD

        .if reason == 1		;DLL_PROCESS_ATTACH
        	mov eax,hInstDLL
        	mov hInst,eax
		mov eax,TRUE
		ret

        .elseif reason == 0	;DLL_PROCESS_DETACH
        .elseif reason == 2	;DLL_THREAD_ATTACH
        .elseif reason == 3	;DLL_THREAD_DETACH
        .endif

        mov eax,FALSE
        ret
LibMain ENDP




op_00:
	ret
	
; NOP
op_00NOP:
	ADD_CYCLES 4	
	inc _PC
	ret

; LD BC,nn
op_01:
	LDIMM16 _B,_C

; LD (BC),A	
op_02:
	STRR8 _B,_C,_A

; INC BC	
op_03:
	inc word ptr [_C]
	ADD_CYCLES 6
	inc _PC
	ret

; INC B
op_04:
	inc _B
	INC8_TAIL

; DEC B
op_05:
	dec _B
	DEC8_TAIL

; LD B,n	
op_06:
	LDIMM8 _B

	
; RLCA
op_07:
	;mov al,0
	;cmp al,Carry
	rol _A,1
	setc Carry
	mov HalfCarry,0
	mov AddSub,0
	ADD_CYCLES 4
	inc _PC
	ret

; EX AF,AF'
op_08:
	mov al,_A
	mov ah,_F
	mov bl,_Ap
	mov bh,_Fp
	mov _A,bl
	mov _F,bh
	mov _Ap,al
	mov _Fp,ah
	
	ADD_CYCLES 4
	inc _PC
	ret
	
; ADD HL,BC
op_09:
	mov ax,word ptr [_C]
	add word ptr [_L],ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 11
	inc _PC
	ret

; LD A,(BC)	
op_0A:
	movzx edx,word ptr [_C]
	invoke mmu_read_byte,edx
	mov _A,al
	ADD_CYCLES 7
	inc _PC
	ret

; DEC BC
op_0B:
	dec word ptr [_C]
	ADD_CYCLES 6
	inc _PC
	ret

; INC C	
op_0C:
	inc _C
	INC8_TAIL

; DEC C	
op_0D:
	dec _C
	DEC8_TAIL
	
; LD C,n
op_0E:
	LDIMM8 _C

; RRCA
op_0F:
	ror _A,1
	setc Carry
	mov HalfCarry,0
	mov AddSub,0
	ADD_CYCLES 4
	inc _PC
	ret	

; DJNZ e
op_10:
	dec _B
	jz djnz_cont
		mov al,[esi]
		add al,2
		movsx ebx,al
		add _PC,bx
		ADD_CYCLES 13
		ret
	djnz_cont:	
		ADD_CYCLES 8
		add _PC,2
		ret

; LD DE,nn
op_11:
	LDIMM16 _D,_E

; LD (DE),A	
op_12:
	STRR8 _D,_E,_A

; INC DE	
op_13:
	inc word ptr [_E]
	ADD_CYCLES 6
	inc _PC
	ret
	
; INC D
op_14:
	inc _D
	INC8_TAIL

; DEC D	
op_15:
	dec _D
	DEC8_TAIL
; LD D,n
op_16:
	LDIMM8 _D

; RLA
op_17:
	mov al,0
	cmp al,Carry
	rcl _A,1
	setc Carry
	mov HalfCarry,0
	mov AddSub,0
	ADD_CYCLES 4
	inc _PC
	ret
	
; JR e
op_18:
	mov al,[esi]
	add al,2
	movsx ebx,al
	add _PC,bx
	ADD_CYCLES 12
	ret

; ADD HL,DE		
op_19:
	mov ax,word ptr [_E]
	add word ptr [_L],ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 11
	inc _PC
	ret

; LD A,(DE)	
op_1A:
	movzx edx,word ptr [_E]
	invoke mmu_read_byte,edx
	mov _A,al
	ADD_CYCLES 7
	inc _PC
	ret

; DEC DE	
op_1B:
	dec word ptr [_E]
	ADD_CYCLES 6
	inc _PC
	ret
	
; INC E
op_1C:
	inc _E
	INC8_TAIL
; DEC E
op_1D:
	dec _E
	DEC8_TAIL

; LD E,n	
op_1E:
	LDIMM8 _E

; RRA
op_1F:
	mov al,0
	cmp al,Carry
	rcr _A,1
	setc Carry
	mov HalfCarry,0
	mov AddSub,0
	ADD_CYCLES 4
	inc _PC
	ret
	
; JR NZ,e
op_20:
	RELJUMP Zero,jnz
	
; LD HL,nn
op_21:
	LDIMM16 _H,_L

; LD (nn),HL	
op_22:
	movzx edx,word ptr [esi]
	movzx eax,word ptr [_L]
	invoke mmu_write_word,edx,eax
	ADD_CYCLES 20
	add _PC,3
	ret
	
; INC HL
op_23:
	inc word ptr [_L]
	ADD_CYCLES 6
	inc _PC
	ret
	
; INC H
op_24:
	inc _H
	INC8_TAIL
; DEC H	
op_25:
	dec _H
	DEC8_TAIL

; LD H,n
op_26:
	LDIMM8 _H

; DAA (STUB)	
op_27:
	mov al,_A
	daa
	sets Sign
	setz Zero
	setpe Overflow
	setc Carry
	mov _A,al
	ADD_CYCLES 4
	inc _PC
	ret
	
; JR Z,e
op_28:
	RELJUMP Zero,jz

; ADD HL,HL	
op_29:
	shl word ptr [_L],1
	setc Carry
	mov AddSub,0
	ADD_CYCLES 11
	inc _PC
	ret

; LD HL,(nn)	
op_2A:
	movzx edx,word ptr [esi]
	invoke mmu_read_byte,edx
	mov _L,al
	inc edx
	invoke mmu_read_byte,edx
	mov _H,al
	ADD_CYCLES 20
	add _PC,3
	ret

; DEC HL
op_2B:
	dec word ptr [_L]
	ADD_CYCLES 6
	inc _PC
	ret

; INC L
op_2C:
	inc _L
	INC8_TAIL
; DEC L	
op_2D:
	dec _L
	DEC8_TAIL

; LD L,N
op_2E:
	LDIMM8 _L

; CPL
op_2F:
	xor _A,0FFh
	mov AddSub,1
	mov HalfCarry,1
	ADD_CYCLES 4
	inc _PC
	ret
	
; JR NC,e
op_30:
	RELJUMP Carry,jnz
	
; LD SP,nn
op_31:
	mov ax,[esi]
	mov _SP,ax
	ADD_CYCLES 10
	add _PC,3
	ret

; LD (nn),A	
op_32:
	movzx edx,word ptr [esi]
	movzx eax,byte ptr [_A]
	invoke mmu_write_byte,edx,eax
	ADD_CYCLES 13
	add _PC,3
	ret

; INC SP
op_33:
	inc _SP
	ADD_CYCLES 4
	inc _PC
	ret

; INC (HL)	
op_34:
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	inc al
	INC8_TAIL
	invoke mmu_write_byte,edx,eax
	ADD_CYCLES 11
	inc _PC
	ret

; DEC (HL)	
op_35:
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	dec al
	DEC8_TAIL
	invoke mmu_write_byte,edx,eax
	ADD_CYCLES 11
	inc _PC
	ret
	
; LD (HL),n
op_36:
	movzx edx,word ptr [_L]
	movzx eax,byte ptr [esi]
	invoke mmu_write_byte,edx,eax
	ADD_CYCLES 10
	add _PC,2
	ret

; SCF	
op_37:
	mov Carry,1
	mov AddSub,0
	mov HalfCarry,0
	ADD_CYCLES 4
	inc _PC
	ret

; JR C,e
op_38:
	RELJUMP Carry,jz

; ADD HL,SP	
op_39:
	mov ax,_SP
	add word ptr [_L],ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 11
	inc _PC
	ret
	
; LD A,(nn)	
op_3A:
	movzx edx,word ptr [esi]
	invoke mmu_read_byte,edx
	mov _A,al
	ADD_CYCLES 13
	add _PC,3
	ret

; DEC SP	
op_3B:
	dec word ptr _SP
	ADD_CYCLES 6
	inc _PC
	ret
	
; INC A	
op_3C:
	inc _A
	INC8_TAIL
; DEC A	
op_3D:
	dec _A
	DEC8_TAIL
; LDA N
op_3E:
	LDIMM8 _A

; CCF	
op_3F:
	xor Carry,1
	mov AddSub,1
	ADD_CYCLES 4
	inc _PC
	ret
	
; LD B,B
op_40:
	ADD_CYCLES 4
	inc _PC
	ret
; LD B,C
op_41:
	MOVETO8 _B,_C

; LD B,D
op_42:
	MOVETO8 _B,_D

; LD B,E
op_43:
	MOVETO8 _B,_E

; LD B,H
op_44:
	MOVETO8 _B,_H

; LD B,L
op_45:
	MOVETO8 _B,_L

; LD B,(HL)
op_46:
	LDM16 _B,_H,_L

; LD B,A
op_47:
	MOVETO8 _B,_A

; LD C,B
op_48:
	MOVETO8 _C,_B
; LD C,C	
op_49:
	ADD_CYCLES 4
	inc _PC
	ret
op_4A:
	MOVETO8 _C,_D
op_4B:
	MOVETO8 _C,_E
op_4C:
	MOVETO8 _C,_H
op_4D:
	MOVETO8 _C,_L
op_4E:
	LDM16 _C,_H,_L

op_4F:
	MOVETO8 _C,_A
op_50:
	MOVETO8 _D,_B
op_51:
	MOVETO8 _D,_C
op_52:
	MOVETO8 _D,_D
op_53:
	MOVETO8 _D,_E
op_54:
	MOVETO8 _D,_H
op_55:
	MOVETO8 _D,_L
op_56:
	LDM16 _D,_H,_L

op_57:
	MOVETO8 _D,_A
op_58:
	MOVETO8 _E,_B
op_59:
	MOVETO8 _E,_C
op_5A:
	MOVETO8 _E,_D
op_5B:
	MOVETO8 _E,_E
op_5C:
	MOVETO8 _E,_H
op_5D:
	MOVETO8 _E,_L
op_5E:
	LDM16 _E,_H,_L

op_5F:
	MOVETO8 _E,_A
	
op_60:
	MOVETO8 _H,_B
op_61:
	MOVETO8 _H,_C
op_62:
	MOVETO8 _H,_D
op_63:
	MOVETO8 _H,_E
op_64:
	MOVETO8 _H,_H
op_65:
	MOVETO8 _H,_L
op_66:
	LDM16 _H,_H,_L

op_67:
	MOVETO8 _H,_A
	
op_68:
	MOVETO8 _L,_B
op_69:
	MOVETO8 _L,_C
op_6A:
	MOVETO8 _L,_D
op_6B:
	MOVETO8 _L,_E
op_6C:
	MOVETO8 _L,_H
op_6D:
	MOVETO8 _L,_L
	
; LD L,(HL)
op_6E:
	LDM16 _L,_H,_L

; LD L,A
op_6F:
	MOVETO8 _L,_A
	
; LD (HL),B
op_70:
	STRR8 _H,_L,_B
; LD (HL),C
op_71:
	STRR8 _H,_L,_C
; LD (HL),D
op_72:
	STRR8 _H,_L,_D
; LD (HL),E
op_73:
	STRR8 _H,_L,_E
; LD (HL),H
op_74:
	STRR8 _H,_L,_H
; LD (HL),L
op_75:
	STRR8 _H,_L,_L
; HALT
op_76:
	mov halted,1
	ret
op_77:
	STRR8 _H,_L,_A

; LD A,B
op_78:
	MOVETO8 _A,_B
op_79:
	MOVETO8 _A,_C
op_7A:
	MOVETO8 _A,_D
op_7B:
	MOVETO8 _A,_E
op_7C:
	MOVETO8 _A,_H
op_7D:
	MOVETO8 _A,_L

; LD A,(HL)
op_7E:
	LDM16 _A,_H,_L

; LD A,A
op_7F:
	ADD_CYCLES 4
	inc _PC
	ret
	
; ADD A,B
op_80:
	mov al,_B
	jmp ADD_A
; ADD A,C
op_81:
	mov al,_C
	jmp ADD_A
; ADD A,D
op_82:
	mov al,_D
	jmp ADD_A
; ADD A,E
op_83:
	mov al,_E
	jmp ADD_A
; ADD A,H
op_84:
	mov al,_H
	jmp ADD_A
; ADD A,L
op_85:
	mov al,_L
	;jmp ADD_A
ADD_A:	
	add _A,al
	setc Carry
	seto Overflow
	setz Zero
	sets Sign
	mov AddSub,0
	ADD_CYCLES 4
	inc _PC
	ret
; ADD A,(HL)
op_86:
	ADD_CYCLES 3
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	jmp ADD_A
; ADD A,A
op_87:
	mov al,_A
	jmp ADD_A
	
	
; ADC A,B
op_88:
	mov al,_B
	jmp ADD_A
; ADC A,C
op_89:
	mov al,_C
	jmp ADD_A
; ADC A,D
op_8A:
	mov al,_D
	jmp ADD_A
; ADC A,E
op_8B:
	mov al,_E
	jmp ADD_A
; ADC A,H
op_8C:
	mov al,_H
	jmp ADD_A
; ADC A,L
op_8D:
	mov al,_L
ADC_A:	
	mov ah,Carry
	add _A,al
	add _A,ah
	setc Carry
	seto Overflow
	setz Zero
	sets Sign
	mov AddSub,0
	ADD_CYCLES 4
	inc _PC
	ret
op_8E:
	ADD_CYCLES 3
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	jmp ADC_A
op_8F:
	mov al,_A
	jmp ADC_A

; SUB A,B
op_90:
	mov al,_B
	jmp SUB_A
; SUB A,C
op_91:
	mov al,_C
	jmp SUB_A
; SUB A,D
op_92:
	mov al,_D
	jmp SUB_A
; SUB A,E
op_93:
	mov al,_E
	jmp SUB_A
; SUB A,H
op_94:
	mov al,_H
	jmp SUB_A
; SUB A,L
op_95:
	mov al,_L
SUB_A:	
	sub _A,al
	setc Carry
	seto Overflow
	setz Zero
	sets Sign
	mov AddSub,1
	ADD_CYCLES 4
	inc _PC
	ret
	
op_96:
	ADD_CYCLES 3
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	jmp SUB_A
; SUB A,A
op_97:
	mov al,_A
	jmp SUB_A
	
; SBC A,B
op_98:
	mov al,_B
	jmp SBC_A
; SBC A,C
op_99:
	mov al,_C
	jmp SBC_A
; SBC A,D
op_9A:
	mov al,_D
	jmp SBC_A
; SBC A,E
op_9B:
	mov al,_E
	jmp SBC_A
; SBC A,H
op_9C:
	mov al,_H
	jmp SBC_A
; SBC A,L
op_9D:
	mov al,_L
SBC_A:
	mov ah,Carry
	sub _A,al
	sub _A,ah
	setc Carry
	seto Overflow
	setz Zero
	sets Sign
	mov AddSub,1
	ADD_CYCLES 4
	inc _PC
	ret
	
op_9E:
	ADD_CYCLES 3
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	jmp SBC_A
; SBC A,A
op_9F:
	mov al,_A
	jmp SBC_A
	
	
; AND A,B
op_A0:
	mov al,_B
	jmp AND_A
; AND A,C
op_A1:
	mov al,_C
	jmp AND_A
; AND A,D
op_A2:
	mov al,_D
	jmp AND_A
; AND A,E
op_A3:
	mov al,_E
	jmp AND_A
; AND A,H
op_A4:
	mov al,_H
	jmp AND_A
; AND A,L
op_A5:
	mov al,_L
AND_A:	
	and _A,al
	setpe Overflow
	setz Zero
	sets Sign
	mov AddSub,0
	mov Carry,0
	ADD_CYCLES 4
	inc _PC
	ret
; AND A,(HL)	
op_A6:
	ADD_CYCLES 3
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	jmp AND_A
op_A7:
	mov al,_A
	jmp AND_A
	
; XOR A,B
op_A8:
	mov al,_B
	jmp XOR_A
; XOR A,C
op_A9:
	mov al,_C
	jmp XOR_A
; XOR A,D
op_AA:
	mov al,_D
	jmp XOR_A
; XOR A,E
op_AB:
	mov al,_E
	jmp XOR_A
; XOR A,H
op_AC:
	mov al,_H
	jmp XOR_A
; XOR A,L
op_AD:
	mov al,_L
XOR_A:	
	xor _A,al
	setpe Overflow
	setz Zero
	sets Sign
	mov AddSub,0
	mov Carry,0
	ADD_CYCLES 4
	inc _PC
	ret
	
; XOR (HL)
op_AE:
	ADD_CYCLES 3
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	jmp XOR_A
	
op_AF:
	mov al,_A
	jmp XOR_A
	
; OR A,B
op_B0:
	mov al,_B
	jmp OR_A
; OR A,C
op_B1:
	mov al,_C
	jmp OR_A
; OR A,D
op_B2:
	mov al,_D
	jmp OR_A
; OR A,E
op_B3:
	mov al,_E
	jmp OR_A
; OR A,H
op_B4:
	mov al,_H
	jmp OR_A
; OR A,L
op_B5:
	mov al,_L
OR_A:	
	or _A,al
	setpe Overflow
	setz Zero
	sets Sign
	mov AddSub,0
	mov Carry,0
	ADD_CYCLES 4
	inc _PC
	ret
	
; OR (HL)
op_B6:
	ADD_CYCLES 3
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	jmp OR_A
	
op_B7:
	mov al,_A
	jmp OR_A
op_B8:
	CPR8 _B
op_B9:
	CPR8 _C
op_BA:
	CPR8 _D
op_BB:
	CPR8 _E
op_BC:
	CPR8 _H
op_BD:
	CPR8 _L
; CP (HL)
op_BE:
	ADD_CYCLES 3
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	mov ah,al
	CPR8 ah
	
op_BF:
	CPR8 _A
	
; RET NZ
op_C0:
	.if Zero == 0
		ADD_CYCLES 1
		jmp op_C9
	.endif
	ADD_CYCLES 5
	inc _PC
	ret

; POP BC
op_C1:
	POPR16 _B,_C
	
; JP NZ,nn	
op_C2:
	.if Zero == 0
		;ADD_CYCLES 1
		jmp op_C3
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; JP nn
op_C3:
	movzx eax,word ptr [esi]
	mov _PC,ax
	ADD_CYCLES 10
	ret

; CALL NZ,nn
op_C4:
	.if Zero == 0
		jmp op_CD
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; PUSH BC	
op_C5:
	PUSHR16 _B,_C
; ADD A,n
op_C6:
	ADD_CYCLES 3
	inc _PC
	mov al,[esi]
	jmp ADD_A

; RST $00
op_C7:
	RST 00h
; RET Z
op_C8:
	.if Zero != 0
		ADD_CYCLES 1
		jmp op_C9
	.endif
	ADD_CYCLES 5
	inc _PC
	ret

; RET
op_C9:
	movzx ebx,_SP
	add ebx,RAM
	mov ax,[ebx]
	mov _PC,ax
	add _SP,2
	ADD_CYCLES 10
	ret

; JP Z,nn	
op_CA:
	.if Zero != 0
		;ADD_CYCLES 1
		jmp op_C3
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; CB prefix	
op_CB:
	movzx ebx,byte ptr [esi]
	inc esi
	jmp dword ptr [cb_opcode_table + ebx*4]
	
; CALL Z,nn
op_CC:
	.if Zero != 0
		jmp op_CD
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; CALL nn
op_CD:
	movzx ebx,_SP
	add ebx,RAM
	mov ax,_PC
	sub _SP,2
	add ax,3
	mov [ebx-2],ax
	mov ax,[esi]
	mov _PC,ax
	ADD_CYCLES 17
	ret
	
; ADC A,n
op_CE:
	ADD_CYCLES 3
	inc _PC
	mov al,[esi]
	jmp ADC_A
	
; RST $38
op_CF:
	RST 08h

; RET NC
op_D0:
	.if Carry == 0
		ADD_CYCLES 1
		jmp op_C9
	.endif
	ADD_CYCLES 5
	inc _PC
	ret

; POP DE
op_D1:
	POPR16 _D,_E

; JP NC,ee
op_D2:
	.if Carry == 0
		;ADD_CYCLES 1
		jmp op_C3
	.endif
	ADD_CYCLES 1
	inc _PC
	ret

; OUT (N),A
op_D3:
	movzx edx,byte ptr [esi]
	movzx eax,_A
	invoke mmu_outpb,edx,eax
	ADD_CYCLES 11
	add _PC,2
	ret

; CALL NC,nn	
op_D4:
	.if Carry == 0
		jmp op_CD
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; PUSH DE
op_D5:
	PUSHR16 _D,_E
	
; SUB N
op_D6:
	mov al,[esi]
	sub _A,al
	setc Carry
	seto Overflow
	setz Zero
	sets Sign
	mov AddSub,1
	ADD_CYCLES 7
	add _PC,2
	ret

; RST $10
op_D7:
	RST 10h

; RET C	
op_D8:
	.if Carry != 0
		ADD_CYCLES 1
		jmp op_C9
	.endif
	ADD_CYCLES 5
	inc _PC
	ret
	
; EXX
op_D9:
	mov ax,word ptr [_C]
	mov bx,word ptr [_Cp]
	mov word ptr [_Cp],ax
	mov word ptr [_C],bx
	mov ax,word ptr [_E]
	mov bx,word ptr [_Ep]
	mov word ptr [_Ep],ax
	mov word ptr [_E],bx
	mov ax,word ptr [_L]
	mov bx,word ptr [_Lp]
	mov word ptr [_Lp],ax
	mov word ptr [_L],bx
	ADD_CYCLES 4
	inc _PC
	ret

; JP C,nn	
op_DA:
	.if Carry != 0
		;ADD_CYCLES 1
		jmp op_C3
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; IN A,(N)
op_DB:
	movzx edx,byte ptr [esi]
	invoke mmu_inpb,edx
	mov _A,al
	ADD_CYCLES 11
	add _PC,2
	ret

; CALL C,nn
op_DC:
	.if Carry != 0
		jmp op_CD
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; DD prefix
op_DD:
	movzx ebx,byte ptr [esi]
	inc esi
	jmp dword ptr [dd_opcode_table + ebx*4]

; SBC A.n
op_DE:
	ADD_CYCLES 3
	inc _PC
	mov al,[esi]
	jmp SBC_A
	
; RST $18	
op_DF:
	RST 18h

; RET PO	
op_E0:
	.if Overflow == 0
		ADD_CYCLES 1
		jmp op_C9
	.endif
	ADD_CYCLES 5
	inc _PC
	ret
	
op_E1:
	POPR16 _H,_L

; JP PO,nn
op_E2:
	.if Overflow == 0
		jmp op_C3
	.endif
	ADD_CYCLES 1
	inc _PC
	ret

; EX (SP),HL	
op_E3:
	movzx edx,_SP
	invoke mmu_read_byte,edx
	mov bl,al
	inc edx
	invoke mmu_read_byte,edx
	mov bh,al
	dec edx
	movzx eax,word ptr [_L]
	mov word ptr [_L],bx
	invoke mmu_write_word,edx,eax
	ADD_CYCLES 19
	inc _PC
	ret

; CALL PO,nn	
op_E4:
	.if Overflow == 0
		jmp op_CD
	.endif
	ADD_CYCLES 1
	inc _PC
	ret

; PUSH HL
op_E5:
	PUSHR16 _H,_L

; AND n	
op_E6:
	ADD_CYCLES 3
	inc _PC
	mov al,[esi]
	jmp AND_A
	
; RST $20
op_E7:
	RST 20h

; RET PE	
op_E8:
	.if Overflow != 0
		ADD_CYCLES 1
		jmp op_C9
	.endif
	ADD_CYCLES 5
	inc _PC
	ret
	
; JP (HL)
op_E9:
	mov ax,word ptr [_L]
	mov _PC,ax
	ADD_CYCLES 4
	ret

; JP PE,ee	
op_EA:
	.if Overflow != 0
		;ADD_CYCLES 1
		jmp op_C3
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; EX DE,HL
op_EB:
	mov ax,word ptr [_L]
	mov bx,word ptr [_E]
	mov word ptr [_L],bx
	mov word ptr [_E],ax
	ADD_CYCLES 4
	inc _PC
	ret

; CALL PE,nn	
op_EC:
	.if Overflow != 0
		jmp op_CD
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; ED prefix
op_ED:
	movzx ebx,byte ptr [esi]
	inc esi
	jmp dword ptr [ed_opcode_table + ebx*4]

; XOR N	
op_EE:
	mov al,[esi]
	xor _A,al
	setpe Overflow
	setz Zero
	sets Sign
	mov AddSub,0
	mov Carry,0
	ADD_CYCLES 7
	add _PC,2
	ret

; RST $28	
op_EF:
	RST 28h

; RET P
op_F0:
	.if Sign == 0
		ADD_CYCLES 1
		jmp op_C9
	.endif
	ADD_CYCLES 5
	inc _PC
	ret
	
; POP AF	
op_F1:
	movzx ebx,_SP
	add ebx,RAM
	mov ax,[ebx]
	add _SP,2
	mov _F,al
	mov _A,ah
	
	test al,1
	setne Carry
	test al,2
	setne AddSub
	test al,4
	setne Overflow
	test al,10h
	setne HalfCarry
	test al,40h
	setne Zero
	test al,80h
	setne Sign
	
	ADD_CYCLES 11
	inc _PC
	ret

; JP P,nn	
op_F2:
	.if Sign == 0
		jmp op_C3
	.endif
	ADD_CYCLES 1
	inc _PC
	ret
	
; DI
op_F3:
	mov IFF1,0
	ADD_CYCLES 4
	inc _PC
	ret

; CALL P,nn	
op_F4:
	.if Sign == 0
		jmp op_CD
	.endif
	ADD_CYCLES 1
	inc _PC
	ret

; PUSH AF	
op_F5:
	movzx ebx,_SP
	add ebx,RAM

	; Combine flags
	mov al,Sign
	shl al,1
	add al,Zero
	shl al,2
	add al,HalfCarry
	shl al,2
	add al,Overflow
	shl al,1
	add al,AddSub
	shl al,1
	add al,Carry
	mov _F,al
	
	mov ah,_A
	mov [ebx-2],ax
	sub _SP,2
	ADD_CYCLES 11
	inc _PC
	ret
	
; OR n
op_F6:
	ADD_CYCLES 3
	mov al,[esi]
	jmp OR_A

; RST $30
op_F7:
	RST 30h

; RET M	
op_F8:
	.if Sign != 0
		ADD_CYCLES 1
		jmp op_C9
	.endif
	ADD_CYCLES 5
	inc _PC
	ret

; LD SP,HL	
op_F9:
	mov ax,word ptr [_L]
	mov _SP,ax
	ADD_CYCLES 6
	inc _PC
	ret

; JP M,nn	
op_FA:
	ABSJUMP Sign,jz
	
; EI
op_FB:
	mov IFF1,1
	ADD_CYCLES 4
	inc _PC
	ret

; CALL M,nn	
op_FC:
	.if Sign != 0
		jmp op_CD
	.endif
	ADD_CYCLES 1
	inc _PC
	ret

; FD prefix	
op_FD:
	movzx ebx,byte ptr [esi]
	inc esi
	jmp dword ptr [fd_opcode_table + ebx*4]
	
; CP N
op_FE:
	mov al,_A
	cmp al,[esi]
	sets Sign
	setz Zero
	seto Overflow
	setc Carry
	mov AddSub,1
	ADD_CYCLES 7
	add _PC,2
	ret

; RST $38	
op_FF:
	RST 38h

;-------------------------------------------------------------------------------------------

; RLC B
op_CB00:
	ROTR8 _B,rol
; RLC C
op_CB01:
	ROTR8 _C,rol
; RLC D
op_CB02:
	ROTR8 _D,rol
op_CB03:
	ROTR8 _E,rol
op_CB04:
	ROTR8 _H,rol
op_CB05:
	ROTR8 _L,rol
op_CB07:
	ROTR8 _A,rol

; RRC B
op_CB08:
	ROTR8 _B,ror
; RRC C
op_CB09:
	ROTR8 _C,ror
op_CB0A:
	ROTR8 _D,ror
op_CB0B:
	ROTR8 _E,ror
op_CB0C:
	ROTR8 _H,ror
op_CB0D:
	ROTR8 _L,ror
op_CB0F:
	ROTR8 _A,ror
	
; RL B
op_CB10:
	ROTR8 _B,rcl
; RL C
op_CB11:
	ROTR8 _C,rcl
op_CB12:
	ROTR8 _D,rcl
op_CB13:
	ROTR8 _E,rcl
op_CB14:
	ROTR8 _H,rcl
op_CB15:
	ROTR8 _L,rcl
op_CB17:
	ROTR8 _A,rcl

op_CB18:
	ROTR8 _B,rcr
op_CB19:
	ROTR8 _C,rcr
op_CB1A:
	ROTR8 _D,rcr
op_CB1B:
	ROTR8 _E,rcr
op_CB1C:
	ROTR8 _H,rcr
op_CB1D:
	ROTR8 _L,rcr
op_CB1E:
	ret
op_CB1F:
	ROTR8 _A,rcr
	
; SLA B
op_CB20:
	ROTR8 _B,sal
; SLA C
op_CB21:
	ROTR8 _C,sal
; SLA D
op_CB22:
	ROTR8 _D,sal
; SLA E
op_CB23:
	ROTR8 _E,sal
; SLA H
op_CB24:
	ROTR8 _H,sal
; SLA L 
op_CB25:
	ROTR8 _L,sal
op_CB26:
	ret
; SLA A
op_CB27:
	ROTR8 _A,sal

op_CB2F:
	ROTR8 _A,sar

op_CB30:
	ROTR8 _B,shl
op_CB31:
	ROTR8 _C,shl
op_CB32:
	ROTR8 _D,shl
op_CB33:
	ROTR8 _E,shl
op_CB34:
	ROTR8 _H,shl
op_CB35:
	ROTR8 _L,shl
	
op_CB38:
	ROTR8 _B,shr
op_CB39:
	ROTR8 _C,shr
op_CB3A:
	ROTR8 _D,shr
op_CB3B:
	ROTR8 _E,shr
op_CB3C:
	ROTR8 _H,shr
op_CB3D:
	ROTR8 _L,shr
op_CB3F:
	ROTR8 _A,shr
	
; BIT 0,B
op_CB40:
	BITR8 _B,0
; BIT 0,C
op_CB41:
	BITR8 _C,0
; BIT 0,D
op_CB42:
	BITR8 _D,0
; BIT 0,E
op_CB43:
	BITR8 _E,0
; BIT 0,H
op_CB44:
	BITR8 _H,0
; BIT 0,L
op_CB45:
	BITR8 _L,0
;STUB
op_CB46:
	ADD_CYCLES 12
	add _PC,2
	ret
; BIT 0,A
op_CB47:
	BITR8 _A,0

; BIT 0,B
op_CB48:
	BITR8 _B,1
; BIT 0,C
op_CB49:
	BITR8 _C,1
; BIT 0,D
op_CB4A:
	BITR8 _D,1
; BIT 0,E
op_CB4B:
	BITR8 _E,1
; BIT 0,H
op_CB4C:
	BITR8 _H,1
; BIT 0,L
op_CB4D:
	BITR8 _L,1
;STUB
op_CB4E:
	ADD_CYCLES 12
	add _PC,2
	ret
; BIT 1,A
op_CB4F:
	BITR8 _A,1

; BIT 2,B
op_CB50:
	BITR8 _B,2
; BIT 2,C
op_CB51:
	BITR8 _C,2
; BIT 2,D
op_CB52:
	BITR8 _D,2
; BIT 2,E
op_CB53:
	BITR8 _E,2
; BIT 2,H
op_CB54:
	BITR8 _H,2
; BIT 2,L
op_CB55:
	BITR8 _L,2
;STUB
op_CB56:
	ADD_CYCLES 12
	add _PC,2
	ret
; BIT 2,A
op_CB57:
	BITR8 _A,2

; BIT 3,B
op_CB58:
	BITR8 _B,3
; BIT 3,C
op_CB59:
	BITR8 _C,3
; BIT 3,D
op_CB5A:
	BITR8 _D,3
; BIT 3,E
op_CB5B:
	BITR8 _E,3
; BIT 3,H
op_CB5C:
	BITR8 _H,3
; BIT 3,L
op_CB5D:
	BITR8 _L,3
; BIT 3,A
op_CB5F:
	BITR8 _A,3
	
	
; BIT 4,B
op_CB60:
	BITR8 _B,4
; BIT 4,C
op_CB61:
	BITR8 _C,4
; BIT 4,D
op_CB62:
	BITR8 _D,4
; BIT 4,E
op_CB63:
	BITR8 _E,4
; BIT 4,H
op_CB64:
	BITR8 _H,4
; BIT 4,L
op_CB65:
	BITR8 _L,4
;STUB
op_CB66:
	ADD_CYCLES 12
	add _PC,2
	ret
; BIT 4,A
op_CB67:
	BITR8 _A,4
	
	
; BIT 5,B
op_CB68:
	BITR8 _B,5
; BIT 5,C
op_CB69:
	BITR8 _C,5
; BIT 5,D
op_CB6A:
	BITR8 _D,5
; BIT 5,E
op_CB6B:
	BITR8 _E,5
; BIT 5,H
op_CB6C:
	BITR8 _H,5
; BIT 5,L
op_CB6D:
	BITR8 _L,5
; BIT 5,A
op_CB6F:
	BITR8 _A,5
	
; BIT 6,B
op_CB70:
	BITR8 _B,6
; BIT 6,C
op_CB71:
	BITR8 _C,6
; BIT 6,D
op_CB72:
	BITR8 _D,6
; BIT 6,E
op_CB73:
	BITR8 _E,6
; BIT 6,H
op_CB74:
	BITR8 _H,6
; BIT 6,L
op_CB75:
	BITR8 _L,6
;STUB
op_CB76:
	ADD_CYCLES 12
	add _PC,2
	ret
; BIT 6,A
op_CB77:
	BITR8 _A,6
	
; BIT 7,B
op_CB78:
	BITR8 _B,7
; BIT 7,C
op_CB79:
	BITR8 _C,7
; BIT 7,D
op_CB7A:
	BITR8 _D,7
; BIT 7,E
op_CB7B:
	BITR8 _E,7
; BIT 7,H
op_CB7C:
	BITR8 _H,7
; BIT 7,L
op_CB7D:
	BITR8 _L,7
;STUB
op_CB7E:
	ADD_CYCLES 12
	add _PC,2
	ret
; BIT 7,A
op_CB7F:
	BITR8 _A,7

; RES 0,(HL)
op_CB86:
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	btr eax,0
	invoke mmu_write_byte,edx,eax
	ADD_CYCLES 15
	add _PC,2
	ret
	
; RES 7,B
op_CBB8:
	RESR8 _B,7

; SET 0,B
op_CBC0:
	SETR8 _B,0
; SET 0,C
op_CBC1:
	SETR8 _C,0
; SET 0,D
op_CBC2:
	SETR8 _D,0
; SET 0,E
op_CBC3:
	SETR8 _E,0
; SET 0,H
op_CBC4:
	SETR8 _H,0
; SET 0,L
op_CBC5:
	SETR8 _L,0
; SET 0,A
op_CBC7:
	SETR8 _A,0

; SET 2,B
op_CBD0:
	SETR8 _B,2
; SET 2,C
op_CBD1:
	SETR8 _C,2
; SET 2,D
op_CBD2:
	SETR8 _D,2
; SET 2,E
op_CBD3:
	SETR8 _E,2
; SET 2,H
op_CBD4:
	SETR8 _H,2
; SET 2,L
op_CBD5:
	SETR8 _L,2
; SET 2,A
op_CBD7:
	SETR8 _A,2
	
;-------------------------------------------------------------------------------------------

; ADD IX,BC
op_DD09:
	mov ax,word ptr [_C]
	add _IX,ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 15
	add _PC,2
	ret

; ADD IX,DE
op_DD19:
	mov ax,word ptr [_E]
	add _IX,ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 15
	add _PC,2
	ret

; LD IX,nn
op_DD21:
	mov ax,[esi]
	mov _IX,ax
	ADD_CYCLES 14
	add _PC,4
	ret

; INC IX
op_DD23:
	inc word ptr _IX
	ADD_CYCLES 10
	add _PC,2
	ret
	
; ADD IX,IX
op_DD29:
	mov ax,_IX
	add _IX,ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 15
	add _PC,2
	ret

; LD (IX+n),N
op_DD36:
	movzx edx,byte ptr [esi]
	add dx,_IX
	movzx eax,byte ptr [esi+1]
	invoke mmu_write_byte,edx,eax
	ADD_CYCLES 19
	add _PC,4
	ret

; ADD IX,SP
op_DD39:
	mov ax,_SP
	add _IX,ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 15
	add _PC,2
	ret

; LD B,(IX+n)
op_DD46:
	LDIXN _IX,_B

; LD C,(IX+n)
op_DD4E:
	LDIXN _IX,_C

; LD D,(IX+n)
op_DD56:
	LDIXN _IX,_D

; LD E,(IX+n)
op_DD5E:
	LDIXN _IX,_E

; LD H,(IX+n)
op_DD66:
	LDIXN _IX,_H
	
; LD L,(IX+n)
op_DD6E:
	LDIXN _IX,_L

; LD (IX+n),B
op_DD70:
	STRIXN _IX,_B
; LD (IX+n),C
op_DD71:
	STRIXN _IX,_C
; LD (IX+n),D
op_DD72:
	STRIXN _IX,_D
; LD (IX+n),E
op_DD73:
	STRIXN _IX,_E
; LD (IX+n),H
op_DD74:
	STRIXN _IX,_H
; LD (IX+n),L
op_DD75:
	STRIXN _IX,_L
; LD (IX+n),A
op_DD77:
	STRIXN _IX,_A
	
; LD A,(IX+n)
op_DD7E:
	LDIXN _IX,_A

; ADD A,(IX+n)
op_DD86:
	ADD_CYCLES 15
	movzx edx,byte ptr [esi]
	add dx,_IX
	invoke mmu_read_byte,edx
	add _PC,2
	jmp ADD_A

; SUB A,(IX+n)
op_DD96:
	ADD_CYCLES 15
	movzx edx,byte ptr [esi]
	add dx,_IX
	invoke mmu_read_byte,edx
	add _PC,2
	jmp SUB_A

; AND A,(IX+n)
op_DDA6:
	ADD_CYCLES 15
	movzx edx,byte ptr [esi]
	add dx,_IX
	invoke mmu_read_byte,edx
	add _PC,2
	jmp AND_A

; ADD A,(IX+n)
op_DDB6:
	ADD_CYCLES 15
	movzx edx,byte ptr [esi]
	add dx,_IX
	invoke mmu_read_byte,edx
	add _PC,2
	jmp OR_A	
	
; POP IX
op_DDE1:
	inc _PC
	ADD_CYCLES 4
	POPR16 _IX,_IX

; PUSH IX
op_DDE5:
	inc _PC
	ADD_CYCLES 4
	PUSHR16 _IX,_IX

; RST 38
op_DDFF:
	RST 38h
	
;-------------------------------------------------------------------------------------------

op_ED00:
	ADD_CYCLES 7
	add _PC,2
	ret
	
; OUT (C),B
op_ED41:
	movzx edx,byte ptr [_C]
	movzx eax,_B
	invoke mmu_outpb,edx,eax
	ADD_CYCLES 12
	add _PC,2
	ret

; SBC HL,BC
op_ED42:
	mov ax,word ptr [_C]
	cmp Carry,1
	cmc
	sbb word ptr [_L],ax
	sets Sign
	setz Zero
	seto Overflow
	setc Carry
	mov AddSub,1
	ADD_CYCLES 15
	add _PC,2
	ret

; LD (nn),BC
op_ED43:
	movzx edx,word ptr [esi]
	mov ax,word ptr [_C]
	invoke mmu_write_word,edx,eax
	ADD_CYCLES 20
	add _PC,4
	ret
; RETN
op_ED45:
	mov al,IFF2
	mov IFF1,al
	movzx ebx,_SP
	add ebx,RAM
	mov ax,[ebx]
	mov _PC,ax
	add _SP,2
	ADD_CYCLES 14
	ret
	
; IM 0
op_ED46:
	mov IMF,0
	ADD_CYCLES 8
	add _PC,2
	ret

; OUT (C),C
op_ED49:
	movzx edx,byte ptr [_C]
	movzx eax,_C
	invoke mmu_outpb,edx,eax
	ADD_CYCLES 12
	add _PC,2
	ret
	
; LD BC,(nn)
op_ED4B:
	movzx edx,word ptr [esi]
	invoke mmu_read_byte,edx
	mov _C,al
	inc edx
	invoke mmu_read_byte,edx
	mov _B,al
	ADD_CYCLES 20
	add _PC,4
	ret

; RETI
op_ED4D:
	movzx ebx,_SP
	add ebx,RAM
	mov ax,[ebx]
	mov _PC,ax
	add _SP,2
	ADD_CYCLES 14
	ret

; OUT (C),D
op_ED51:
	movzx edx,byte ptr [_C]
	movzx eax,_D
	invoke mmu_outpb,edx,eax
	ADD_CYCLES 12
	add _PC,2
	ret

; SBC HL,DE
op_ED52:
	mov ax,word ptr [_E]
	cmp Carry,1
	cmc
	sbb word ptr [_L],ax
	sets Sign
	setz Zero
	seto Overflow
	setc Carry
	mov AddSub,1
	ADD_CYCLES 15
	add _PC,2
	ret
	
; IM 1
op_ED56:
	mov IMF,2
	ADD_CYCLES 8
	add _PC,2
	ret

; OUT (C),H
op_ED61:
	movzx edx,byte ptr [_C]
	movzx eax,byte ptr [_H]
	invoke mmu_outpb,edx,eax
	ADD_CYCLES 12
	add _PC,2
	ret
	
; OUT (C),L
op_ED69:
	movzx edx,byte ptr [_C]
	movzx eax,byte ptr [_L]
	invoke mmu_outpb,edx,eax
	ADD_CYCLES 12
	add _PC,2
	ret
	
; OUT (C),0
op_ED71:
	movzx edx,byte ptr [_C]
	xor eax,eax
	invoke mmu_outpb,edx,eax
	ADD_CYCLES 12
	add _PC,2
	ret
	
; IN A,(C)
op_ED78:
	movzx edx,byte ptr [_C]
	invoke mmu_inpb,edx
	mov _A,al
	ADD_CYCLES 12
	add _PC,2
	ret

; OUT (C),A
op_ED79:
	movzx edx,byte ptr [_C]
	movzx eax,byte ptr [_A]
	invoke mmu_outpb,edx,eax
	ADD_CYCLES 12
	add _PC,2
	ret

; LDI
op_EDA0:
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	movzx edx,word ptr [_E]
	invoke mmu_write_byte,edx,eax
	inc word ptr [_L]
	inc word ptr [_E]
	dec word ptr [_C]
	setz Overflow
	mov HalfCarry,0
	mov AddSub,0
	ADD_CYCLES 16
	add _PC,2
	ret
	
; CPI
op_EDA1:
	movzx edx,word ptr [_L]
	invoke mmu_read_byte,edx
	cmp _A,al
	sets Sign
	setz Zero
	seto Overflow
	setc Carry
	mov AddSub,1
	inc word ptr [_L]
	dec word ptr [_C]
	ADD_CYCLES 16
	add _PC,2
	ret

; OUTI
op_EDA3:
	movzx ebx,word ptr [_L]
	invoke mmu_read_byte,ebx
	movzx edx,_C
	invoke mmu_outpb,edx,eax 
	inc bx
	dec _B
	setz Zero
	mov AddSub,1
	mov word ptr [_L],bx
	ADD_CYCLES 16
	add _PC,2
	ret

; LDIR
op_EDB0:
	@@ldir_loop:
		movzx edx,word ptr [_L]
		invoke mmu_read_byte,edx
		movzx edx,word ptr [_E]
		invoke mmu_write_byte,edx,eax
		inc word ptr [_L]
		inc word ptr [_E]
		dec word ptr [_C]
		cmp word ptr [_C],0
		je @@ldir_end
		add ecx,400h
	jmp @@ldir_loop
	@@ldir_end:
	mov HalfCarry,0
	mov Overflow,0
	mov AddSub,0
	ADD_CYCLES 21
	add _PC,2
	ret

; OTIR
op_EDB3:
	movzx edx,_C
	movzx ebx,word ptr [_L]
	@@otir_loop:
		push ebx
		push edx
		invoke mmu_read_byte,ebx
		invoke mmu_outpb,edx,eax 
		pop edx
		pop ebx
		inc bx
		dec _B
		add ecx,400h
		cmp _B,0
		je @@otir_end
	jmp @@otir_loop
	@@otir_end:
	mov Zero,1
	mov word ptr [_L],bx
	ADD_CYCLES 21
	add _PC,2
	ret

; LDDR
op_EDB8:
	@@lddr_loop:
		cmp word ptr [_C],0
		je @@lddr_end
		movzx edx,word ptr [_L]
		invoke mmu_read_byte,edx
		movzx edx,word ptr [_E]
		invoke mmu_write_byte,edx,eax
		dec word ptr [_L]
		dec word ptr [_C]
		dec word ptr [_E]
		add ecx,400h
	jmp @@lddr_loop
	@@lddr_end:
	mov HalfCarry,0
	mov Overflow,0
	mov AddSub,0
	ADD_CYCLES 21
	add _PC,2
	ret
	

;-------------------------------------------------------------------------------------------

; ADD IY,BC
op_FD09:
	mov ax,word ptr [_C]
	add _IY,ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 15
	add _PC,2
	ret

; ADD IY,DE
op_FD19:
	mov ax,word ptr [_E]
	add _IY,ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 15
	add _PC,2
	ret

; LD IY,nn
op_FD21:
	mov ax,[esi]
	mov _IY,ax
	ADD_CYCLES 14
	add _PC,4
	ret

; INC IY
op_FD23:
	inc word ptr _IY
	ADD_CYCLES 10
	add _PC,2
	ret
	
; ADD IY,IY
op_FD29:
	mov ax,_IY
	add _IY,ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 15
	add _PC,2
	ret

; ADD IY,SP
op_FD39:
	mov ax,_SP
	add _IY,ax
	setc Carry
	mov AddSub,0
	ADD_CYCLES 15
	add _PC,2
	ret

; LD B,(IY+n)
op_FD46:
	LDIXN _IY,_B

; LD C,(IY+n)
op_FD4E:
	LDIXN _IY,_C

; LD D,(IY+n)
op_FD56:
	LDIXN _IY,_D

; LD E,(IY+n)
op_FD5E:
	LDIXN _IY,_E

; LD H,(IY+n)
op_FD66:
	LDIXN _IY,_H
	
; LD L,(IY+n)
op_FD6E:
	LDIXN _IY,_L

; LD (IY+n),B
op_FD70:
	STRIXN _IY,_B
; LD (IY+n),C
op_FD71:
	STRIXN _IY,_C
; LD (IY+n),D
op_FD72:
	STRIXN _IY,_D
; LD (IY+n),E
op_FD73:
	STRIXN _IY,_E
; LD (IY+n),H
op_FD74:
	STRIXN _IY,_H
; LD (IY+n),L
op_FD75:
	STRIXN _IY,_L
; LD (IY+n),A
op_FD77:
	STRIXN _IY,_A
	
; LD A,(IY+n)
op_FD7E:
	LDIXN _IY,_A
	
; ADD A,(IY+n)
op_FD86:
	ADD_CYCLES 15
	movzx edx,byte ptr [esi]
	add dx,_IY
	invoke mmu_read_byte,edx
	add _PC,2
	jmp ADD_A

; SUB A,(IY+n)
op_FD96:
	ADD_CYCLES 15
	movzx edx,byte ptr [esi]
	add dx,_IY
	invoke mmu_read_byte,edx
	add _PC,2
	jmp SUB_A

; AND A,(IY+n)
op_FDA6:
	ADD_CYCLES 15
	movzx edx,byte ptr [esi]
	add dx,_IY
	invoke mmu_read_byte,edx
	add _PC,2
	jmp AND_A

; ADD A,(IY+n)
op_FDB6:
	ADD_CYCLES 15
	movzx edx,byte ptr [esi]
	add dx,_IY
	invoke mmu_read_byte,edx
	add _PC,2
	jmp OR_A	
	
; POP IY
op_FDE1:
	inc _PC
	ADD_CYCLES 4
	POPR16 _IY,_IY

; PUSH IY
op_FDE5:
	inc _PC
	ADD_CYCLES 4
	PUSHR16 _IY,_IY

; RST 38
op_FDFF:
	RST 38h
	
;-------------------------------------------------------------------------------------------

	
opcode_table:
	dd OFFSET op_00NOP
	dd OFFSET op_01
	dd OFFSET op_02
	dd OFFSET op_03
	dd OFFSET op_04
	dd OFFSET op_05
	dd OFFSET op_06
	dd OFFSET op_07
	dd OFFSET op_08
	dd OFFSET op_09
	dd OFFSET op_0A
	dd OFFSET op_0B
	dd OFFSET op_0C
	dd OFFSET op_0D
	dd OFFSET op_0E
	dd OFFSET op_0F

	dd OFFSET op_10
	dd OFFSET op_11
	dd OFFSET op_12
	dd OFFSET op_13
	dd OFFSET op_14
	dd OFFSET op_15
	dd OFFSET op_16
	dd OFFSET op_17
	dd OFFSET op_18
	dd OFFSET op_19
	dd OFFSET op_1A
	dd OFFSET op_1B
	dd OFFSET op_1C
	dd OFFSET op_1D
	dd OFFSET op_1E
	dd OFFSET op_1F

	dd OFFSET op_20
	dd OFFSET op_21
	dd OFFSET op_22
	dd OFFSET op_23
	dd OFFSET op_24
	dd OFFSET op_25
	dd OFFSET op_26
	dd OFFSET op_27
	dd OFFSET op_28
	dd OFFSET op_29
	dd OFFSET op_2A
	dd OFFSET op_2B
	dd OFFSET op_2C
	dd OFFSET op_2D
	dd OFFSET op_2E
	dd OFFSET op_2F

	dd OFFSET op_30
	dd OFFSET op_31
	dd OFFSET op_32
	dd OFFSET op_33
	dd OFFSET op_34
	dd OFFSET op_35
	dd OFFSET op_36
	dd OFFSET op_37
	dd OFFSET op_38
	dd OFFSET op_39
	dd OFFSET op_3A
	dd OFFSET op_3B
	dd OFFSET op_3C
	dd OFFSET op_3D
	dd OFFSET op_3E
	dd OFFSET op_3F

	dd OFFSET op_40
	dd OFFSET op_41
	dd OFFSET op_42
	dd OFFSET op_43
	dd OFFSET op_44
	dd OFFSET op_45
	dd OFFSET op_46
	dd OFFSET op_47
	dd OFFSET op_48
	dd OFFSET op_49
	dd OFFSET op_4A
	dd OFFSET op_4B
	dd OFFSET op_4C
	dd OFFSET op_4D
	dd OFFSET op_4E
	dd OFFSET op_4F

	dd OFFSET op_50
	dd OFFSET op_51
	dd OFFSET op_52
	dd OFFSET op_53
	dd OFFSET op_54
	dd OFFSET op_55
	dd OFFSET op_56
	dd OFFSET op_57
	dd OFFSET op_58
	dd OFFSET op_59
	dd OFFSET op_5A
	dd OFFSET op_5B
	dd OFFSET op_5C
	dd OFFSET op_5D
	dd OFFSET op_5E
	dd OFFSET op_5F

	dd OFFSET op_60
	dd OFFSET op_61
	dd OFFSET op_62
	dd OFFSET op_63
	dd OFFSET op_64
	dd OFFSET op_65
	dd OFFSET op_66
	dd OFFSET op_67
	dd OFFSET op_68
	dd OFFSET op_69
	dd OFFSET op_6A
	dd OFFSET op_6B
	dd OFFSET op_6C
	dd OFFSET op_6D
	dd OFFSET op_6E
	dd OFFSET op_6F

	dd OFFSET op_70
	dd OFFSET op_71
	dd OFFSET op_72
	dd OFFSET op_73
	dd OFFSET op_74
	dd OFFSET op_75
	dd OFFSET op_76
	dd OFFSET op_77
	dd OFFSET op_78
	dd OFFSET op_79
	dd OFFSET op_7A
	dd OFFSET op_7B
	dd OFFSET op_7C
	dd OFFSET op_7D
	dd OFFSET op_7E
	dd OFFSET op_7F

	dd OFFSET op_80
	dd OFFSET op_81
	dd OFFSET op_82
	dd OFFSET op_83
	dd OFFSET op_84
	dd OFFSET op_85
	dd OFFSET op_86
	dd OFFSET op_87
	dd OFFSET op_88
	dd OFFSET op_89
	dd OFFSET op_8A
	dd OFFSET op_8B
	dd OFFSET op_8C
	dd OFFSET op_8D
	dd OFFSET op_8E
	dd OFFSET op_8F

	dd OFFSET op_90
	dd OFFSET op_91
	dd OFFSET op_92
	dd OFFSET op_93
	dd OFFSET op_94
	dd OFFSET op_95
	dd OFFSET op_96
	dd OFFSET op_97
	dd OFFSET op_98
	dd OFFSET op_99
	dd OFFSET op_9A
	dd OFFSET op_9B
	dd OFFSET op_9C
	dd OFFSET op_9D
	dd OFFSET op_9E
	dd OFFSET op_9F

	dd OFFSET op_A0
	dd OFFSET op_A1
	dd OFFSET op_A2
	dd OFFSET op_A3
	dd OFFSET op_A4
	dd OFFSET op_A5
	dd OFFSET op_A6
	dd OFFSET op_A7
	dd OFFSET op_A8
	dd OFFSET op_A9
	dd OFFSET op_AA
	dd OFFSET op_AB
	dd OFFSET op_AC
	dd OFFSET op_AD
	dd OFFSET op_AE
	dd OFFSET op_AF

	dd OFFSET op_B0
	dd OFFSET op_B1
	dd OFFSET op_B2
	dd OFFSET op_B3
	dd OFFSET op_B4
	dd OFFSET op_B5
	dd OFFSET op_B6
	dd OFFSET op_B7
	dd OFFSET op_B8
	dd OFFSET op_B9
	dd OFFSET op_BA
	dd OFFSET op_BB
	dd OFFSET op_BC
	dd OFFSET op_BD
	dd OFFSET op_BE
	dd OFFSET op_BF

	dd OFFSET op_C0
	dd OFFSET op_C1
	dd OFFSET op_C2
	dd OFFSET op_C3
	dd OFFSET op_C4
	dd OFFSET op_C5
	dd OFFSET op_C6
	dd OFFSET op_C7
	dd OFFSET op_C8
	dd OFFSET op_C9
	dd OFFSET op_CA
	dd OFFSET op_CB
	dd OFFSET op_CC
	dd OFFSET op_CD
	dd OFFSET op_CE
	dd OFFSET op_CF

	dd OFFSET op_D0
	dd OFFSET op_D1
	dd OFFSET op_D2
	dd OFFSET op_D3
	dd OFFSET op_D4
	dd OFFSET op_D5
	dd OFFSET op_D6
	dd OFFSET op_D7
	dd OFFSET op_D8
	dd OFFSET op_D9
	dd OFFSET op_DA
	dd OFFSET op_DB
	dd OFFSET op_DC
	dd OFFSET op_DD
	dd OFFSET op_DE
	dd OFFSET op_DF

	dd OFFSET op_E0
	dd OFFSET op_E1
	dd OFFSET op_E2
	dd OFFSET op_E3
	dd OFFSET op_E4
	dd OFFSET op_E5
	dd OFFSET op_E6
	dd OFFSET op_E7
	dd OFFSET op_E8
	dd OFFSET op_E9
	dd OFFSET op_EA
	dd OFFSET op_EB
	dd OFFSET op_EC
	dd OFFSET op_ED
	dd OFFSET op_EE
	dd OFFSET op_EF

	dd OFFSET op_F0
	dd OFFSET op_F1
	dd OFFSET op_F2
	dd OFFSET op_F3
	dd OFFSET op_F4
	dd OFFSET op_F5
	dd OFFSET op_F6
	dd OFFSET op_F7
	dd OFFSET op_F8
	dd OFFSET op_F9
	dd OFFSET op_FA
	dd OFFSET op_FB
	dd OFFSET op_FC
	dd OFFSET op_FD
	dd OFFSET op_FE
	dd OFFSET op_FF	


; Opcodes with ED prefix
cb_opcode_table:
	dd OFFSET op_CB00
	dd OFFSET op_CB01
	dd OFFSET op_CB02
	dd OFFSET op_CB03
	dd OFFSET op_CB04
	dd OFFSET op_CB05
	dd OFFSET op_00
	dd OFFSET op_CB07
	dd OFFSET op_CB08
	dd OFFSET op_CB09
	dd OFFSET op_CB0A
	dd OFFSET op_CB0B
	dd OFFSET op_CB0C
	dd OFFSET op_CB0D
	dd OFFSET op_00
	dd OFFSET op_CB0F

	dd OFFSET op_CB10
	dd OFFSET op_CB11
	dd OFFSET op_CB12
	dd OFFSET op_CB13
	dd OFFSET op_CB14
	dd OFFSET op_CB15
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_CB18
	dd OFFSET op_CB19
	dd OFFSET op_CB1A
	dd OFFSET op_CB1B
	dd OFFSET op_CB1C
	dd OFFSET op_CB1D
	dd OFFSET op_00
	dd OFFSET op_CB1F

	dd OFFSET op_CB20
	dd OFFSET op_CB21
	dd OFFSET op_CB22
	dd OFFSET op_CB23
	dd OFFSET op_CB24
	dd OFFSET op_CB25
	dd OFFSET op_CB26
	dd OFFSET op_CB27
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_CB2F

	dd OFFSET op_CB30
	dd OFFSET op_CB31
	dd OFFSET op_CB32
	dd OFFSET op_CB33
	dd OFFSET op_CB34
	dd OFFSET op_CB35
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_CB38
	dd OFFSET op_CB39
	dd OFFSET op_CB3A
	dd OFFSET op_CB3B
	dd OFFSET op_CB3C
	dd OFFSET op_CB3D
	dd OFFSET op_00
	dd OFFSET op_CB3F

	dd OFFSET op_CB40
	dd OFFSET op_CB41
	dd OFFSET op_CB42
	dd OFFSET op_CB43
	dd OFFSET op_CB44
	dd OFFSET op_CB45
	dd OFFSET op_CB46
	dd OFFSET op_CB47
	dd OFFSET op_CB48
	dd OFFSET op_CB49
	dd OFFSET op_CB4A
	dd OFFSET op_CB4B
	dd OFFSET op_CB4C
	dd OFFSET op_CB4D
	dd OFFSET op_CB4E
	dd OFFSET op_CB4F

	dd OFFSET op_CB50
	dd OFFSET op_CB51
	dd OFFSET op_CB52
	dd OFFSET op_CB53
	dd OFFSET op_CB54
	dd OFFSET op_CB55
	dd OFFSET op_CB56
	dd OFFSET op_CB57
	dd OFFSET op_CB58
	dd OFFSET op_CB59
	dd OFFSET op_CB5A
	dd OFFSET op_CB5B
	dd OFFSET op_CB5C
	dd OFFSET op_CB5D
	dd OFFSET op_00
	dd OFFSET op_CB5F

	dd OFFSET op_CB60
	dd OFFSET op_CB61
	dd OFFSET op_CB62
	dd OFFSET op_CB63
	dd OFFSET op_CB64
	dd OFFSET op_CB65
	dd OFFSET op_00
	dd OFFSET op_CB67
	dd OFFSET op_CB68
	dd OFFSET op_CB69
	dd OFFSET op_CB6A
	dd OFFSET op_CB6B
	dd OFFSET op_CB6C
	dd OFFSET op_CB6D
	dd OFFSET op_00
	dd OFFSET op_CB6F

	dd OFFSET op_CB70
	dd OFFSET op_CB71
	dd OFFSET op_CB72
	dd OFFSET op_CB73
	dd OFFSET op_CB74
	dd OFFSET op_CB75
	dd OFFSET op_00
	dd OFFSET op_CB77
	dd OFFSET op_CB78
	dd OFFSET op_CB79
	dd OFFSET op_CB7A
	dd OFFSET op_CB7B
	dd OFFSET op_CB7C
	dd OFFSET op_CB7D
	dd OFFSET op_CB7E
	dd OFFSET op_CB7F

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_CB86
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; 9
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; A
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_CBB8
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_CBC0
	dd OFFSET op_CBC1	; C
	dd OFFSET op_CBC2
	dd OFFSET op_CBC3
	dd OFFSET op_CBC4
	dd OFFSET op_CBC5
	dd OFFSET op_00
	dd OFFSET op_CBC7
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_CBD0
	dd OFFSET op_CBD1
	dd OFFSET op_CBD2
	dd OFFSET op_CBD3
	dd OFFSET op_CBD4
	dd OFFSET op_CBD5
	dd OFFSET op_00
	dd OFFSET op_CBD7
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00	; E
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	
	
	
; Opcodes with DD prefix
dd_opcode_table:
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD09
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD19
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_DD21
	dd OFFSET op_00
	dd OFFSET op_DD23
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD29
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD36
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD39
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD46
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; 4
	dd OFFSET op_DD4E
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD56
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD5E
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; 6
	dd OFFSET op_DD66
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD6E
	dd OFFSET op_00	

	dd OFFSET op_DD70
	dd OFFSET op_DD71
	dd OFFSET op_DD72
	dd OFFSET op_DD73
	dd OFFSET op_DD74
	dd OFFSET op_DD75
	dd OFFSET op_00
	dd OFFSET op_DD77
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD7E
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD86
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; 8
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00		

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DD96
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DDA6
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; A
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DDB6
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; C
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_DDE1
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DDE5
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_DDFF	
	
	
; Opcodes with ED prefix
ed_opcode_table:
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_ED41
	dd OFFSET op_ED42
	dd OFFSET op_ED43
	dd OFFSET op_00
	dd OFFSET op_ED45
	dd OFFSET op_ED46
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_ED49
	dd OFFSET op_00
	dd OFFSET op_ED4B
	dd OFFSET op_00
	dd OFFSET op_ED4D
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_ED51
	dd OFFSET op_ED52
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_ED56
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_ED61
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_ED69
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_ED71
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; 7
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_ED78
	dd OFFSET op_ED79
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; 9
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_EDA0
	dd OFFSET op_EDA1
	dd OFFSET op_00
	dd OFFSET op_EDA3
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_EDB0
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_EDB3
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00
	dd OFFSET op_ED00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	
	

; Opcodes with FD prefix
fd_opcode_table:
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FD09
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FD19
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_FD21
	dd OFFSET op_00
	dd OFFSET op_FD23
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FD29
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FD39
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	;5
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_FD70
	dd OFFSET op_FD71
	dd OFFSET op_FD72
	dd OFFSET op_FD73
	dd OFFSET op_FD74
	dd OFFSET op_FD75
	dd OFFSET op_00
	dd OFFSET op_FD77
	dd OFFSET op_00
	dd OFFSET op_00	; 7
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FD86
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FD96
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; 9
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FDA6
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00		

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FDB6
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; B
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	; D
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_FDE1
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FDE5
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00	

	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_00
	dd OFFSET op_FDFF	
	
	

zoft80_init PROC pcount:DWORD
	invoke mmu_get_ram_base
	sub eax,0C000h
	mov RAM,eax
	
	mov cycle,0
	mov eax,pcount
	mov _PC,ax
	
	mov eax,OFFSET _A
	ret
zoft80_init ENDP


zoft80_reset PROC
	pusha
	mov edi,OFFSET start_regs
	mov ecx,OFFSET end_regs
	sub ecx,edi
	inc ecx
	mov eax,0
	rep stosb
	mov breakpoint,80000000h
	popa
	ret
zoft80_reset ENDP


zoft80_set_pc PROC pcount:DWORD
	mov eax,pcount
	mov _PC,ax
	ret
zoft80_set_pc ENDP



zoft80_set_breakpoint PROC breakp:DWORD
	mov eax,breakp
	mov breakpoint,eax
	ret
zoft80_set_breakpoint ENDP


zoft80_execute PROC threshold:DWORD
	LOCAL lastCyc:DWORD
	
	pusha
	
	mov ecx,cycle

	cmp halted,0
	je @@exe_loop
	@@is_halted:
	popa
	mov eax,threshold
	add eax,400h
	mov cycle,eax
	ret
	
	@@exe_loop:
		cmp ecx,threshold
		jge @@exe_done
		mov lastCyc,ecx
		
		movzx eax,_PC
		cmp eax,breakpoint
		jne @@no_breakpoint
		jmp @@at_breakpoint
		@@no_breakpoint:
		
		mov lastPC,ax
		push eax
		call mmu_read_byte
		inc esi
		
		call dword ptr [opcode_table + eax*4]

		test halted,1
		jnz @@is_halted
		
		;cmp _PC,08000h
		;jae @@bad_pc
		
		; !! Temp fix !!		
		cmp ecx,lastCyc
		jne @@exe_loop
		mov ecx,1
	;jmp @@exe_loop
	 
	@@exe_done:
	mov cycle,ecx
	popa
	
	mov eax,cycle
	ret

	@@bad_pc:
		popa
		mov eax,2
		ret
	@@at_breakpoint:
		popa
		mov eax,3
		ret
zoft80_execute ENDP


END LibMain
