/*
	Skelton for Z-80 PC Emulator

	Author : Takeda.Toshiya
	Date   : 2004.04.29 -

	[ Emulation Core (win32) ]
*/

#include "emu.h"
#include "vm/vm.h"
#include "config.h"

extern config_t config;

// ----------------------------------------------------------------------------
// initialize
// ----------------------------------------------------------------------------

EMU::EMU(HWND hwnd)
{
	// open debug logfile
	open_debug();
	
	// store main window handle
	main_window_handle = hwnd;
	
	// load sound config
	int frequency, latency;
	
	switch(config.sound_frequency)
	{
#ifdef _WIN32_WCE
		case 0: frequency = 1000; break;
		case 1: frequency = 2000; break;
		case 2: frequency = 3000; break;
		case 3: frequency = 4000; break;
		default:
			frequency = 2000;
			config.sound_frequency = 1;
#else
		case 0: frequency = 11025; break;
		case 1: frequency = 22050; break;
		case 2: frequency = 44100; break;
		case 3: frequency = 48000; break;
		default:
			frequency = 22050;
			config.sound_frequency = 1;
#endif
	}
	switch(config.sound_latency)
	{
		case 0: latency =  6; break;	// 0.1sec
		case 1: latency = 11; break;	// 0.2sec
		case 2: latency = 17; break;	// 0.3sec
		case 3: latency = 22; break;	// 0.4sec
		default:
			latency = 11;
			config.sound_latency = 1;
	}
	sound_enable = config.sound_enable;
	
	// initialize disk uif
	for(int i = 0; i < 4; i++) {
		insert_request[i] = false;
		eject_wait[i] = 0;
	}
	
	// initialize
	vm = new VM(this);
	
	initialize_input();
	initialize_media();
	initialize_screen();
	initialize_sound(frequency, (int)(frequency * latency / FRAMES_PER_SEC + 0.5));
	initialize_socket();
	
	update_timer();
}

EMU::~EMU()
{
	release_input();
	release_media();
	release_screen();
	release_sound();
	release_socket();
	if(vm)
		delete vm;
	close_debug();
}

// ----------------------------------------------------------------------------
// drive machine
// ----------------------------------------------------------------------------

void EMU::run()
{
	mute = false;
	
	// run virtual machine
	vm->run();
	
	// run real machine
	update_input();
	update_sound();
	update_timer();
	update_socket();
	
	// disk uif
	for(int drv = 0; drv < 4; drv++) {
		if(eject_wait[drv]) {
			if(--eject_wait[drv] == 0 && insert_request[drv]) {
				vm->insert_disk(insert_path[drv], drv);
				insert_request[drv] = false;
			}
		}
	}
}

void EMU::reset()
{
	// reset virtual machine
	vm->reset();
	
	// stop media
	stop_media();
}

void EMU::ipl_reset()
{
	// reset virtual machine
	vm->ipl_reset();
}

void EMU::application_path(_TCHAR* path)
{
	GetModuleFileName(NULL, path, _MAX_PATH);
	int pt = _tcslen(path);
	while(path[pt] != '\\')
		pt--;
	path[pt + 1] = '\0';
}

// ----------------------------------------------------------------------------
// debug log
// ----------------------------------------------------------------------------

void EMU::open_debug()
{
#ifdef _DEBUG_LOG_
	debug = fopen("d:\\debug.log", "w");
#endif
}

void EMU::close_debug()
{
#ifdef _DEBUG_LOG_
	fclose(debug);
#endif
}

void EMU::out_debug(const _TCHAR* format, ...)
{
#ifdef _DEBUG_LOG_
	va_list ap;
	_TCHAR buffer[1024];
	
	va_start(ap, format);
	_vstprintf(buffer, format, ap);
	_ftprintf(debug, _T("%s"), buffer);
	va_end(ap);
#endif
}

// ----------------------------------------------------------------------------
// user interface
// ----------------------------------------------------------------------------

void EMU::insert_disk(_TCHAR* filename, int drv)
{
	if(vm->disk_inserted(drv)) {
		vm->eject_disk(drv);
		eject_wait[drv] = 55;
	}
	if(eject_wait[drv]) {
		_tcscpy(insert_path[drv], filename);
		insert_request[drv] = true;
	}
	else
		vm->insert_disk(filename, drv);
}

void EMU::eject_disk(int drv)
{
	if(vm->disk_inserted(drv)) {
		vm->eject_disk(drv);
		eject_wait[drv] = 55;
	}
}

void EMU::update_config()
{
	// vm class
	vm->update_config();
	
	// emu class
	if(sound_enable != config.sound_enable) {
		sound_enable = config.sound_enable;
		if(sound_enable)
			play_sound();
		else
			stop_sound();
	}
}

