/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.09.05 -

	[ mouse ]
*/

#include "mouse.h"
#include "../fileio.h"

#define CANCEL_EVENT(event) { \
	if(regist_id[event] != -1) { \
		vm->cancel_callback(regist_id[event]); \
		regist_id[event] = -1; \
	} \
}
#define REGIST_EVENT(event, wait) { \
	if(regist_id[event] != -1) { \
		vm->cancel_callback(regist_id[event]); \
		regist_id[event] = -1; \
	} \
	vm->regist_callback(this, event, wait, false, &regist_id[event]); \
}

void MOUSE::initialize()
{
	regist_id[0] = regist_id[1] = regist_id[2] = -1;
	mouse_select = over_flow = nextrecv_irq = false;
	rega_num = regb_num = fifo_r = fifo_w = 0;
	_memset(rega, 0, sizeof(rega));
	_memset(regb, 0, sizeof(regb));
	_memset(fifo, 0, sizeof(fifo));
	
	port_select = 0;
	mouse_stat = emu->mouse_buffer();
}

void MOUSE::reset()
{
	regist_id[0] = regist_id[1] = regist_id[2] = -1;
	mouse_select = over_flow = nextrecv_irq = false;
	rega_num = regb_num = fifo_r = fifo_w = 0;
	_memset(rega, 0, sizeof(rega));
	_memset(regb, 0, sizeof(regb));
	_memset(fifo, 0, sizeof(fifo));
}

void MOUSE::write_io8(uint16 addr, uint8 data)
{
	// port select
	if((addr & 0xf0) == 0xa0) {
		// 0xa1, 0xa3
		if(port_select & 0x80)
			return;
	}
	else if((addr & 0xf0) == 0xb0) {
		// 0xb1, 0xb3
		if(!(port_select & 0x80))
			return;
	}
	
	switch(addr & 0xff)
	{
		case 0xa1:
		case 0xb1:
			// cmd a
			if(rega_num == 0) {
				if((data & 0x38) == 0x18) {
					rega_num = 0;
					_memset(rega, 0, sizeof(rega));
				}
			}
			rega[rega_num] = data;
			rega_num = (rega_num == 0) ? data & 0x7 : 0;
			break;
			
		case 0xa3:
		case 0xb3:
			// cmd b
			if(regb_num == 0) {
				if((data & 0x38) == 0x18) {
					CANCEL_EVENT(EVENT_RECV1);
					CANCEL_EVENT(EVENT_RECV2);
					CANCEL_EVENT(EVENT_RECV3);
					regb_num = fifo_r = fifo_w = 0;
					over_flow = nextrecv_irq = false;
					_memset(regb, 0, sizeof(regb));
					_memset(fifo, 0, sizeof(fifo));
				}
				else if((data & 0x38) == 0x20)
					nextrecv_irq = true;
				else if((data & 0x38) == 0x30)
					over_flow = false;
			}
			else if(regb_num == 1) {
				if(!(data & 0x18))
					vm->cancel_interrupt(IRQ_SERIAL);
			}
			else if(regb_num == 5) {
				if(regb[3] == 0xc1 && regb[4] == 0x44 && regb[5] == 0x68 && data == 0xe8 && mouse_select) {
					// mouse status
					recv_data[0] = (mouse_stat[0] >= 128 ? 0x10 : mouse_stat[0] < -128 ? 0x20 : 0) |
					               (mouse_stat[1] >= 128 ? 0x40 : mouse_stat[1] < -128 ? 0x80 : 0) |
					               ((mouse_stat[2] & 0x1) ? 0x1 : 0) | ((mouse_stat[2] & 0x2) ? 2 : 0);
					recv_data[1] = (uint8)mouse_stat[0];
					recv_data[2] = (uint8)mouse_stat[1];
					
					// recieve events
					fifo_r = fifo_w = 0;
					REGIST_EVENT(EVENT_RECV1, 2083);	// 4800bps, 10bit
					REGIST_EVENT(EVENT_RECV2, 4167);
					REGIST_EVENT(EVENT_RECV3, 6250);
				}
			}
			regb[regb_num] = data;
			regb_num = (regb_num == 0) ? data & 0x7 : 0;
			break;
			
		case 0xcd:
			port_select = data;
			break;
	}
}

uint8 MOUSE::read_io8(uint16 addr)
{
	// port select
	if((addr & 0xf0) == 0xa0) {
		// 0xa0, 0xa1, 0xa2, 0xa3
		if(port_select & 0x80)
			return 0xff;
	}
	else if((addr & 0xf0) == 0xb0) {
		// 0xb0, 0xb1, 0xb2, 0xb3
		if(!(port_select & 0x80))
			return 0xff;
	}
	
	switch(addr & 0xff)
	{
		case 0xa0:
		case 0xb0:
			// data a
			return 0xff;
			
		case 0xa1:
		case 0xb1:
		{
			// state a
			uint8 val = 0xff;
			
			if(rega_num == 0) {
				val = 0x44;
				val |= (vm->interrupt_status(IRQ_SERIAL) == 1) ? 0x2 : 0;
			}
			else if(rega_num == 1)
				val = 0x8f;
			rega_num = 0;
			return val;
		}
		case 0xa2:
		case 0xb2:
			// data b
			return pop();
			
		case 0xa3:
		case 0xb3:
		{
			// state b
			uint8 val = 0xff;
			
			if(regb_num == 0) {
				val = 0x44;
				val |= (fifo_r != fifo_w) ? 0x1 : 0;
			}
			else if(regb_num == 1) {
				val = 0x8f;
				val |= over_flow ? 0x20 : 0;
			}
			else if(regb_num == 2)
				val = regb[2];
			regb_num = 0;
			return val;
		}
	}
	return 0xff;
}

void MOUSE::write_signal(int ch, uint32 data)
{
	if(ch == SIGNAL_SOUND)
		mouse_select = data ? true : false;
}

void MOUSE::event_callback(int event_id, int err)
{
	if(event_id == EVENT_RECV1) {
		if(push(recv_data[0])) {
			// success
			if(regb[1] & 0x18) {
				uint8 vector = (regb[1] & 0x4) ? (regb[2] & 0xf0) | 0x4 : regb[2];
				vm->request_interrupt(IRQ_SERIAL, vector, true);
				nextrecv_irq = false;
			}
		}
		else {
			// failed
			uint8 vector = (regb[1] & 0x4) ? (regb[2] & 0xf0) | 0x6 : regb[2];
			vm->request_interrupt(IRQ_SERIAL, vector, true);
			nextrecv_irq = false;
		}
		regist_id[0] = -1;
	}
	else if(event_id == EVENT_RECV2) {
		if(push(recv_data[1])) {
			// success
			if((regb[1] & 0x10) || nextrecv_irq) {
				uint8 vector = (regb[1] & 0x4) ? (regb[2] & 0xf0) | 0x4 : regb[2];
				vm->request_interrupt(IRQ_SERIAL, vector, true);
				nextrecv_irq = false;
			}
		}
		else {
			// failed
			uint8 vector = (regb[1] & 0x4) ? (regb[2] & 0xf0) | 0x6 : regb[2];
			vm->request_interrupt(IRQ_SERIAL, vector, true);
			nextrecv_irq = false;
		}
		regist_id[1] = -1;
	}
	else if(event_id == EVENT_RECV3) {
		if(push(recv_data[2])) {
			// success
			if((regb[1] & 0x10) || nextrecv_irq) {
				uint8 vector = (regb[1] & 0x4) ? (regb[2] & 0xf0) | 0x4 : regb[2];
				vm->request_interrupt(IRQ_SERIAL, vector, true);
				nextrecv_irq = false;
			}
		}
		else {
			// failed
			uint8 vector = (regb[1] & 0x4) ? (regb[2] & 0xf0) | 0x6 : regb[2];
			vm->request_interrupt(IRQ_SERIAL, vector, true);
			nextrecv_irq = false;
		}
		regist_id[2] = -1;
	}
}

bool MOUSE::push(uint8 data)
{
	int cnt = (fifo_w >= fifo_r) ? fifo_w - fifo_r : fifo_w - fifo_r + 8;
	if(cnt < 4) {
		fifo[fifo_w] = data;
		fifo_w = (fifo_w + 1) & 0x7;
		return true;
	}
	else {
		over_flow = true;
		return false;
	}
}

uint8 MOUSE::pop()
{
	if(fifo_r != fifo_w) {
		uint8 val = fifo[fifo_r];
		fifo_r = (fifo_r + 1) & 0x7;
		return val;
	}
	return 0;
}

