/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.31 -

	[ timer (i8253) ]
*/

#include "timer.h"

// ----------------------------------------------------------------------------
// JE^P
// ----------------------------------------------------------------------------

COUNTER::COUNTER()
{
	count = 0x10000;
	count_reg = 0;
	
	mode = rl10 = 3;
	start = bcd = latch = mode0 = false;
	delay = readcnt = 0;
}

void COUNTER::reg_to_count()
{
	if(bcd) {
		int a = (count_reg >>  0) & 0xf;
		int b = (count_reg >>  4) & 0xf;
		int c = (count_reg >>  8) & 0xf;
		int d = (count_reg >> 12) & 0xf;
		count = a + b * 10 + c * 100 + d * 1000;
	}
	else
		count = (count_reg == 0) ? 0x10000 : (count_reg == 1 && mode == 3) ? 0x10001 : count_reg;
}

void COUNTER::count_to_latch()
{
	if(bcd) {
		int a = count % 10;
		int b = ((int)(count / 10)) % 10;
		int c = ((int)(count / 100)) % 10;
		int d = ((int)(count / 1000)) % 10;
		count_latch = a | (b << 4) | (c << 8) | (d << 12);
	}
	else
		count_latch = count & 0xffff;
}

void COUNTER::write_ctrl(uint8 data)
{
	if((data >> 4) & 3 == 0) {
		// JE^b`Iy[V
		count_to_latch();
		latch = true;
		
		readcnt = (rl10 == 1 || rl10 == 2) ? 1 : 2;
		return;
	}
	int new_mode = (data >> 1) & ((data & 4) ? 3 : 7);
//	if(mode != new_mode)
//			start = false;
	mode = new_mode;
	rl10 = (data >> 4) & 3;
	bcd = (data & 1) ? true : false;
	mode0 = false;
}

void COUNTER::write_count(uint8 data)
{
	if(rl10 == 1)
		count_reg = data;
	else if(rl10 == 2)
		count_reg = data << 8;
	else
		count_reg = ((count_reg >> 8) & 0x00ff) | (data << 8);
	
	if(mode == 0 || mode == 3 || mode == 4) {
		start = true;
		delay = 1;
	}
	else if(mode == 2) {
		if(!start)
			delay = 1;
		start = true;
	}
	mode0 = false;
}

uint8 COUNTER::read_count()
{
	if(!readcnt) {
		if(!latch)
			count_to_latch();
		latch = false;
		
		if(rl10 == 2)
			count_latch >>= 8;
		readcnt = (rl10 == 1 || rl10 == 2) ? 1 : 2;
	}
	uint8 val = count_latch & 0xff;
	count_latch >>= 8;
	readcnt--;
	return val;
}

void COUNTER::write_gate()
{
	// H->L->H
	if(mode == 2 || mode == 3)
		reg_to_count();
	else if(mode == 1 || mode == 5)
		delay = 1;
	if(!(mode == 0 || mode == 4))
		start = true;
}

bool COUNTER::write_clock()
{
	if(start) {
		if(delay && --delay == 0) {
			// PNbNœǂݍ݁A̎JEgJn
			reg_to_count();
			return false;
		}
		if(count == 0)
			count = 0xffff;
		else if(--count == 0) {
			// H->L->Ho
			if(mode == 0) {
				if(mode0)
					return false;
				mode0 = true;
			}
			else if(mode == 2 || mode == 3)
				reg_to_count();
			else
				start = false;
			return true;
		}
	}
	return false;
}

// ----------------------------------------------------------------------------
// i8253{
// ----------------------------------------------------------------------------

void TIMER::initialize()
{
	for(int i = 0; i < 3; i++)
		counter[i] = new COUNTER();
	
	// 32usec -> 31.25KHz
	int regist_id;
	vm->regist_callback(this, EVENT_TIMER, 32, true, &regist_id);
}

void TIMER::release()
{
	for(int i = 0; i < 3; i++) {
		if(counter[i])
			delete counter[i];
	}
}

void TIMER::write_io8(uint16 addr, uint8 data)
{
	switch(addr & 0xff)
	{
		case 0xe4: // count reg #0
		case 0xe5: // count reg #1
		case 0xe6: // count reg #2
			counter[addr & 3]->write_count(data);
			break;
		case 0xe7:
			// ctrl reg
			if((data & 0xc0) != 0xc0)
				counter[(data >> 6) & 3]->write_ctrl(data);
			break;
		case 0xf0: case 0xf1: case 0xf2: case 0xf3:
			// gate signal
			counter[0]->write_gate();
			counter[1]->write_gate();
//			counter[2]->write_gate();
			break;
	}
}

uint8 TIMER::read_io8(uint16 addr)
{
	switch(addr & 0xff)
	{
		case 0xe4: // count reg #0
		case 0xe5: // count reg #1
		case 0xe6: // count reg #2
			return counter[addr & 3]->read_count();
	}
	return 0xff;
}

void TIMER::event_callback(int event_id, int err)
{
	for(int i = 0; i < 3; i++) {
		// JオmF
		if(!counter[i]->write_clock())
			return;
		// IRQ
		if(i == 0)
			vm->request_interrupt(IRQ_TIMER, vector, false);
	}
}

