/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.10 -

	[ virtual machine ]
*/

#include "vm.h"
#include "../emu.h"
#include "device.h"

#include "z80.h"
#include "memory.h"
#include "io.h"
#include "interrupt.h"

#include "calendar.h"
#include "cassette.h"
#include "crtc.h"
#include "emm.h"
#include "extrom.h"
#include "fdc.h"
#include "joystick.h"
#include "kanji.h"
#include "keyboard.h"
#include "mouse.h"
#include "romfile.h"
#include "sasi.h"
#include "sound.h"
#include "timer.h"
#include "w3100a.h"

#include "../config.h"

extern config_t config;

// ----------------------------------------------------------------------------
// initialize
// ----------------------------------------------------------------------------

VM::VM(EMU* parent_emu) : emu(parent_emu)
{
	// set cpu clocks
	int powers[4] = {1, 2, 4, 8};
	if(!(0 <= config.cpu_clock && config.cpu_clock <= 3))
		config.cpu_clock = 0;
	set_clocks(powers[config.cpu_clock]);
	
	// clear all event
	for(int i = 0; i < CALLBACK_MAX; i++) {
		event[i].enable = false;
		event[i].device = NULL;
	}
	event_clock = -1;
	past_clock = 0;
	
	// create devices
	first_device = last_device = NULL;
	
	cpu = new Z80(this, emu);
	memory = new MEMORY(this, emu);
	io = new IO(this, emu);
	interrupt = new INTERRUPT(this, emu);
	
	calendar = new CALENDAR(this, emu);
	cassette = new CASSETTE(this, emu);
	crtc = new CRTC(this, emu);
	emm = new EMM(this, emu);
	extrom = new EXTROM(this, emu);
	fdc = new FDC(this, emu);
	joystick = new JOYSTICK(this, emu);
	kanji = new KANJI(this, emu);
	keyboard = new KEYBOARD(this, emu);
	mouse = new MOUSE(this, emu);
	romfile = new ROMFILE(this, emu);
	sasi = new SASI(this, emu);
	sound = new SOUND(this, emu);
	timer = new TIMER(this, emu);
	network = new W3100A(this, emu);
	
	dummy = new DEVICE(this, emu);
	
	// initialize all devices
	DEVICE* device = first_device;
	while(device) {
		device->initialize();
		device = device->next_device;
	}
	
	// set context
	cpu->memory = memory;
	cpu->io = io;
	cpu->interrupt = interrupt;
	
	device = first_device;
	while(device) {
		io->regist_iomap(device);
		device = device->next_device;
	}
}

VM::~VM()
{
	for(int i = 0; i < CALLBACK_MAX; i++)
		event[i].device = NULL;
	
	// delete all devices
	for(DEVICE* device = first_device; device;) {
		DEVICE* next = device->next_device;
		device->release();
		delete device;
		device = next;
	}
}

void VM::set_clocks(int power)
{
	int sum_cpu = (int)(power * CPU_CLOCKS / FRAMES_PER_SEC / LINES_PER_FRAME + 0.5);
	int remain_cpu = sum_cpu;
	int sum_vm = (int)(CPU_CLOCKS / FRAMES_PER_SEC / LINES_PER_FRAME + 0.5);
	int remain_vm = sum_vm;
	
	for(int i = 0; i < 27; i++) {
		clocks_cpu[i] = (int)(sum_cpu / 27);
		remain_cpu -= clocks_cpu[i];
		clocks_vm[i] = (int)(sum_vm / 27);
		remain_vm -= clocks_vm[i];
	}
	for(int i = 0; i < remain_cpu; i++)
		clocks_cpu[(int)(27 * i / remain_cpu)]++;
	for(int i = 0; i < remain_vm; i++)
		clocks_vm[(int)(27 * i / remain_vm)]++;
}

// ----------------------------------------------------------------------------
// drive virtual machine
// ----------------------------------------------------------------------------

void VM::reset()
{
	// clear events (except loop event)
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(!(event[i].enable && event[i].loop)) {
			event[i].enable = false;
			event[i].device = NULL;
		}
	}
	// get next event clock
	event_clock = -1;
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(event[i].enable && (event[i].clock < event_clock || event_clock < 0))
			event_clock = event[i].clock;
	}
	if(event_clock < 0)
		past_clock = 0;
	
	// reset all devices
	DEVICE* device = first_device;
	while(device) {
		device->reset();
		device = device->next_device;
	}
}

void VM::ipl_reset()
{
	// clear events (except loop event)
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(!(event[i].enable && event[i].loop)) {
			event[i].enable = false;
			event[i].device = NULL;
		}
	}
	// get next event clock
	event_clock = -1;
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(event[i].enable && (event[i].clock < event_clock || event_clock < 0))
			event_clock = event[i].clock;
	}
	if(event_clock < 0)
		past_clock = 0;
	
	// ipl reset all devices
	DEVICE* device = first_device;
	while(device) {
		device->reset();
		device = device->next_device;
	}
	memory->ipl_reset();
}

void VM::run()
{
	// run virtual machine for 1 frame period
	for(int v = 0; v < LINES_PER_FRAME; v++) {
		crtc->set_vsync(v);
		for(int h = 0, hh = 0; h < 108; h += 4, hh++) {
			crtc->set_hsync(h);
//			drive_vm(CLOCKS_PER_32DOTS);
			drive_vm(clocks_cpu[hh], clocks_vm[hh]);
		}
		sound->update_sound();
	}
	calendar->update_calendar();
	joystick->update_input();
	keyboard->update_input();
}

void VM::drive_vm(int clock_cpu, int clock_vm)
{
	// run cpu
	cpu->run(clock_cpu);
	
	// check next event clock
	if(event_clock < 0)
		return;
	
	event_clock -= clock_vm;
	past_clock += clock_vm;
	
	if(event_clock > 0)
		return;
	
	// run events
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(!event[i].enable)
			continue;
		// event is active
		event[i].clock -= past_clock;
		while(event[i].clock <= 0) {
			//int usec = (int)(event[i].clock * 1000000. / CPU_CLOCKS + 0.5);
			int usec = (int)(event[i].clock / 6 + 0.5);
			event[i].device->event_callback(event[i].event_id, usec);
			if(event[i].loop) {
				// loop event
				event[i].clock += event[i].loop;
			}
			else {
				// not loop event
				event[i].device = NULL;
				event[i].enable = false;
				break;
			}
		}
	}
	past_clock = 0;
	
	// get next event clock
	event_clock = -1;
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(event[i].enable && (event[i].clock < event_clock || event_clock < 0))
			event_clock = event[i].clock;
	}
}

void VM::regist_callback(DEVICE* dev, int event_id, int usec, bool loop, int* regist_id)
{
	// regist_id = -1 when failed to regist event
	*regist_id = -1;
	
	// check if events exist or not
	bool no_exist = true;
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(event[i].enable) {
			no_exist = false;
			break;
		}
	}
	if(no_exist)
		past_clock = 0;
	
	// regist event
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(!event[i].enable) {
			//int clock = (int)(CPU_CLOCKS / 1000000. * usec + 0.5);
			int clock = 6 * usec;
			event[i].enable = true;
			event[i].device = dev;
			event[i].event_id = event_id;
			event[i].clock = clock + past_clock;
			event[i].loop = loop ? clock : 0;
			
			*regist_id = i;
			break;
		}
	}
	
	// get next event clock
	event_clock = -1;
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(event[i].enable && (event[i].clock < event_clock || event_clock < 0))
			event_clock = event[i].clock;
	}
	if(event_clock < 0)
		past_clock = 0;
}

void VM::cancel_callback(int regist_id)
{
	// cancel registered event
	if(0 <= regist_id && regist_id < CALLBACK_MAX) {
		event[regist_id].device = NULL;
		event[regist_id].enable = false;
	}
	
	// get next event clock
	event_clock = -1;
	for(int i = 0; i < CALLBACK_MAX; i++) {
		if(event[i].enable && (event[i].clock < event_clock || event_clock < 0))
			event_clock = event[i].clock;
	}
	if(event_clock < 0)
		past_clock = 0;
}

void VM::request_interrupt(int dev, uint8 vector, bool pending)
{
	interrupt->request_interrupt(dev, vector, pending);
}

void VM::cancel_interrupt(int dev)
{
	interrupt->cancel_interrupt(dev);
}

int VM::interrupt_status(int dev)
{
	return interrupt->interrupt_status(dev);
}

// ----------------------------------------------------------------------------
// draw screen
// ----------------------------------------------------------------------------

void VM::draw_screen()
{
	crtc->draw_screen();
}

// ----------------------------------------------------------------------------
// soud generation
// ----------------------------------------------------------------------------

void VM::initialize_sound(int rate, int samples)
{
	sound->initialize_sound(rate, samples);
}

uint16* VM::create_sound(int samples, bool fill)
{
	return sound->create_sound(samples, fill);
}

// ----------------------------------------------------------------------------
// network
// ----------------------------------------------------------------------------

void VM::network_connected(int ch)
{
	network->connected(ch);
}

void VM::network_disconnected(int ch)
{
	network->disconnected(ch);
}

uint8* VM::get_sendbuffer(int ch, int* size)
{
	return network->get_sendbuffer(ch, size);
}

void VM::inc_sendbuffer_ptr(int ch, int size)
{
	network->inc_sendbuffer_ptr(ch, size);
}

uint8* VM::get_recvbuffer0(int ch, int* size0, int* size1)
{
	return network->get_recvbuffer0(ch, size0, size1);
}

uint8* VM::get_recvbuffer1(int ch)
{
	return network->get_recvbuffer1(ch);
}

void VM::inc_recvbuffer_ptr(int ch, int size)
{
	network->inc_recvbuffer_ptr(ch, size);
}

// ----------------------------------------------------------------------------
// user interface
// ----------------------------------------------------------------------------

void VM::update_config()
{
	DEVICE* device = first_device;
	while(device) {
		device->update_config();
		device = device->next_device;
	}
	
	// update cpu clocks
	int powers[4] = {1, 2, 4, 8};
	if(!(0 <= config.cpu_clock && config.cpu_clock <= 3))
		config.cpu_clock = 0;
	set_clocks(powers[config.cpu_clock]);
}

void VM::insert_disk(_TCHAR* filename, int drv)
{
	fdc->insert_disk(filename, drv);
}

void VM::eject_disk(int drv)
{
	fdc->eject_disk(drv);
}

bool VM::disk_inserted(int drv)
{
	return fdc->disk_inserted(drv);
}

