/*
	Skelton for Z-80 PC Emulator

	Author : Takeda.Toshiya
	Date   : 2004.04.29 -

	[ winmain (win32) ]
*/

/*
	pre-processor

	_WIN32_WCE	windows ce
	_WIN32_HPC	windows ce (hpc2000)
	_WIN32_PPC	windows ce (ppc2002)
	_WIN32_SIG	windows ce (sigmarion 3)
	
	_USE_GAPI_	use game api
*/


#ifdef _WIN32_PPC
//#define _USE_GAPI_
#elif defined _WIN32_HPC
//#define _USE_GAPI_
#elif defined _WIN32_SIG
//#define _USE_GAPI_
#endif

#include <windows.h>
#include <windowsx.h>
#include <winsock.h>
#include <commdlg.h>
#include <commctrl.h>
#ifdef _WIN32_PPC
#include <aygshell.h>
#endif
#include <mmsystem.h>
#include <stdio.h>
#ifdef _USE_GAPI_
#include "gx.h"
#endif
#include "res/resource.h"
#include "config.h"
#include "emu.h"

#ifdef _WIN32_WCE
#define timeGetTime() GetTickCount()
#endif

#define WINDOW_WIDTH 640
#define WINDOW_HEIGHT 400

// config
extern config_t config;

// emulation core
EMU* emu;
BOOL emulation_running, emulation_freeze;

// command bar
#ifdef _WIN32_WCE
HINSTANCE hInst;
HWND hCmdBar;
BOOL commandbar_show, sip_on;
#endif

// uif
void update_menu(HMENU hMenu, int index);
void open_disk(HWND hWnd, int drv);
void open_tape(HWND hWnd);
void set_window(HWND hwnd, BOOL full);
BOOL fullscreen_now = FALSE;
BOOL screen_active = TRUE;

// windows main
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPTSTR szCmdLine, int iCmdShow);
LRESULT CALLBACK WndProc(HWND hWnd, UINT iMsg, WPARAM wParam, LPARAM lParam);

// timing control
int intervals[FRAMES_PER_10SECS];

void init_interval()
{
	int remain = 1000;
	
	for(int i = 0; i < FRAMES_PER_10SECS; i++) {
		intervals[i] = (int)(1000 / FRAMES_PER_10SECS);
		remain -= intervals[i];
	}
	for(int i = 0; i < remain; i++)
		intervals[(int)(FRAMES_PER_10SECS * i / remain)]++;
}

int get_interval()
{
	static int cnt = 0;
	int interval = 0;
	
	for(int i = 0; i < 10; i++) {
		interval += intervals[cnt];
		cnt = (cnt + 1 < FRAMES_PER_10SECS) ? cnt + 1 : 0;
	}
	return interval;
}

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPTSTR szCmdLine, int iCmdShow)
{
	// create window
	WNDCLASS wndclass;
	wndclass.style = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc = (WNDPROC)WndProc;
	wndclass.cbClsExtra = 0;
	wndclass.cbWndExtra = 0;
	wndclass.hInstance = hInstance;
	wndclass.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON1));
	wndclass.hCursor = 0;
	wndclass.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
#ifdef _WIN32_WCE
	wndclass.lpszMenuName = 0;
#else
	wndclass.lpszMenuName = MAKEINTRESOURCE(IDR_MENU1);
#endif
	wndclass.lpszClassName = _T("CWINDOW");
	RegisterClass(&wndclass);
	
#ifdef _WIN32_WCE
	// for command bar
	hInst = hInstance;
#endif
	
	// show window
#ifdef _WIN32_PPC
	// ppc2002
	HWND hWnd = CreateWindow(_T("CWINDOW"), _T("EmuZ-2500"), WS_VISIBLE,
	                         0, 0, 240, 320,
	                         NULL, NULL, hInstance, NULL);
	SHFullScreen(hWnd, SHFS_HIDESTARTICON | SHFS_HIDETASKBAR | SHFS_SHOWSIPBUTTON);
#elif defined(_WIN32_HPC) || defined(_WIN32_SIG)
	// hpc2000, sigmarion 3
	HWND hWnd = CreateWindow(_T("CWINDOW"), _T("EmuZ-2500"), WS_VISIBLE,
	                         CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
	                         NULL, NULL, hInstance, NULL);
#else
	// windows pc
	RECT rect = {0, 0, WINDOW_WIDTH, WINDOW_HEIGHT};
	AdjustWindowRectEx(&rect, WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MAXIMIZEBOX | WS_VISIBLE, TRUE, 0);
	
	HDC hdc = CreateDC(_T("DISPLAY"), NULL, NULL, NULL);
	int width = GetDeviceCaps(hdc, HORZRES);
	int height = GetDeviceCaps(hdc, VERTRES);
	int dest_x = (int)((width - (rect.right - rect.left)) / 2);
	int dest_y = (int)((height - (rect.bottom - rect.top)) / 2);
	dest_x = (dest_x < 0) ? 0 : dest_x;
	dest_y = (dest_y < 0) ? 0 : dest_y;
	
	HWND hWnd = CreateWindow(_T("CWINDOW"), _T("EmuZ-2500"),
	                         WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_VISIBLE,
	                         dest_x, dest_y, rect.right - rect.left, rect.bottom - rect.top,
	                         NULL, NULL, hInstance, NULL);
#endif
	
	ShowWindow(hWnd, iCmdShow);
	UpdateWindow(hWnd);
	
#ifndef _WIN32_WCE
	// accelerator
	HACCEL hAccel = LoadAccelerators(hInstance, MAKEINTRESOURCE(IDR_ACCELERATOR1));
#endif
	
#ifdef _WIN32_WCE
	// show menu bar
	if(hCmdBar)
		CommandBar_Show(hCmdBar, TRUE);
	commandbar_show = TRUE;
#endif
	
#ifdef _WIN32_PPC
	// get sip status
	SIPINFO si;
	memset(&si,0, sizeof(si));
	si.cbSize = sizeof(si);
	if(SHSipInfo(SPI_GETSIPINFO, 0, &si, 0))
		sip_on = (si.fdwFlags & SIPF_ON) ? TRUE : FALSE;
#else
	// disenable ime
	ImmAssociateContext(hWnd, 0);
#endif
	
#ifdef _USE_GAPI_
	// initialize gapi
	GXOpenDisplay(hWnd, GX_FULLSCREEN);
#endif
	
	// initialize emulation core
	load_config();
	emu = new EMU(hWnd);
	emu->set_screen_size(WINDOW_WIDTH, WINDOW_HEIGHT);
	
	// main loop
	init_interval();
	int current_interval = get_interval(), next_interval;
	int skip_frames = 0, min_skip_frames = config.skip_frames, fps = 0;
	
	DWORD next_time = timeGetTime();
	DWORD fps_time = next_time + 1000;
	
	emulation_running = TRUE;
	emulation_freeze = FALSE;
	MSG msg;
	
	while(emulation_running) {
		// check window message
		if(PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE)) {
			if(!GetMessage(&msg,NULL,0,0)) {
				emulation_running = FALSE;
			}
#ifdef _WIN32_WCE
			TranslateMessage(&msg);
			DispatchMessage(&msg);
#else
			if(!TranslateAccelerator(hWnd, hAccel, &msg)) {
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
#endif
		}
		else if(!emulation_freeze) {
			// get next period
			next_time += current_interval;
			next_interval = get_interval();
			
			// drive machine
			emu->run();
			
			if(next_time > timeGetTime()) {
				// update window if enough time
				if(skip_frames >= min_skip_frames) {
#ifdef _USE_GAPI_
#ifdef _WIN32_PPC
					if(screen_active && !commandbar_show && !sip_on) {
#else
					if(screen_active) {
#endif
						LPWORD lpDraw = (LPWORD)GXBeginDraw();
						if(lpDraw) {
							emu->draw_gapi(lpDraw);
							GXEndDraw();
						}
					}
#else
					emu->draw_screen();
#endif
					skip_frames = 0;
					min_skip_frames = config.skip_frames;
					fps++;
				}
				else
					skip_frames++;
				
				// sleep 1 frame priod if need
				if((int)(next_time - timeGetTime()) >= next_interval)
					Sleep(next_interval);
			}
			else if(++skip_frames > 10) {
				// update window at least once per 10 frames
				emu->draw_screen();
				fps++;
				
				skip_frames = 0;
				min_skip_frames = config.skip_frames;
				next_time = timeGetTime();
			}
			current_interval = next_interval;
			Sleep(0);
		}
		else {
			// now freeze
			Sleep(10);
		}
		
		// calc frame rate
		if(fps_time < timeGetTime()) {
#ifndef _WIN32_PPC
			_TCHAR buf[32];
			int ratio = (int)(100 * fps / FRAMES_PER_SEC + 0.5);
			_stprintf(buf, _T("EmuZ-2500 - %d fps (%d %%)"), fps, ratio);
			SetWindowText(hWnd, buf);
#endif
			fps_time += 1000;
			fps = 0;
		}
	}
	
	// exit fullscreen
#ifdef _WIN32_PPC
	SHFullScreen(hWnd, SHFS_SHOWSTARTICON | SHFS_SHOWTASKBAR | SHFS_SHOWSIPBUTTON);
#endif
#ifndef _WIN32_WCE
	if(fullscreen_now)
		ChangeDisplaySettings(NULL, 0);
#endif
	
	// release emulation core
	delete emu;
	save_config();
	
#ifdef _USE_GAPI_
	// release gapi
	GXCloseDisplay();
#endif
	return msg.wParam;
}

#ifdef _WIN32_PPC
// character code convert matrix for pocket pc sip
static const WORD keycode[0x80] = {
	0x000, 0x001, 0x002, 0x003, 0x004, 0x005, 0x006, 0x007,
	0x008, 0x009, 0x00a, 0x00b, 0x00c, 0x00d, 0x00e, 0x00f,
	0x010, 0x011, 0x012, 0x013, 0x014, 0x015, 0x016, 0x017,
	0x018, 0x019, 0x01a, 0x01b, 0x01c, 0x01d, 0x01e, 0x01f,
	0x020, 0x131, 0x132, 0x133, 0x134, 0x135, 0x136, 0x137,
	0x138, 0x139, 0x1ba, 0x1bb, 0x0bc, 0x0bd, 0x0be, 0x0bf,
	0x030, 0x031, 0x032, 0x033, 0x034, 0x035, 0x036, 0x037,
	0x038, 0x039, 0x0ba, 0x0bb, 0x1bc, 0x1bd, 0x1be, 0x1bf,
	0x0c0, 0x141, 0x142, 0x143, 0x144, 0x145, 0x146, 0x147,
	0x148, 0x149, 0x14a, 0x14b, 0x14c, 0x14d, 0x14e, 0x14f,
	0x150, 0x151, 0x152, 0x153, 0x154, 0x155, 0x156, 0x157,
	0x158, 0x159, 0x15a, 0x0db, 0x0dc, 0x0dd, 0x0de, 0x05f,
	0x140, 0x041, 0x042, 0x043, 0x044, 0x045, 0x046, 0x047,
	0x048, 0x049, 0x04a, 0x04b, 0x04c, 0x04d, 0x04e, 0x04f,
	0x050, 0x051, 0x052, 0x053, 0x054, 0x055, 0x056, 0x057,
	0x058, 0x059, 0x05a, 0x1db, 0x1dc, 0x1dd, 0x1de, 0x07f
};
#endif

LRESULT CALLBACK WndProc(HWND hWnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
#ifdef _WIN32_PPC
	SIPINFO si;
#endif
	HDC hdc;
	PAINTSTRUCT ps;
	int ch;
	
	switch(iMsg)
	{
#ifdef _WIN32_WCE
		case WM_CREATE:
			hCmdBar = CommandBar_Create(hInst, hWnd, 1);
#ifdef _WIN32_HPC
			CommandBar_InsertMenubar(hCmdBar, hInst, IDR_MENU2, 0);
#elif defined _WIN32_PPC
			CommandBar_InsertMenubar(hCmdBar, hInst, IDR_MENU3, 0);
#endif
			CommandBar_AddAdornments(hCmdBar, 0, 0);
			break;
#endif

		case WM_DESTROY:
#ifdef _WIN32_WCE
			CommandBar_Destroy(hCmdBar);
#endif
			PostQuitMessage(0);
			emulation_running = FALSE;
			break;

#ifdef _USE_GAPI_
		case WM_SETFOCUS:
			GXResume();
			screen_active = TRUE;
			break;
			
		case WM_KILLFOCUS:
			GXSuspend();
			screen_active = FALSE;
			break;
#endif

		case WM_PAINT:
			if(emu) {
				hdc = BeginPaint(hWnd, &ps);
				emu->update_screen(hdc);
				EndPaint(hWnd, &ps);
			}
			return 0;
			
		case MM_WOM_DONE:
			if(emu)
				emu->notify_sound();
			break;

#ifdef _WIN32_PPC
		case WM_LBUTTONDOWN:
			if(hCmdBar)
				CommandBar_Show(hCmdBar, (commandbar_show = !commandbar_show));
			break;
#endif

#ifdef _WIN32_PPC
		case WM_SETTINGCHANGE:
			memset(&si,0, sizeof(si));
			si.cbSize = sizeof(si);
			if(SHSipInfo(SPI_GETSIPINFO, 0, &si, 0))
				sip_on = (si.fdwFlags & SIPF_ON) ? TRUE : FALSE;
			break;

		case WM_KEYDOWN:
			if(emu && !sip_on)
				emu->key_down_ppc(LOBYTE(wParam));
			break;

		case WM_KEYUP:
			if(emu && !sip_on)
				emu->key_up_ppc(LOBYTE(wParam));
			break;

		case WM_CHAR:
			if(emu && sip_on) {
				if(LOBYTE(wParam) < 0x80) {
					int code = keycode[LOBYTE(wParam)];
					if(code & 0x100)
						emu->key_down_sip(0x10);
					emu->key_down_sip(code & 0xff);
				}
			}
			break;
#else
		case WM_KEYDOWN:
			if(emu)
				emu->key_down(LOBYTE(wParam));
			break;

		case WM_KEYUP:
			if(emu)
				emu->key_up(LOBYTE(wParam));
			break;

		case WM_SYSKEYDOWN:
			if(emu)
				emu->key_down(LOBYTE(wParam));
			//break;
			return 0; // not activate menu when hit ALT/F10

		case WM_SYSKEYUP:
			if(emu)
				emu->key_up(LOBYTE(wParam));
			//break;
			return 0; // not activate menu when hit ALT/F10
#endif

#ifndef _WIN32_WCE
		case WM_MOVING:
			// mute
			if(emu)
				emu->mute_sound();
			break;
#endif

		case WM_INITMENUPOPUP:
			// mute and open menu
			if(emu)
				emu->mute_sound();
			update_menu((HMENU)wParam, LOWORD(lParam));
			break;

		case WM_SOCKET0: ch = 0; goto socket;
		case WM_SOCKET1: ch = 1; goto socket;
		case WM_SOCKET2: ch = 2; goto socket;
		case WM_SOCKET3: ch = 3;
socket:
			if(!emu)
				break;
			if(WSAGETSELECTERROR(lParam) != 0) {
				emu->disconnect_socket(ch);
				emu->socket_disconnected(ch);
				break;
			}
			if(emu->get_socket(ch) != (int)wParam)
				break;
			switch(WSAGETSELECTEVENT(lParam))
			{
				case FD_CONNECT:
					emu->socket_connected(ch);
					break;
				case FD_CLOSE:
					emu->socket_disconnected(ch);
					break;
				case FD_WRITE:
					emu->send_data(ch);
					break;
				case FD_READ:
					emu->recv_data(ch);
					break;
			}
			break;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
				case ID_EXIT:
					SendMessage(hWnd, WM_CLOSE, 0, 0L);
					emulation_running = FALSE;
					break;

				case ID_RESET:
					if(emu)
						emu->reset();
					break;
				case ID_IPLRESET:
					if(emu)
						emu->ipl_reset();
					break;

				case ID_INSERT_FD1: if(emu) open_disk(hWnd, 0); break;
				case ID_INSERT_FD2: if(emu) open_disk(hWnd, 1); break;
				case ID_INSERT_FD3: if(emu) open_disk(hWnd, 2); break;
				case ID_INSERT_FD4: if(emu) open_disk(hWnd, 3); break;

				case ID_EJECT_FD1: if(emu) emu->eject_disk(0); break;
				case ID_EJECT_FD2: if(emu) emu->eject_disk(1); break;
				case ID_EJECT_FD3: if(emu) emu->eject_disk(2); break;
				case ID_EJECT_FD4: if(emu) emu->eject_disk(3); break;

				case ID_RECENT_FD11: if(emu) emu->insert_disk(config.recent_disk[0][0], 0); break;
				case ID_RECENT_FD12: if(emu) emu->insert_disk(config.recent_disk[0][1], 0); break;
				case ID_RECENT_FD13: if(emu) emu->insert_disk(config.recent_disk[0][2], 0); break;
				case ID_RECENT_FD14: if(emu) emu->insert_disk(config.recent_disk[0][3], 0); break;
				case ID_RECENT_FD15: if(emu) emu->insert_disk(config.recent_disk[0][4], 0); break;
				case ID_RECENT_FD16: if(emu) emu->insert_disk(config.recent_disk[0][5], 0); break;
				case ID_RECENT_FD17: if(emu) emu->insert_disk(config.recent_disk[0][6], 0); break;
				case ID_RECENT_FD18: if(emu) emu->insert_disk(config.recent_disk[0][7], 0); break;

				case ID_RECENT_FD21: if(emu) emu->insert_disk(config.recent_disk[1][0], 1); break;
				case ID_RECENT_FD22: if(emu) emu->insert_disk(config.recent_disk[1][1], 1); break;
				case ID_RECENT_FD23: if(emu) emu->insert_disk(config.recent_disk[1][2], 1); break;
				case ID_RECENT_FD24: if(emu) emu->insert_disk(config.recent_disk[1][3], 1); break;
				case ID_RECENT_FD25: if(emu) emu->insert_disk(config.recent_disk[1][4], 1); break;
				case ID_RECENT_FD26: if(emu) emu->insert_disk(config.recent_disk[1][5], 1); break;
				case ID_RECENT_FD27: if(emu) emu->insert_disk(config.recent_disk[1][6], 1); break;
				case ID_RECENT_FD28: if(emu) emu->insert_disk(config.recent_disk[1][7], 1); break;

				case ID_RECENT_FD31: if(emu) emu->insert_disk(config.recent_disk[2][0], 2); break;
				case ID_RECENT_FD32: if(emu) emu->insert_disk(config.recent_disk[2][1], 2); break;
				case ID_RECENT_FD33: if(emu) emu->insert_disk(config.recent_disk[2][2], 2); break;
				case ID_RECENT_FD34: if(emu) emu->insert_disk(config.recent_disk[2][3], 2); break;
				case ID_RECENT_FD35: if(emu) emu->insert_disk(config.recent_disk[2][4], 2); break;
				case ID_RECENT_FD36: if(emu) emu->insert_disk(config.recent_disk[2][5], 2); break;
				case ID_RECENT_FD37: if(emu) emu->insert_disk(config.recent_disk[2][6], 2); break;
				case ID_RECENT_FD38: if(emu) emu->insert_disk(config.recent_disk[2][7], 2); break;

				case ID_RECENT_FD41: if(emu) emu->insert_disk(config.recent_disk[3][0], 3); break;
				case ID_RECENT_FD42: if(emu) emu->insert_disk(config.recent_disk[3][1], 3); break;
				case ID_RECENT_FD43: if(emu) emu->insert_disk(config.recent_disk[3][2], 3); break;
				case ID_RECENT_FD44: if(emu) emu->insert_disk(config.recent_disk[3][3], 3); break;
				case ID_RECENT_FD45: if(emu) emu->insert_disk(config.recent_disk[3][4], 3); break;
				case ID_RECENT_FD46: if(emu) emu->insert_disk(config.recent_disk[3][5], 3); break;
				case ID_RECENT_FD47: if(emu) emu->insert_disk(config.recent_disk[3][6], 3); break;
				case ID_RECENT_FD48: if(emu) emu->insert_disk(config.recent_disk[3][7], 3); break;

				case ID_INSERT_TAPE: if(emu) open_tape(hWnd); break;

				case ID_EJECT_TAPE: if(emu) emu->eject_media(); break;

				case ID_RECENT_TAP1: if(emu) emu->insert_media(config.recent_tape[0]); break;
				case ID_RECENT_TAP2: if(emu) emu->insert_media(config.recent_tape[1]); break;
				case ID_RECENT_TAP3: if(emu) emu->insert_media(config.recent_tape[2]); break;
				case ID_RECENT_TAP4: if(emu) emu->insert_media(config.recent_tape[3]); break;
				case ID_RECENT_TAP5: if(emu) emu->insert_media(config.recent_tape[4]); break;
				case ID_RECENT_TAP6: if(emu) emu->insert_media(config.recent_tape[5]); break;
				case ID_RECENT_TAP7: if(emu) emu->insert_media(config.recent_tape[6]); break;
				case ID_RECENT_TAP8: if(emu) emu->insert_media(config.recent_tape[7]); break;

				case ID_SCREEN_WINDOW:
					if(fullscreen_now)
						set_window(hWnd, FALSE);
					break;
				case ID_SCREEN_FULL:
					if(!fullscreen_now)
						set_window(hWnd, TRUE);
					break;

				case ID_CONFIG_6MHZ:
					config.cpu_clock = 0;
					if(emu)
						emu->update_config();
					break;
				case ID_CONFIG_12MHZ:
					config.cpu_clock = 1;
					if(emu)
						emu->update_config();
					break;
				case ID_CONFIG_24MHZ:
					config.cpu_clock = 2;
					if(emu)
						emu->update_config();
					break;
				case ID_CONFIG_48MHZ:
					config.cpu_clock = 3;
					if(emu)
						emu->update_config();
					break;

				case ID_CONFIG_M1CYCLE:
					config.m1_cycle = !config.m1_cycle;
					if(emu)
						emu->update_config();
					break;

				case ID_CONFIG_TIMER:
					config.fast_timer = !config.fast_timer;
					if(emu)
						emu->update_config();
					break;

				case ID_CONFIG_SKIP0:
					config.skip_frames = 0;
					break;
				case ID_CONFIG_SKIP1:
					config.skip_frames = 1;
					break;
				case ID_CONFIG_SKIP2:
					config.skip_frames = 2;
					break;
				case ID_CONFIG_SKIP3:
					config.skip_frames = 3;
					break;

				case ID_CONFIG_A400L:
					config.monitor_type = 0;
					if(emu)
						emu->update_config();
					break;
				case ID_CONFIG_D400L:
					config.monitor_type = 1;
					if(emu)
						emu->update_config();
					break;
				case ID_CONFIG_A200L:
					config.monitor_type = 2;
					if(emu)
						emu->update_config();
					break;
				case ID_CONFIG_D200L:
					config.monitor_type = 3;
					if(emu)
						emu->update_config();
					break;

				case ID_CONFIG_SCANLINE:
					config.scan_line = !config.scan_line;
					if(emu)
						emu->update_config();
					break;

				case ID_CONFIG_SOUNDENB:
					config.sound_enable = !config.sound_enable;
					if(emu)
						emu->update_config();
					break;

				case ID_CONFIG_FREQ0:
					config.sound_frequency = 0;
					break;
				case ID_CONFIG_FREQ1:
					config.sound_frequency = 1;
					break;
				case ID_CONFIG_FREQ2:
					config.sound_frequency = 2;
					break;
				case ID_CONFIG_FREQ3:
					config.sound_frequency = 3;
					break;

				case ID_CONFIG_LATE0:
					config.sound_latency = 0;
					break;
				case ID_CONFIG_LATE1:
					config.sound_latency = 1;
					break;
				case ID_CONFIG_LATE2:
					config.sound_latency = 2;
					break;
				case ID_CONFIG_LATE3:
					config.sound_latency = 3;
					break;

#ifndef _WIN32_WCE
				// accelerator
				case ID_SCREEN:
					// mute and open menu
					if(emu)
						emu->mute_sound();
					if(fullscreen_now)
						set_window(hWnd, FALSE);
					else
						set_window(hWnd, TRUE);
					break;
					
				case ID_MOUSE:
					if(emu)
						emu->toggle_mouse();
					break;
#endif
			}
			return 0;
	}
	return DefWindowProc(hWnd, iMsg, wParam, lParam) ;
}

void update_menu(HMENU hMenu, int index)
{
	switch(index)
	{
		case 0:
		{
			if(config.cpu_clock == 0)
				CheckMenuRadioItem(hMenu, ID_CONFIG_6MHZ, ID_CONFIG_48MHZ, ID_CONFIG_6MHZ, MF_BYCOMMAND);
			else if(config.cpu_clock == 1)
				CheckMenuRadioItem(hMenu, ID_CONFIG_6MHZ, ID_CONFIG_48MHZ, ID_CONFIG_12MHZ, MF_BYCOMMAND);
			else if(config.cpu_clock == 2)
				CheckMenuRadioItem(hMenu, ID_CONFIG_6MHZ, ID_CONFIG_48MHZ, ID_CONFIG_24MHZ, MF_BYCOMMAND);
			else
				CheckMenuRadioItem(hMenu, ID_CONFIG_6MHZ, ID_CONFIG_48MHZ, ID_CONFIG_48MHZ, MF_BYCOMMAND);
			
			CheckMenuItem(hMenu, ID_CONFIG_M1CYCLE, config.m1_cycle ? MF_CHECKED : MF_UNCHECKED);
			CheckMenuItem(hMenu, ID_CONFIG_TIMER, config.fast_timer ? MF_CHECKED : MF_UNCHECKED);
			break;
		}
		case 1:
		{
			// fdd 1
			bool flag = false;
			DeleteMenu(hMenu, ID_RECENT_FD11, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD12, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD13, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD14, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD15, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD16, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD17, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD18, MF_BYCOMMAND);
			if(_tcscmp(config.recent_disk[0][0], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD11, config.recent_disk[0][0]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[0][1], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD12, config.recent_disk[0][1]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[0][2], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD13, config.recent_disk[0][2]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[0][3], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD14, config.recent_disk[0][3]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[0][4], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD15, config.recent_disk[0][4]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[0][5], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD16, config.recent_disk[0][5]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[0][6], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD17, config.recent_disk[0][6]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[0][7], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD18, config.recent_disk[0][7]);
				flag = true;
			}
			if(!flag)
				AppendMenu(hMenu, MF_GRAYED | MF_STRING, ID_RECENT_FD11, _T("None"));
			break;
		}
		case 2:
		{
			// fdd 2
			bool flag = false;
			DeleteMenu(hMenu, ID_RECENT_FD21, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD22, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD23, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD24, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD25, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD26, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD27, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD28, MF_BYCOMMAND);
			if(_tcscmp(config.recent_disk[1][0], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD21, config.recent_disk[1][0]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[1][1], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD22, config.recent_disk[1][1]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[1][2], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD23, config.recent_disk[1][2]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[1][3], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD24, config.recent_disk[1][3]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[1][4], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD25, config.recent_disk[1][4]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[1][5], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD26, config.recent_disk[1][5]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[1][6], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD27, config.recent_disk[1][6]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[1][7], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD28, config.recent_disk[1][7]);
				flag = true;
			}
			if(!flag)
				AppendMenu(hMenu, MF_GRAYED | MF_STRING, ID_RECENT_FD21, _T("None"));
			break;
		}
#ifndef _WIN32_PPC
		case 3:
		{
			// fdd 3
			bool flag = false;
			DeleteMenu(hMenu, ID_RECENT_FD31, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD32, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD33, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD34, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD35, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD36, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD37, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD38, MF_BYCOMMAND);
			if(_tcscmp(config.recent_disk[2][0], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD31, config.recent_disk[2][0]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[2][1], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD32, config.recent_disk[2][1]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[2][2], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD33, config.recent_disk[2][2]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[2][3], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD34, config.recent_disk[2][3]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[2][4], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD35, config.recent_disk[2][4]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[2][5], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD36, config.recent_disk[2][5]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[2][6], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD37, config.recent_disk[2][6]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[2][7], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD38, config.recent_disk[2][7]);
				flag = true;
			}
			if(!flag)
				AppendMenu(hMenu, MF_GRAYED | MF_STRING, ID_RECENT_FD31, _T("None"));
			break;
		}
		case 4:
		{
			// fdd 4
			bool flag = false;
			DeleteMenu(hMenu, ID_RECENT_FD41, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD42, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD43, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD44, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD45, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD46, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD47, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_FD48, MF_BYCOMMAND);
			if(_tcscmp(config.recent_disk[3][0], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD41, config.recent_disk[3][0]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[3][1], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD42, config.recent_disk[3][1]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[3][2], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD43, config.recent_disk[3][2]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[3][3], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD44, config.recent_disk[3][3]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[3][4], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD45, config.recent_disk[3][4]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[3][5], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD46, config.recent_disk[3][5]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[3][6], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD47, config.recent_disk[3][6]);
				flag = true;
			}
			if(_tcscmp(config.recent_disk[3][7], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_FD48, config.recent_disk[3][7]);
				flag = true;
			}
			if(!flag)
					AppendMenu(hMenu, MF_GRAYED | MF_STRING, ID_RECENT_FD41, _T("None"));
			break;
		}
		case 5:
		{
			// tape
			bool flag = false;
			DeleteMenu(hMenu, ID_RECENT_TAP1, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_TAP2, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_TAP3, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_TAP4, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_TAP5, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_TAP6, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_TAP7, MF_BYCOMMAND);
			DeleteMenu(hMenu, ID_RECENT_TAP8, MF_BYCOMMAND);
			if(_tcscmp(config.recent_tape[0], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_TAP1, config.recent_tape[0]);
				flag = true;
			}
			if(_tcscmp(config.recent_tape[1], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_TAP2, config.recent_tape[1]);
				flag = true;
			}
			if(_tcscmp(config.recent_tape[2], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_TAP3, config.recent_tape[2]);
				flag = true;
			}
			if(_tcscmp(config.recent_tape[3], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_TAP4, config.recent_tape[3]);
				flag = true;
			}
			if(_tcscmp(config.recent_tape[4], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_TAP5, config.recent_tape[4]);
				flag = true;
			}
			if(_tcscmp(config.recent_tape[5], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_TAP6, config.recent_tape[5]);
				flag = true;
			}
			if(_tcscmp(config.recent_tape[6], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_TAP7, config.recent_tape[6]);
				flag = true;
			}
			if(_tcscmp(config.recent_tape[7], _T(""))) {
				AppendMenu(hMenu, MF_STRING, ID_RECENT_TAP8, config.recent_tape[7]);
				flag = true;
			}
			if(!flag)
				AppendMenu(hMenu, MF_GRAYED | MF_STRING, ID_RECENT_TAP1, _T("None"));
			break;
		}
#endif
#ifdef _WIN32_PPC
		case 3:
#else
		case 6:
#endif
		{
			// screen menu
#ifndef _WIN32_WCE
			EnableMenuItem(hMenu, ID_SCREEN_WINDOW, fullscreen_now ? MF_ENABLED : MF_GRAYED);
			EnableMenuItem(hMenu, ID_SCREEN_FULL, fullscreen_now ? MF_GRAYED : MF_ENABLED);
#endif
			if(config.monitor_type == 0)
				CheckMenuRadioItem(hMenu, ID_CONFIG_A400L, ID_CONFIG_D200L, ID_CONFIG_A400L, MF_BYCOMMAND);
			else if(config.monitor_type == 1)
				CheckMenuRadioItem(hMenu, ID_CONFIG_A400L, ID_CONFIG_D200L, ID_CONFIG_D400L, MF_BYCOMMAND);
			else if(config.monitor_type == 2)
				CheckMenuRadioItem(hMenu, ID_CONFIG_A400L, ID_CONFIG_D200L, ID_CONFIG_A200L, MF_BYCOMMAND);
			else
				CheckMenuRadioItem(hMenu, ID_CONFIG_A400L, ID_CONFIG_D200L, ID_CONFIG_D200L, MF_BYCOMMAND);
			
			if(config.skip_frames == 0)
				CheckMenuRadioItem(hMenu, ID_CONFIG_SKIP0, ID_CONFIG_SKIP3, ID_CONFIG_SKIP0, MF_BYCOMMAND);
			else if(config.skip_frames == 1)
				CheckMenuRadioItem(hMenu, ID_CONFIG_SKIP0, ID_CONFIG_SKIP3, ID_CONFIG_SKIP1, MF_BYCOMMAND);
			else if(config.skip_frames == 2)
				CheckMenuRadioItem(hMenu, ID_CONFIG_SKIP0, ID_CONFIG_SKIP3, ID_CONFIG_SKIP2, MF_BYCOMMAND);
			else
				CheckMenuRadioItem(hMenu, ID_CONFIG_SKIP0, ID_CONFIG_SKIP3, ID_CONFIG_SKIP3, MF_BYCOMMAND);
			
			CheckMenuItem(hMenu, ID_CONFIG_SCANLINE, config.scan_line ? MF_CHECKED : MF_UNCHECKED);
			break;
		}
#ifndef _WIN32_PPC
		case 7:
		{
			// sound menu
			if(config.sound_frequency == 0)
				CheckMenuRadioItem(hMenu, ID_CONFIG_FREQ0, ID_CONFIG_FREQ3, ID_CONFIG_FREQ0, MF_BYCOMMAND);
			else if(config.sound_frequency == 1)
				CheckMenuRadioItem(hMenu, ID_CONFIG_FREQ0, ID_CONFIG_FREQ3, ID_CONFIG_FREQ1, MF_BYCOMMAND);
			else if(config.sound_frequency == 2)
				CheckMenuRadioItem(hMenu, ID_CONFIG_FREQ0, ID_CONFIG_FREQ3, ID_CONFIG_FREQ2, MF_BYCOMMAND);
			else
				CheckMenuRadioItem(hMenu, ID_CONFIG_FREQ0, ID_CONFIG_FREQ3, ID_CONFIG_FREQ3, MF_BYCOMMAND);
				
			if(config.sound_latency == 0)
				CheckMenuRadioItem(hMenu, ID_CONFIG_LATE0, ID_CONFIG_LATE3, ID_CONFIG_LATE0, MF_BYCOMMAND);
			else if(config.sound_latency == 1)
				CheckMenuRadioItem(hMenu, ID_CONFIG_LATE0, ID_CONFIG_LATE3, ID_CONFIG_LATE1, MF_BYCOMMAND);
			else if(config.sound_latency == 2)
				CheckMenuRadioItem(hMenu, ID_CONFIG_LATE0, ID_CONFIG_LATE3, ID_CONFIG_LATE2, MF_BYCOMMAND);
			else
				CheckMenuRadioItem(hMenu, ID_CONFIG_LATE0, ID_CONFIG_LATE3, ID_CONFIG_LATE3, MF_BYCOMMAND);
			
			CheckMenuItem(hMenu, ID_CONFIG_SOUNDENB, config.sound_enable ? MF_CHECKED : MF_UNCHECKED);
			break;
		}
#endif
	}
}

void open_disk(HWND hWnd, int drv)
{
	_TCHAR szFile[_MAX_PATH] = _T("");
	OPENFILENAME OpenFileName;
	_memset(&OpenFileName, 0, sizeof(OpenFileName));
	OpenFileName.lStructSize = sizeof(OPENFILENAME);
	OpenFileName.hwndOwner = hWnd;
	OpenFileName.lpstrFilter = _T("MZ-2500 disk image (*.d88)\0*.d88\0All Files (*.*)\0*.*\0\0");
	OpenFileName.lpstrFile = szFile;
	OpenFileName.nMaxFile = _MAX_PATH;
	OpenFileName.lpstrTitle = _T("Open MZ-2500 disk image");
	
	if(GetOpenFileName(&OpenFileName)) {
		if(_tcscmp(config.recent_disk[drv][0], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_disk[drv][1], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_disk[drv][2], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_disk[drv][3], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_disk[drv][4], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_disk[drv][5], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_disk[drv][6], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_disk[drv][7], OpenFileName.lpstrFile) == 0) {
		}
		else {
			_tcscpy(config.recent_disk[drv][7], config.recent_disk[drv][6]);
			_tcscpy(config.recent_disk[drv][6], config.recent_disk[drv][5]);
			_tcscpy(config.recent_disk[drv][5], config.recent_disk[drv][4]);
			_tcscpy(config.recent_disk[drv][4], config.recent_disk[drv][3]);
			_tcscpy(config.recent_disk[drv][3], config.recent_disk[drv][2]);
			_tcscpy(config.recent_disk[drv][2], config.recent_disk[drv][1]);
			_tcscpy(config.recent_disk[drv][1], config.recent_disk[drv][0]);
			_tcscpy(config.recent_disk[drv][0], OpenFileName.lpstrFile);
		}
		emu->insert_disk(OpenFileName.lpstrFile, drv);
	}
}

void open_tape(HWND hWnd)
{
	_TCHAR szFile[_MAX_PATH] = _T("");
	OPENFILENAME OpenFileName;
	_memset(&OpenFileName, 0, sizeof(OpenFileName));
	OpenFileName.lStructSize = sizeof(OPENFILENAME);
	OpenFileName.hwndOwner = hWnd;
	OpenFileName.lpstrFilter = _T("MZ-2500 tape image (*.m3u)\0*.m3u\0All Files (*.*)\0*.*\0\0");
	OpenFileName.lpstrFile = szFile;
	OpenFileName.nMaxFile = _MAX_PATH;
	OpenFileName.lpstrTitle = _T("Open MZ-2500 tape image");
	
	if(GetOpenFileName(&OpenFileName)) {
		if(_tcscmp(config.recent_tape[0], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_tape[1], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_tape[2], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_tape[3], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_tape[4], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_tape[5], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_tape[6], OpenFileName.lpstrFile) == 0) {
		}
		else if(_tcscmp(config.recent_tape[7], OpenFileName.lpstrFile) == 0) {
		}
		else {
			_tcscpy(config.recent_tape[7], config.recent_tape[6]);
			_tcscpy(config.recent_tape[6], config.recent_tape[5]);
			_tcscpy(config.recent_tape[5], config.recent_tape[4]);
			_tcscpy(config.recent_tape[4], config.recent_tape[3]);
			_tcscpy(config.recent_tape[3], config.recent_tape[2]);
			_tcscpy(config.recent_tape[2], config.recent_tape[1]);
			_tcscpy(config.recent_tape[1], config.recent_tape[0]);
			_tcscpy(config.recent_tape[0], OpenFileName.lpstrFile);
		}
		emu->insert_media(OpenFileName.lpstrFile);
	}
}

void set_window(HWND hwnd, BOOL full)
{
#ifndef _WIN32_WCE
	static LONG style = WS_VISIBLE;
	static int dest_x = 0, dest_y = 0;
	WINDOWPLACEMENT place;
	place.length = sizeof(WINDOWPLACEMENT);
	
	if(full && !fullscreen_now) {
		// fullscreen
		DEVMODE dev;
		HDC hdc = GetDC(NULL);
		ZeroMemory(&dev, sizeof(dev));
		dev.dmSize = sizeof(dev);
		dev.dmFields = DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;
		dev.dmBitsPerPel = GetDeviceCaps(hdc, BITSPIXEL);
		dev.dmPelsWidth = 640;
		dev.dmPelsHeight = 480;
		ReleaseDC(NULL, hdc);
		
		if(ChangeDisplaySettings(&dev, CDS_TEST) == DISP_CHANGE_SUCCESSFUL) {
			GetWindowPlacement(hwnd, &place);
			dest_x = place.rcNormalPosition.left;
			dest_y = place.rcNormalPosition.top;
			ChangeDisplaySettings(&dev, CDS_FULLSCREEN);
			style = GetWindowLong(hwnd, GWL_STYLE);
			SetWindowLong(hwnd, GWL_STYLE, WS_VISIBLE);
			SetWindowPos(hwnd, HWND_TOP, 0, 0, 640, 480, SWP_SHOWWINDOW);
			SetCursorPos(320, 200);
			fullscreen_now = TRUE;
		}
		
		// set screen size to emu class
		if(emu)
			emu->set_screen_size(640, 480 - 18);
	}
	else if(!full && fullscreen_now) {
		// window
		RECT rect = {0, 0, WINDOW_WIDTH, WINDOW_HEIGHT};
		AdjustWindowRect(&rect, WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_VISIBLE, TRUE);
		ChangeDisplaySettings(NULL, 0);
		SetWindowLong(hwnd, GWL_STYLE, style);
		SetWindowPos(hwnd, HWND_BOTTOM, dest_x, dest_y, rect.right - rect.left, rect.bottom - rect.top, SWP_SHOWWINDOW);
		fullscreen_now = FALSE;
		
		// set screen size to emu class
		if(emu)
			emu->set_screen_size(WINDOW_WIDTH, WINDOW_HEIGHT);
	}
#endif
}

