/*
	Skelton for Z-80 PC Emulator

	Author : Takeda.Toshiya
	Date   : 2004.04.29 -

	[ Emulation Core (win32) ]
*/

#ifndef _EMU_H_
#define _EMU_H_

#ifndef _WIN32_WCE
#define _USE_DSOUND_
//#define _DEBUG_LOG_
#elif defined(_WIN32_HPC) || defined(_WIN32_SIG)
#define _USE_WAVEOUT_
#endif

#define MEDIA_MAX 32
#define SOCKET_MAX 4
#define SOCKET_BUFFER_MAX 0x100000
#define WM_SOCKET0 (WM_USER + 1)
#define WM_SOCKET1 (WM_USER + 2)
#define WM_SOCKET2 (WM_USER + 3)
#define WM_SOCKET3 (WM_USER + 4)

#define DIRECTSOUND_VERSION 0x300
#define DIRECTDRAW_VERSION 0x500

#include <windows.h>
#include <windowsx.h>
#include <winsock.h>
#ifdef _USE_DSOUND_
#include <dsound.h>
#endif
#include <mmsystem.h>
#include <stdio.h>
#include "common.h"

class VM;

class EMU
{
protected:
	VM* vm;
private:
	// ----------------------------------------
	// debug log
	// ----------------------------------------
#ifdef _DEBUG_LOG_
	FILE* debug;
#endif
	void open_debug();
	void close_debug();
	
	// ----------------------------------------
	// window
	// ----------------------------------------
	
	HWND main_window_handle;
	int window_width, window_height;
	
	// ----------------------------------------
	// input
	// ----------------------------------------
	
	void initialize_input();
	void release_input();
	void update_input();
	
	uint8 key_status[256];	// windows key code mapping
	uint8 joy_status[2];	// joystick #1, #2 (b0 = up, b1 = down, b2 = left, b3 = right, b4-b7 = trigger #1-#4
	int mouse_status[3];	// x, y, button (b0 = left, b1 = right)
	bool mouse_enable;
	
#ifndef _WIN32_WCE
	int joy_num;
	uint32 joy_xmin[2], joy_xmax[2];
	uint32 joy_ymin[2], joy_ymax[2];
	JOYCAPS joycaps[2];
#endif
	
	// ----------------------------------------
	// screen
	// ----------------------------------------
	
	void initialize_screen();
	void release_screen();
	
	HDC hdcDIB;
	HBITMAP hBMP;
	LPBYTE lpBuf;
	LPWORD lpBMP;
	LPBITMAPINFO lpDIB;
	
#ifdef _WIN32_PPC
	WORD buffer[SCREEN_HEIGHT][SCREEN_WIDTH];
#endif
	
	// ----------------------------------------
	// sound
	// ----------------------------------------
	
	void initialize_sound(int rate, int samples);
	void release_sound();
	void update_sound();
	
#ifdef _USE_DSOUND_
	// use direct sound
	LPDIRECTSOUND lpds;
	LPDIRECTSOUNDBUFFER lpdsb, lpdsp;
	bool first_half;
#elif defined _USE_WAVEOUT_
	// use waveOut API
	HWAVEOUT hwo;
	WAVEHDR wavehdr[4];
	int play_block;
#endif
	
	int sound_rate, sound_samples;
	DWORD sound_buffer_size;
	bool sound_ok;
	bool sound_enable;
	bool mute;
	
	// ----------------------------------------
	// media
	// ----------------------------------------
	
	void initialize_media();
	void release_media();
	
	_TCHAR media_path[MEDIA_MAX][_MAX_PATH];
	int media_cnt;
	
	// ----------------------------------------
	// timer
	// ----------------------------------------
	
	void update_timer();
	SYSTEMTIME sTime;
	
	// ----------------------------------------
	// network
	// ----------------------------------------
	
	void initialize_socket();
	void release_socket();
	void update_socket();
	
	int soc[SOCKET_MAX];
	bool is_tcp[SOCKET_MAX];
	struct sockaddr_in udpaddr[SOCKET_MAX];
	int socket_delay[SOCKET_MAX];
	char recv_buffer[SOCKET_MAX][SOCKET_BUFFER_MAX];
	int recv_r_ptr[SOCKET_MAX], recv_w_ptr[SOCKET_MAX];
	
	// ----------------------------------------
	// others
	// ----------------------------------------
	
	_TCHAR insert_path[4][_MAX_PATH];
	bool insert_request[4];
	int eject_wait[4];
	
public:
	// debug log
	void out_debug(const _TCHAR* format, ...);
	
	// ----------------------------------------
	// initialize
	// ----------------------------------------
	
	EMU(HWND hwnd);
	~EMU();
	
	void application_path(_TCHAR* path);
	
	// ----------------------------------------
	// for windows
	// ----------------------------------------
	
	// drive virtual machine
	
	void run();
	void reset();
	void ipl_reset();
	
	// user interface
	
	void insert_disk(_TCHAR* filename, int drv);
	void eject_disk(int drv);
	void insert_media(_TCHAR* filename);
	void eject_media();
	
	void update_config();
	
	// input device
	
	void key_down(int code);
	void key_up(int code);
	void key_down_ppc(int code);
	void key_up_ppc(int code);
	void key_down_sip(int code);
	
	void enable_mouse();
	void disenable_mouse();
	void toggle_mouse();
	
	// screen
	
	void set_screen_size(int width, int height);
	void draw_screen();
	void draw_gapi(LPWORD lpDraw);
	void update_screen(HDC hdc);
	
	// sound
	
	void play_sound();
	void stop_sound();
	void notify_sound();
	void mute_sound();
	
	// network
	
	int get_socket(int ch) { return soc[ch]; }
	void socket_connected(int ch);
	void socket_disconnected(int ch);
	void send_data(int ch);
	void recv_data(int ch);
	
	// ----------------------------------------
	// for virtual machine
	// ----------------------------------------
	
	// get timer
	
	void get_timer(int time[]);
	
	// play media
	
	int media_count();
	void play_media(int trk);
	void stop_media();
	
	// get input device status
	
	uint8* key_buffer() { return key_status; }
	uint8* joy_buffer() { return joy_status; }
	int* mouse_buffer() { return mouse_status; }
	
	// get screen buffer
	
#ifdef _WIN32_PPC
	uint16* screen_buffer(int y) { return buffer[y]; }
#else
	uint16* screen_buffer(int y) { return &lpBMP[SCREEN_WIDTH * (SCREEN_HEIGHT - y - 1)]; }
#endif
	
	// network
	
	bool init_socket_tcp(int ch);
	bool init_socket_udp(int ch);
	bool connect_socket(int ch, uint32 ipaddr, int port);
	void disconnect_socket(int ch);
	bool listen_socket(int ch);
	void send_data_tcp(int ch);
	void send_data_udp(int ch, uint32 ipaddr, int port);
};

#endif
