/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.31 -

	[ fdc (MB8877) ]
*/

#ifndef _FDC_H_ 
#define _FDC_H_

#include "vm.h"
#include "../emu.h"

#include "device.h"
#include "disk.h"

#define FDC_ST_BUSY 0x01	// busy
#define FDC_ST_INDEX 0x02	// index hole
#define FDC_ST_DRQ 0x02		// data request
#define FDC_ST_TRACK00 0x04	// track0
#define FDC_ST_LOSTDATA 0x04	// data lost
#define FDC_ST_CRCERR 0x08	// crc error
#define FDC_ST_SEEKERR 0x10	// seek error
#define FDC_ST_RECNFND 0x10	// sector not found
#define FDC_ST_HEADENG 0x20	// head engage
#define FDC_ST_RECTYPE 0x20	// record type
#define FDC_ST_WRITEFAULT 0x20	// write fault
#define FDC_ST_WRITEP 0x40	// write protect
#define FDC_ST_NOTREADY 0x80	// media not inserted

#define FDC_CMD_TYPE1	1
#define FDC_CMD_RD_SEC	2
#define FDC_CMD_RD_MSEC	3
#define FDC_CMD_WR_SEC	4
#define FDC_CMD_WR_MSEC	5
#define FDC_CMD_RD_ADDR	6
#define FDC_CMD_RD_TRK	7
#define FDC_CMD_WR_TRK	8
#define FDC_CMD_TYPE4	0x80

#define EVENT_MAX 7

// 6msec, 12msec, 20msec, 30msec
static const int seek_wait[4] = {6000, 12000, 20000, 30000};

class FDC : public DEVICE
{
private:
	// config, patch
	bool ignore_crc;
	bool stat_multisec;
	bool laydock_patch;
	
	// disk info
	DISK* disk[4];
	
	// drive info
	typedef struct {
		int track;
		int index;
		bool access;
	} fdc_t;
	fdc_t fdc[4];
	
	// registor
	uint8 status;
	uint8 cmdreg;
	uint8 trkreg;
	uint8 secreg;
	uint8 datareg;
	uint8 drvreg;
	uint8 sidereg;
	uint8 cmdtype;
	
	bool drive_reverse;
	
	// event
	int regist_id[EVENT_MAX];
	
	// status
	bool now_search;
	bool now_seek;
	int seektrk;
	bool seekvct;
	int indexcnt;
	int sectorcnt;
	
	// image handler
	uint8 search_track();
	uint8 search_sector(int trk, int side, int sct, bool compare);
	uint8 search_addr();
	bool make_track();
	
	// command
	void process_cmd();
	void cmd_restore();
	void cmd_seek();
	void cmd_step();
	void cmd_stepin();
	void cmd_stepout();
	void cmd_readdata();
	void cmd_writedata();
	void cmd_readaddr();
	void cmd_readtrack();
	void cmd_writetrack();
	void cmd_forceint();
	
public:
	FDC(VM* parent_vm, EMU* parent_emu) : DEVICE(parent_vm, parent_emu) {}
	~FDC() {}
	
	// common functions
	void initialize();
	void release();
	void reset();
	void update_config();
	
	void write_io8(uint16 addr, uint8 data);
	uint8 read_io8(uint16 addr);
	void write_signal(int ch, uint32 data);
	
	int iomap_write(int index) {
		static const int map[8] = { 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, -1 };
		return map[index];
	}
	int iomap_read(int index) {
		static const int map[5] = { 0xd8, 0xd9, 0xda, 0xdb, -1 };
		return map[index];
	}
	
	void event_callback(int event_id, int err);
	
	// unique function
	void insert_disk(_TCHAR path[], int drv);
	void eject_disk(int drv);
	bool disk_inserted(int drv);
	
	bool access_lamp(int drv) { return fdc[drv & 3].access; }
	bool access_all() { return fdc[0].access | fdc[1].access | fdc[2].access | fdc[3].access; }
	void access_reset() { fdc[0].access = fdc[1].access = fdc[2].access = fdc[3].access = false; }
};

#endif
