/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.30 -

	[ keyboard (Z-80PIO) ]
*/

#include "keyboard.h"
#include "cassette.h"
#include "crtc.h"

void KEYBOARD::initialize()
{
	for(int i = 0; i < 2; i++) {
		port[i].reg = 0;
		port[i].mode = 0x40;
		port[i].ctrl = 0;
		port[i].dir = 0xff;
		port[i].mask = 0xff;
		port[i].vector = 0;
		port[i].set_dir = false;
		port[i].set_mask = false;
		port[i].prv_req = false;
	}
	key_stat = emu->key_buffer();
}

void KEYBOARD::reset()
{
	for(int i = 0; i < 2; i++) {
		port[i].reg = 0;
		port[i].mode = 0x40;
		port[i].ctrl = 0;
		port[i].dir = 0xff;
		port[i].mask = 0xff;
		port[i].vector = 0;
		port[i].set_dir = false;
		port[i].set_mask = false;
		port[i].prv_req = false;
	}
}

void KEYBOARD::write_io8(uint16 addr, uint8 data)
{
	switch(addr & 0xff)
	{
		case 0xe8:
			// reg a
			port[0].reg = data;
			check_interrupt(0);
			create_keystat();
			check_interrupt(1);
			// crtc column size
			vm->crtc->write_signal(SIGNAL_KEYBOARD, (data & 0x20) ? 1 : 0);
			break;
		case 0xe9:
			// ctrl a
			if(port[0].set_dir) {
				port[0].dir = data;
				port[0].set_dir = false;
			}
			else if(port[0].set_mask) {
				port[0].mask = data;
				port[0].set_mask = false;
			}
			else if(!(data & 0x01))
				port[0].vector = data;
			else if((data & 0x0f) == 0x07) {
				if((port[0].mode & 0xc0) == 0xc0 && (data & 0x10))
					port[0].set_mask = true;
				if(data & 0x10)
					vm->cancel_interrupt(IRQ_KEYBOARD);
				port[0].ctrl = data;
			}
			else if((data & 0x0f) == 0x0f) {
				if((data & 0xc0) == 0xc0)
					port[0].set_dir = true;
				port[0].mode = data;
			}
			check_interrupt(0);
			break;
		case 0xea:
			// reg b
			break;
		case 0xeb:
			// ctrl b
			if(port[1].set_dir) {
				port[1].dir = data;
				port[1].set_dir = false;
			}
			else if(port[1].set_mask) {
				port[1].mask = data;
				port[1].set_mask = false;
			}
			else if(!(data & 0x01))
				port[1].vector = data;
			else if((data & 0x0f) == 0x07) {
				if((port[1].mode & 0xc0) == 0xc0 && (data & 0x10))
					port[1].set_mask = true;
				if(data & 0x10)
					vm->cancel_interrupt(IRQ_KEYBOARD);
				port[1].ctrl = data;
			}
			else if((data & 0x0f) == 0x0f) {
				if((data & 0xc0) == 0xc0)
					port[1].set_dir = true;
				port[1].mode = data;
			}
			check_interrupt(1);
			break;
	}
}

uint8 KEYBOARD::read_io8(uint16 addr)
{
	switch(addr & 0xff)
	{
		case 0xe8:
			// reg a
			return port[0].reg;
		case 0xea:
			// reg b
			return port[1].reg;
	}
	return 0xff;
}

void KEYBOARD::update_input()
{
	// update key status
	key_stat[0] = 0;
	
	keys[0xf] = 0xff;
	for(int i = 0; i <= 0x0d; i++) {
		uint8 tmp = 0;
		for(int j = 0; j < 8; j++)
			tmp |= (key_stat[key_map[i][j]]) ? 0 : (1 << j);
		keys[i] = tmp;
		keys[0xf] &= tmp;
	}
	
	// update port-b
	create_keystat();
	check_interrupt(1);
}

void KEYBOARD::create_keystat()
{
	// update port-b
	if(!(port[0].reg & 0x10))
		port[1].reg = keys[0xf];
	else {
		if((port[0].reg & 0xf) > 0xd)
			port[1].reg = 0xff;
		else
			port[1].reg = keys[port[0].reg & 0xf];
	}
	vm->cassette->write_signal(SIGNAL_KEYBOARD, port[1].reg);
}

void KEYBOARD::check_interrupt(int p)
{
	// not mode-3, or interrupt disenabled
	if((port[p].mode & 0xc0) != 0xc0 || !(port[p].ctrl & 0x80)) {
		vm->cancel_interrupt(IRQ_KEYBOARD);
		port[p].prv_req = false;
		return;
	}
	
	// check mode-3
	bool next_req = false;
	if((port[p].ctrl & 0x60) == 0x00 && (port[p].reg & port[p].mask) != port[p].mask)
		next_req = true;
	else if((port[p].ctrl & 0x60) == 0x20 && (port[p].reg & port[p].mask) != 0x00)
		next_req = true;
	else if((port[p].ctrl & 0x60) == 0x40 && (port[p].reg & port[p].mask) == 0x00)
		next_req = true;
	else if((port[p].ctrl & 0x60) == 0x60 && (port[p].reg & port[p].mask) == port[p].mask)
		next_req = true;
	
	// request interrupt when activated first
	if(!port[p].prv_req && next_req)
		vm->request_interrupt(IRQ_KEYBOARD, port[p].vector, true);
	// cancel request when not activated
	if(!next_req)
		vm->cancel_interrupt(IRQ_KEYBOARD);
	port[p].prv_req = next_req;
}

