/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.30 -

	[ memory bus ]
*/

#include "memory.h"
#include "crtc.h"
#include "../fileio.h"

void MEMORY::initialize()
{
	// initialize memory
	_memset(ram, 0, sizeof(ram));
	_memset(vram, 0, sizeof(vram));
	_memset(tvram, 0, sizeof(tvram));
	_memset(pcg, 0, sizeof(pcg));
	
	_memset(ipl, 0xff, sizeof(ipl));
	_memset(kanji, 0xff, sizeof(kanji));
	_memset(dic, 0xff, sizeof(dic));
	_memset(phone, 0xff, sizeof(phone));
	
	_memset(dummy_r, 0xff, sizeof(dummy_r));
	
	// load rom image
	_TCHAR app_path[_MAX_PATH], file_path[_MAX_PATH];
	emu->application_path(app_path);
	FILEIO* fio = new FILEIO();
	
	_stprintf(file_path, _T("%s%s"), app_path, _T("IPL.ROM"));
	if(fio->Fopen(file_path, FILEIO_READ_BINARY)) {
		fio->Fread(ipl, sizeof(ipl), 1);
		fio->Fclose();
	}
	
	_stprintf(file_path, _T("%s%s"), app_path, _T("KANJI.ROM"));
	if(fio->Fopen(file_path, FILEIO_READ_BINARY)) {
		fio->Fread(kanji, sizeof(kanji), 1);
		fio->Fclose();
	}
	
	_stprintf(file_path, _T("%s%s"), app_path, _T("DICT.ROM"));
	if(fio->Fopen(file_path, FILEIO_READ_BINARY)) {
		fio->Fread(dic, sizeof(dic), 1);
		fio->Fclose();
	}
	
	_stprintf(file_path, _T("%s%s"), app_path, _T("PHONE.ROM"));
	if(fio->Fopen(file_path, FILEIO_READ_BINARY)) {
		fio->Fread(phone, sizeof(phone), 1);
		fio->Fclose();
	}
	
	delete fio;
	
	// initialize memory map
	bank = 0;
	set_map(0x34);
	set_map(0x35);
	set_map(0x36);
	set_map(0x37);
	set_map(0x04);
	set_map(0x05);
	set_map(0x06);
	set_map(0x07);
}

void MEMORY::reset()
{
	// initialize memory map
	bank = 0;
	set_map(0x00);
	set_map(0x01);
	set_map(0x02);
	set_map(0x03);
	set_map(0x04);
	set_map(0x05);
	set_map(0x06);
	set_map(0x07);
}

void MEMORY::ipl_reset()
{
	// initialize memory map
	bank = 0;
	set_map(0x34);
	set_map(0x35);
	set_map(0x36);
	set_map(0x37);
	set_map(0x04);
	set_map(0x05);
	set_map(0x06);
	set_map(0x07);
}

void MEMORY::write_data8(uint16 addr, uint8 data)
{
	if(page_type[addr >> 13] == PAGE_TYPE_MODIFY) {
		if(page[addr >> 13] == 0x30)
			vm->crtc->write_data8((addr & 0x1fff) + 0x0000, data);
		else if(page[addr >> 13] == 0x31)
			vm->crtc->write_data8((addr & 0x1fff) + 0x2000, data);
		else if(page[addr >> 13] == 0x32)
			vm->crtc->write_data8((addr & 0x1fff) + 0x4000, data);
		else
			vm->crtc->write_data8((addr & 0x1fff) + 0x6000, data);
	}
	else
		bank_w[addr >> 11][addr & 0x07ff] = data;
}

uint8 MEMORY::read_data8(uint16 addr)
{
	if(page_type[addr >> 13] == PAGE_TYPE_MODIFY) {
		if(page[addr >> 13] == 0x30)
			return vm->crtc->read_data8((addr & 0x1fff) + 0x0000);
		else if(page[addr >> 13] == 0x31)
			return vm->crtc->read_data8((addr & 0x1fff) + 0x2000);
		else if(page[addr >> 13] == 0x32)
			return vm->crtc->read_data8((addr & 0x1fff) + 0x4000);
		else
			return vm->crtc->read_data8((addr & 0x1fff) + 0x6000);
	}
	else
		return bank_r[addr >> 11][addr & 0x07ff];
}

void MEMORY::write_io8(uint16 addr, uint8 data)
{
	switch(addr & 0xff)
	{
		case 0xb4:
			// map bank
			bank = data & 0x7;
			break;
		case 0xb5:
			// map reg
			set_map(data & 0x3f);
			break;
		case 0xce:
			// dictionary bank
			dic_bank = data & 0x1f;
			for(int i = 0; i < 8; i++) {
				if(page_type[i] == PAGE_TYPE_DIC) {
					for(int j = 0; j < 4; j++)
						bank_r[i * 4 + j] = dic + dic_bank * 0x2000 + 0x800 * j;
					for(int j = 0; j < 4; j++)
						bank_w[i * 4 + j] = dummy_w;
				}
			}
			break;
		case 0xcf:
			// kanji bank
			kanji_bank = data;
			for(int i = 0; i < 8; i++) {
				if(page_type[i] == PAGE_TYPE_KANJI) {
					bank_r[i * 4 + 0] = (!(kanji_bank & 0x80)) ? pcg : kanji + (kanji_bank & 0x7f) * 0x800;
					bank_w[i * 4 + 0] = (!(kanji_bank & 0x80)) ? pcg : dummy_w;
				}
			}
			break;
	}
}

uint8 MEMORY::read_io8(uint16 addr)
{
	switch(addr & 0xff)
	{
		case 0xb4:
			// map bank
			return bank;
		case 0xb5:
			// map reg
			uint8 val = page[bank];
			bank = (bank + 1) & 0x7;
			return val;
	}
	return 0xff;
}

void MEMORY::set_map(uint8 data)
{
	if(data <= 0x1f) {
		// main ram
		for(int i = 0; i < 4; i++)
			bank_r[bank * 4 + i] = ram + 0x2000 * data + 0x800 * i;
		for(int i = 0; i < 4; i++)
			bank_w[bank * 4 + i] = ram + 0x2000 * data + 0x800 * i;
		page_type[bank] = PAGE_TYPE_NORMAL;
	}
	else if(0x20 <= data && data <= 0x2f) {
		// vram
		int offset = 0;
		switch (data) {
			case 0x20: offset = 0x0; break;
			case 0x21: offset = 0x1; break;
			case 0x22: offset = 0x4; break;
			case 0x23: offset = 0x5; break;
			case 0x24: offset = 0x8; break;
			case 0x25: offset = 0x9; break;
			case 0x26: offset = 0xc; break;
			case 0x27: offset = 0xd; break;
			case 0x28: offset = 0x2; break;
			case 0x29: offset = 0x3; break;
			case 0x2a: offset = 0x6; break;
			case 0x2b: offset = 0x7; break;
			case 0x2c: offset = 0xa; break;
			case 0x2d: offset = 0xb; break;
			case 0x2e: offset = 0xe; break;
			case 0x2f: offset = 0xf; break;
		}
		for(int i = 0; i < 4; i++)
			bank_r[bank * 4 + i] = vram + 0x2000 * offset + 0x800 * i;
		for(int i = 0; i < 4; i++)
			bank_w[bank * 4 + i] = vram + 0x2000 * offset + 0x800 * i;
		page_type[bank] = PAGE_TYPE_NORMAL;
	}
	else if(0x30 <= data && data <= 0x33) {
		// read modify write
		for(int i = 0; i < 4; i++)
			bank_r[bank * 4 + i] = dummy_r;
		for(int i = 0; i < 4; i++)
			bank_w[bank * 4 + i] = dummy_w;
		page_type[bank] = PAGE_TYPE_MODIFY;
	}
	else if(0x34 <= data && data <= 0x37) {
		// ipl rom
		for(int i = 0; i < 4; i++)
			bank_r[bank * 4 + i] = ipl + 0x2000 * (data - 0x34) + 0x800 * i;
		for(int i = 0; i < 4; i++)
			bank_w[bank * 4 + i] = dummy_w;
		page_type[bank] = PAGE_TYPE_NORMAL;
	}
	else if(data == 0x38) {
		// text vram
		for(int i = 0; i < 3; i++)
			bank_r[bank * 4 + i] = tvram + 0x800 * i;
		for(int i = 0; i < 3; i++)
			bank_w[bank * 4 + i] = tvram + 0x800 * i;
		bank_r[bank * 4 + 3] = dummy_r;
		bank_w[bank * 4 + 3] = dummy_w;
		page_type[bank] = PAGE_TYPE_NORMAL;
	}
	else if(data == 0x39) {
		// kanji rom, pcg
		for(int i = 1; i < 4; i++)
			bank_r[bank * 4 + i] = pcg + 0x800 * i;
		for(int i = 1; i < 4; i++)
			bank_w[bank * 4 + i] = pcg + 0x800 * i;
		bank_r[bank * 4 + 0] = (!(kanji_bank & 0x80)) ? pcg : kanji + (kanji_bank & 0x7f) * 0x800;
		bank_w[bank * 4 + 0] = (!(kanji_bank & 0x80)) ? pcg : dummy_w;
		page_type[bank] = PAGE_TYPE_KANJI;
	}
	else if(data == 0x3a) {
		// dictionary rom
		for(int i = 0; i < 4; i++)
			bank_r[bank * 4 + i] = dic + dic_bank * 0x2000 + 0x800 * i;
		for(int i = 0; i < 4; i++)
			bank_w[bank * 4 + i] = dummy_w;
		page_type[bank] = PAGE_TYPE_DIC;
	}
	else if(0x3c <= data && data <= 0x3f) {
		// phone rom
		for(int i = 0; i < 4; i++)
			bank_r[bank * 4 + i] = phone + 0x2000 * (data - 0x3c) + 0x800 * i;
		for(int i = 0; i < 4; i++)
			bank_w[bank * 4 + i] = dummy_w;
		page_type[bank] = PAGE_TYPE_NORMAL;
	}
	else {
		// n.c
		for(int i = 0; i < 4; i++)
			bank_r[bank * 4 + i] = dummy_r;
		for(int i = 0; i < 4; i++)
			bank_w[bank * 4 + i] = dummy_w;
		page_type[bank] = PAGE_TYPE_NORMAL;
	}
	page[bank] = data;
	bank = (bank + 1) & 0x07;
}

