/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.30 -

	[ sound (YM2203) ]
*/

#include "sound.h"
#include "crtc.h"
#include "fdc.h"
#include "mouse.h"
#include "../config.h"

extern config_t config;

void SOUND::initialize()
{
	// config
	sound_enable = config.sound_enable;
	monitor_200line = (config.monitor_type & 2) ? true : false;
	
	// initialize ym2203
	opn = new FM::OPN;
	//opn = new OPN();
	frame_usec = (int)(1000000. / FRAMES_PER_SEC / LINES_PER_FRAME + 0.5);
	
	// initialize sound buffer
	sound_tmp = NULL;
	sound_buffer = NULL;
	buffer_ptr = 0;
	accum_cnt = 0;
	
	alarm_signal = false;
	pcm_signal = false;
	pcm_ptr = 0;
	
	// initialize recovery info
	_memset(regs, 0, sizeof(regs));
	_memset(reg_28h, 0, sizeof(reg_28h));
	_memset(reg_a0h, 0, sizeof(reg_a0h));
	_memset(reg_a1h, 0, sizeof(reg_a1h));
	_memset(reg_a2h, 0, sizeof(reg_a2h));
	_memset(reg_a4h, 0, sizeof(reg_a4h));
	_memset(reg_a5h, 0, sizeof(reg_a5h));
	_memset(reg_a6h, 0, sizeof(reg_a6h));
	_memset(reg_a8h, 0, sizeof(reg_a8h));
	_memset(reg_a9h, 0, sizeof(reg_a9h));
	_memset(reg_aah, 0, sizeof(reg_aah));
	_memset(reg_ach, 0, sizeof(reg_ach));
	_memset(reg_adh, 0, sizeof(reg_adh));
	_memset(reg_aeh, 0, sizeof(reg_aeh));
}

void SOUND::release()
{
	if(opn)
		delete opn;
	if(sound_tmp)
		free(sound_tmp);
	if(sound_buffer)
		free(sound_buffer);
}

void SOUND::update_config()
{
	if(sound_enable != config.sound_enable) {
		if(sound_buffer)
			_memset(sound_buffer, 0, sound_samples * 2);
		buffer_ptr = 0;
		pcm_ptr = 0;
		sound_enable = config.sound_enable;
	}
}

void SOUND::reset()
{
	if(opn)
		opn->Reset();
	opn->SetReg(0x27, 0);
	
	if(sound_buffer)
		_memset(sound_buffer, 0, sound_samples * 2);
	buffer_ptr = 0;
	
	pcm_signal = false;
	pcm_ptr = 0;
	
	// initialize recovery info
	_memset(regs, 0, sizeof(regs));
	_memset(reg_28h, 0, sizeof(reg_28h));
	_memset(reg_a0h, 0, sizeof(reg_a0h));
	_memset(reg_a1h, 0, sizeof(reg_a1h));
	_memset(reg_a2h, 0, sizeof(reg_a2h));
	_memset(reg_a4h, 0, sizeof(reg_a4h));
	_memset(reg_a5h, 0, sizeof(reg_a5h));
	_memset(reg_a6h, 0, sizeof(reg_a6h));
	_memset(reg_a8h, 0, sizeof(reg_a8h));
	_memset(reg_a9h, 0, sizeof(reg_a9h));
	_memset(reg_aah, 0, sizeof(reg_aah));
	_memset(reg_ach, 0, sizeof(reg_ach));
	_memset(reg_adh, 0, sizeof(reg_adh));
	_memset(reg_aeh, 0, sizeof(reg_aeh));
}

void SOUND::write_io8(uint16 addr, uint8 data)
{
	switch(addr & 0xff)
	{
		case 0xc8:
			// sound reg num
			reg_num = data;
			// prescaler
			if(reg_num == 0x2d || reg_num == 0x2e || reg_num == 0x2f)
				opn->SetReg(reg_num, 0);
			break;
		case 0xc9:
			// sound reg
			if(reg_num == 0xe) {
				vm->fdc->write_signal(SIGNAL_SOUND, (data & 0x2) ? 1 : 0);
				vm->crtc->write_signal(SIGNAL_SOUND, (data & 0x4) ? 0 : 1);
				vm->mouse->write_signal(SIGNAL_SOUND, (data & 0x8) ? 1 : 0);
			}
			opn->SetReg(reg_num, data);
			
			// set recovery info
			regs[reg_num] = data;
			switch(reg_num) {
				case 0x28: reg_28h[data & 0x3] = data; break;
				case 0x2d: regs[0x2d] = 1; regs[0x2e] = 0; regs[0x2f] = 0; break;
				case 0x2e: regs[0x2d] = 0; regs[0x2e] = 1; regs[0x2f] = 0; break;
				case 0x2f: regs[0x2d] = 0; regs[0x2e] = 0; regs[0x2f] = 1; break;
				case 0xa0: reg_a0h[(regs[0xa4] >> 3) & 0x7] = data; break;
				case 0xa1: reg_a1h[(regs[0xa5] >> 3) & 0x7] = data; break;
				case 0xa2: reg_a2h[(regs[0xa6] >> 3) & 0x7] = data; break;
				case 0xa4: reg_a4h[(regs[0xa4] >> 3) & 0x7] = data; break;
				case 0xa5: reg_a5h[(regs[0xa5] >> 3) & 0x7] = data; break;
				case 0xa6: reg_a6h[(regs[0xa6] >> 3) & 0x7] = data; break;
				case 0xa8: reg_a8h[(regs[0xac] >> 3) & 0x7] = data; break;
				case 0xa9: reg_a9h[(regs[0xad] >> 3) & 0x7] = data; break;
				case 0xaa: reg_aah[(regs[0xae] >> 3) & 0x7] = data; break;
				case 0xac: reg_ach[(regs[0xac] >> 3) & 0x7] = data; break;
				case 0xad: reg_adh[(regs[0xad] >> 3) & 0x7] = data; break;
				case 0xae: reg_aeh[(regs[0xae] >> 3) & 0x7] = data; break;
			}
			break;
	}
}

uint8 SOUND::read_io8(uint16 addr)
{
	switch(addr & 0xff)
	{
		case 0xc8:
			// sound state
			return opn->ReadStatus();
		case 0xc9:
			// sound reg
			if(reg_num == 0x0f)
				return 0x37 | (monitor_200line ? 0x40 : 0) | (alarm_signal ? 0x00 : 0x80);
			else
				return opn->GetReg(reg_num);
	}
	return 0xff;
}

void SOUND::write_signal(int ch, uint32 data)
{
	if(ch == SIGNAL_CALENDAR)
		alarm_signal = data ? true : false;
	else if(ch == SIGNAL_CASSETTE)
		pcm_signal = data ? true : false;
}

void SOUND::initialize_sound(int rate, int samples)
{
	opn->Init(SOUND_CLOCK, rate, false, NULL);
	opn->Reset();
	opn->SetReg(0x27, 0);
	
	if(sound_tmp)
		free(sound_tmp);
	sound_tmp = NULL;
	
	if(sound_buffer)
		free(sound_buffer);
	sound_buffer = NULL;
	
	sound_rate = rate;
	sound_samples = samples;
	// (samples per line) * 1024
	frame_cnt = (int)(1024. * sound_rate / FRAMES_PER_SEC / LINES_PER_FRAME + 0.5);
	
	sound_tmp = (int32*)malloc(sound_samples * sizeof(int32) * 2); // 32bit, stereo
	sound_buffer = (uint16*)malloc(sound_samples * sizeof(uint16)); // 16bit, mono
	_memset(sound_buffer, 0, sound_samples * 2);
	buffer_ptr = 0;
	
	pcm_signal = false;
	pcm_ptr = 0;
}

void SOUND::update_sound()
{
	accum_cnt += frame_cnt;
	int cnt = accum_cnt >> 10;
	accum_cnt -= cnt << 10;
	if(sound_enable) {
		create_sound(cnt, false);
		if(pcm_ptr < PCM_SAMPLES - 1)
			pcm_buffer[pcm_ptr++] = pcm_signal;
	}
	opn->Count(frame_usec);
}

uint16* SOUND::create_sound(int samples, bool fill)
{
	// get samples to be created
	int cnt = 0;
	if(fill)
		cnt = sound_samples - buffer_ptr;
	else
		cnt = (sound_samples - buffer_ptr < samples) ? sound_samples - buffer_ptr : samples;
	
	// create sound waves
	_memset(sound_tmp, 0, sizeof(int32) * 2 * cnt);
#if 1
	opn->Mix(sound_tmp, cnt);
#else
	try {
		opn->Mix(sound_tmp, cnt);
	}
	catch(...) {
		// initialize fmgen
		if(opn)
			delete opn;
		opn = new FM::OPN;
		
		opn->Init(SOUND_CLOCK, sound_rate, false, NULL);
		opn->Reset();
		opn->SetReg(0x27, 0);
		
		// recovery regs
		for(int i = 0; i <= 0x0f; i++)
			opn->SetReg(i, regs[i]);
		for(int i = 0x24; i <= 0x27; i++)
			opn->SetReg(i, regs[i]);
		for(int i = 0; i < 3; i++)
			opn->SetReg(0x28, reg_28h[i]);
		if(regs[0x2d])
			opn->SetReg(0x2d, regs[0x2d]);
		if(regs[0x2e])
			opn->SetReg(0x2e, regs[0x2e]);
		if(regs[0x2f])
			opn->SetReg(0x2f, regs[0x2f]);
		for(int i = 0x30; i <= 0x9e; i++)
			opn->SetReg(i, regs[i]);
		for(int i = 0; i < 8; i++) {
			opn->SetReg(0xa4, reg_a4h[i]);
			opn->SetReg(0xa0, reg_a0h[i]);
			opn->SetReg(0xa5, reg_a5h[i]);
			opn->SetReg(0xa1, reg_a1h[i]);
			opn->SetReg(0xa6, reg_a6h[i]);
			opn->SetReg(0xa2, reg_a2h[i]);
			opn->SetReg(0xac, reg_ach[i]);
			opn->SetReg(0xa8, reg_a8h[i]);
			opn->SetReg(0xad, reg_adh[i]);
			opn->SetReg(0xa9, reg_a9h[i]);
			opn->SetReg(0xae, reg_aeh[i]);
			opn->SetReg(0xaa, reg_aah[i]);
		}
		for(int i = 0xb0; i <= 0xb2; i++)
			opn->SetReg(i, regs[i]);
		
		// clear buffer
		_memset(sound_tmp, 0, cnt * 4);
	}
#endif
	
	// update sound buffer
	for(int i = 0, j = 0; i < cnt; i++, j += 2) {
		// stereo -> mono
#ifdef _WIN32_WCE
		int dat = sound_tmp[j];
#else
		int dat = sound_tmp[j] + (pcm_buffer[(int)(pcm_ptr * i / cnt)] ? PCM_VOLUME : 0);
#endif
		uint16 highlow = (uint16)(dat & 0x0000ffff);
		
		if((dat > 0) && (highlow >= 0x8000)) {
			sound_buffer[buffer_ptr++] = 0x7fff;
			continue;
		}
		if((dat < 0) && (highlow < 0x8000)) {
			sound_buffer[buffer_ptr++] = 0x8000;
			continue;
		}
		sound_buffer[buffer_ptr++] = highlow;
	}
	
	if(fill)
		buffer_ptr = 0;
	pcm_ptr = 0;
	return sound_buffer;
}

