/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.10 -

	[ virtual machine ]
*/

#ifndef _VM_H_
#define _VM_H_

#include "../common.h"

// interrup
#define IRQ_MAX		6
#define IRQ_KEYBOARD	0
#define IRQ_SERIAL	1
#define IRQ_VBLANK	2
#define IRQ_TIMER	3
#define IRQ_PRINTER	4
#define IRQ_CALENDAR	5

// event callback
#define CALLBACK_MAX	17
// calendar
#define EVENT_1HZ	0
#define EVENT_16HZ	1
// cassette
#define EVENT_PRE	0
#define EVENT_SIGNAL	1
#define EVENT_AFTER	2
// crtc
#define EVENT_BLINK	0
// fdc
#define EVENT_SEEK	0
#define EVENT_SEEKEND	1
#define EVENT_SEARCH	2
#define EVENT_TYPE4	3
#define EVENT_MULTI1	4
#define EVENT_MULTI2	5
#define EVENT_LOST	6
// mouse
#define EVENT_RECV1	0
#define EVENT_RECV2	1
#define EVENT_RECV3	2
// timer
#define EVENT_TIMER	0

// signal source
#define SIGNAL_CALENDAR	0
#define SIGNAL_CASSETTE	1
#define SIGNAL_CRTC	2
#define SIGNAL_KEYBOARD	3
#define SIGNAL_SOUND	4

class EMU;
class DEVICE;

class Z80;
class MEMORY;
class IO;
class INTERRUPT;

class CALENDAR;
class CASSETTE;
class CRTC;
class FDC;
class EMM;
class EXTROM;
class JOYSTICK;
class KANJI;
class KEYBOARD;
class MOUSE;
class ROMFILE;
class SASI;
class SOUND;
class TIMER;
class W3100A;

class VM
{
	// define friend
	friend Z80;
	friend MEMORY;
	friend IO;
	friend INTERRUPT;
	
	friend CALENDAR;
	friend CASSETTE;
	friend CRTC;
	friend FDC;
	friend KEYBOARD;
	friend MOUSE;
	//friend PRINTER;
	friend SASI;
	friend SOUND;
	friend TIMER;
protected:
	EMU* emu;
	
	Z80* cpu;
	MEMORY* memory;
	IO* io;
	INTERRUPT* interrupt;
	
	CALENDAR* calendar;
	CASSETTE* cassette;
	CRTC* crtc;
	EMM* emm;
	EXTROM* extrom;
	FDC* fdc;
	JOYSTICK* joystick;
	KANJI* kanji;
	KEYBOARD* keyboard;
	MOUSE* mouse;
	ROMFILE* romfile;
	SASI* sasi;
	SOUND* sound;
	TIMER* timer;
	W3100A* network;
	// i/o non connected
	DEVICE* dummy;
private:
	// event
	void drive_vm(int clock_cpu, int clock_vm);
	void set_clocks(int power);
	int clocks_cpu[27];
	int clocks_vm[27];
	
	typedef struct {
		bool enable;
		DEVICE* device;
		int event_id;
		int clock;
		int loop;
	} event_t;
	event_t event[CALLBACK_MAX];
	int event_clock;
	int past_clock;
	
public:
	// ----------------------------------------
	// initialize
	// ----------------------------------------
	
	VM(EMU* parent_emu);
	~VM();
	
	// ----------------------------------------
	// for emulation class
	// ----------------------------------------
	
	// drive virtual machine
	void run();
	void reset();
	void ipl_reset();
	
	// draw screen
	void draw_screen();
	
	// sound generation
	void initialize_sound(int rate, int samples);
	uint16* create_sound(int samples, bool fill);
	
	// network
	void network_connected(int ch);
	void network_disconnected(int ch);
	uint8* get_sendbuffer(int ch, int* size);
	void inc_sendbuffer_ptr(int ch, int size);
	uint8* get_recvbuffer0(int ch, int* size0, int* size1);
	uint8* get_recvbuffer1(int ch);
	void inc_recvbuffer_ptr(int ch, int size);
	
	// user interface
	void insert_disk(_TCHAR* filename, int drv);
	void eject_disk(int drv);
	bool disk_inserted(int drv);
	void update_config();
	
	// ----------------------------------------
	// for each device
	// ----------------------------------------
	
	// interrupt
	void request_interrupt(int dev, uint8 vector, bool pending);
	void cancel_interrupt(int dev);
	int interrupt_status(int dev);
	
	// event callback
	void regist_callback(DEVICE* device, int event_id, int usec, bool loop, int* regist_id);
	void cancel_callback(int regist_id);
	
	// devices
	DEVICE* first_device;
	DEVICE* last_device;
};

#endif
