/*
	Skelton for Z-80 PC Emulator

	Author : Takeda.Toshiya
	Date   : 2004.04.29 -

	[ sound (win32) ]
*/

#include <stdio.h>
#include "emu.h"
#include "vm/vm.h"

#define DSOUND_BUFFER_SIZE (DWORD)(sound_samples * 4)
#define DSOUND_BUFFER_HALF (DWORD)(sound_samples * 2)

#define WAVEOUT_BUFFER_SIZE (DWORD)(sound_samples * 2)

void EMU::initialize_sound(int rate, int samples)
{
	sound_rate = rate;
	sound_samples = samples;
	vm->initialize_sound(sound_rate, sound_samples);
	sound_ok = false;
	
#ifdef _USE_DSOUND_
	// initialize direct sound
	PCMWAVEFORMAT pcmwf;
	DSBUFFERDESC dsbd;
	WAVEFORMATEX wfex;
	
	if(FAILED(DirectSoundCreate(NULL, &lpds, NULL)))
		return;
	if(FAILED(lpds->SetCooperativeLevel(main_window_handle, DSSCL_PRIORITY)))
		return;
	
	// primary buffer
	ZeroMemory(&dsbd, sizeof(dsbd));
	dsbd.dwSize = sizeof(dsbd);
	dsbd.dwFlags = DSBCAPS_PRIMARYBUFFER;
	if(FAILED(lpds->CreateSoundBuffer(&dsbd, &lpdsp, NULL)))
		return;
	ZeroMemory(&wfex, sizeof(wfex));
	wfex.wFormatTag = WAVE_FORMAT_PCM;
	wfex.nChannels = 1;
	wfex.nSamplesPerSec = sound_rate;
	wfex.nBlockAlign = 2;
	wfex.nAvgBytesPerSec = wfex.nSamplesPerSec * wfex.nBlockAlign;
	wfex.wBitsPerSample = 16;
	if(FAILED(lpdsp->SetFormat(&wfex)))
		return;
	
	// secondary buffer
	ZeroMemory(&pcmwf, sizeof(pcmwf));
	pcmwf.wf.wFormatTag = WAVE_FORMAT_PCM;
	pcmwf.wf.nChannels = 1;
	pcmwf.wf.nSamplesPerSec = sound_rate;
	pcmwf.wf.nBlockAlign = 2;
	pcmwf.wf.nAvgBytesPerSec = pcmwf.wf.nSamplesPerSec * pcmwf.wf.nBlockAlign;
	pcmwf.wBitsPerSample = 16;
	ZeroMemory(&dsbd, sizeof(dsbd));
	dsbd.dwSize = sizeof(dsbd);
	dsbd.dwFlags = DSBCAPS_STICKYFOCUS | DSBCAPS_GETCURRENTPOSITION2;
	dsbd.dwBufferBytes = DSOUND_BUFFER_SIZE;
	dsbd.lpwfxFormat = (LPWAVEFORMATEX)&pcmwf;
	if(FAILED(lpds->CreateSoundBuffer(&dsbd, &lpdsb, NULL)))
		return;
	
	// start play
	if(sound_enable)
		lpdsb->Play(0, 0, DSBPLAY_LOOPING);
	first_half = true;
#elif defined _USE_WAVEOUT_
	// initialize waveOut
	ZeroMemory(wavehdr, sizeof(wavehdr));
	for(int i = 0; i < 4; i++) {
		wavehdr[i].lpData = (LPSTR)malloc(WAVEOUT_BUFFER_SIZE);
		ZeroMemory(wavehdr[i].lpData, WAVEOUT_BUFFER_SIZE);
		wavehdr[i].dwBufferLength = WAVEOUT_BUFFER_SIZE;
	}
	
	// format setting
	WAVEFORMATEX wf;
	ZeroMemory(&wf, sizeof(WAVEFORMATEX));
	wf.wFormatTag = WAVE_FORMAT_PCM;
	wf.nChannels = 1;
	wf.nSamplesPerSec = sound_rate;
	wf.wBitsPerSample = 16;
	wf.nBlockAlign = wf.nChannels * wf.wBitsPerSample / 8;
	wf.nAvgBytesPerSec = wf.nSamplesPerSec * wf.nBlockAlign;
	
	// start play
	if(waveOutOpen(&hwo, WAVE_MAPPER, &wf, (DWORD)main_window_handle, 0, CALLBACK_WINDOW) != MMSYSERR_NOERROR)
		return;
	for(int i = 0; i < 4; i++) {
		wavehdr[i].dwUser = 0;
		wavehdr[i].dwFlags = 0;
		wavehdr[i].dwLoops = 0;
		wavehdr[i].lpNext = NULL;
		wavehdr[i].reserved = 0;
		if(!waveOutPrepareHeader(hwo, &wavehdr[i], sizeof(WAVEHDR))) {
			if(!waveOutWrite(hwo, &wavehdr[i], sizeof(WAVEHDR)))
				continue;
			// failed
			waveOutUnprepareHeader(hwo, &wavehdr[i], sizeof(WAVEHDR));
		}
		wavehdr[i].dwFlags = 0;
	}
	if(!sound_enable)
		waveOutPause(hwo);
	play_block = 0;
#endif
	sound_ok = true;
}

void EMU::release_sound()
{
#ifdef _USE_DSOUND_
	// release direct sound
	if(lpdsp)
		lpdsp->Release();
	if(lpdsb)
		lpdsb->Release();
	if(lpds)
		lpds->Release();
	lpdsp = NULL;
	lpdsb = NULL;
	lpds = NULL;
#elif defined _USE_WAVEOUT_
	// release waveOut
	if(hwo) {
		while(waveOutReset(hwo) == MMSYSERR_HANDLEBUSY)
			Sleep(10);
		for(int i = 0; i < 4; i++) {
			if(wavehdr[i].dwFlags & WHDR_PREPARED)
				waveOutUnprepareHeader(hwo, &wavehdr[i], sizeof(WAVEHDR));
		}
		while(waveOutClose(hwo) == MMSYSERR_HANDLEBUSY)
			Sleep(10);
		hwo = NULL;
	}
	for(int i = 0; i < 4; i++)
		if(wavehdr[i].lpData)
			free(wavehdr[i].lpData);
#endif
}

void EMU::play_sound()
{
	// play sound
	if(sound_ok) {
#ifdef _USE_DSOUND_
		lpdsb->Play(0, 0, DSBPLAY_LOOPING);
		first_half = true;
#elif defined _USE_WAVEOUT_
		waveOutRestart(hwo);
#endif
	}
}

void EMU::stop_sound()
{
	// stop sound
	if(sound_ok) {
#ifdef _USE_DSOUND_
		lpdsb->Stop();
#elif defined _USE_WAVEOUT_
		waveOutPause(hwo);
#endif
	}
}

void EMU::update_sound()
{
	if(sound_ok) {
#ifdef _USE_DSOUND_
		// check current position
		DWORD play_c, write_c, offset, size1, size2;
		WORD *ptr1, *ptr2;
		
		if(FAILED(lpdsb->GetCurrentPosition(&play_c, &write_c)))
			return;
		if(first_half) {
			if(play_c < DSOUND_BUFFER_HALF)
				return;
			offset = 0;
		}
		else {
			if(play_c > DSOUND_BUFFER_HALF)
				return;
			offset = DSOUND_BUFFER_HALF;
		}
		
		// sound buffer must be updated
		if(mute || !sound_enable) {
			if(lpdsb->Lock(offset, DSOUND_BUFFER_HALF, (void **)&ptr1, &size1, (void **)&ptr2, &size2, 0) == DSERR_BUFFERLOST)
				lpdsb->Restore();
			if(ptr1)
				ZeroMemory(ptr1, size1);
			if(ptr2)
				ZeroMemory(ptr2, size2);
			lpdsb->Unlock(ptr1, size1, ptr2, size2);
		}
		else {
			uint16* sound_buffer = vm->create_sound(0, true);
			if(lpdsb->Lock(offset, DSOUND_BUFFER_HALF, (void **)&ptr1, &size1, (void **)&ptr2, &size2, 0) == DSERR_BUFFERLOST)
				lpdsb->Restore();
			if(ptr1)
				CopyMemory(ptr1, sound_buffer, size1);
			if(ptr2)
				CopyMemory(ptr2, sound_buffer + size1, size2);
			lpdsb->Unlock(ptr1, size1, ptr2, size2);
		}
		first_half = !first_half;
#endif
	}
}

void EMU::notify_sound()
{
	if(sound_ok) {
#ifdef _USE_WAVEOUT_
		int prev_block = play_block;
		play_block = (play_block + 1) & 0x3;
		int update_block = (play_block + 2) & 0x3;
		
		// prepare next block
		wavehdr[play_block].dwUser = wavehdr[play_block].dwFlags = 0;
		wavehdr[play_block].dwLoops = wavehdr[play_block].reserved = 0;
		wavehdr[play_block].lpNext = NULL;
		if(!waveOutPrepareHeader(hwo, &wavehdr[play_block], sizeof(WAVEHDR)))
			if(waveOutWrite(hwo, &wavehdr[play_block], sizeof(WAVEHDR)))
				waveOutUnprepareHeader(hwo, &wavehdr[play_block], sizeof(WAVEHDR));
		
		// unprepare block
		if(wavehdr[prev_block].dwFlags & WHDR_PREPARED)
			waveOutUnprepareHeader(hwo, &wavehdr[prev_block], sizeof(WAVEHDR));
		wavehdr[prev_block].dwFlags = 0;
		
		// update next's next block
		if(mute || !sound_enable)
			ZeroMemory(wavehdr[update_block].lpData, WAVEOUT_BUFFER_SIZE);
		else {
			uint16* sound_buffer = vm->create_sound(0, true);
			CopyMemory(wavehdr[update_block].lpData, sound_buffer, WAVEOUT_BUFFER_SIZE);
		}
#endif
	}
}

void EMU::mute_sound()
{
	if(!mute && sound_ok) {
#ifdef _USE_DSOUND_
		// check current position
		DWORD size1, size2;
		WORD *ptr1, *ptr2;
		
		if(lpdsb->Lock(0, DSOUND_BUFFER_SIZE, (void **)&ptr1, &size1, (void**)&ptr2, &size2, 0) == DSERR_BUFFERLOST)
			lpdsb->Restore();
		if(ptr1)
			ZeroMemory(ptr1, size1);
		if(ptr2)
			ZeroMemory(ptr2, size2);
		lpdsb->Unlock(ptr1, size1, ptr2, size2);
#endif
	}
	mute = true;
}

