/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.09.05 -

	[ calendar (RP-5C15) ]
*/

#include "calendar.h"
#include "sound.h"

#define YEAR		time[0]
#define MONTH		time[1]
#define DAY		time[2]
#define DAY_OF_WEEK	time[3]
#define HOUR		time[4]
#define MINUTE		time[5]
#define SECOND		time[6]

void CALENDAR::initialize()
{
	irq_done = agree = false;
	signal_1hz = signal_16hz = false;
	emu->get_timer(time);
	
	int regist_id;
	vm->regist_callback(this, EVENT_1HZ, 500000, true, &regist_id);
	vm->regist_callback(this, EVENT_16HZ, 31250, true, &regist_id);
}

void CALENDAR::reset()
{
	irq_done = agree = false;
}

void CALENDAR::write_io8(uint16 addr, uint8 data)
{
	uint8 haddr = addr >> 8;
	
	switch(addr & 0xff)
	{
		case 0xcc:
			switch(haddr)
			{
				case 0x00:
				case 0x01:
				case 0x02:
				case 0x03:
				case 0x04:
				case 0x05:
				case 0x06:
				case 0x07:
				case 0x08:
				case 0x09:
				case 0x0a:
				case 0x0b:
				case 0x0c:
					if(regs[0xd] & 0x1)
						regs[haddr] = data;
					break;
				case 0x0d:
				case 0x0e:
				case 0x0f:
					regs[haddr] = data;
					break;
			}
			break;
	}
}

uint8 CALENDAR::read_io8(uint16 addr)
{
	uint8 haddr = addr >> 8;
	
	switch(addr & 0xff)
	{
		case 0xcc:
			switch(haddr)
			{
				case 0x00:
					return (regs[0xd] & 0x1) ? regs[haddr] : SECOND % 10;
				case 0x01:
					return (regs[0xd] & 0x1) ? regs[haddr] : (uint8)(SECOND / 10);
				case 0x02:
					return (regs[0xd] & 0x1) ? regs[haddr] : MINUTE % 10;
				case 0x03:
					return (regs[0xd] & 0x1) ? regs[haddr] : (uint8)(MINUTE / 10);
				case 0x04:
					return (regs[0xd] & 0x1) ? regs[haddr] : ((regs[0xa] & 0x1) ? HOUR : HOUR % 12) % 10;
				case 0x05:
					if(regs[0xd] & 0x1)
						return regs[haddr];
					else {
						if(regs[0xa] & 0x1)
							return (uint8)(HOUR / 10);
						else
							return (uint8)((HOUR % 12) / 10) | (HOUR < 12 ? 0 : 2);
					}
				case 0x06:
					return (regs[0xd] & 0x1) ? regs[haddr] : DAY_OF_WEEK;
				case 0x07:
					return (regs[0xd] & 0x1) ? regs[haddr] : DAY % 10;
				case 0x08:
					return (regs[0xd] & 0x1) ? regs[haddr] : (uint8)(DAY / 10);
				case 0x09:
					return (regs[0xd] & 0x1) ? regs[haddr] : MONTH % 10;
				case 0x0a:
					return (regs[0xd] & 0x1) ? regs[haddr] : (uint8)(MONTH / 10);
				case 0x0b:
					if(regs[0xd] & 0x1) {
						if(((YEAR - 0) % 4) == 0 && (((YEAR - 0) % 100) != 0 || ((YEAR - 0) % 400) == 0))
							return 0;
						else if(((YEAR - 1) % 4) == 0 && (((YEAR - 1) % 100) != 0 || ((YEAR - 1) % 400) == 0))
							return 1;
						else if(((YEAR - 2) % 4) == 0 && (((YEAR - 2) % 100) != 0 || ((YEAR - 2) % 400) == 0))
							return 2;
						else
							return 3;
					}
					else
						return YEAR % 10;
				case 0x0c:
					return (regs[0xd] & 0x1) ? regs[haddr] : (uint8)((YEAR % 100) / 10);
			}
			return 0xff;
	}
	return 0xff;
}

void CALENDAR::event_callback(int event_id, int err)
{
	if(event_id == EVENT_1HZ)
		signal_1hz = !signal_1hz;
	else if(event_id == EVENT_16HZ)
		signal_16hz = !signal_16hz;
	
	bool signal = agree;
	signal |= (regs[0xf] & 0x8) ? false : signal_1hz;
	signal |= (regs[0xf] & 0x4) ? false : signal_16hz;
	vm->sound->write_signal(SIGNAL_CALENDAR, signal ? 1 : 0);
}

void CALENDAR::update_calendar()
{
	emu->get_timer(time);
	
	// check interrupt
	int minute, hour, day;
	
	minute = (regs[2] & 0xf) + (regs[3] & 0x7) * 10;
	if(regs[0xa] & 1)
		hour = (regs[4] & 0xf) + (regs[5] & 0x3) * 10;
	else
		hour = (regs[4] & 0xf) + (regs[5] & 0x1) * 10 + (regs[5] & 0x2 ? 12 : 0);
	day = (regs[7] & 0xf) + (regs[8] & 0x3) * 10;
	agree = ((regs[0xd] & 0x4) && minute == MINUTE && hour == HOUR && day == DAY) ? true : false;
	
	if(agree && !irq_done) {
		vm->request_interrupt(IRQ_CALENDAR, vector, false);
		irq_done = true;
	}
	else
		irq_done = false;
}

