/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.31 -

	[ cassette (i8255) ]
*/

#include "cassette.h"
#include "sound.h"

//  (0.2sec)
#define REGIST_PRE() { \
	now_signal = false; \
	vm->regist_callback(this, EVENT_PRE, 200000, false, &regist_pre); \
}
// M (0.5sec)
#define REGIST_SIGNAL() { \
	now_signal = true; \
	vm->regist_callback(this, EVENT_SIGNAL, 500000, false, &regist_signal); \
}
//  (0.3sec)
#define REGIST_AFTER() { \
	now_signal = false; \
	vm->regist_callback(this, EVENT_AFTER, 300000, false, &regist_after); \
}

#define CANCEL_EVENT() { \
	if(regist_pre != -1) \
		vm->cancel_callback(regist_pre); \
	if(regist_signal != -1) \
		vm->cancel_callback(regist_signal); \
	if(regist_after != -1) \
		vm->cancel_callback(regist_after); \
	regist_pre = regist_signal = regist_after = -1; \
}

void CASSETTE::initialize()
{
	areg = breg = creg = 0xff;
	
	now_signal = now_playing = false;
	fw_rw_mode = 0;
	track = 0.0;
	
	regist_pre = regist_signal = regist_after = -1;
}

void CASSETTE::release()
{
	stop_media();
}

void CASSETTE::reset()
{
	regist_pre = regist_signal = regist_after = -1;
}

void CASSETTE::write_io8(uint16 addr, uint8 data)
{
	switch(addr & 0xff)
	{
		case 0xe0:
			// port a
			write_porta(data);
			break;
		case 0xe1:
			// port b
			write_portb(data);
			break;
		case 0xe2:
			// port c
			write_portc(data);
			break;
		case 0xe3:
			// ctrl reg
			if(!(data & 0x80)) {
				uint8 mask = 1 << ((data >> 1) & 7);
				write_portc((creg & ~mask) | ((data & 1) ? mask : 0));
				vm->sound->write_signal(SIGNAL_CASSETTE, creg & 4);
			}
			break;
	}
}

uint8 CASSETTE::read_io8(uint16 addr)
{
	switch(addr & 0xff)
	{
		case 0xe0:
			// port a
			return areg;
		case 0xe1:
		{
			// port b
			uint8 val = (fw_rw_mode ? 0 : 8) | (now_signal ? 0x40 : 0);
			return (key_stat & 0x80) | (vblank ? 0 : 1) | val;
		}
		case 0xe2:
			// port c
			return creg;
	}
	return 0xff;
}

void CASSETTE::write_signal(int ch, uint32 data)
{
	if(ch == SIGNAL_KEYBOARD)
		key_stat = (uint8)data;
	else if(ch == SIGNAL_CRTC)
		vblank = data ? true : false;
}

void CASSETTE::event_callback(int event_id, int err)
{
	if(event_id == EVENT_PRE) {
		regist_pre = -1;
		REGIST_SIGNAL();
	}
	else if(event_id == EVENT_SIGNAL) {
		regist_after = -1;
		REGIST_AFTER();
	}
	else if(event_id == EVENT_AFTER) {
		regist_after = -1;
		if(fw_rw_mode == 1) {
			track = (float)((int)track + 1) + 0.1F;
			if((int)(track + 0.5) > emu->media_count()) {
				// reach last
				if(areg & 0x20)
					fw_rw_mode = 0; // stop now
				else {
					fw_rw_mode = -1; // auto rewind
					REGIST_PRE();
				}
				now_playing = false;
			}
			else {
				if(now_playing)
					play_media(); // play next track
				else {
					REGIST_PRE();
				}
			}
		}
		else if(fw_rw_mode == -1) {
			track = (float)((int)track - 1) + 0.9F;
			if(track < 1.0) {
				// reach top
				if(areg & 0x40)
					fw_rw_mode = 0; // stop now
				else {
					fw_rw_mode = 1; // auto play
					play_media();
				}
			}
			else {
				REGIST_PRE();
			}
		}
	}
}

void CASSETTE::write_porta(uint8 data)
{
	if((areg & 1) && !(data & 1)) {
		// rewind
		stop_media();
		fw_rw_mode = -1;
		CANCEL_EVENT();
		REGIST_PRE();
	}
	if((areg & 2) && !(data & 2)) {
		// forward
		stop_media();
		fw_rw_mode = 1;
		CANCEL_EVENT();
		REGIST_PRE();
	}
	if((areg & 4) && !(data & 4)) {
		// start play
		play_media();
		fw_rw_mode = 1;
		CANCEL_EVENT();
	}
	if((areg & 8) && !(data & 8)) {
		// stop
		stop_media();
		fw_rw_mode = 0;
		CANCEL_EVENT();
	}
	areg = data;
}

void CASSETTE::write_portb(uint8 data)
{
	breg = data;
}

void CASSETTE::write_portc(uint8 data)
{
	if(!(creg & 2) && (data & 2))
		vm->reset();
	if(!(creg & 8) && (data & 8))
		vm->ipl_reset();
	creg = data;
}

void CASSETTE::play_media()
{
	if((int)(track + 0.5) <= emu->media_count() && (int)(track + 0.5) <= track) {
		// current position is after signal
		track = (track < 1.0) ? 1.1F : (int)(track + 0.5) + 0.1F;
		emu->play_media((int)track);
	}
	else {
		// current position is pre signal
		REGIST_PRE();
	}
	now_playing = true;
}

void CASSETTE::stop_media()
{
	emu->stop_media();
	now_playing = false;
}

