/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.30 -

	[ crtc ]
*/

#include "crtc.h"
#include "z80.h"
#include "cassette.h"
#include "memory.h"
#include "../config.h"

extern config_t config;

void CRTC::initialize()
{
	// config
	monitor_200line = (config.monitor_type & 2) ? true : false;;
	scan_line = scan_tmp = config.scan_line;
	monitor_digital = monitor_tmp = (config.monitor_type & 1) ? true : false;
	
	// get memory pointer
	vram_b = vm->memory->get_vram() + 0x00000;
	vram_r = vm->memory->get_vram() + 0x08000;
	vram_g = vm->memory->get_vram() + 0x10000;
	vram_i = vm->memory->get_vram() + 0x18000;
	tvram1 = vm->memory->get_tvram() + 0x0000;
	attrib = vm->memory->get_tvram() + 0x0800;
	tvram2 = vm->memory->get_tvram() + 0x1000;
	kanji1 = vm->memory->get_kanji() + 0x00000;
	kanji2 = vm->memory->get_kanji() + 0x20000;
	pcg0 = vm->memory->get_pcg() + 0x0000;
	pcg1 = vm->memory->get_pcg() + 0x0800;
	pcg2 = vm->memory->get_pcg() + 0x1000;
	pcg3 = vm->memory->get_pcg() + 0x1800;
	
	// set text palette
	for(int i = 0; i < 8; i++)
		palette16[i + 16] = palette256[i + 256] = RGB_COLOR(((i & 0x02) ? 0x1f : 0), ((i & 0x04) ? 0x1f : 0), ((i & 0x01) ? 0x1f : 0));
	for(int i = 0; i < 64; i++) {
		palette256[i + 256 + 16] = RGB_COLOR(((i & 0x02) ? 0x08 : 0) | ((i & 0x10) ? 0x10 : 0), 
		                                     ((i & 0x04) ? 0x08 : 0) | ((i & 0x20) ? 0x10 : 0), 
		                                     ((i & 0x01) ? 0x08 : 0) | ((i & 0x08) ? 0x10 : 0));
	}
	
	// set cg palette
	for(int i = 0; i < 16; i++) {
		uint8 r, g, b, r8, g8, b8;
		if((i & 0x0f) == 0x08) {
			// gray
			r = r8 = 0x13;
			g = g8 = 0x13;
			b = b8 = 0x13;
		}
		else {
			r = ((i & 0x0a) == 0x0a) ? 0x1f : ((i & 0x0a) == 0x02) ? 0x0f : 0;
			g = ((i & 0x0c) == 0x0c) ? 0x1f : ((i & 0x0c) == 0x04) ? 0x0f : 0;
			b = ((i & 0x09) == 0x09) ? 0x1f : ((i & 0x09) == 0x01) ? 0x0f : 0;
			r8 = (i & 0x02) ? 0x1f : 0;
			g8 = (i & 0x04) ? 0x1f : 0;
			b8 = (i & 0x01) ? 0x1f : 0;
		}
		
		if(monitor_digital)
			palette16[i] = RGB_COLOR(r8, g8, b8);
		else
			palette16[i] = RGB_COLOR(r, g, b);
		
		palette4096r[i] = r;
		palette4096g[i] = g;
		palette4096b[i] = b;
		palette4096[i] = RGB_COLOR(r, g, b);
	}
	for(int i = 0; i < 256; i++) {
		palette256[i] = RGB_COLOR((((i & 0x20) ? 0x10 : 0) | ((i & 0x02) ? 0x08 : 0) | ((i & 0x80) ? 0x04 : 0)),
		                          (((i & 0x40) ? 0x10 : 0) | ((i & 0x04) ? 0x08 : 0) | ((i & 0x80) ? 0x04 : 0)),
		                          (((i & 0x10) ? 0x10 : 0) | ((i & 0x01) ? 0x08 : 0) | ((i & 0x80) ? 0x04 : 0)));
	}
	
	// extract cg optimize matrix
	for(int p1 = 0; p1 < 256; p1++) {
		for(int p2 = 0; p2 < 256; p2++) {
			for(int i = 0; i < 8; i++) {
				cg_matrix0[p1][p2][i] = (p1 & (0x01 << i) ? 0x01 : 0) | (p2 & (0x01 << i) ? 0x02 : 0);
				cg_matrix1[p1][p2][i] = (p1 & (0x01 << i) ? 0x04 : 0) | (p2 & (0x01 << i) ? 0x08 : 0);
				cg_matrix2[p1][p2][i] = (p1 & (0x01 << i) ? 0x10 : 0) | (p2 & (0x01 << i) ? 0x20 : 0);
				cg_matrix3[p1][p2][i] = (p1 & (0x01 << i) ? 0x40 : 0) | (p2 & (0x01 << i) ? 0x80 : 0);
			}
		}
	}
	
	// initialize crtc
	_memset(textreg, 0, sizeof(textreg));
	_memset(cgreg, 0, sizeof(cgreg));
	
	cgreg_num = 0x80;
	cgreg[0x00] = cgreg[0x01] = cgreg[0x02] = cgreg[0x03] = cgreg[0x06] = 0xff;
	GDEVS =   0; cgreg[0x08] = 0x00; cgreg[0x09] = 0x00;
	GDEVE = 400; cgreg[0x0a] = 0x90; cgreg[0x0b] = 0x01;
	GDEHS =   0; cgreg[0x0c] = 0x00;
	GDEHE =  80; cgreg[0x0d] = 0x50;
	
	for(int i = 0; i < 16; i++)
		palette_reg[i] = i;
	
	for(int i = 0; i < 16; i++) {
		for(int j = 1; j < 8; j++)
			priority[i][j] = j + 16;
		priority[i][0] = i; // transparent black
		priority[i][8] = 0 + 16; // non transparent black
	}
	for(int i = 0; i < 256; i++) {
		for(int j = 1; j < 16 + 64; j++)
			priority256[i][j] = j + 256;
		priority256[i][0] = i; // transparent black
		priority256[i][8] = 0 + 256; // non transparent black
		priority256[i][16] = i; // transparent black (64 colors)
	}
	
	vector = 0;
	scrn_size = SCRN_320x200;
	font_size = true;
	column_size = false;
	cg_mask = 0x0f;
	clear_flag = 0;
	pal_select = false;
	blink = false;
	latch[0] = latch[1] = latch[2] = latch[3] = 0;
	
#ifdef OPTIMIZE_TEXT
	map_init = text_init = trans_init = true;
#else
	map_init = trans_init = true;
#endif
	
	// 500msec
	int regist_id;
	vm->regist_callback(this, EVENT_BLINK, 500000, true, &regist_id);
}

void CRTC::update_config()
{
	//monitor_200line = (config.monitor_type ? 2) ? true : false;;
	scan_tmp = config.scan_line;
	monitor_tmp = (config.monitor_type & 1) ? true : false;
#ifdef OPTIMIZE_TEXT
	map_init = text_init = trans_init = true;
#else
	map_init = trans_init = true;
#endif
}

void CRTC::reset()
{
	_memset(textreg, 0, sizeof(textreg));
	_memset(cgreg, 0, sizeof(cgreg));
	cgreg[0] = cgreg[1] = cgreg[2] = cgreg[3] = cgreg[6] = 0xff;
	
	scrn_size = SCRN_320x200;
	font_size = true;
	column_size = false;
	vector = 0;
	
#ifdef OPTIMIZE_TEXT
	map_init = text_init = trans_init = true;
#else
	map_init = trans_init = true;
#endif
}

void CRTC::write_data8(uint16 addr, uint8 data)
{
	// read modify write
	if(cgreg[0xe] == 0x3) {
		// 4 colors
		if((cgreg[5] & 0xc0) == 0x00) {
			// REPLACE
			if(addr & 0x4000) {
				if(cgreg[5] & 0x01) {
					vram_g[addr & 0x3fff] &= ~cgreg[6];
					vram_g[addr & 0x3fff] |= (cgreg[4] & 0x01) ? (data & cgreg[0] & cgreg[6]) : 0x00;
				}
				if(cgreg[5] & 0x02) {
					vram_i[addr & 0x3fff] &= ~cgreg[6];
					vram_i[addr & 0x3fff] |= (cgreg[4] & 0x02) ? (data & cgreg[1] & cgreg[6]) : 0x00;
				}
			}
			else {
				if(cgreg[5] & 0x01) {
					vram_b[addr & 0x3fff] &= ~cgreg[6];
					vram_b[addr & 0x3fff] |= (cgreg[4] & 0x01) ? (data & cgreg[0] & cgreg[6]) : 0x00;
				}
				if(cgreg[5] & 0x02) {
					vram_r[addr & 0x3fff] &= ~cgreg[6];
					vram_r[addr & 0x3fff] |= (cgreg[4] & 0x02) ? (data & cgreg[1] & cgreg[6]) : 0x00;
				}
			}
		}
		else if((cgreg[5] & 0xc0) == 0x40) {
			// PSET
			if(addr & 0x4000) {
				if(cgreg[5] & 0x01) {
					vram_g[addr & 0x3fff] &= ~data;
					vram_g[addr & 0x3fff] |= (cgreg[4] & 0x01) ? (data & cgreg[0]) : 0x00;
				}
				if(cgreg[5] & 0x02) {
					vram_i[addr & 0x3fff] &= ~data;
					vram_i[addr & 0x3fff] |= (cgreg[4] & 0x02) ? (data & cgreg[1]) : 0x00;
				}
			}
			else {
				if(cgreg[5] & 0x01) {
					vram_b[addr & 0x3fff] &= ~data;
					vram_b[addr & 0x3fff] |= (cgreg[4] & 0x01) ? (data & cgreg[0]) : 0x00;
				}
				if(cgreg[5] & 0x02) {
					vram_r[addr & 0x3fff] &= ~data;
					vram_r[addr & 0x3fff] |= (cgreg[4] & 0x02) ? (data & cgreg[1]) : 0x00;
				}
			}
		}
	}
	else {
		if((cgreg[5] & 0xc0) == 0x00) {
			// REPLACE
			if(cgreg[5] & 0x01) {
				vram_b[addr & 0x7fff] &= ~cgreg[6];
				vram_b[addr & 0x7fff] |= (cgreg[4] & 0x01) ? (data & cgreg[0] & cgreg[6]) : 0x00;
			}
			if(cgreg[5] & 0x02) {
				vram_r[addr & 0x7fff] &= ~cgreg[6];
				vram_r[addr & 0x7fff] |= (cgreg[4] & 0x02) ? (data & cgreg[1] & cgreg[6]) : 0x00;
			}
			if(cgreg[5] & 0x04) {
				vram_g[addr & 0x7fff] &= ~cgreg[6];
				vram_g[addr & 0x7fff] |= (cgreg[4] & 0x04) ? (data & cgreg[2] & cgreg[6]) : 0x00;
			}
			if(cgreg[5] & 0x08) {
				vram_i[addr & 0x7fff] &= ~cgreg[6];
				vram_i[addr & 0x7fff] |= (cgreg[4] & 0x08) ? (data & cgreg[3] & cgreg[6]) : 0x00;
			}
		}
		else if((cgreg[5] & 0xc0) == 0x40) {
			// PSET
			if(cgreg[5] & 0x01) {
				vram_b[addr & 0x7fff] &= ~data;
				vram_b[addr & 0x7fff] |= (cgreg[4] & 0x01) ? (data & cgreg[0]) : 0x00;
			}
			if(cgreg[5] & 0x02) {
				vram_r[addr & 0x7fff] &= ~data;
				vram_r[addr & 0x7fff] |= (cgreg[4] & 0x02) ? (data & cgreg[1]) : 0x00;
			}
			if(cgreg[5] & 0x04) {
				vram_g[addr & 0x7fff] &= ~data;
				vram_g[addr & 0x7fff] |= (cgreg[4] & 0x04) ? (data & cgreg[2]) : 0x00;
			}
			if(cgreg[5] & 0x08) {
				vram_i[addr & 0x7fff] &= ~data;
				vram_i[addr & 0x7fff] |= (cgreg[4] & 0x08) ? (data & cgreg[3]) : 0x00;
			}
		}
	}
}

uint8 CRTC::read_data8(uint16 addr)
{
	// read modify write
	uint8 b, r, g, i, pl;
	
	if(cgreg[0xe] == 0x3) {
		// 4 colors
		b = latch[0] = (addr & 0x4000) ? vram_g[addr & 0x3fff] : vram_b[addr & 0x3fff];
		r = latch[1] = (addr & 0x4000) ? vram_i[addr & 0x3fff] : vram_r[addr & 0x3fff];
		g = latch[2] = 0;
		i = latch[3] = 0;
		pl = cgreg[7] & 1;
	}
	else {
		b = latch[0] = vram_b[addr & 0x7fff];
		r = latch[1] = vram_r[addr & 0x7fff];
		g = latch[2] = vram_g[addr & 0x7fff];
		i = latch[3] = vram_i[addr & 0x7fff];
		pl = cgreg[7] & 3;
	}
	
	if(cgreg[7] & 0x10) {
		uint8 compare = cgreg[7] & 0xf;
		uint8 val = (compare == (((b & 0x80) >> 7) | ((r & 0x80) >> 6) | ((g & 0x80) >> 5) | ((i & 0x80) >> 4))) ? 0x80 : 0x00;
		val |= (compare == (((b & 0x40) >> 6) | ((r & 0x40) >> 5) | ((g & 0x40) >> 4) | ((i & 0x40) >> 3))) ? 0x40 : 0x00;
		val |= (compare == (((b & 0x20) >> 5) | ((r & 0x20) >> 4) | ((g & 0x20) >> 3) | ((i & 0x20) >> 2))) ? 0x20 : 0x00;
		val |= (compare == (((b & 0x10) >> 4) | ((r & 0x10) >> 3) | ((g & 0x10) >> 2) | ((i & 0x10) >> 1))) ? 0x10 : 0x00;
		val |= (compare == (((b & 0x08) >> 3) | ((r & 0x08) >> 2) | ((g & 0x08) >> 1) | ((i & 0x08) >> 0))) ? 0x08 : 0x00;
		val |= (compare == (((b & 0x04) >> 2) | ((r & 0x04) >> 1) | ((g & 0x04) >> 0) | ((i & 0x04) << 1))) ? 0x04 : 0x00;
		val |= (compare == (((b & 0x02) >> 1) | ((r & 0x02) >> 0) | ((g & 0x02) << 1) | ((i & 0x02) << 2))) ? 0x02 : 0x00;
		val |= (compare == (((b & 0x01) >> 0) | ((r & 0x01) << 1) | ((g & 0x01) << 2) | ((i & 0x01) << 3))) ? 0x01 : 0x00;
		return val;
	}
	else
		return latch[pl];
}

void CRTC::write_io8(uint16 addr, uint8 data)
{
	uint8 haddr = (addr >> 8) & 0xff;
	
	switch(addr & 0xff)
	{
		case 0xae:
		{
			// 4096 palette reg
			uint8 num = (haddr & 0x1f) >> 1;
			uint8 r = palette4096r[num];
			uint8 g = palette4096g[num];
			uint8 b = palette4096b[num];
			if(haddr & 1)
				g = (data & 0x0f) << 1;
			else {
				r = (data & 0xf0) >> 3;
				b = (data & 0x0f) << 1;
			}
			palette4096r[num] = r;
			palette4096g[num] = g;
			palette4096b[num] = b;
			palette4096[num] = RGB_COLOR(r, g, b);
			// never change palette 0
			//palette4096[0] = 0;
			break;
		}
		case 0xbc:
			// cgreg num
			cgreg_num = data;
			break;
		case 0xbd:
		{
			// cgreg
			cgreg[cgreg_num & 0x1f] = data;
			
			// clear screen
			if((cgreg_num & 0x1f) == 0x05 && (data & 0xc0) == 0x80) {
				uint16 st, sz;
				switch(cgreg[0x0e]) {
					case 0x03: case 0x14: case 0x15: case 0x17: case 0x1d:
						// clear 0x0000 - 0x4000
						st = 0x0000;
						sz = 0x4000;
						break;
					case 0x94: case 0x95: case 0x97: case 0x9d:
						// clear 0x4000 - 0x7fff
						st = 0x4000;
						sz = 0x4000;
						break;
					default:
						// clear 0x0000 - 0x7fff
						st = 0x0000;
						sz = 0x8000;
				}
				if(cgreg[5] & 0x01)
					_memset(vram_b + st, 0, sz);
				if(cgreg[5] & 0x02)
					_memset(vram_r + st, 0, sz);
				if(cgreg[5] & 0x04)
					_memset(vram_g + st, 0, sz);
				if(cgreg[5] & 0x08)
					_memset(vram_i + st, 0, sz);
				clear_flag = 1;
			}
			// screen size
			if((cgreg_num & 0x1f) == 0x0e) {
				switch(data) {
					case 0x15: case 0x14: case 0x1d: case 0x95: case 0x94: case 0x9d:
						scrn_size = SCRN_320x200;
						break;
					case 0x17: case 0x97:
						scrn_size = SCRN_640x200;
						break;
					case 0x03: case 0x93:
						scrn_size = SCRN_640x400;
						break;
				}
			}
			// view range
			if((cgreg_num & 0x1f) == 0x08)
				cgreg[0x09] = 0;
			if((cgreg_num & 0x1f) == 0x0a)
				cgreg[0x0b] = 0;
			if(0x08 <= (cgreg_num & 0x1f) && (cgreg_num & 0x1f) <= 0x0d) {
				GDEVS = (cgreg[0x08] | ((cgreg[0x09] & 0x01) << 8)); //* ((scrn_size == SCRN_640x400) ? 1 : 2);
				GDEVE = (cgreg[0x0a] | ((cgreg[0x0b] & 0x01) << 8)); //* ((scrn_size == SCRN_640x400) ? 1 : 2);
				GDEHS = cgreg[0x0c] & 0x7f;
				GDEHE = cgreg[0x0d] & 0x7f;
			}
			// refresh scroll
			if(0x0e <= (cgreg_num & 0x1f) && (cgreg_num & 0x1f) <= 0x17)
				map_init = true;
			// inc cgreg num
			if(cgreg_num & 0x80)
				cgreg_num = (cgreg_num & 0xfc) | ((cgreg_num + 1) & 0x03);
			break;
		}
		case 0xf4:
			// textreg num
			textreg_num = data;
			break;
		case 0xf5:
		{
			// textreg
#ifdef OPTIMIZE_TEXT
			switch(textreg_num)
			{
				case 0:
				case 1:
				case 2:
				case 3:
				case 5:
				case 7:
				case 8:
				case 9:
					if(textreg[textreg_num] != data)
						text_init = true;
			}
#endif
			if(textreg_num == 0 && (textreg[0] & 2) != (data & 2))
				trans_init = true;
			
			if(textreg_num < 0x10)
				textreg[textreg_num] = data;
			else if(0x80 <= textreg_num && textreg_num < 0x90)
				textreg[textreg_num - 0x70] = data;
			if(textreg_num == 0x0a) {
				// update 256 colors palette
				for(int i = 0; i < 256; i++) {
					uint8 b0 = (data & 0x03) >> 0;
					uint8 r0 = (data & 0x0c) >> 2;
					uint8 g0 = (data & 0x30) >> 4;
					uint16 b = ((i & 0x10) ? 0x10 : 0) | ((i & 0x1) ? 0x8 : 0) | ((b0 == 0 && (i & 0x80)) || (b0 == 1 && (i & 0x8)) || (b0 == 2) ? 0x4 : 0);
					uint16 r = ((i & 0x20) ? 0x10 : 0) | ((i & 0x2) ? 0x8 : 0) | ((r0 == 0 && (i & 0x80)) || (r0 == 1 && (i & 0x8)) || (r0 == 2) ? 0x4 : 0);
					uint16 g = ((i & 0x40) ? 0x10 : 0) | ((i & 0x4) ? 0x8 : 0) | ((g0 == 0 && (i & 0x80)) || (g0 == 1 && (i & 0x8)) || (g0 == 2) ? 0x4 : 0);
					palette256[i] = RGB_COLOR(r, g, b);
				}
			}
			if(0x80 <= textreg_num && textreg_num < 0x90) {
				// set palette reg
				palette_reg[textreg_num & 0x0f] = data & 0x0f;
				// update priority
				for(int i = 1; i < 8; i++)
					priority[textreg_num & 0x0f][i] = (data & 0x10) ? textreg_num & 0x0f : i + 16;
				priority[textreg_num & 0x0f][0] = textreg_num & 0x0f; // transparent black
				priority[textreg_num & 0x0f][8] = (data & 0x10) ? textreg_num & 0x0f : 0 + 16; // non transparent black
				// update priority (256 colors)
				for(int i = 0; i < 16; i++) {
					for(int j = 1; j < 16 + 64; j++)
						priority256[(textreg_num & 0x0f) * 16 + i][j] = (data & 0x10) ? (data & 0x0f) * 16 + i : j + 256;
					priority256[(textreg_num & 0x0f) * 16 + i][0] = (data & 0x0f) * 16 + i; // transparent black
					priority256[(textreg_num & 0x0f) * 16 + i][8] = (data & 0x10) ? (data & 0x0f) * 16 + i : 0 + 256; // non transparent black
					priority256[(textreg_num & 0x0f) * 16 + i][16] = (data & 0x0f) * 16 + i; // transparent black (64 colors)
				}
			}
			
			// kugyokuden 400line patch
			if(vm->cpu->prvPC == 0xc27e && !monitor_200line) {
				if(textreg[3] == 0x26 && textreg[5] == 0xee) {
					textreg[3] = 0x11;
					textreg[5] = 0xd9;
				}
			}
			break;
		}
		case 0xf6:
			// cg mask reg
			cg_mask = (data & 0x07) | 0x08;
			break;
		case 0xf7:
		{
			// font size reg
#ifdef OPTIMIZE_TEXT
			bool next_size = (data & 0x01) ? true : false;
			if(font_size != next_size)
				text_init = true;
			font_size = next_size;
#else
			font_size = (data & 0x01) ? true : false;
#endif
			break;
		}
	}
}

uint8 CRTC::read_io8(uint16 addr)
{
	switch(addr & 0xff)
	{
		case 0xbc:
			// read plane b
			if(cgreg[7] & 0x10) {
				uint8 b = latch[0];
				uint8 r = latch[1];
				uint8 g = latch[2];
				uint8 i = latch[3];
				uint8 compare = cgreg[7] & 0xf;
				
				uint8 val = (compare == (((b & 0x80) >> 7) | ((r & 0x80) >> 6) | ((g & 0x80) >> 5) | ((i & 0x80) >> 4))) ? 0x80 : 0x00;
				val |= (compare == (((b & 0x40) >> 6) | ((r & 0x40) >> 5) | ((g & 0x40) >> 4) | ((i & 0x40) >> 3))) ? 0x40 : 0x00;
				val |= (compare == (((b & 0x20) >> 5) | ((r & 0x20) >> 4) | ((g & 0x20) >> 3) | ((i & 0x20) >> 2))) ? 0x20 : 0x00;
				val |= (compare == (((b & 0x10) >> 4) | ((r & 0x10) >> 3) | ((g & 0x10) >> 2) | ((i & 0x10) >> 1))) ? 0x10 : 0x00;
				val |= (compare == (((b & 0x08) >> 3) | ((r & 0x08) >> 2) | ((g & 0x08) >> 1) | ((i & 0x08) >> 0))) ? 0x08 : 0x00;
				val |= (compare == (((b & 0x04) >> 2) | ((r & 0x04) >> 1) | ((g & 0x04) >> 0) | ((i & 0x04) << 1))) ? 0x04 : 0x00;
				val |= (compare == (((b & 0x02) >> 1) | ((r & 0x02) >> 0) | ((g & 0x02) << 1) | ((i & 0x02) << 2))) ? 0x02 : 0x00;
				val |= (compare == (((b & 0x01) >> 0) | ((r & 0x01) << 1) | ((g & 0x01) << 2) | ((i & 0x01) << 3))) ? 0x01 : 0x00;
				return val;
			}
			else
				return latch[0];
		case 0xbd:
			// read plane r
			if(cgreg[7] & 0x10)
				return (vblank ? 0 : 0x80) | clear_flag;
			else
				return latch[1];
		case 0xbe:
			// read plane g
			return latch[2];
		case 0xbf:
			// read plane i
			return latch[3];
		case 0xf4: case 0xf5: case 0xf6: case 0xf7:
			// get blank state
			return (vblank ? 0 : 1) | (hblank ? 0 : 2);
	}
	return 0xff;
}

void CRTC::write_signal(int ch, uint32 data)
{
	if(ch == SIGNAL_KEYBOARD) {
#ifdef OPTIMIZE_TEXT
		bool next_size = data ? true : false;
		if(column_size != next_size)
			text_init = true;
		column_size = next_size;
#else
		column_size = data ? true : false;
#endif
	}
	else if(ch == SIGNAL_SOUND)
		pal_select = data ? true : false;
}

void CRTC::event_callback(int event_id, int err)
{
	blink = !blink;
}

void CRTC::set_hsync(int cnt)
{
	hblank = (GDEHS <= cnt && cnt < GDEHE) ? false : true;
	hblank |= vblank;
}

void CRTC::set_vsync(int cnt)
{
	vblank = (GDEVS <= cnt && cnt < GDEVE) ? false : true;
	vm->cassette->write_signal(SIGNAL_CRTC, vblank ? 1 : 0);
	
	if(cnt == GDEVE && GDEVS != GDEVE)
		vm->request_interrupt(IRQ_VBLANK, vector, false);
	// complete clear screen
	if(cnt == 400)
		clear_flag = 0;
}

