/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.30 -

	[ crtc - draw screen ]
*/

#include "crtc.h"
#include "fdc.h"
#include "sasi.h"

void CRTC::draw_screen()
{
	// update config
	scan_line = scan_tmp;
	
	if(monitor_digital != monitor_tmp) {
		monitor_digital = monitor_tmp;
		// set 16 colors palette
		for(int i = 0; i < 16; i++) {
			uint8 r, g, b, r8, g8, b8;
			if((i & 0x0f) == 0x08) {
				// gray
				r = r8 = 0x13;
				g = g8 = 0x13;
				b = b8 = 0x13;
			}
			else {
				r = ((i & 0x0a) == 0x0a) ? 0x1f : ((i & 0x0a) == 0x02) ? 0x0f : 0;
				g = ((i & 0x0c) == 0x0c) ? 0x1f : ((i & 0x0c) == 0x04) ? 0x0f : 0;
				b = ((i & 0x09) == 0x09) ? 0x1f : ((i & 0x09) == 0x01) ? 0x0f : 0;
				r8 = (i & 0x02) ? 0x1f : 0;
				g8 = (i & 0x04) ? 0x1f : 0;
				b8 = (i & 0x01) ? 0x1f : 0;
			}
			
			if(monitor_digital)
				palette16[i] = RGB_COLOR(r8, g8, b8);
			else
				palette16[i] = RGB_COLOR(r, g, b);
		}
	}
	
	// back color
	uint16 palette16tmp[16 + 8], palette4096tmp[16 + 8];
	uint16 palette16txt[9], palette4096txt[9], palette256txt[16+64];
	
	uint8 back16 = ((textreg[0xb] & 0x4) >> 2) | ((textreg[0xb] & 0x20) >> 4) | ((textreg[0xc] & 0x1) << 2) | ((textreg[0xb] & 0x1) << 3);
	uint16 back256 = RGB_COLOR((textreg[0xb] & 0x38) >> 1, ((textreg[0xb] & 0xc0) >> 4) | ((textreg[0xc] & 0x1) << 4), (textreg[0xb] & 0x07) << 2);
	
	for(int i = 0; i < 16 + 8; i++) {
		palette16tmp[i] = palette16[(i & 16) ? i : (palette_reg[i]) ? (palette_reg[i] & cg_mask) : (back16 & cg_mask)];
		uint8 col = (i == 16) ? 0 : (i & 16) ? (i & 0xf) + 8 : i;
		palette4096tmp[i] = palette4096[(palette_reg[col]) ? (palette_reg[col] & cg_mask) : (back16 & cg_mask)];
	}
	palette256[0] = back256;
	
	_memcpy(palette16txt, &palette16tmp[16], sizeof(uint16) * 8);
	palette16txt[0] = (back16 == 0 && palette_reg[0] == 2) ? 0 : palette16[palette_reg[back16]]; // tower of doruaga
	palette16txt[8] = 0;
	_memcpy(palette4096txt, &palette4096tmp[16], sizeof(uint16) * 8);
	palette4096txt[0] = palette4096[palette_reg[back16]];
	palette4096txt[8] = 0;
	_memcpy(palette256txt, &palette256[256], sizeof(uint16) * (16 + 64));
	palette256txt[0] = back256;
	palette256txt[8] = 0;
	
	// draw cg screen
	_memset(cg, 0, sizeof(cg));
	draw_cg();
	
	// draw text screen
#ifdef OPTIMIZE_TEXT
	if(text_init)
		_memset(text, 0, sizeof(text));
	draw_text();
	_memcpy(tvram1_cmp, tvram1, sizeof(tvram1_cmp));
	_memcpy(tvram2_cmp, tvram2, sizeof(tvram2_cmp));
	_memcpy(attrib_cmp, attrib, sizeof(attrib_cmp));
	text_init = false;
#else
	_memset(text, 0, sizeof(text));
	draw_text();
#endif
	
	// view port
	int vs = (GDEVS <= GDEVE) ? GDEVS * (scrn_size == SCRN_640x400 ? 1 : 2) : 0;
	int ve = (GDEVS <= GDEVE) ? GDEVE * (scrn_size == SCRN_640x400 ? 1 : 2) : 400;
	int hs = (GDEHS <= GDEHE && GDEHS < 80) ? GDEHS * 8 : 0;
	int he = (GDEHS <= GDEHE && GDEHE < 80) ? GDEHE * 8 : 640;
	
	// mix screens
	if(cgreg[0x0e] == 0x1d || cgreg[0x0e] == 0x9d) {
		// 256 colors
		for(int y = 0; y < vs && y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_text = &text[640 * y];
			for(int x = 0; x < 640; x++)
				dest[x] = palette256txt[src_text[x]];
		}
		for(int y = vs; y < ve && y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_cg = &cg[640 * y], *src_text = &text[640 * y];
			for(int x = 0; x < hs && x < 640; x++)
				dest[x] = palette256txt[src_text[x]];
			for(int x = hs; x < he && x < 640; x++)
				dest[x] = palette256[priority256[src_cg[x]][src_text[x]]];
			for(int x = he; x < 640; x++)
				dest[x] = palette256txt[src_text[x]];
		}
		for(int y = ve; y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_text = &text[640 * y];
			for(int x = 0; x < 640; x++)
				dest[x] = palette256txt[src_text[x]];
		}
	}
	else if(!pal_select) {
		// 16 colors
		for(int y = 0; y < vs && y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_cg = &cg[640 * y], *src_text = &text[640 * y];
			for(int x = 0; x < 640; x++)
				dest[x] = palette16txt[src_text[x]];
		}
		for(int y = vs; y < ve && y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_cg = &cg[640 * y], *src_text = &text[640 * y];
			for(int x = 0; x < hs && x < 640; x++)
				dest[x] = palette16txt[src_text[x]];
			for(int x = hs; x < he && x < 640; x++)
				dest[x] = palette16tmp[priority[src_cg[x]][src_text[x]]];
			for(int x = he; x < 640; x++)
				dest[x] = palette16txt[src_text[x]];
		}
		for(int y = ve; y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_cg = &cg[640 * y], *src_text = &text[640 * y];
			for(int x = 0; x < 640; x++)
				dest[x] = palette16txt[src_text[x]];
		}
	}
	else {
		// 4096 colors
		for(int y = 0; y < vs && y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_cg = &cg[640 * y], *src_text = &text[640 * y];
			for(int x = 0; x < 640; x++)
				dest[x] = palette4096txt[src_text[x]];
		}
		for(int y = vs; y < ve && y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_cg = &cg[640 * y], *src_text = &text[640 * y];
			for(int x = 0; x < hs && x < 640; x++)
				dest[x] = palette4096txt[src_text[x]];
			for(int x = hs; x < he && x < 640; x++)
				dest[x] = palette4096tmp[priority[src_cg[x]][src_text[x]]];
			for(int x = he; x < 640; x++)
				dest[x] = palette16txt[src_text[x]];
		}
		for(int y = ve; y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			uint8 *src_cg = &cg[640 * y], *src_text = &text[640 * y];
			for(int x = 0; x < 640; x++)
				dest[x] = palette4096txt[src_text[x]];
		}
	}
	
	// access lamp
	bool r = vm->fdc->access_lamp(0) | vm->fdc->access_lamp(2);
	bool g = vm->fdc->access_lamp(1) | vm->fdc->access_lamp(3);
	bool b = vm->sasi->access_lamp();
	vm->fdc->access_reset();
	vm->sasi->access_reset();
	
	if(r || g || b) {
		uint16 col = r ? RGB_COLOR(0x1f, 0, 0) : g ? RGB_COLOR(0, 0x1f, 0) : RGB_COLOR(0, 0, 0x1f);
		for(int y = 400 - 8; y < 400; y++) {
			uint16 *dest = emu->screen_buffer(y);
			for(int x = 640 - 8; x < 640; x++)
				dest[x] = col;
		}
	}
}

// ----------------------------------------------------------------------------
// draw text screen
// ----------------------------------------------------------------------------

void CRTC::draw_text()
{
	// extract text optimize matrix
	if(trans_init) {
		trans_color = (textreg[0] & 2) ? 8 : 0;
		for(int pat = 0; pat < 256; pat++) {
			for(int col = 0; col <= 8; col++) {
				text_matrix[pat][col][0] = text_matrixw[pat][col][ 0] = text_matrixw[pat][col][ 1] = (pat & 0x80) ? col : trans_color;
				text_matrix[pat][col][1] = text_matrixw[pat][col][ 2] = text_matrixw[pat][col][ 3] = (pat & 0x40) ? col : trans_color;
				text_matrix[pat][col][2] = text_matrixw[pat][col][ 4] = text_matrixw[pat][col][ 5] = (pat & 0x20) ? col : trans_color;
				text_matrix[pat][col][3] = text_matrixw[pat][col][ 6] = text_matrixw[pat][col][ 7] = (pat & 0x10) ? col : trans_color;
				text_matrix[pat][col][4] = text_matrixw[pat][col][ 8] = text_matrixw[pat][col][ 9] = (pat & 0x08) ? col : trans_color;
				text_matrix[pat][col][5] = text_matrixw[pat][col][10] = text_matrixw[pat][col][11] = (pat & 0x04) ? col : trans_color;
				text_matrix[pat][col][6] = text_matrixw[pat][col][12] = text_matrixw[pat][col][13] = (pat & 0x02) ? col : trans_color;
				text_matrix[pat][col][7] = text_matrixw[pat][col][14] = text_matrixw[pat][col][15] = (pat & 0x01) ? col : trans_color;
			}
		}
		trans_init = false;
	}
	
	// draw text
	if(column_size)
		draw_80column_screen();
	else
		draw_40column_screen();
	
	// display period
	int SL, EL, SC, EC;
	if(monitor_200line) {
		// 200 lines
		SL = (textreg[3] - 38) << 1;
		EL = (textreg[5] - 38) << 1;
		if(column_size) {
			// 80 column
			SC = (textreg[7] & 0x7f) - 11;
			EC = (textreg[8] & 0x7f) - 11;
		}
		else {
			// 40 column
			SC = (textreg[7] & 0x7f) - 10;
			EC = (textreg[8] & 0x7f) - 10;
		}
	}
	else {
		// 400 lines
		SL = (textreg[3] - 17) << 1;
		EL = (textreg[5] - 17) << 1;
		if(column_size) {
			// 80 colums
			SC = (textreg[7] & 0x7f) - 9;
			EC = (textreg[8] & 0x7f) - 9;
		}
		else {
			// 40 column
			SC = (textreg[7] & 0x7f) - 8;
			EC = (textreg[8] & 0x7f) - 8;
		}
	}
	SL = (SL < 0) ? 0 : (SL > 400) ? 400 : SL;
	EL = (EL < 0) ? 0 : (EL > 400) ? 400 : EL;
	SC = (SC < 0) ? 0 : (SC > 80) ? 80 : SC;
	EC = (EC < 0) ? 0 : (EC > 80) ? 80 : EC;
	if(EL >= SL) {
		for(int y = 0; y < SL; y++)
			_memset(text + 640 * y, 0, 640);
		for(int y = EL; y < 400; y++)
			_memset(text + 640 * y, 0, 640);
	}
	else {
		for(int y = EL; y < SL; y++)
			_memset(text + 640 * y, 0, 640);
	}
	if(EC >= SC) {
		for(int y = 0; y < 400; y++) {
			_memset(text + 640 * y, 0, SC * 8);
			_memset(text + 640 * y + EC * 8, 0, (80 - EC) * 8);
		}
	}
	else {
		for(int y = 0; y < 400; y++)
			_memset(text + 640 * y + EC * 8, 0, (SC - EC) * 8);
	}
}

void CRTC::draw_80column_screen()
{
	uint16 src = textreg[1] | ((textreg[2] & 0x07) << 8);
	uint8 line = (textreg[0] & 0x10) ? 2 : 0;
	uint8 height = (textreg[0] & 0x10) ? 20 : 16;
	uint8 vd = (textreg[9] & 0x0f) << 1;
	
	// 80x20(25)
	for(int y = line; y < 416; y += height) {
		int dest = (y - vd) * 640;
		for(int x = 0; x < 80; x++) {
			draw_80column_font((src++) & 0x7ff, dest, y - vd);
			dest += 8;
		}
	}
}

void CRTC::draw_40column_screen()
{
	uint16 src1 = textreg[1] | ((textreg[2] & 0x07) << 8);
	uint16 src2 = src1 + 0x400;
	uint8 line = (textreg[0] & 0x10) ? 2 : 0;
	uint8 height = (textreg[0] & 0x10) ? 20 : 16;
	uint8 vd = (textreg[9] & 0x0f) << 1;
	
	if((textreg[0] & 0x0c) == 0x00) {
		// 40x20(25), 64colors
		for(int y = line; y < 416; y += height) {
			int dest1 = (y - vd) * 640;
			int dest2 = (y - vd) * 640 + 640 * 480;
			for(int x = 0; x < 40; x++) {
				draw_40column_font((src1++) & 0x7ff, dest1, y - vd);
				draw_40column_font((src2++) & 0x7ff, dest2, y - vd);
				dest1 += 16;
				dest2 += 16;
			}
		}
		for(int y = 0; y < 400; y++) {
			uint32 src1 = 640 * y;
			uint32 src2 = 640 * y + 640 * 480;
			uint32 dest = 640 * y;
			uint8 col;
			for(int x = 0; x < 640; x++) {
				//if((text[src1] & 0x8) | (text[src2] & 0x8))
				if((text[src1] & 0x8) && (text[src2] & 0x8))
					// non transparent black
					col = 8;
				else
					col = (((text[src1] & 0x7) << 3) | (text[src2] & 0x7)) + 16;
				text[dest++] = col;
				src1++;
				src2++;
			}
		}
	}
	else if((textreg[0] & 0x0c) == 0x04) {
		// 40x20(25), No.1 Screen
		for(int y = line; y < 416; y += height) {
			int dest = (y - vd) * 640;
			for(int x = 0; x < 40; x++) {
				draw_40column_font((src1++) & 0x7ff, dest, y - vd);
				dest += 16;
			}
		}
	}
	else if((textreg[0] & 0x0c) == 0x08) {
		// 40x20(25), No.2 Screen
		for(int y = line; y < 416; y += height) {
			int dest = (y - vd) * 640;
			for(int x = 0; x < 40; x++) {
				draw_40column_font((src2++) & 0x7ff, dest, y - vd);
				dest += 16;
			}
		}
	}
	else if((textreg[0] & 0x0c) == 0x0c) {
		// 40x20(25), No.1 + No.2 Screens (No.1 > No.2)
#if 1
		for(int y = line; y < 416; y += height) {
			int dest = (y - vd) * 640;
			for(int x = 0; x < 40; x++) {
				draw_40column_font((src1++) & 0x7ff, dest, y - vd);
				dest += 16;
			}
		}
		for(int y = line; y < 416; y += height) {
			int dest = (y - vd) * 640 + 640 * 480;
			for(int x = 0; x < 40; x++) {
				draw_40column_font((src2++) & 0x7ff, dest, y - vd);
				dest += 16;
			}
		}
		for(int y = line; y < 400; y++) {
			int dest = (y - vd) * 640;
			uint8* tsrc1 = &text[dest];
			uint8* tsrc2 = &text[dest + 640 * 480];
			uint8* tdest = &text[dest];
			for(int x = 0; x < 640; x++)
				tdest[x] = (tsrc1[x] & 0x7) ? tsrc1[x] : (tsrc2[x] & 0x7) ? tsrc2[x] : ((tsrc1[x] & 0x8) | (tsrc2[x] & 0x8));
		}
#else
		for(int y = line; y < 416; y += height) {
			int dest = (y - vd) * 640;
			for(int x = 0; x < 40; x++) {
				draw_40column_font((src2++) & 0x7ff, dest, y - vd);
				dest += 16;
			}
		}
		uint8 text2[640*400];
		_memcpy(text2, text, 640 * 400);
		for(int y = line; y < 416; y += height) {
			int dest = (y - vd) * 640;
			for(int x = 0; x < 40; x++) {
				draw_40column_font((src1++) & 0x7ff, dest, y - vd);
				dest += 16;
			}
		}
		for(int y = line; y < 400; y++) {
			int dest = (y - vd) * 640;
			uint8* tsrc1 = &text2[dest];
			uint8* tsrc2 = &text[dest];
			uint8* tdest = &text[dest];
			for(int x = 0; x < 640; x++)
				tdest[x] = (tsrc1[x] & 0x7) ? tsrc1[x] : (tsrc2[x] & 0x7) ? tsrc2[x] : ((tsrc1[x] & 0x8) | (tsrc2[x] & 0x8));
		}
#endif
	}
}

void CRTC::draw_80column_font(uint16 src, int dest, int y)
{
	// draw char (80 column)
	uint8* pattern1;
	uint8* pattern2;
	uint8* pattern3;
	
	uint32 code;
	uint8 sel, col, pat1, pat2, pat3;
	
	// none draw
#ifdef OPTIMIZE_TEXT
	if(!text_init && tvram1[src] == tvram1_cmp[src] && tvram2[src] == tvram2_cmp[src] && attrib[src] == attrib_cmp[src])
		return;
	if((tvram1[src] & 0x7c) == 0x00 && ((tvram2[src] & 0xfe) == 0x8c || (tvram2[src] & 0xfe) == 0x80) && (attrib[src] & 0xf8) == 0x00) {
		if(!text_init) {
			for(int i = 0; i < 16; i++) {
				if(dest >= 0)
					_memset(&text[dest], 0, 8);
				dest += 640;
			}
		}
		return;
	}
#else
#ifdef _WIN32_WCE
	if((tvram1[src] & 0x7c) == 0x00 && ((tvram2[src] & 0xfe) == 0x8c || (tvram2[src] & 0xfe) == 0x80) && (attrib[src] & 0xf8) == 0x00)
		return;
#endif
#endif
	
	// select char type
	sel = (tvram2[src] & 0xc0) | (attrib[src] & 0x38);
	switch(sel) {
		case 0x00: case 0x40:
			pattern1 = pcg0;
			break;
		case 0x80:
			pattern1 = kanji1;
			break;
		case 0xc0:
			pattern1 = kanji2;
			break;
		case 0x10: case 0x50: case 0x90: case 0xd0:
			pattern1 = pcg1;
			break;
		case 0x20: case 0x60: case 0xa0: case 0xe0:
			pattern1 = pcg2;
			break;
		case 0x30: case 0x70: case 0xb0: case 0xf0:
			pattern1 = pcg3;
			break;
		default:
			pattern1 = pcg1;
			pattern2 = pcg2;
			pattern3 = pcg3;
			break;
	}
	if(sel & 0x08) {
		// PCG1 + PCG2 + PCG3 8colors
		
		// generate addr
		code = font_size ? tvram1[src] << 3 : (tvram1[src] & 0xfe) << 3;
		// draw
		if(font_size) {
			for(int i = 0; i < 8; i++) {
				// check end line of screen
				if(!(y < 639))
					break;
				y += 2;
				
				// reverse, blink
				if(dest >= 0) {
					if(attrib[src] & 0x40) {
						pat1 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern1[code + i];
						pat2 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern2[code + i];
						pat3 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern3[code + i];
					}
					else {
						pat1 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern1[code + i];
						pat2 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern2[code + i];
						pat3 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern3[code + i];
					}
					uint8* tdest = &text[dest];
					
					if(!scan_line) {
						tdest[0] = tdest[640] = ((pat1 & 0x80) ? 1 : 0) | ((pat2 & 0x80) ? 2 : 0) | ((pat3 & 0x80) ? 4 : 0);
						tdest[1] = tdest[641] = ((pat1 & 0x40) ? 1 : 0) | ((pat2 & 0x40) ? 2 : 0) | ((pat3 & 0x40) ? 4 : 0);
						tdest[2] = tdest[642] = ((pat1 & 0x20) ? 1 : 0) | ((pat2 & 0x20) ? 2 : 0) | ((pat3 & 0x20) ? 4 : 0);
						tdest[3] = tdest[643] = ((pat1 & 0x10) ? 1 : 0) | ((pat2 & 0x10) ? 2 : 0) | ((pat3 & 0x10) ? 4 : 0);
						tdest[4] = tdest[644] = ((pat1 & 0x08) ? 1 : 0) | ((pat2 & 0x08) ? 2 : 0) | ((pat3 & 0x08) ? 4 : 0);
						tdest[5] = tdest[645] = ((pat1 & 0x04) ? 1 : 0) | ((pat2 & 0x04) ? 2 : 0) | ((pat3 & 0x04) ? 4 : 0);
						tdest[6] = tdest[646] = ((pat1 & 0x02) ? 1 : 0) | ((pat2 & 0x02) ? 2 : 0) | ((pat3 & 0x02) ? 4 : 0);
						tdest[7] = tdest[647] = ((pat1 & 0x01) ? 1 : 0) | ((pat2 & 0x01) ? 2 : 0) | ((pat3 & 0x01) ? 4 : 0);
					}
					else {
						tdest[0] = ((pat1 & 0x80) ? 1 : 0) | ((pat2 & 0x80) ? 2 : 0) | ((pat3 & 0x80) ? 4 : 0);
						tdest[1] = ((pat1 & 0x40) ? 1 : 0) | ((pat2 & 0x40) ? 2 : 0) | ((pat3 & 0x40) ? 4 : 0);
						tdest[2] = ((pat1 & 0x20) ? 1 : 0) | ((pat2 & 0x20) ? 2 : 0) | ((pat3 & 0x20) ? 4 : 0);
						tdest[3] = ((pat1 & 0x10) ? 1 : 0) | ((pat2 & 0x10) ? 2 : 0) | ((pat3 & 0x10) ? 4 : 0);
						tdest[4] = ((pat1 & 0x08) ? 1 : 0) | ((pat2 & 0x08) ? 2 : 0) | ((pat3 & 0x08) ? 4 : 0);
						tdest[5] = ((pat1 & 0x04) ? 1 : 0) | ((pat2 & 0x04) ? 2 : 0) | ((pat3 & 0x04) ? 4 : 0);
						tdest[6] = ((pat1 & 0x02) ? 1 : 0) | ((pat2 & 0x02) ? 2 : 0) | ((pat3 & 0x02) ? 4 : 0);
						tdest[7] = ((pat1 & 0x01) ? 1 : 0) | ((pat2 & 0x01) ? 2 : 0) | ((pat3 & 0x01) ? 4 : 0);
					}
				}
				dest += 1280;
			}
		}
		else {
			for(int i = 0; i < 16; i++) {
				// check end line of screen
				if(!(y++ < 640))
					break;
				
				// reverse, blink
				if(dest >= 0) {
					if(attrib[src] & 0x40) {
						pat1 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern1[code + i];
						pat2 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern2[code + i];
						pat3 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern3[code + i];
					}
					else {
						pat1 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern1[code + i];
						pat2 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern2[code + i];
						pat3 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern3[code + i];
					}
					uint8* tdest = &text[dest];
					
					col = ((pat1 & 0x80) ? 1 : 0) | ((pat2 & 0x80) ? 2 : 0) | ((pat3 & 0x80) ? 4 : 0);
					tdest[0] = col ? col : trans_color;
					col = ((pat1 & 0x40) ? 1 : 0) | ((pat2 & 0x40) ? 2 : 0) | ((pat3 & 0x40) ? 4 : 0);
					tdest[1] = col ? col : trans_color;
					col = ((pat1 & 0x20) ? 1 : 0) | ((pat2 & 0x20) ? 2 : 0) | ((pat3 & 0x20) ? 4 : 0);
					tdest[2] = col ? col : trans_color;
					col = ((pat1 & 0x10) ? 1 : 0) | ((pat2 & 0x10) ? 2 : 0) | ((pat3 & 0x10) ? 4 : 0);
					tdest[3] = col ? col : trans_color;
					col = ((pat1 & 0x08) ? 1 : 0) | ((pat2 & 0x08) ? 2 : 0) | ((pat3 & 0x08) ? 4 : 0);
					tdest[4] = col ? col : trans_color;
					col = ((pat1 & 0x04) ? 1 : 0) | ((pat2 & 0x04) ? 2 : 0) | ((pat3 & 0x04) ? 4 : 0);
					tdest[5] = col ? col : trans_color;
					col = ((pat1 & 0x02) ? 1 : 0) | ((pat2 & 0x02) ? 2 : 0) | ((pat3 & 0x02) ? 4 : 0);
					tdest[6] = col ? col : trans_color;
					col = ((pat1 & 0x01) ? 1 : 0) | ((pat2 & 0x01) ? 2 : 0) | ((pat3 & 0x01) ? 4 : 0);
					tdest[7] = col ? col : trans_color;
				}
				dest += 640;
			}
		}
	}
	else {
		// monochrome
		
		// generate addr
		if(font_size) {
			if(sel == 0x80 || sel == 0xc0)
				code = ((tvram2[src] & 0x3f) << 11) | (tvram1[src] << 3);
			else
				code = tvram1[src] << 3;
		}
		else {
			if(sel == 0x80 || sel == 0xc0)
				code = ((tvram2[src] & 0x3f) << 11) | ((tvram1[src] & 0xfe) << 3);
			else
				code = (tvram1[src] & 0xfe) << 3;
		}
		// color
		col = (attrib[src] & 0x07) ? attrib[src] & 0x07 : 8;
		// draw
		if(font_size) {
			uint32 dest1 = dest;
			uint32 dest2 = (dest >= 640 * 399) ? dest - 640 * 399 : dest + 640;
			for(int i = 0; i < 8; i++) {
				// check end line of screen
				if(!(y < 639))
					break;
				y = y + 2;
				
				// reverse, blink
				if(attrib[src] & 0x40)
					pat1 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern1[code + i];
				else
					pat1 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern1[code + i];
				if(dest >= 0)
					_memcpy(&text[dest], text_matrix[pat1][col], 8);
				dest += 640;
				if(dest >= 0 && !scan_line)
					_memcpy(&text[dest], text_matrix[pat1][col], 8);
				dest += 640;
			}
		}
		else {
			for(int i = 0; i < 16; i++) {
				// check end line of screen
				if(!(y++ < 640))
					break;
				
				// reverse, blink
				if(attrib[src] & 0x40)
					pat1 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern1[code + i];
				else
					pat1 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern1[code + i];
				if(dest >= 0)
					_memcpy(&text[dest], text_matrix[pat1][col], 8);
				dest += 640;
			}
		}
	}
}

void CRTC::draw_40column_font(uint16 src, int dest, int y)
{
	// draw char (40 column)
	uint8* pattern1;
	uint8* pattern2;
	uint8* pattern3;
	
	uint32 code;
	uint8 sel, col, pat1, pat2, pat3;
	
	// none draw
#ifdef OPTIMIZE_TEXT
	if(!text_init && tvram1[src] == tvram1_cmp[src] && tvram2[src] == tvram2_cmp[src] && attrib[src] == attrib_cmp[src])
		return;	
	if((tvram1[src] & 0x7c) == 0x00 && ((tvram2[src] & 0xfe) == 0x8c || (tvram2[src] & 0xfe) == 0x80) && (attrib[src] & 0xf8) == 0x00) {
		if(!text_init) {
			for(int i = 0; i < 16; i++) {
				if(dest >= 0)
					_memset(&text[dest], 0, 16);
				dest += 640;
			}
		}
		return;
	}
#else
#ifdef _WIN32_WCE
	if((tvram1[src] & 0x7c) == 0x00 && ((tvram2[src] & 0xfe) == 0x8c || (tvram2[src] & 0xfe) == 0x80) && (attrib[src] & 0xf8) == 0x00)
		return;
#endif
#endif

	// select char type
	sel = (tvram2[src] & 0xc0) | (attrib[src] & 0x38);
	switch(sel) {
		case 0x00: case 0x40:
			pattern1 = pcg0;
			break;
		case 0x80:
			pattern1 = kanji1;
			break;
		case 0xc0:
			pattern1 = kanji2;
			break;
		case 0x10: case 0x50: case 0x90: case 0xd0:
			pattern1 = pcg1;
			break;
		case 0x20: case 0x60: case 0xa0: case 0xe0:
			pattern1 = pcg2;
			break;
		case 0x30: case 0x70: case 0xb0: case 0xf0:
			pattern1 = pcg3;
			break;
		default:
			pattern1 = pcg1;
			pattern2 = pcg2;
			pattern3 = pcg3;
			break;
	}
	if(sel & 0x08) {
		// PCG1 + PCG2 + PCG3 8colors
		
		// generate addr
		code = font_size ? tvram1[src] << 3 : (tvram1[src] & 0xfe) << 3;
		// draw
		if(font_size) {
			for(int i = 0; i < 8; i++) {
				// check end line of screen
				if(!(y < 639))
					break;
				y = y + 2;
				
				// reverse, blink
				if(dest >= 0) {
					if(attrib[src] & 0x40) {
						pat1 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern1[code + i];
						pat2 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern2[code + i];
						pat3 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern3[code + i];
					}
					else {
						pat1 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern1[code + i];
						pat2 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern2[code + i];
						pat3 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern3[code + i];
					}
					uint8* tdest = &text[dest];
					
					if(!scan_line) {
						tdest[ 0] = tdest[ 1] = tdest[640] = tdest[641] = ((pat1 & 0x80) ? 1 : 0) | ((pat2 & 0x80) ? 2 : 0) | ((pat3 & 0x80) ? 4 : 0);
						tdest[ 2] = tdest[ 3] = tdest[642] = tdest[643] = ((pat1 & 0x40) ? 1 : 0) | ((pat2 & 0x40) ? 2 : 0) | ((pat3 & 0x40) ? 4 : 0);
						tdest[ 4] = tdest[ 5] = tdest[644] = tdest[645] = ((pat1 & 0x20) ? 1 : 0) | ((pat2 & 0x20) ? 2 : 0) | ((pat3 & 0x20) ? 4 : 0);
						tdest[ 6] = tdest[ 7] = tdest[646] = tdest[647] = ((pat1 & 0x10) ? 1 : 0) | ((pat2 & 0x10) ? 2 : 0) | ((pat3 & 0x10) ? 4 : 0);
						tdest[ 8] = tdest[ 9] = tdest[648] = tdest[649] = ((pat1 & 0x08) ? 1 : 0) | ((pat2 & 0x08) ? 2 : 0) | ((pat3 & 0x08) ? 4 : 0);
						tdest[10] = tdest[11] = tdest[650] = tdest[651] = ((pat1 & 0x04) ? 1 : 0) | ((pat2 & 0x04) ? 2 : 0) | ((pat3 & 0x04) ? 4 : 0);
						tdest[12] = tdest[13] = tdest[652] = tdest[653] = ((pat1 & 0x02) ? 1 : 0) | ((pat2 & 0x02) ? 2 : 0) | ((pat3 & 0x02) ? 4 : 0);
						tdest[14] = tdest[15] = tdest[654] = tdest[655] = ((pat1 & 0x01) ? 1 : 0) | ((pat2 & 0x01) ? 2 : 0) | ((pat3 & 0x01) ? 4 : 0);
					}
					else {
						tdest[ 0] = tdest[ 1] = ((pat1 & 0x80) ? 1 : 0) | ((pat2 & 0x80) ? 2 : 0) | ((pat3 & 0x80) ? 4 : 0);
						tdest[ 2] = tdest[ 3] = ((pat1 & 0x40) ? 1 : 0) | ((pat2 & 0x40) ? 2 : 0) | ((pat3 & 0x40) ? 4 : 0);
						tdest[ 4] = tdest[ 5] = ((pat1 & 0x20) ? 1 : 0) | ((pat2 & 0x20) ? 2 : 0) | ((pat3 & 0x20) ? 4 : 0);
						tdest[ 6] = tdest[ 7] = ((pat1 & 0x10) ? 1 : 0) | ((pat2 & 0x10) ? 2 : 0) | ((pat3 & 0x10) ? 4 : 0);
						tdest[ 8] = tdest[ 9] = ((pat1 & 0x08) ? 1 : 0) | ((pat2 & 0x08) ? 2 : 0) | ((pat3 & 0x08) ? 4 : 0);
						tdest[10] = tdest[11] = ((pat1 & 0x04) ? 1 : 0) | ((pat2 & 0x04) ? 2 : 0) | ((pat3 & 0x04) ? 4 : 0);
						tdest[12] = tdest[13] = ((pat1 & 0x02) ? 1 : 0) | ((pat2 & 0x02) ? 2 : 0) | ((pat3 & 0x02) ? 4 : 0);
						tdest[14] = tdest[15] = ((pat1 & 0x01) ? 1 : 0) | ((pat2 & 0x01) ? 2 : 0) | ((pat3 & 0x01) ? 4 : 0);
					}
				}
				dest += 1280;
			}
		}
		else {
			for(int i = 0; i < 16; i++) {
				// check end line of screen
				if(!(y++ < 640))
					break;
				
				// reverse, blink
				if(dest >= 0) {
					if(attrib[src] & 0x40) {
						pat1 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern1[code + i];
						pat2 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern2[code + i];
						pat3 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern3[code + i];
					}
					else {
						pat1 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern1[code + i];
						pat2 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern2[code + i];
						pat3 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern3[code + i];
					}
					uint8* tdest = &text[dest];
					
					col = ((pat1 & 0x80) ? 1 : 0) | ((pat2 & 0x80) ? 2 : 0) | ((pat3 & 0x80) ? 4 : 0);
					tdest[ 0] = tdest[ 1] = col ? col : trans_color;
					col = ((pat1 & 0x40) ? 1 : 0) | ((pat2 & 0x40) ? 2 : 0) | ((pat3 & 0x40) ? 4 : 0);
					tdest[ 2] = tdest[ 3] = col ? col : trans_color;
					col = ((pat1 & 0x20) ? 1 : 0) | ((pat2 & 0x20) ? 2 : 0) | ((pat3 & 0x20) ? 4 : 0);
					tdest[ 4] = tdest[ 5] = col ? col : trans_color;
					col = ((pat1 & 0x10) ? 1 : 0) | ((pat2 & 0x10) ? 2 : 0) | ((pat3 & 0x10) ? 4 : 0);
					tdest[ 6] = tdest[ 7] = col ? col : trans_color;
					col = ((pat1 & 0x08) ? 1 : 0) | ((pat2 & 0x08) ? 2 : 0) | ((pat3 & 0x08) ? 4 : 0);
					tdest[ 8] = tdest[ 9] = col ? col : trans_color;
					col = ((pat1 & 0x04) ? 1 : 0) | ((pat2 & 0x04) ? 2 : 0) | ((pat3 & 0x04) ? 4 : 0);
					tdest[10] = tdest[11] = col ? col : trans_color;
					col = ((pat1 & 0x02) ? 1 : 0) | ((pat2 & 0x02) ? 2 : 0) | ((pat3 & 0x02) ? 4 : 0);
					tdest[12] = tdest[13] = col ? col : trans_color;
					col = ((pat1 & 0x01) ? 1 : 0) | ((pat2 & 0x01) ? 2 : 0) | ((pat3 & 0x01) ? 4 : 0);
					tdest[14] = tdest[15] = col ? col : trans_color;
				}
				dest += 640;
			}
		}
	}
	else {
		// monochrome
		
		// generate addr
		if(font_size) {
			if(sel == 0x80 || sel == 0xc0)
				code = ((tvram2[src] & 0x3f) << 11) | (tvram1[src] << 3);
			else
				code = tvram1[src] << 3;
		}
		else {
			if(sel == 0x80 || sel == 0xc0)
				code = ((tvram2[src] & 0x3f) << 11) | ((tvram1[src] & 0xfe) << 3);
			else
				code = (tvram1[src] & 0xfe) << 3;
		}
		// color
		col = (attrib[src] & 0x07) ? attrib[src] & 0x07 : 8;
		// draw
		if(font_size) {
			for(int i = 0; i < 8; i++) {
				// check end line of screen
				if(!(y < 639))
					break;
				y = y + 2;
				
				// reverse, blink
				if(attrib[src] & 0x40)
					pat1 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern1[code + i];
				else
					pat1 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern1[code + i];
				if(dest >= 0)
					_memcpy(&text[dest], text_matrixw[pat1][col], 16);
				dest += 640;
				if(dest >= 0 && !scan_line)
					_memcpy(&text[dest], text_matrixw[pat1][col], 16);
				dest += 640;
			}
		}
		else {
			for(int i = 0; i < 16; i++) {
				// check end line of screen
				if(!(y++ < 640))
					break;
				
				// reverse, blink
				if(attrib[src] & 0x40)
					pat1 = ((attrib[src] & 0x80) && blink) ? 0xff : ~pattern1[code + i];
				else
					pat1 = ((attrib[src] & 0x80) && blink) ? 0x00 : pattern1[code + i];
				if(dest >= 0)
					_memcpy(&text[dest], text_matrixw[pat1][col], 16);
				dest += 640;
			}
		}
	}
}

// ----------------------------------------------------------------------------
// drive cg screen
// ----------------------------------------------------------------------------

void CRTC::draw_cg()
{
	// draw cg screen
	switch(cgreg[0x0e]) {
		case 0x03:
			if(map_init)
				create_640x400map();
			draw_640x400x4screen();
			break;
		case 0x14:
			if(map_init)
				create_320x200map();
			draw_320x200x16screen(0);
			draw_320x200x16screen(1);
			break;
		case 0x15:
			if(map_init)
				create_320x200map();
			draw_320x200x16screen(1);
			draw_320x200x16screen(0);
			break;
		case 0x17:
			if(map_init)
				create_640x200map();
			draw_640x200x16screen(0);
			break;
		case 0x1d:
			if(map_init)
				create_320x200map();
			draw_320x200x256screen(0);
			break;
		case 0x93:
			if(map_init)
				create_640x400map();
			draw_640x400x16screen();
			break;
		case 0x94:
			if(map_init)
				create_320x200map();
			draw_320x200x16screen(2);
			draw_320x200x16screen(3);
			break;
		case 0x95:
			if(map_init)
				create_320x200map();
			draw_320x200x16screen(3);
			draw_320x200x16screen(2);
			break;
		case 0x97:
			if(map_init)
				create_640x200map();
			draw_640x200x16screen(1);
			break;
		case 0x9d:
			if(map_init)
				create_320x200map();
			draw_320x200x256screen(1);
			break;
	}
	
	// fill scan line
	if(!scan_line && !(cgreg[0x0e] == 0x03 || cgreg[0x0e] == 0x93)) {
		for(int y = 0; y < 400; y += 2)
			_memcpy(cg + (y + 1) * 640, cg + y * 640, 640);
	}
}

void CRTC::draw_320x200x16screen(uint8 pl)
{
	uint8 B, R, G, I, col;
	uint32 dest = 0;
	
	for(int y = 0; y < 200; y++) {
		for(int x = 0; x < 40; x++) {
			uint16 src = map_addr[y][x] | (0x2000 * pl);
			uint32 dest2 = dest + map_hdsc[y][x];
			dest += 16;
			
			if(pl == 0 || pl == 2) {
				B = (cgreg[0x18] & 0x01) ? vram_b[src] : 0;
				R = (cgreg[0x18] & 0x02) ? vram_r[src] : 0;
				G = (cgreg[0x18] & 0x04) ? vram_g[src] : 0;
				I = (cgreg[0x18] & 0x08) ? vram_i[src] : 0;
			}
			else {
				B = (cgreg[0x18] & 0x10) ? vram_b[src] : 0;
				R = (cgreg[0x18] & 0x20) ? vram_r[src] : 0;
				G = (cgreg[0x18] & 0x40) ? vram_g[src] : 0;
				I = (cgreg[0x18] & 0x80) ? vram_i[src] : 0;
			}
			
			uint8* cdest = &cg[dest2];
			col = cg_matrix0[B][R][0] | cg_matrix1[G][I][0];
			cdest[ 0] = cdest[ 1] = col ? col : cdest[ 0];
			col = cg_matrix0[B][R][1] | cg_matrix1[G][I][1];
			cdest[ 2] = cdest[ 3] = col ? col : cdest[ 2];
			col = cg_matrix0[B][R][2] | cg_matrix1[G][I][2];
			cdest[ 4] = cdest[ 5] = col ? col : cdest[ 4];
			col = cg_matrix0[B][R][3] | cg_matrix1[G][I][3];
			cdest[ 6] = cdest[ 7] = col ? col : cdest[ 6];
			col = cg_matrix0[B][R][4] | cg_matrix1[G][I][4];
			cdest[ 8] = cdest[ 9] = col ? col : cdest[ 8];
			col = cg_matrix0[B][R][5] | cg_matrix1[G][I][5];
			cdest[10] = cdest[11] = col ? col : cdest[10];
			col = cg_matrix0[B][R][6] | cg_matrix1[G][I][6];
			cdest[12] = cdest[13] = col ? col : cdest[12];
			col = cg_matrix0[B][R][7] | cg_matrix1[G][I][7];
			cdest[14] = cdest[15] = col ? col : cdest[14];
		}
		dest += 640;
	}
}

void CRTC::draw_320x200x256screen(uint8 pl)
{
	uint8 B0, B1, R0, R1, G0, G1, I0, I1;
	uint32 dest = 0;
	
	for(int y = 0; y < 200; y++) {
		for(int x = 0; x < 40; x++) {
			uint16 src = map_addr[y][x] | (0x4000 * pl);
			uint32 dest2 = dest + map_hdsc[y][x];
			dest += 16;
			
			B1 = (cgreg[0x18] & 0x01) ? vram_b[src + 0x0000] : 0;
			B0 = (cgreg[0x18] & 0x10) ? vram_b[src + 0x2000] : 0;
			R1 = (cgreg[0x18] & 0x02) ? vram_r[src + 0x0000] : 0;
			R0 = (cgreg[0x18] & 0x20) ? vram_r[src + 0x2000] : 0;
			G1 = (cgreg[0x18] & 0x04) ? vram_g[src + 0x0000] : 0;
			G0 = (cgreg[0x18] & 0x40) ? vram_g[src + 0x2000] : 0;
			I1 = (cgreg[0x18] & 0x08) ? vram_i[src + 0x0000] : 0;
			I0 = (cgreg[0x18] & 0x80) ? vram_i[src + 0x2000] : 0;
			
			uint8* cdest = &cg[dest2];
			cdest[ 0] = cdest[ 1] = cg_matrix0[B0][R0][0] | cg_matrix1[G0][I0][0] | cg_matrix2[B1][R1][0] | cg_matrix3[G1][I1][0];
			cdest[ 2] = cdest[ 3] = cg_matrix0[B0][R0][1] | cg_matrix1[G0][I0][1] | cg_matrix2[B1][R1][1] | cg_matrix3[G1][I1][1];
			cdest[ 4] = cdest[ 5] = cg_matrix0[B0][R0][2] | cg_matrix1[G0][I0][2] | cg_matrix2[B1][R1][2] | cg_matrix3[G1][I1][2];
			cdest[ 6] = cdest[ 7] = cg_matrix0[B0][R0][3] | cg_matrix1[G0][I0][3] | cg_matrix2[B1][R1][3] | cg_matrix3[G1][I1][3];
			cdest[ 8] = cdest[ 9] = cg_matrix0[B0][R0][4] | cg_matrix1[G0][I0][4] | cg_matrix2[B1][R1][4] | cg_matrix3[G1][I1][4];
			cdest[10] = cdest[11] = cg_matrix0[B0][R0][5] | cg_matrix1[G0][I0][5] | cg_matrix2[B1][R1][5] | cg_matrix3[G1][I1][5];
			cdest[12] = cdest[13] = cg_matrix0[B0][R0][6] | cg_matrix1[G0][I0][6] | cg_matrix2[B1][R1][6] | cg_matrix3[G1][I1][6];
			cdest[14] = cdest[15] = cg_matrix0[B0][R0][7] | cg_matrix1[G0][I0][7] | cg_matrix2[B1][R1][7] | cg_matrix3[G1][I1][7];
		}
		dest += 640;
	}
}

void CRTC::draw_640x200x16screen(uint8 pl)
{
	uint8 B, R, G, I;
	uint32 dest = 0;
	
	for(int y = 0; y < 200; y++) {
		for(int x = 0; x < 80; x++) {
			uint16 src = map_addr[y][x] | (0x4000 * pl);
			uint32 dest2 = dest + map_hdsc[y][x];
			dest += 8;
			
			B = (cgreg[0x18] & 0x01) ? vram_b[src] : 0;
			R = (cgreg[0x18] & 0x02) ? vram_r[src] : 0;
			G = (cgreg[0x18] & 0x04) ? vram_g[src] : 0;
			I = (cgreg[0x18] & 0x08) ? vram_i[src] : 0;
			
			uint8* cdest = &cg[dest2];
			cdest[0] = cg_matrix0[B][R][0] | cg_matrix1[G][I][0];
			cdest[1] = cg_matrix0[B][R][1] | cg_matrix1[G][I][1];
			cdest[2] = cg_matrix0[B][R][2] | cg_matrix1[G][I][2];
			cdest[3] = cg_matrix0[B][R][3] | cg_matrix1[G][I][3];
			cdest[4] = cg_matrix0[B][R][4] | cg_matrix1[G][I][4];
			cdest[5] = cg_matrix0[B][R][5] | cg_matrix1[G][I][5];
			cdest[6] = cg_matrix0[B][R][6] | cg_matrix1[G][I][6];
			cdest[7] = cg_matrix0[B][R][7] | cg_matrix1[G][I][7];
		}
		dest += 640;
	}
}

void CRTC::draw_640x400x4screen()
{
	uint8 B, R;
	uint32 dest = 0;
	
	for(int y = 0; y < 400; y++) {
		for(int x = 0; x < 80; x++) {
			uint16 src = map_addr[y][x];
			uint32 dest2 = dest + map_hdsc[y][x];
			dest += 8;
			
			B = (cgreg[0x18] & 0x01) ? ((src & 0x4000) ? vram_g[src & 0x3fff] : vram_b[src]) : 0;
			R = (cgreg[0x18] & 0x02) ? ((src & 0x4000) ? vram_i[src & 0x3fff] : vram_r[src]) : 0;
			cg[dest2++] = cg_matrix0[B][R][0];
			cg[dest2++] = cg_matrix0[B][R][1];
			cg[dest2++] = cg_matrix0[B][R][2];
			cg[dest2++] = cg_matrix0[B][R][3];
			cg[dest2++] = cg_matrix0[B][R][4];
			cg[dest2++] = cg_matrix0[B][R][5];
			cg[dest2++] = cg_matrix0[B][R][6];
			cg[dest2++] = cg_matrix0[B][R][7];
		}
	}
}

void CRTC::draw_640x400x16screen()
{
	uint8 B, R, G, I;
	uint32 dest = 0;
	
	for(int y = 0; y < 400; y++) {
		for(int x = 0; x < 80; x++) {
			uint16 src = map_addr[y][x];
			uint32 dest2 = dest + map_hdsc[y][x];
			dest += 8;
			
			B = vram_b[src];
			R = vram_r[src];
			G = vram_g[src];
			I = vram_i[src];
			cg[dest2++] = cg_matrix0[B][R][0] | cg_matrix1[G][I][0];
			cg[dest2++] = cg_matrix0[B][R][1] | cg_matrix1[G][I][1];
			cg[dest2++] = cg_matrix0[B][R][2] | cg_matrix1[G][I][2];
			cg[dest2++] = cg_matrix0[B][R][3] | cg_matrix1[G][I][3];
			cg[dest2++] = cg_matrix0[B][R][4] | cg_matrix1[G][I][4];
			cg[dest2++] = cg_matrix0[B][R][5] | cg_matrix1[G][I][5];
			cg[dest2++] = cg_matrix0[B][R][6] | cg_matrix1[G][I][6];
			cg[dest2++] = cg_matrix0[B][R][7] | cg_matrix1[G][I][7];
		}
	}
}

void CRTC::create_320x200map()
{
	uint8 HDSC = cgreg[0x0f] & 0x07;
	uint16 SAD0 = cgreg[0x10] | ((cgreg[0x11] & 0x1f) << 8);
	uint16 SAD1 = cgreg[0x12] | ((cgreg[0x13] & 0x1f) << 8);
	uint16 SAD2 = cgreg[0x14] | ((cgreg[0x15] & 0x1f) << 8);
	uint16 SLN1 = cgreg[0x16] | ((cgreg[0x17] & 0x01) << 8);
	if(SLN1 > 200) SLN1 = 200;
	
	for(int y = 0; y < SLN1; y++) {
		for(int x = 0; x < 40; x++) {
			map_hdsc[y][x] = HDSC;
			map_addr[y][x] = (SAD0++) & 0x1fff;
			if(SAD0 > SAD1) SAD0 = 0;
		}
	}
	for(int y = SLN1; y < 200; y++) {
		for(int x = 0; x < 40; x++) {
			map_hdsc[y][x] = 0;
			map_addr[y][x] = (SAD2++) & 0x1fff;
		}
	}
}

void CRTC::create_640x200map()
{
	uint8 HDSC = cgreg[0x0f] & 0x07;
	uint16 SAD0, SAD1, SAD2, SLN1;
	
	if((((cgreg[0x10] & 0x0f) == 0x0d && 0x56 <= cgreg[0x11] && cgreg[0x11] <= 0x72) || (cgreg[0x10] == 0x05 && cgreg[0x11] == 0x72)) &&
	   cgreg[0x12] == 0x7f && cgreg[0x13] == 0x3e && cgreg[0x14] == 0x00 && cgreg[0x15] == 0x00 && cgreg[0x16] == 0xff && cgreg[0x17] == 0x01) {
		// BASIC-M25 fppb`
		SAD0 = (cgreg[0x10] | ((cgreg[0x11] & 0x7f) << 8)) - 0x40e0;
		SAD1 = cgreg[0x12] | ((cgreg[0x13] & 0x7f) << 8);
		SAD2 = cgreg[0x14] | ((cgreg[0x15] & 0x7f) << 8);
		SLN1 = cgreg[0x16] | ((cgreg[0x17] & 0x01) << 8);
	}
	else {
		SAD0 = cgreg[0x10] | ((cgreg[0x11] & 0x3f) << 8);
		SAD1 = cgreg[0x12] | ((cgreg[0x13] & 0x3f) << 8);
		SAD2 = cgreg[0x14] | ((cgreg[0x15] & 0x3f) << 8);
		SLN1 = cgreg[0x16] | ((cgreg[0x17] & 0x01) << 8);
	}
	if(SLN1 > 200) SLN1 = 200;
	
	for(int y = 0; y < SLN1; y++) {
		for(int x = 0; x < 80; x++) {
			map_hdsc[y][x] = HDSC;
			map_addr[y][x] = (SAD0++) & 0x3fff;
			if(SAD0 > SAD1) SAD0 = 0;
		}
	}
	for(int y = SLN1; y < 200; y++) {
		for(int x = 0; x < 80; x++) {
			map_hdsc[y][x] = 0;
			map_addr[y][x] = (SAD2++) & 0x3fff;
		}
	}
	map_init = false;
}

void CRTC::create_640x400map()
{
	uint8 HDSC = cgreg[0x0f] & 0x07;
	uint16 SAD0 = cgreg[0x10] | ((cgreg[0x11] & 0x7f) << 8);
	uint16 SAD1 = cgreg[0x12] | ((cgreg[0x13] & 0x7f) << 8);
	uint16 SAD2 = cgreg[0x14] | ((cgreg[0x15] & 0x7f) << 8);
	uint16 SLN1 = cgreg[0x16] | ((cgreg[0x17] & 0x01) << 8);
	if(SLN1 > 400) SLN1 = 400;
	
	for(int y = 0; y < SLN1; y++) {
		for(int x = 0; x < 80; x++) {
			map_hdsc[y][x] = HDSC;
			map_addr[y][x] = (SAD0++) & 0x7fff;
			if(SAD0 > SAD1) SAD0 = 0;
		}
	}
	for(int y = SLN1; y < 400; y++) {
		for(int x = 0; x < 80; x++) {
			map_hdsc[y][x] = 0;
			map_addr[y][x] = (SAD2++) & 0x7fff;
		}
	}
	map_init = false;
}
