/*
	SHARP MZ-2500 Emulator 'EmuZ-2500'
	(Skelton for Z-80 PC Emulator)

	Author : Takeda.Toshiya
	Date   : 2004.08.31 -

	[ timer (i8253) ]
*/

#include <stdio.h>
#include "timer.h"
#include "z80.h"
#include "../config.h"

extern config_t config;

void TIMER::initialize()
{
	for(int i = 0; i < 3; i++) {
		counter[i].count = 0x10000;
		counter[i].count_reg = 0;
		counter[i].ctrl_reg = 0x34;
		counter[i].mode = 3;
		counter[i].mode0_flag = false;
		counter[i].r_cnt = 0;
		counter[i].delay = 0;
		counter[i].start = false;
	}
	
	// 32usec -> 31.25KHz
	int regist_id;
	vm->regist_callback(this, EVENT_TIMER, 32, true, &regist_id);
	
	fast_timer = config.fast_timer;
}

void TIMER::update_config()
{
	fast_timer = config.fast_timer;
}

#define COUNT_VALUE(n) ((!counter[n].count_reg) ? 0x10000 : (counter[n].mode == 3 && counter[n].count_reg == 1) ? 0x10001 : counter[n].count_reg)

void TIMER::write_io8(uint16 addr, uint8 data)
{
	switch(addr & 0xff)
	{
		case 0xe4: // count reg #0
		case 0xe5: // count reg #1
		case 0xe6: // count reg #2
		{
			int ch = addr & 3;
			if((counter[ch].ctrl_reg & 0x30) == 0x10)
				counter[ch].count_reg = data;
			else if((counter[ch].ctrl_reg & 0x30) == 0x20)
				counter[ch].count_reg = data << 8;
			else
				counter[ch].count_reg = ((counter[ch].count_reg & 0xff00) >> 8) | (data << 8);
			// start count now if mode is not 1 or 5
			if(!(counter[ch].mode == 1 || counter[ch].mode == 5))
				counter[ch].start = true;
			// copy to counter at 1st clock, start count at 2nd clock
			counter[ch].delay = 1;
			counter[ch].mode0_flag = false;
			
			// thexder patch
//			if((addr & 0xff) == 0xe4 && counter[0].count_reg == 0x34 && counter[0].ctrl_reg == 0x34) {
//				if(vm->cpu->prvPC == 0x8ca6 || vm->cpu->prvPC == 0xc015)
//					counter[0].count_reg = 0x1a;
//			}
			break;
		}
		case 0xe7:
		{
			// ctrl reg
			if((data & 0xc0) == 0xc0)
				break;
			uint8 ch = (data >> 6) & 0x3;
			
			if((data & 0x30) == 0x00) {
				// counter latch
				if((counter[ch].ctrl_reg & 0x30) == 0x10) {
					// lower byte
					counter[ch].latch = (uint16)counter[ch].count;
					counter[ch].r_cnt = 1;
				}
				else if((counter[ch].ctrl_reg & 0x30) == 0x20) {
					// upper byte
					counter[ch].latch = (uint16)(counter[ch].count >> 8);
					counter[ch].r_cnt = 1;
				}
				else {
					// lower -> upper
					counter[ch].latch = (uint16)counter[ch].count;
					counter[ch].r_cnt = 2;
				}
			}
			else {
				counter[ch].ctrl_reg = data;
				counter[ch].mode = ((data & 0xe) == 0xc) ? 2 : ((data & 0xe) == 0xe) ? 3 : (data & 0xe) >> 1;
				counter[ch].count_reg = 0;
			}
			break;
		}
		case 0xf0: case 0xf1: case 0xf2: case 0xf3:
			// gate signal
			if(!(counter[0].mode == 0 || counter[0].mode == 4))
				counter[0].count = COUNT_VALUE(0);
			if(!(counter[1].mode == 0 || counter[1].mode == 4))
				counter[1].count = COUNT_VALUE(1);
			counter[0].start = counter[1].start = true;
			counter[0].mode0_flag = counter[1].mode0_flag = false;
			break;
	}
}

uint8 TIMER::read_io8(uint16 addr)
{
	switch(addr & 0xff)
	{
		case 0xe4: // count reg #0
		case 0xe5: // count reg #1
		case 0xe6: // count reg #2
		{
			int ch = addr & 3;
			if(!counter[ch].r_cnt) {
				// not latched (through current count)
				if((counter[ch].ctrl_reg & 0x30) == 0x10) {
					// lower byte
					counter[ch].latch = (uint16)counter[ch].count;
					counter[ch].r_cnt = 1;
				}
				else if((counter[ch].ctrl_reg & 0x30) == 0x20) {
					// upper byte
					counter[ch].latch = (uint16)(counter[ch].count >> 8);
					counter[ch].r_cnt = 1;
				}
				else {
					// lower -> upper
					counter[ch].latch = (uint16)counter[ch].count;
					counter[ch].r_cnt = 2;
				}
			}
			uint8 val = (uint8)(counter[ch].latch & 0xff);
			counter[ch].latch >>= 8;
			counter[ch].r_cnt--;
			return val;
		}
	}
	return 0xff;
}

void TIMER::event_callback(int event_id, int err)
{
	bool carry = update_count(0);
	if(fast_timer)
		carry |= update_count(0);
	if(carry) {
		if(update_count(1))
			update_count(2);
	}
}

bool TIMER::update_count(int ch)
{
	bool carry = false;
	
	if(!counter[ch].start)
		return false;
	if(counter[ch].delay) {
		if(counter[ch].delay == 1)
			counter[ch].count = COUNT_VALUE(ch);
		counter[ch].delay--;
		return false;
	}
	if(--counter[ch].count == 0) {
		if(counter[ch].mode == 0 || counter[ch].mode == 2 || counter[ch].mode == 3)
			counter[ch].count = COUNT_VALUE(ch);
		else
			counter[ch].start = false;
		if(!(counter[ch].mode == 0 && counter[ch].mode0_flag))
			carry = true;
		counter[ch].mode0_flag = true;
		if(ch == 0)
			vm->request_interrupt(IRQ_TIMER, vector, false);
	}
	return carry;
}

