/*
	Skelton for Z-80 PC Emulator

	Author : Takeda.Toshiya
	Date   : 2006.08.27 -

	[ network (win32) ]
*/

#include <stdio.h>
#include "emu.h"
#include "vm/vm.h"

void EMU::initialize_socket()
{
	// init winsock
	WSADATA wsaData;
	WSAStartup(0x0101, &wsaData);
	
	// init sockets
	for(int i = 0; i < SOCKET_MAX; i++) {
		soc[i] = INVALID_SOCKET;
		socket_delay[i] = 0;
		recv_r_ptr[i] = recv_w_ptr[i] = 0;
	}
}

void EMU::release_socket()
{
	// release sockets
	for(int i = 0; i < SOCKET_MAX; i++) {
		if(soc[i] != INVALID_SOCKET) {
			shutdown(soc[i], 2);
			closesocket(soc[i]);
		}
	}
	
	// release winsock
	WSACleanup();
}

void EMU::socket_connected(int ch)
{
	out_debug("socket_connected(%d)\n", ch);
	
	// winmain notify that network is connected
	vm->network_connected(ch);
}

void EMU::socket_disconnected(int ch)
{
	out_debug("socket_disconnected(%d)\n", ch);
	
	// winmain notify that network is disconnected
#if 1
	if(!socket_delay[ch])
		socket_delay[ch] = 1;//56;
	out_debug("socket_disconnected(%d) delay = %d\n", ch, socket_delay[ch]);
#else
	vm->network_disconnected(ch);
#endif
}

void EMU::update_socket()
{
	for(int i = 0; i < SOCKET_MAX; i++) {
		if(recv_r_ptr[i] < recv_w_ptr[i]) {
			// get buffer
			int size0, size1;
			uint8* buf0 = vm->get_recvbuffer0(i, &size0, &size1);
			uint8* buf1 = vm->get_recvbuffer1(i);
			
			int size = recv_w_ptr[i] - recv_r_ptr[i];
			if(size > size0 + size1)
				size = size0 + size1;
			char* src = &recv_buffer[i][recv_r_ptr[i]];
			recv_r_ptr[i] += size;
			
			if(size <= size0)
				_memcpy(buf0, src, size);
			else {
				_memcpy(buf0, src, size0);
				_memcpy(buf1, src + size0, size - size0);
			}
			vm->inc_recvbuffer_ptr(i, size);
//			out_debug("buffer transfered %d bytes\n", size);
		}
		else if(socket_delay[i] != 0) {
			 if(--socket_delay[i] == 0) {
				out_debug("socket_disconnected(%d) done\n", i);
				vm->network_disconnected(i);
			}
		}
	}
}

bool EMU::init_socket_tcp(int ch)
{
	out_debug("init_socket_tcp(%d)\n", ch);
	
	is_tcp[ch] = true;
	
	if(soc[ch] != INVALID_SOCKET)
		disconnect_socket(ch);
	if((soc[ch] = socket(PF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET) {
		out_debug("init_socket_tcp(%d)\tsocket() failed\n", ch);
		return false;
	}
	if(WSAAsyncSelect(soc[ch], main_window_handle, WM_SOCKET0 + ch, FD_CONNECT | FD_WRITE | FD_READ | FD_CLOSE) == SOCKET_ERROR) {
		out_debug("init_socket_tcp(%d)\tWSAAsyncSelect() failed\n", ch);
		closesocket(soc[ch]);
		soc[ch] = INVALID_SOCKET;
		return false;
	}
	recv_r_ptr[ch] = recv_w_ptr[ch] = 0;
	return true;
}

bool EMU::init_socket_udp(int ch)
{
	out_debug("init_socket_udp(%d)\n", ch);
	
	is_tcp[ch] = false;
	
	disconnect_socket(ch);
	if((soc[ch] = socket(PF_INET, SOCK_DGRAM, 0)) == INVALID_SOCKET) {
		out_debug("init_socket_udp(%d)\tsocket() failed\n", ch);
		return false;
	}
	if(WSAAsyncSelect(soc[ch], main_window_handle, WM_SOCKET0 + ch, FD_CONNECT | FD_WRITE | FD_READ | FD_CLOSE) == SOCKET_ERROR) {
		out_debug("init_socket_udp(%d)\tWSAAsyncSelect() failed\n", ch);
		closesocket(soc[ch]);
		soc[ch] = INVALID_SOCKET;
		return false;
	}
	recv_r_ptr[ch] = recv_w_ptr[ch] = 0;
	return true;
}

bool EMU::connect_socket(int ch, uint32 ipaddr, int port)
{
	out_debug("connect_socket(%d, %8x, %d)\n", ch, ipaddr, port);
	
	struct sockaddr_in tcpaddr;
	tcpaddr.sin_family = AF_INET;
	tcpaddr.sin_addr.s_addr = ipaddr;
	tcpaddr.sin_port = htons((unsigned short)port);
	_memset(tcpaddr.sin_zero, (int)0, sizeof(tcpaddr.sin_zero));
	
	if(connect(soc[ch], (struct sockaddr *)&tcpaddr, sizeof(tcpaddr)) == SOCKET_ERROR) {
		if(WSAGetLastError() != WSAEWOULDBLOCK) {
			out_debug("connect_socket(%d, %8x, %d)\tconnect() failed\n", ch, ipaddr, port);
			return false;
		}
	}
	return true;
}

void EMU::disconnect_socket(int ch)
{
	out_debug("disconnect_socket(%d)\n", ch);
	
	if(soc[ch] != INVALID_SOCKET) {
		shutdown(soc[ch], 2);
		closesocket(soc[ch]);
		soc[ch] = INVALID_SOCKET;
	}
	vm->network_disconnected(ch);
}

bool EMU::listen_socket(int ch)
{
	out_debug("listen_socket(%d)\n", ch);
	
	return false;
}

void EMU::send_data_tcp(int ch)
{
	out_debug("send_data_tcp(%d)\n", ch);
	
	if(is_tcp[ch])
		send_data(ch);
}

void EMU::send_data_udp(int ch, uint32 ipaddr, int port)
{
	out_debug("send_data_udp(%d, %8x, %d)\n", ch, ipaddr, port);
	
	if(!is_tcp[ch]) {
		udpaddr[ch].sin_family = AF_INET;
		udpaddr[ch].sin_addr.s_addr = ipaddr;
		udpaddr[ch].sin_port = htons((unsigned short)port);
		_memset(udpaddr[ch].sin_zero, (int)0, sizeof(udpaddr[ch].sin_zero));
		
		send_data(ch);
	}
}

void EMU::send_data(int ch)
{
	out_debug("send_data(%d)\n", ch);
	
	// loop while send buffer is not emupty or not WSAEWOULDBLOCK
	while(1) {
		// get send buffer and data size
		int size;
		uint8* buf = vm->get_sendbuffer(ch, &size);
		
		if(!size) {
			out_debug("send_data(%d)\tno data\n", ch);
			return;
		}
		if(is_tcp[ch]) {
			if((size = send(soc[ch], (char *)buf, size, 0)) == SOCKET_ERROR) {
				// if WSAEWOULDBLOCK, WM_SOCKET* and FD_WRITE will come later
				if(WSAGetLastError() != WSAEWOULDBLOCK) {
					out_debug("send_data(%d)\tsend() failed\n", ch);
					disconnect_socket(ch);
					socket_disconnected(ch);
				}
				return;
			}
		}
		else {
			if((size = sendto(soc[ch], (char *)buf, size, 0, (struct sockaddr *)&udpaddr[ch], sizeof(udpaddr[ch]))) == SOCKET_ERROR) {
				// if WSAEWOULDBLOCK, WM_SOCKET* and FD_WRITE will come later
				if(WSAGetLastError() != WSAEWOULDBLOCK) {
					out_debug("send_data(%d)\tsend() failed\n", ch);
					disconnect_socket(ch);
					socket_disconnected(ch);
				}
				return;
			}
		}
		out_debug("send_data(%d)\tsend() = %d\n", ch, size);
		for(int i = 0; i < size; i++)
			out_debug("%c", (char *)buf[i]);
		out_debug("\n");
		vm->inc_sendbuffer_ptr(ch, size);
	}
}

void EMU::recv_data(int ch)
{
	out_debug("recv_data(%d)\n", ch);
	
	if(is_tcp[ch]) {
		int size = SOCKET_BUFFER_MAX - recv_w_ptr[ch];
		char* buf = &recv_buffer[ch][recv_w_ptr[ch]];
		if((size = recv(soc[ch], buf, size, 0)) == SOCKET_ERROR) {
			out_debug("recv_data(%d) recv() failed\n", ch);
			disconnect_socket(ch);
			socket_disconnected(ch);
			return;
		}
		recv_w_ptr[ch] += size;
		out_debug("recv_data(%d) recv() = %d\n", ch, size);
	}
	else {
#if 1
		SOCKADDR_IN addr;
		int len = sizeof(addr);
		int size = SOCKET_BUFFER_MAX - recv_w_ptr[ch];
		char* buf = &recv_buffer[ch][recv_w_ptr[ch]];
		
		if(size < 8) {
			out_debug("recv_data(%d) buffer full\n", ch);
			return;
		}
		if((size = recvfrom(soc[ch], buf + 8, size - 8, 0, (struct sockaddr *)&addr, &len)) == SOCKET_ERROR) {
			out_debug("recv_data(%d) recv() failed\n", ch);
			disconnect_socket(ch);
			socket_disconnected(ch);
			return;
		}
		size += 8;
		buf[0] = size >> 8;
		buf[1] = size;
		buf[2] = (char)addr.sin_addr.s_addr;
		buf[3] = (char)(addr.sin_addr.s_addr >> 8);
		buf[4] = (char)(addr.sin_addr.s_addr >> 16);
		buf[5] = (char)(addr.sin_addr.s_addr >> 24);
		buf[6] = (char)addr.sin_port;
		buf[7] = (char)(addr.sin_port >> 8);
		recv_w_ptr[ch] += size;
		out_debug("recv_data(%d) recv() = %d\n", ch, size);
#else
		// get buffer
		int size0, size1;
		uint8* buf0 = vm->get_recvbuffer0(ch, &size0, &size1);
		uint8* buf1 = vm->get_recvbuffer1(ch);
		int size = size0 + size1;
		out_debug("size = %d, %d\n", size0, size1);
		
		SOCKADDR_IN addr;
		int len = sizeof(addr);
		char buf[0x2000];
		
		if(size < 8) {
			out_debug("recv_data(%d) buffer full\n", ch);
			return;
		}
		if((size = recvfrom(soc[ch], buf + 8, size - 8, 0, (struct sockaddr *)&addr, &len)) == SOCKET_ERROR) {
			out_debug("recv_data(%d) recv() failed\n", ch);
			disconnect_socket(ch);
			socket_disconnected(ch);
			return;
		}
		size += 8;
		buf[0] = size >> 8;
		buf[1] = size;
		buf[2] = (char)addr.sin_addr.s_addr;
		buf[3] = (char)(addr.sin_addr.s_addr >> 8);
		buf[4] = (char)(addr.sin_addr.s_addr >> 16);
		buf[5] = (char)(addr.sin_addr.s_addr >> 24);
		buf[6] = (char)addr.sin_port;
		buf[7] = (char)(addr.sin_port >> 8);
		
		out_debug("recv_data(%d) recv() = %d\n", ch, size);
		for(int i = 0; i < size; i++)
			out_debug("%c", buf[i]);
		out_debug("\n");
		
		if(size <= size0)
			_memcpy(buf0, buf, size);
		else {
			_memcpy(buf0, buf, size0);
			_memcpy(buf1, buf + size0, size - size0);
		}
		vm->inc_recvbuffer_ptr(ch, size);
#endif
	}
}

