/************************************************************************\
 * File Version Information
 * $Header: /Altair32v3/88hdsk.c 1     3/28/14 10:04p Racini $
 ************************************************************************/
/************************************************************************\
  MITS Altair Emulator
  MITS Hard Disk (88-HDSK) emulation

  General Comments:

	The 88-HDSK from MITS/Pertec consists of a 5mb removable platter and
	a fixed 5mb platter. Each platter is double sided. Head 0 and 1 are the
	top and bottom surface of the removable platter and head 2 and 3 are the
	top and bottom surface of the fixed platter. Hard disk BASIC treats the
	two platters as two separate drives. Each platter has 406 cylinders
	with 24 sectors per track and 256 bytes per sector.

	The disk image file starts with head 0, track 0, sector 0 (0,0,0) through 
	(0,0,23), followed by head 1, track 0, sector 0 (1,0,0) through (1,0,23).
	The pattern then repeats starting with (0,1,0).

	The external hard disk is accessed through eight ports of a 4-PIO card 
	at I/O addresses A0h-A7h. 

	Written by Mike Douglas March, 2014
	Disk images provided by Martin Eberhard 

Change Log:
  2014/03/22  MWD -- Initial release
  2016/02/20  RAC -- RELEASE MARKER -- v3.34.0900
\************************************************************************/
#define _CRT_SECURE_NO_WARNINGS			// BAD thing to do
#include <windows.h>	// required for all Windows applications
#include <windowsx.h>	// message cracker macros
#include <commdlg.h>	// Windows common dialogs
#include <commctrl.h>	// Windows common controls
#include <stdlib.h>		// C standard lib
#include <string.h>		// C-lib - strings
#include <stdio.h>		// C-lib 
#include "altair32.h"	// includes "resource.h" and exports from other major modules
#include "comcthlp.h"	// common control macros
#include "i8080.h"		// processor emulation

/**  Typedefs & Defines  **************************************/
#define HDSK_SECTOR_SIZE		256				/* size of sector */
#define HDSK_SECTORS_PER_TRACK	24				/* sectors per track */
#define	HDSK_NUM_HEADS			2				/* heads per disk */
#define	HDSK_NUM_TRACKS			406				/* tracks per surface */
#define HDSK_TRACK_SIZE			(HDSK_SECTOR_SIZE * HDSK_SECTORS_PER_TRACK)
#define HDSK_CYLINDER_SIZE		(HDSK_TRACK_SIZE * 2)
#define HDSK_CAPACITY			(HDSK_CYLINDER_SIZE * HDSK_NUM_TRACKS)	
#define HDSK_NUMBER				8				/* number of hard disks	*/
#define	IO_IN					0				/* I/O operation is input */
#define	IO_OUT					1				/* I/O operation is output */

// Disk controller commands are in upper nibble of command high byte.

#define	CMD_SHIFT		4		// shift right 4 places
#define	CMD_MASK		0x0f	// mask after shifting
#define CMD_SEEK		0		// seek to track
#define	CMD_WRITE_SEC	2		// write sector from buf n
#define	CMD_READ_SEC	3		// read sector into buf n
#define	CMD_WRITE_BUF	4		// load buffer n from CPU
#define	CMD_READ_BUF	5		// read buffer n into CPU
#define	CMD_READ_STATUS	6		// read controller IV byte
#define	CMD_SET_IV_BYTE	8		// write controller IV byte
#define	CMD_READ_UNFMT	10		// read unformatted sector
#define	CMD_FORMAT		12
#define	CMD_INITIALIZE	14

// Other disk controller bit fields

#define	UNIT_SHIFT		2		// shift right 2 places
#define	UNIT_MASK		0x03	// mask after shifting

#define	BUFFER_MASK		0x03	// mask - no shift needed

#define	TRACK_SHIFTH	8		// shift left 8 places into MSbyte
#define	TRACK_MASKH		0x01	// msb of track number 
#define	TRACK_MASKL		0xff	// entire lsb of track number

#define	HEAD_SHIFT		5		// shift right 5 places
#define	HEAD_MASK		0x03	// mask after shifting (no heads 4-7)

#define	SECTOR_MASK		0x1f	// mask - no shift needed

// Command status equates

#define	CSTAT_WRITE_PROTECT		0x80	// disk is write protected
#define CSTAT_NOT_READY			0x01	// drive not ready
#define CSTAT_BAD_SECTOR		0x02	// invalid sector number

/**  Forward Prototypes  **************************************/

static void doRead(void);
static void doWrite(void);

/**  Module Globals - Private  ********************************/
static uint32 selectedDisk = 0;			// current active disk
static uint32 selectedSector = 0;		// current sector
static uint32 selectedTrack = 0;		// current track
static uint32 selectedHead = 0;			// current head
static uint32 selectedBuffer = 0;		// current buffer # in use
static uint32 bufferIdx = 0;			// current index into selected buffer
static uint32 maxBufferIdx = 256;		// maximum buffer index allowed
static uint32 cmdLowByte = 0;			// low byte of command
static FILE *diskFile[] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };

//  Controller status bytes

static uint8 cstat = 0;			// command status from controller

// The hard disk controller support four 256 byte disk buffers */

static uint8 diskBuf1[HDSK_SECTOR_SIZE];
static uint8 diskBuf2[HDSK_SECTOR_SIZE];
static uint8 diskBuf3[HDSK_SECTOR_SIZE];
static uint8 diskBuf4[HDSK_SECTOR_SIZE];
static uint8 *diskBuf[] = { diskBuf1, diskBuf2, diskBuf3, diskBuf4 };


/**************************************************************\
*   I/O Instruction Handlers  *********************************|
***************************************************************|
*
*	I/O instruction handlers are called from the CPU module
*  	when an IN or OUT instruction is issued.
*
*  	Each function is passed an 'io' flag, where 0 means a read
*  	from the port, and 1 means a write to the port.  On input,
*	the actual input is passed as the return value.  On output,
*	'data' is written to the device.
*
\**************************************************************/

/*-------------------------------------------------------------------------------------
 hdReturnReady - common I/O handler for several hard disk status ports which set 
	bit 7 when the corresponding hard disk function is ready. In the emulator,
	we're always ready for the next step, so we simply return ready all the time.

	0xA0 - CREADY register. Accessed through the status/control register of 4-PIO
		port 1-A. Returns the "ready for command" status byte.

	0xA2 - ACSTA register. Accessed through the status/control register of 4-PIO
		port 1-B. Returns the "command received" status byte.
	
	0xA4 - CDSTA register. Accessed through the status/control register of 4-PIO 
		port 2-A. Returns the "command data available" status byte.

	0xA6 - ADSTA register. Accessed through the status/control register of 4-PIO 
		port 2-B. Returns the "available to write" status byte.
		
---------------------------------------------------------------------------------------*/
int hdRetRdy(int io, int data)
{
	return(0x80);		// always indicate ready

// output operations have no effect
}

/*------------------------------------------------------------
 hdCstat (0xA1) CSTAT register. Accessed through the 
	data register of 4-PIO port 1-A.

	Comments:	Returns error code byte of the most recent 
				operation. Reading this byte also clears
				the CRDY bit, but this isn't actually done
				in the emulation since we're always ready.
-------------------------------------------------------------*/
int hdCstat(int io, int data)
{
	return(cstat);

// output operations have no effect
}

/*------------------------------------------------------------
 hdAcmd (0xA3) ACMD register. Accessed  through the 
	data register of 4-PIO port 1-B.

	Comments:	The high byte of a command is written to
				this register and initiates the command.
				The low byte of a command is assumed to
				have already been written and stored in
				cmdLowByte;
-------------------------------------------------------------*/
int hdAcmd(int io, int data)
{
	uint32 command;				// command field from command msb
	uint32 unit;				// unit number from command msb
	uint32 buffer;				// buffer number from command msb

// if not an OUT command, exit

	if (io != IO_OUT)
		return(0);

// extract command and possible unit and buffer fields.

	cstat = 0;					// assume command success
	command = (data >> CMD_SHIFT) & CMD_MASK;
	unit = (data >> UNIT_SHIFT) & UNIT_MASK;
	buffer = data & BUFFER_MASK;

// SEEK command. Updated selectedTrack.

	if (command == CMD_SEEK) {
		selectedTrack = cmdLowByte + ((data & TRACK_MASKH) << TRACK_SHIFTH);
		if (selectedTrack >= HDSK_NUM_TRACKS)
			selectedTrack = HDSK_NUM_TRACKS-1;
	}

// READ, READ UNFORMATTED or WRITE SECTOR command.

	else if ((command==CMD_WRITE_SEC) || (command==CMD_READ_SEC) || (command==CMD_READ_UNFMT)) {
		selectedHead = (cmdLowByte >> HEAD_SHIFT) & HEAD_MASK;
		selectedDisk = (selectedHead >> 1) + unit * 2 ;
		selectedSector = cmdLowByte & SECTOR_MASK;
		selectedBuffer = buffer;
		if (((selectedHead & 1) == 0) && (selectedTrack== 0) && (selectedSector == 0)) {
			cstat = CSTAT_WRITE_PROTECT & selectedTrack;
		}
		if (diskFile[selectedDisk] == NULL)		// make sure a file is attached
			cstat = CSTAT_NOT_READY;
		else {
			if (command == CMD_WRITE_SEC)
				doWrite();
			else
				doRead();
		}
	}

// READ or WRITE BUFFER command. Initiates reading/loading specified buffer. 

	else if ((command == CMD_WRITE_BUF) || (command == CMD_READ_BUF)) {
		selectedBuffer = buffer;
		maxBufferIdx = cmdLowByte;
		if (maxBufferIdx == 0) 
			maxBufferIdx = 256;
		bufferIdx = 0;
	}

// READ STATUS command (read IV byte)

	else if (command == CMD_READ_STATUS) {
	}

// SET IV byte command

	else if (command == CMD_SET_IV_BYTE) {
	}

// FORMAT command

	else if (command == CMD_FORMAT) {
	}

// INITIALIZE command

	else if (command == CMD_INITIALIZE) {
	}

	return(0);
}

/*------------------------------------------------------------
 hdCdata (0xA5) Cdata register. Accessed through the 
	data register of 4-PIO port 1-B.
	
	Comments:	Returns data from the read buffer
-------------------------------------------------------------*/
int hdCdata(int io, int data)
{
	if (io == IO_IN) {
		if (bufferIdx < maxBufferIdx)
			return(diskBuf[selectedBuffer][bufferIdx++]);
	}
	return(0);

// output operations have no effect
}

/*------------------------------------------------------------
 hdAdata (0xA7) ADATA register. Accessed through the 
	data register of 4-PIO port 2-B.
	
	Comments:	Accepts data into the current buffer 
			    and is also the low byte of a command.
-------------------------------------------------------------*/
int hdAdata(int io, int data)
{
	if (io == IO_OUT) {
		cmdLowByte = data & 0xff;
		if (bufferIdx < maxBufferIdx)
			diskBuf[selectedBuffer][bufferIdx++] = data;
	}
	return(0);
}


/**************************************************************\
*   Public Routines  ******************************************|
\**************************************************************/
/*-- MITS_Hdsk_Shutdown ---------------------------------------------
	
	Params:		none
	Uses:		diskFile[] array
	Returns:	none
	Comments:	Close any open disk image files
-------------------------------------------------------------*/
void MITS_Hdsk_Shutdown(void)
{
	int i;

	for (i=0; i < HDSK_NUMBER; i++){
		if(diskFile[i] != NULL)
			fclose(diskFile[i]);
		diskFile[i] = NULL;
	}
}


/*-- MITS_Hdsk_OpenImageFile  --------------------------------
	Called from main window message handler (menu item) to open
	MITS Hard Disk image files. The open file dialog box is
	displayed over and over with a drive number in the title
	bar until the operator hits cancel. This allows opening
	of multiple drives.
	
	Params:		None
	Uses:		diskFile[] array
	Returns:	nothing
	Comments:	
-------------------------------------------------------------*/
void MITS_Hdsk_OpenImageFile(void)
{
	BOOL bNewDisk = FALSE;
	char DlgCaption[] = {"Open MITS Hard Disk File for Drive #"};
	char outfile[_MAX_PATH];
	FILE *pfp;
	char *driveNumPtr;	
	uint32 driveNum;
	static  char szFilter[] = "MITS Hard Disk Image Files (*.DSK)\0*.dsk\0" \
                              "All Files (*.*)\0*.*\0\0";

/* Close all open disks, then put up the open file dialog to allow opening
	multiple drives until the user cancels. */

	MITS_Hdsk_Shutdown();					// close open drives
	driveNumPtr = strrchr(DlgCaption, '#');			// 
	for (driveNum = 0; driveNum < 8; driveNum++) {
		*driveNumPtr = driveNum + '0';		// put drive number in caption
		if(!GetFilenameFromUser(DlgCaption, szFilter, outfile, &bNewDisk))
			return;							// GetOpenFileName was cancelled
		if(bNewDisk)						// user forced name of a non-existent file
			return;
		if((pfp=fopen(outfile, "rb+")) == NULL)
			return;							// file open error
		diskFile[driveNum] = pfp;
	}
}

/*--  doRead  -------------------------------------------------
	Performs read from MITS Hard Disk image file
	
	Params:		nothing
	Uses:		selectedTrack, selectedHead, selectedSector
				selectedDisk, diskFile[], diskBuf[]
	Returns:	nothing (updates cstat directly)
	Comments:	
-------------------------------------------------------------*/
static void doRead(void)
{
	uint32 fileOffset;
	
	fileOffset = HDSK_CYLINDER_SIZE * selectedTrack + 
					HDSK_TRACK_SIZE * (selectedHead & 0x01) + 
					HDSK_SECTOR_SIZE * selectedSector;
	if (fseek(diskFile[selectedDisk], fileOffset, SEEK_SET)) 
		cstat = CSTAT_NOT_READY;						/* seek error */
	else if (fread(diskBuf[selectedBuffer], HDSK_SECTOR_SIZE, 1, diskFile[selectedDisk]) != 1) 
		cstat = CSTAT_NOT_READY;						/* read error */
}


/*--  doWrite  ------------------------------------------------
	Performs write to MITS Hard Disk image file
	
	Params:		none
	Uses:		selectedTrack, selectedHead, selectedSector
				selectedDisk, diskFile[], diskBuf[]
	Returns:	nothing (updates cstat directly)
	Comments:	
-------------------------------------------------------------*/
static void doWrite(void)
{
	uint32 fileOffset;
	
	fileOffset = HDSK_CYLINDER_SIZE * selectedTrack + 
					HDSK_TRACK_SIZE * (selectedHead & 0x01) + 
					HDSK_SECTOR_SIZE * selectedSector;
	if (fseek(diskFile[selectedDisk], fileOffset, SEEK_SET)) 
		cstat = CSTAT_NOT_READY;					/* seek error */
	else if (fwrite(diskBuf[selectedBuffer], HDSK_SECTOR_SIZE, 1, diskFile[selectedDisk]) != 1)
		cstat = CSTAT_NOT_READY;					/* write error */
}
/* end of file: 88hdsk.c */
