/************************************************************************\
 * File Version Information
 * $Header: /Altair32v3/altair32.h 102   12/20/13 9:55p Racini $
 ************************************************************************/
/************************************************************************\
  MITS Altair Emulator
  Master include file
 
  Copyright (c) 2000-2016 Richard A. Cini
 
  2001/12/28  RAC -- New master header file for all common routines
  2002/01/02  RAC -- Added new struct members to winstate_t and synced
						with Solace v3.0
  2002/01/31  RAC -- RELEASE MARKER -- v2.2
  2002/05/17  RAC -- Added signed types for i16, i32 and timer_t type; 
						added new source files from Solace v3.1 to handle
						source-level debugging.
  2002/08/23  RAC -- RELEASE MARKER -- v2.30.10
  2002/09/12  RAC -- Added support for swappable console_io function ptrs
  2002/11/15  RAC -- RELEASE MARKER -- v2.40.2100
  2002/11/23  RAC -- Added reference to HTMLHelp header.
  2003/03/03  RAC -- Merged changes from Joe Forgione
  2003/04/26  RAC -- RELEASE MARKER -- v2.50.2045
  2003/06/27  RAC -- Added "hidden" i8080 register pair WZ to struct
  2004/01/20  RAC -- Diff'ed changes from FJS (Fred J. Scipione)
  2004/03/09  RAC -- Diff'ed more changes from FJS (Fred J. Scipione)
  2004/06/01  SML -- GCC changes
  2004/07/30  RAC -- RELEASE MARKER -- v3.00.0135  
  2004/08/17  FJS -- added DWORD_PTR typedef for VC6.0  
  2005/04/21  RAC -- added SASI defs; FP stuff
  2006/05/03  HH  -- added define for depreciated functions
  2006/05/12  RAC -- RELEASE MARKER -- v3.10.0200
  2006/06/02  RAC -- Moved hdisk defines to hdisk module
  2006/06/09  RAC -- Added exports for rtc, dazzler joystick
  2006/06/13  RAC -- simstate timeslicing definitions changed from uint32
  						to int32 to sync with Solace 3.2
  2006/07/29  RAC -- Added members to simstate to support better thread
  						exiting.
  2006/11/15  RAC -- RELEASE MARKER -- v3.20.0400
  2011/09/17  RAC -- RELEASE MARKER -- v3.30.0800
  2013/02/03  RAC -- RELEASE MARKER -- v3.32.1100
  2013/12/19  MWD -- Support interrupts for SIO and 2SIO console ports
  2013/12/31  RAC -- RELEASE MARKER -- v3.33.2100
  2014/03/22  MWD -- Add exported functions from 88HDSK.C
  2016/02/20  RAC -- RELEASE MARKER -- v3.34.0900
\************************************************************************/
#ifndef _ALTAIR32_H_
#define _ALTAIR32_H_

#pragma warning( disable : 4005 )	// disable warnings for macro redef
#pragma warning( disable : 4996 )	// disable _CRT_SECURE_NO_WARNINGS
//#define STRICT		// define strict typechecking
//#include <windows.h>	// include for all Windows programs
//#include <windowsx.h>	// Windows "extensions" and crackers

// The POSIX name for these functions is deprecated as of VS.NET 2005.
// Instead, use the ISO C++ conformant names.
#if defined(_CRT_SECURE_NO_DEPRECATE)
#define strdup _strdup
#define lseek _lseek
#define open _open
#define close _close
#define read _read
#define write _write
#endif

// FJS (for LCC, etc.) -
#ifndef DWORD_PTR
typedef DWORD DWORD_PTR;
#endif

#include <basetsd.h>
#include <stdio.h>		// used for FILE* in this header
#include "resource.h"


/**  Base Types  ************************************************/
typedef signed   char  offset;
typedef unsigned char  uchar;
#ifdef _LCC_
typedef unsigned char  byte;
#endif
typedef unsigned char  uint8;
typedef unsigned short ushort;
typedef unsigned short word;
typedef unsigned short uint16;
typedef          short int16;
typedef unsigned int   uint32;
typedef          int   int32;
typedef unsigned __int64 uint64;
typedef          __int64 int64;
typedef unsigned __int64 timer_t;


/**  Defines  ***************************************************/
//#define VER_NT_ONLY 0		// set to 1 to enforce running on only NT
							// remove to enable using A32 on Win95/98/ME
#ifndef VER_NT_ONLY
#define VER_NT_ONLY 1		// default to NT-only configuration
#endif

#if 1
#include <assert.h>
#define ASSERT(x) assert(x)
#else
#define ASSERT(x)
#endif

// General GDI params 
#ifdef IMSAI
#define SZWINDOWTEXT    "IMSAI 8080 Emulator"     // Text for window title bar
#define	ALTAIRBMP_W	600		// IMSAI bitmap sizes
//#define	ALTAIRBMP_W	750	// IMSAI bitmap sizes
#define	ALTAIRBMP_H	320	
//#define	ALTAIRBMP_H	400
#define ALTAIRLED_HW  12
#define	PWR_ON	 SW_UP		// switch directions
#define	PWR_OFF	 SW_DOWN
#define SW_STEP (pos == SW_UP) || (pos == SW_DOWN)
#define ICON_NAME "IMSAIICON"	// names
//#define PANEL_NAME "imsaipanel800"		//"imsaipanel"
#define PANEL_NAME "imsaipanel"
#else
#define SZWINDOWTEXT    "Altair 8800 Emulator"    // Text for window title bar
#define	ALTAIRBMP_W	796		//640		// Altair bitmap sizes
#define	ALTAIRBMP_H	336		//267
#define ALTAIRLED_HW  16	//12
#define	PWR_ON	 SW_DOWN	// switch directions
#define	PWR_OFF	 SW_UP
#define SW_STEP pos == SW_UP
#define ICON_NAME "ALTAIRICON"	// names
#define PANEL_NAME "ALTAIRPANEL800"		//"altairpanel"
#endif

// RUN/STOP happens to follow ON/OFF -
#define	SW_RUN	 PWR_ON		
#define	SW_STOP	 PWR_OFF

// Switch positions - values are significant
// 2-state switches cycle mod 2, 3-state mod 3
#define	SW_UP	 0
#define	SW_DOWN	 1
#define	SW_NEUT	 2

#define SZHELPFILE  "Altair32_main.htm"
#define UNTITLED    " (untitled)"

#ifndef ABS
#define ABS(x) ((x)>=0?(x):(-(x)))
#endif

// Boundary testing macro for cursor hit_tests. (x,y) are hit coordinates.
// LRTB is a bounding box to test against.
#define IN_RECT(x,y,L,R,T,B) \
	(((x)>=(L))&&((x)<=(R))&&((y)>=(T))&&((y)<=(B)))

// Statbar positions. DO NOT CHANGE OR REORDER NUMBERS AS THEY
// RELATE DIRECTLY TO THE POSITION (FROM L TO R) OF EACH "PART"
// OF THE STATBAR.
#define STATBAR_READY	0
#define STATBAR_CPU		1
#define STATBAR_LPR		2
#define STATBAR_DISK	3
#define STATBAR_CASS	4
#define STATBAR_CONS	5
#define STATBAR_JOY		6

typedef struct tagWINSTATE {
	char      szAppName[64];
	char      szFileName[_MAX_PATH];           // path to file
	char      szTitleName[_MAX_PATH+_MAX_EXT]; // application title
	char      szBaseDir[_MAX_PATH];            // path to app EXE
	char      szFilesDir[_MAX_PATH];
	FILE	  *logfile;
	HINSTANCE hInst;		// global instance handle
	HMENU     hMenu;		// global handle to main menu
	HWND      hWnd;			// global handle to main window
	HWND      hDbgWnd;		// handle to debugger window
	
	/* Because the DC is private, we don't need to keep getting the
	 * window DC by doing a makecompatibleDC, selecting the bitmap,
	 * then deleting the DC for every use--it is way too slow.
	 * Instead, we do it once then just use this copy of it.
	 */
	HDC     hdcWin; 		// = GetDC(winstate.hWnd);
	HCURSOR hNullCursor;
	HCURSOR hArrowCursor;
	int  exitcode;			// what WM_QUIT returned (ugly hack)
	BOOL useHPTimer;		// high-precision timer is available
	BOOL useLPTimer;		// high-precision timer is available
	__int64 tick_freq;		// # timer ticks per second
} winstate_t;

// Definitions for external memory read and write routines
typedef unsigned char ( _cdecl *MEMRPROC)(unsigned short);
typedef void ( _cdecl *MEMWPROC)(unsigned short, unsigned char);

// ROM Loader structure. Used in loader.c and syscfg.c.
typedef struct tagLOADROM {
	int  iUsed;				// allocated
	char szROMName[_MAX_FNAME+_MAX_EXT];       // ROM name
	int  iROMStart;			// start address
	int  iROMSize;
	BOOL Read_only;
} LOADROM;

/* Structure to define special memory mapped hardware device handler
 * routines. Used primarily in 65xx/68xx processors as memory-mapped
 * I/O is the only way to do things. Might prove useful here if we
 * need to emulate a memory-mapped hardware device.
 */
typedef struct tagMEMHANDLERS {
	unsigned short usStart;	// starting memory address
	unsigned short usLen;	// length of mapped range in bytes
	MEMRPROC pfn_read;		// *read
	MEMWPROC pfn_write;		// *write
} MEMHANDLERS;

// Structure to pass to CPU emulator generic memory handler routines
typedef struct tagEMUHANDLERS {
   MEMRPROC pfn_read;
   MEMWPROC pfn_write;
} EMUHANDLERS;

// The UI notifies the core simulator of certain debugger events
typedef enum tagSYSNOTIFY {
	DEBUG_KEY = 1,  // halt CPU and start debugger
	RUN_KEY,        // run if halted; halt if running
	FORCERUN_KEY,   // run
	STEP_KEY,       // step over N instructions
	STEPI_KEY,      // step into N instructions
	CLOSE_WIN       // close the debugger and run
} sysnotify_t;

typedef enum tagDBGWINMSG {
	DW_Activate,
	DW_ActivateStep,
	DW_Inactivate,
	DW_Close,
	DW_Update
} DbgWinMsg_t;

typedef union tagPAIR {
  struct { byte l,h; } B;   // little-endian struct
  word W;
} pair;

// used for reporting why the processor halted
typedef enum tagBRKPREASON {
	BRKPT_BRK,
	BRKPT_HALT,
	BRKPT_ILLEGAL,
	BRKPT_DEBUG } brkpt_t;

// types of breakpoints
typedef enum tagBRKPTYPE {
	BRK_INVALID, 
	BRK_TEMP, 
	BRK_OP, 
	BRK_IN, 
	BRK_OUT,
	BRK_RD,
	BRK_RD16,
	BRK_WR,
	BRK_WR16 } break_t;

// there are a number of breakpoints we support
typedef struct tagBREAKPOINT {
	break_t btype;		// type of breakpoint
	int     enabled;	// is breakpoint enabled
	int     sernum; 	// breakpoint serial #
	int     addr;   	// address to break on
	int     data;   	// 8 or 16b data value to trigger on
	int     mask;   	// 8 or 16b data mask to trigger on
	int     triggered;  // this breakpoint was detected
	int     dlo;    	// rd16/wr16 low  datum (-1 means not touched)
	int     dhi;    	// rd16/wr16 high datum (-1 means not touched)
} breakpoint_t;

// CPU definitional structure
typedef struct tagI8080 {
	pair nAF,nBC,nDE,nHL,nPC,nSP,nWZ; /* Main registers           */
	pair nAFp,nBCp,nDEp,nHLp,nIX,nIY; /* FJS Z80 alt registers... */
	word Disp;				/* FJS Z80 displacement register	   */
	word IFF;				/* interrupt flag                      */
	word CPUstatus;			/* state of status register            */
	word startPC;			/* if debugging, PC of 1st byte of op  */
	word CPUcycle;			/* current CPU cycle 0-2               */
	int  brkpt;				/* set if breakpoint is detected       */
	int  brknum;			/* serial number of breakpoint         */
	int  ICount,XCount;		/* cycle count, backup cycle count     */
	int  IBackup;			/* Private, don't touch                */
	brkpt_t reason;			/* reason why breakpoint occurred      */
} I8080;

// ----- Function pointer typedefs -----
// Must match ultimate routine prototype
// processor
typedef byte rd8080fp(word);
typedef void wr8080fp(word, byte);
typedef byte exec8080fp(I8080 *);

// console
typedef int  conread_fp(void);     // read from console channel
typedef void conwrite_fp(byte);    // write to console channel
typedef int  constato_fp(void);    // get output channel status
typedef int  constati_fp(void);    // get input channel status
typedef BOOL conshtdn_fp(void);    // shutdown funciton

// SIO port swap
typedef int	ioport_fp(int, int);	


// uP speed. 500KHz speed is for slow-stepping
typedef enum tagSPEED {
	MHZ_0_50,
	MHZ_1_02,
	MHZ_2_04,
	MHZ_4_09,
	MHZ_unregulated
} speed_t;

// uP type. i8080 and Z80 options <FJS added cycle variants> -
typedef enum tagCPUT {
	CPU_8080 = 0, // FJS must start at 0
	CPU_Z80,
	CPU_8080_Cycle, // FJS future
	CPU_Z80_Cycle
} cputype_t;

// simstate holds things related to running the emulator that aren't
// part of the physical system. Bookkeeping type things live here.
typedef struct tagSIMSTATE {
	enum {			// CPU state - used in scheduler
		RESET,		// CPU is in reset
		HALTED,		// emulator is not stepping 8080
		RUNNING,	// 8080 is free running until breakpoint
		STEP,		// step over N 8080 instructions, then -> HALTED
		STEPRUN,	// run until temp breakpoint, then -> STEP
		STEPI		// step N 8080 instructions, then -> HALTED
	} runstate;	// used in notify_coresim

	enum {			// front panel state
		ACTIVE,		// enabled and accepting input
		INACTIVE,	// not accepting input
		DISABLED	// front panel not available
	} fpstate;
	
	// Handles
	HANDLE hFrontPanel;	// handle to front panel device

	// On-screen FP control. Flows to scheduler, too.
	// Is there a reason why these should be word??
	word bPrun;            // 'WAIT' LED
	word bPpower;          // 'Power' status and LED
	word bPprotect;        // memory write-protect/'PROT' LED

	// CPU definition
	cputype_t	up_type; 			// processor type
    speed_t		up_speed;			// processor speed
	speed_t		orig_up_speed;		// backup speed for single-step throttling
	
	// Function pointer for normal *or* debugging routines
	// switched in altair.c
	exec8080fp *exec8080_func;
	rd8080fp *di80_func;    // byte Rd80(word Addr)
	rd8080fp *op80_func;    // byte Rd80(word Addr)
	rd8080fp *rd80_func;    // byte Rd80(word Addr)
	wr8080fp *wr80_func;    // void Wr80(word Addr, byte Value)
	rd8080fp *in80_func;    // byte In80(word Port)
	wr8080fp *out80_func;   // void Out80(word Port, byte Value)

	// Timeslicing stuff. Changed uint32 to int32 6/13/06
	HANDLE hThread;				// handle to scheduler thread
	int32  timeslice_ms;		// how often we resync
	__int64  tick_freq;			// # timer ticks per second
	int32 sol_ticks_per_ms;		// # of 8080 clocks per millisecond
	int32 sol_ticks_per_slice;		// # of 8080 clocks per timeslice
	int32 x86_ticks_per_slice;		// # of x86 clocks per timeslice
	int throttle;     				// 1=slow down 8080 emulation
	int prog_detect;				// # timeslices since PC was outside OS
	int stepcount;
	int firststep;
	int inactivate_timer;	// how many slices before disabling dbg window

	// Console selection. Switched in syscfg.c and altair.c
	int iConsoleType;				// console type 0=telnet; 1=windows console; 2=serial
	ushort usTelnetPort; 			// telnet port
	conread_fp  *conread_func;		// read from console channel
	conwrite_fp *conwrite_func;		// write to console channel
	constato_fp *constato_func;		// get output channel status
	constati_fp *constati_func;		// get input channel status
	conshtdn_fp *conshtdn_func; 	// console shutdown

	int iSioType;              		// Serial port type 0=file; 1=telnet; 2=COM
	conread_fp  *Sioread_func;		// read from console channel
	conwrite_fp *Siowrite_func; 	// write to console channel
	constato_fp *Siostato_func;		// get output channel status
	constati_fp *Siostati_func;		// get input channel status
	conshtdn_fp *Sioshtdn_func;		// console shutdown

#if 0
	ioport_fp *sioastat_func;		// port A status register function
	ioport_fp *sioadata_func;		// port A data register function
	ioport_fp *siobstat_func;		// port B status register function
	ioport_fp *siobdata_func;		// port B data register function
#endif
} simstate_t;

// Video window control messages
typedef enum tagVIDWINMSG {
	VWM_Activate,
    VWM_Inactivate,
    VWM_Close,
	VWM_Create
} VidWinMsg_t;


/** exported by init.c  *****************************************/
extern void InitEmu( void );
extern char szHelpDir[_MAX_PATH];


/**  exported by 88disk.c  **************************************/
extern int  iAttachCount;      // # drives attached
extern HWND hwndDISK;          // hwnd for dialog
extern void OpenDiskWin( void );
extern void DISK_InitController( void );
extern void DISK_Shutdown( void );
extern int  dsk08h( int, int );
extern int  dsk09h( int, int );
extern int  dsk0ah( int, int );


/**  exported by 88hdsk.c  **************************************/
extern void MITS_Hdsk_Shutdown(void);
extern void MITS_Hdsk_OpenImageFile(void);
extern int  hdRetRdy( int, int );	// 0xA0,A2,A4,A6 - I/O vector handlers
extern int  hdCstat( int, int );	// 0xA1
extern int  hdAcmd( int, int );		// 0xA3
extern int  hdCdata( int, int );	// 0xA5
extern int  hdAdata( int, int );	// 0xA7


/**  exported by altair.c  **************************************/
#define SETLED_UPD      0       // update led's according to data
#define SETLED_CLR      1       // clear all led's
#define SETLED_SET      2       // set all led's
#define SETLED_REFRESH  3       // refresh LED's regardless of current status
#define RUN_OFF			0
#define RUN_ON			1
#define MEM_UNPROT		0
#define MEM_PROT		1
extern winstate_t winstate;	// UI state
extern simstate_t simstate;	// simulation state
extern ushort Dswitches;		// used to save position

extern LRESULT CALLBACK MainWndProc( HWND, UINT, WPARAM, LPARAM );
extern void SetLEDs( short );
extern void startrun( void );
extern void Sys_DbgNotify( sysnotify_t event, int param );
extern void bkptchange_coresim( void );
extern void SysKeyPress( byte );
//extern int  UI_HandleMessages( int * );
extern void DoSwitch( short, short );


/**  exported by dazzler.c  *************************************/
extern void DAZZLER_Init( void );
extern void DAZZLER_Destroy(void);
extern void DAZZLER_VideoWin(VidWinMsg_t msg, int arg);
extern int dazz0eh(int io, int data);
extern int dazz0fh(int io, int data);
extern int joybtns( int, int );
extern int joy1x( int, int );
extern int joy1y( int, int );
extern int joy2x( int, int );
extern int joy2y( int, int );
extern int joy1z( int, int );
extern int joy2z( int, int );


/**  exported by debug.c  ***************************************/
extern int DAsm( char *S, ushort adrs, int extended );
extern int OpLen( ushort adr );


/**  exported by debugger.c  ************************************/
extern void breakpoint_init( void );
extern void breakpoint_genmap( void );
extern void breakpoint_temp_add( word addr );
extern void breakpoint_temp_kill( void );
/* FJS corrected first arg from brkpt_t to break_t -> */
extern int  breakpoint_test( break_t btype, word addr, word data, int *num );
extern int  breakpoint_verify( int *num );
extern int  any_breakpoints( break_t btype );
extern int  check_brk( word addr, break_t type );
extern void dbg_interp( char *str );
extern int  DebuggerWindowGetSize( RECT *rc, int *visible );
extern void DebuggerWindowSetSize( RECT *rc );


/**  exported by hexfile.c  *************************************/
extern int  parse_hex_line( char *, int [], int *, int *, int * );


/**  exported by i8080.c  ***************************************/
// This should be kept only in one place (i8080.h) -
#if 1
#include "i8080.h"
#else
#define MEMSIZE 65536
extern  simstate_t simstate;
extern  I8080   I80Regs;
extern  byte    Mem[MEMSIZE];
extern  byte    PageAccessRights[MEMSIZE]; // FJS was int
extern  word    IOPANEL;
extern  word    MBR;   // buffer register for data buss
extern	word	ABR;	// buffer register for Address buss
extern  word    Aswitches;	// switch register
extern  byte    CLK_OP;

extern  byte    GetMem( word );
extern  byte    GetMem_brkpt ( word );
extern  void    SetMem( word , byte );
extern  void    SetMem_brkpt( word , byte );
extern  byte    GetIO( word );
extern  byte    GetIO_brkpt( word );
extern  void    SetIO( word , byte );
extern  void    SetIO_brkpt( word , byte );
extern  byte    clock8080( I8080 * );
extern  byte    clock8080_brkpt( I8080 * );
extern  void    Reset8080( void );

// Used to get and set the emulated CPU type and speed -
extern  void    SYS_Set8080Speed( speed_t );
// extern  speed_t SYS_Get8080Speed( void ); // not used
extern	void	SYS_SetCPUType( cputype_t );
// extern	cputype_t	SYS_GetCPUType( void ); // not used

extern  void    CpuPOR( void );
extern  void    LoadBootProm( void );
extern  BOOL    CoreDump( HWND, PSTR );
extern  void    report_curstate( void );
extern  void    brkpt_notify( void );
extern  void    simstep( void );
extern  void    simrun( void );
extern  void    simstepover( void );
#endif


/**  exported by loader.c  **************************************/
extern BOOL CALLBACK lpAskLoadAddr( HWND, UINT, WPARAM, LPARAM );
extern BOOL CALLBACK lpAskSaveAddr( HWND, UINT, WPARAM, LPARAM );
extern BOOL ReadBinaryFile( HWND, PSTR, PSTR, int );
//extern uint32 ReadBinaryFile( HWND, PSTR, PSTR, int );
// possible exit codes for RBF:
//#define RBF_SUCCESS					0
//#define RBF_ERR_CANNOT_OPEN_IMAGE	0x80000000
//#define RBF_ERR_IMAGE_TOO_LARGE		0x80000001
//#define RBF_ERR_IMAGE_LOAD_ERROR	0x80000002
extern uint32 hextoi( char * );
extern long TapeFileLength( char * );


/**  exported by scheduler.c  ***********************************/
typedef void (*timercb_t)( uint32 arg1, uint32 arg2 );
// FWIW, the timebase should be double.
#define TIMER_US(f) ((int)(14.318180*(f)+0.5))  // used with TimerCreate
#define TIMER_MS(f) ((int)(14318.180*(f)+0.5))
extern unsigned __stdcall  SCHED_DoTimeSlicing( void* pArguments );
extern void    SCHED_TimerInit( void );
extern timer_t TimeAbs( void );
extern int     TimerCreate( uint32, timercb_t, uint32, uint32 );
extern timer_t TimeDiff( timer_t );
extern uint32  TimeDiffSat( timer_t );
extern void    TimerKill( int );
extern int     TimerRemaining( int );  // in base ticks, not CPU ticks
extern void    TimerRollover( uint32, uint32 );
extern void    TimerTick( uint32 );
extern void    TimerTickScale( int );  // specify # of base ticks per CPU clock
extern void    UpdateTimerInfo( void );


/**  exported by 88lpc.c  ***************************************/
extern int lpc02h( int, int );
extern int lpc03h( int, int );
extern void LPR_OpenWin( void );
extern void LPR_Shutdown( void );
extern void LPR_Destroy( int );


/**  exported by taperdr.c  *************************************/
extern int  sio12h( int, int );   //SIO#2 RDR/PUN
extern int  sio13h( int, int );   //SIO#2 RDR/PUN


/** The next module is related to taperdr.c  **/
/**  exported by ptp.c  *****************************************/
extern HWND hwndPTP;
extern void PTP_OpenWin( void );
extern void PTP_Shutdown( void );


/**  exported by cass.c  ****************************************/
extern int acr06h(int io, int data);
extern int acr07h(int io, int data);
extern HWND hwndCASS;
extern void CASS_OpenWin( void );
extern void CASS_Init( void );
extern void CASS_Shutdown( void );
extern void CASS_Destroy( int );

/**  exported by console.c  **************************************/
extern void Check_Console( void );	// check console I/O for interrupt purposes
extern void	Console_Init( void );	// intialize
extern int  sio00h( int, int );		//SIO console port status and control
extern int  sio01h( int, int );		//SIO console port data
extern int  sio10h( int, int );		//2SIO console port status and control
extern int  sio11h( int, int );		//2SIO console port data

// exported by tcp_io.c
#define TCP_INIT        0       // Ready for initialization
#define TCP_LISTENING   1       // Socket created - waiting for a connection
#define TCP_CONNECTED   2       // Socket connected to client
#define TCP_DATA_READY  4       // Data in buffer
#define TCP_SHUTDOWN    99      // Socket closed - shut down

// extern int  TcpState;		// FJS
extern BOOL TELNET_TcpInit( void );
extern BOOL TELNET_TcpShutdown( void );
extern int  TELNET_TcpInStatus( void );   // get read data status
extern int  TELNET_TcpOutStatus( void );  // get output ready status
extern int  TELNET_TcpRead( void );       // read in a single character
extern void TELNET_TcpWrite( UCHAR );     // write a character
extern BOOL TELNET_TcpGetData( void );
extern BOOL TELNET_TcpPutData( UCHAR );

// exported by wincon.c
extern BOOL WINCON_Shutdown( void );
extern int  WINCON_InStatus( void );     // get read data status
extern int  WINCON_OutStatus( void );    // get output ready status
extern int  WINCON_Read( void );         // read in a single character
extern void WINCON_Write( UCHAR );       // write a character

// exported by ser_io.c
extern BOOL SERIAL_SerShutdown( void );
extern int  SERIAL_SerInStatus( void );   // get read data status
extern int  SERIAL_SerOutStatus( void );  // get output ready status
extern int  SERIAL_SerRead( void );       // read in a single character
extern void SERIAL_SerWrite( UCHAR );     // write a character
void Populate_ComboBox ( HWND, char * );
HANDLE  hComm;						// port handle
DCB     config_;					// DCB
OVERLAPPED combuff;				// buffer
const char*  comport;				// port number
int     baudrate;					// baud rate
CHAR    S_port[80];				// buffer
unsigned short int Text_Color;		// Default Green
unsigned short int Back_Color;		// Default Black FJS


/**  exported by statbar.c  *************************************/
// Global PART IDs to remove hard-coded values from code
#define STATBAR_READY	0
#define STATBAR_CPU		1
#define STATBAR_LPR		2
#define STATBAR_DISK	3
#define STATBAR_CASS	4
#define STATBAR_CONS	5
#define STATBAR_JOY		6
extern HWND hwndStatusBar;
extern HWND InitStatusBar( HWND hwndParent );
extern HWND RebuildStatusBar( HWND hwndParent, WORD wFlag );
extern void StatusBarMessage( HWND hwndSB, WORD wMsg );
extern LRESULT Statusbar_MenuSelect( HWND, WPARAM, LPARAM );


/**  exported by windbg.c  **************************************/
extern uint32 brkpt_map[2048];     // a bit indicating if anything special is going on
extern byte   brkpt_iomap[256];    // is anything special is going on
#define NUM_BREAKPOINTS 32          // # of possible breakpoints
//extern brkpt_t breakpoints[NUM_BREAKPOINTS];
//extern int      breaknum;     // # of defined breakpoints FJS
//extern int    breaknxt;       // breakpoint serial number FJS

// add a message to the end of the debugger log window.
// it can contain embedded newlines.
extern void UI_DbgWinLog( char *str, int bold );
extern int  UI_DbgWinActive( void);  // is debugger window active?
extern void UI_DbgWin( DbgWinMsg_t msg, int arg );
extern void bkptchange_coresim( void );
extern void InitDebugger( void );
extern void DestroyDebugger( void );


/**  exported by syscfg.c  **************************************/
#define MAXROMS 6
extern HWND hwndConCfg;
extern HWND hwndMemCfg;
extern HWND hwndOPTS;
// extern HWND hwndPrinterCfg;
// extern HWND hwndRomCfg;
extern HWND hwndMiscCfg;
extern HWND hwndSpeedCfg;
extern void SYSCFG_WriteConfigInfo( char *, char *, char * );
extern char *SYSCFG_ReadConfigInfo( char *, char * );
extern void SYSCFG_OpenOPTS( void );
extern void SYSCFG_LoadOptions( void );
extern void SYSCFG_SaveOptions( void );

// Memory configuration
extern word ramstart;
extern word ramend;
extern word romstart;
extern word romend;
LOADROM load_rom[MAXROMS];

// 'Miscellaneous' options flags
BOOL Bell;
BOOL WordWrap;
BOOL Ansi;
BOOL Disksnd;
BOOL Fansnd;
BOOL StripParity;
BOOL Mirror2SIOPorts;
int  LptCharSet;   // 0=ASCII7, 1=MITS

// Terminal 
int ConTermWidth;

// Front Panel options
BOOL bFPsound;

#if 0
/**  exported by xebec.c  ***************************************/
extern BOOL S1410_InitXebec1410( void );
extern void S1410_ShutdownXebec1410( void );


/**  exported by sasihost.c  ************************************/
extern int xebec30h( int, int );	/* register 0	*/
extern int xebec31h( int, int ); 	/* register 1	*/
extern int xebec32h( int, int );	/* register 2	*/
extern int xebec33h( int, int );	/* register 3	*/
#endif


/**  exported by hdisk.c  ***************************************/
// FURTURE - drive commands
typedef enum tagHDCMD {
	HDISK_NONE,		// 0 no current command/idle
	HDISK_RESET,	// 1 reset disk
	HDISK_READ,		// 2 absolute disk read 0=success; "select" is implied
	HDISK_WRITE		// 3 absolute disk write
} hdcmd_t;

typedef struct tagHDPRMBLK {	// parameter block
	hdcmd_t	hdskLastCommand;	// command to be executed...
	uint32	selectedDisk;		// on disk [x]...
	uint32	selectedSector;	// sector [y]...
	uint32	selectedTrack;		// track [z]
//	uint32	selectedLOGSECT;	// possible: selected logical sector	
	uint32	selectedDMA;		// DMA address
	uint32	udata1;			// user data field
	uint32	hdskTrace;			// trace enabled
} hdprmblk_t;

extern BOOL hdsk_boot(hdprmblk_t dpb);
extern BOOL HDISK_IoCtl(uint32);
extern BOOL HDISK_OpenContainerFile(void);
extern int  hdskfdh(int io, int data);


/**  exported by utilfunc.c  ************************************/
extern BOOL GetFilenameFromUser(char *, char *, char *, BOOL *);
extern char *just_filename(char *fullname);
extern void HOST_ThrowErrMsg( char * );
extern int  HOST_IsAbsolutePath( char *name );		// Should these
extern char *HOST_ExtractPath( char *name );		// be const char
extern char *HOST_MakeAbsolutePath( char *name );	// and not char?
extern void DoCaption( HWND, PSTR, PSTR );
extern char *HOST_UniqueFilename(const char *dir, const char *prefix);


/**  exported by seh.c  *****************************************/
extern void InstallFaultHandler(void);
extern void UninstallFaultHandler(void);


/**  exported by ct_rtc.c  **************************************/
extern int rtc80h(int, int);

#endif
/* end of file: altair32.h */
