/************************************************************************\
 * File Version Information
 * $Header: /Altair32v3/console.c 12    12/20/13 9:55p Racini $
 ************************************************************************/
/************************************************************************\
  MITS Altair Emulator
  MITS 88-SIO2 device driver

  Copyright (c) 2000-2016 Richard A. Cini

  This module emulates the RS232 serial interface card used in the
  Altair. Choices were the 88-SIO or 88-SIO2 (dual port) interface cards.
  The original card, the SIO, was based on the COM2502 UAR/T (equivalent
  to the GI AY-3-1013 or AY-3-1015 chips). The 2SIO dual-port card had
  two RS-232 ports based on the 6850 ACIA chip from Motorola. MITS
  assigned standard port addresses for the cards as specified below.
  Port assignments were modifyable based on a limited selection. Baud rates
  ranged from 110 to 9600 although most hardware ran only at 300 baud.

  A typical later-period configuration would include a "glass TTY" terminal
  (commonly known as a VDT) connected to the first port. 

  Here's the layout of the port configuration.

	   Port 	Dir		Function
	   ----		---		--------
	   00/01 88-SIO		Legacy console port for old software
   	   00		W		Configuration register
	   00		R		Status register - console
	   01		W		Write console data
	   01		R		Read console data

	   10/11 88-SIO2-1	Console port A - used by CP/M disk images
	   10		W		Configuration register
	   10		R		Status register
	   11		W		Write console data
	   11		R		Read console data

  Tape BASIC programs are saved by using the logging capabilities of the
  telnet client or the Windows Console. Alternatively, a program can be
  loaded into BASIC by cutting and pasting into the terminal client.
  This is a crude but effective emulation of the way programs were
  loaded and saved when using an ASR33 -- set the mode to "punch" and
  LIST the program. LOADing was done by setting the mode to "reader"
  and running the tape -- the characters were jammed into the BASIC input
  buffer as if the user typed them.

  To handle true paper tape images (i.e., programs like BASIC 3.2 itself)
  there is programmatic access through ports 24h-25h by connecting a tape
  image file through the punch control dialog. CP/M supports a paper
  tape punch which uses the second port of the 2SIO card. See taperdr.c.

  Support of interrupt driven I/O has been added for both SIO and 2SIO ports.
  "Generic" status and configuration registers -- independent of the emulated
  port type -- are used for I/O status and control in this module. When 8080 
  software executes IN and OUT instructions to the actual SIO or 2SIO ports,
  information from the common status and configuration registers is used to 
  form the SIO or 2SIO specific register content and behavior.

Change Log: 
  2000/12/06  RAC -- Began coding.
  2000/12/20  RAC -- Decided to use file capture route. Implemented.
  2001/01/03  RAC -- Added power control flag for conditionals.
  2001/02/20  RAC -- "Module global" fix; return parens; string term.
  2001/07/16  RAC -- Began integration and testing of TCP connection.
  2001/07/31  TMP -- Telnet code passes loopback verification test.
  2001/08/08  TMP -- Changes to telnet code for better error flag handling.
  2001/08/19  RAC -- RELEASE MARKER -- v2.0
  2001/08/29  RAC -- Comments updated.
  2001/09/17  RAC -- Added PTP mode code.
  2001/10/15  RAC -- Changed prototypes to Hex
  2001/11/16  RAC -- Made temporary changes to prove CP/M operation.
  2001/11/25  RAC -- Added "CPM" conditional
  2001/11/26  RAC -- Separated tape console from disk console; removed
  						CPM conditional.
  2001/12/03  RAC -- Separated LPR code into separate source file. Added
  						control code trap in console code for BELL (^G)
  						which doesn't seem to work yet.
  2001/12/14  RAC -- RELEASE MARKER -- v2.1
  2001/12/20  RAC -- Changes to comments.
  2002/01/02  RAC -- Moved routines to new master include file and merged
  						all serial devices into one export header (sio.h
  						is new serial device header)
  2002/01/31  RAC -- Fixed (hopefully) minor problem in sio01h to get BASIC
  						3.2 working.
  2002/01/31  RAC -- RELEASE MARKER -- v2.2
  2002/07/07  RAC -- RELEASE MARKER -- v2.3
  2002/08/23  RAC -- RELEASE MARKER -- v2.30.10
  2002/10/04  RAC -- Made changes to support calling through simstate.con_func
  						in support of swappable console modes (telnet/WinCon)
  2002/11/15  RAC -- RELEASE MARKER -- v2.40.2100
  2003/04/26  RAC -- RELEASE MARKER -- v2.50.2045
  2003/06/30  RAC -- Comments updated
  2003/10/16  RAC -- Began changes to move PUN/RDR to 2SIO/2, LIST to 88LPC.
  2004/07/30  RAC -- RELEASE MARKER -- v3.00.0135
  2006/05/12  RAC -- RELEASE MARKER -- v3.10.0200
  2006/11/15  RAC -- RELEASE MARKER -- v3.20.0400
  2009/09/24  RAC -- Re-added support for original 88-SIO ports at 0/1
  2011/09/17  RAC -- RELEASE MARKER -- v3.30.0800
  2013/02/03  RAC -- RELEASE MARKER -- v3.32.1100
  2013/12/19  MWD -- Support interrupts for SIO and 2SIO console ports
  2016/02/17  MWD -- Minor fixes to interrupt code for use over real serial port
  2016/02/20  RAC -- RELEASE MARKER -- v3.34.0900
\************************************************************************/
#define _CRT_SECURE_NO_WARNINGS			// BAD thing to do
#include <windows.h>
#include <commdlg.h>
#include <stdio.h>		// C-lib
#include <mmsystem.h>
#include "altair32.h"	// includes "resource.h"


// Console status and control register (common for SIO and 2SIO)
static byte consoleStatus;		// Console status register
#define	CON_RDRF		0x01	// Receive data register full (1=DAV)
#define	CON_TDRE		0x02	// Transmit data register empty (1=ready to xmit)
#define	CON_DATA_FLAGS	0x03	// bit mask for both data flags
#define	CON_INT_ACTIVE	0x04	// INT to 8080 asserted (1=interrupt asserted)

static byte consoleControl;		// Console control register
#define	CON_RINT_ENABLE	0x01	// Rcv interrupts enabled (1=interrupts enabled)
#define	CON_XINT_ENABLE	0x02	// Xmit interrupts enabled (1=interrupts enabled)
#define	CON_INT_FLAGS	0x03	// bit mask for both interrupt flags

// Status and control register defines for 88-SIO
#define SIO_RDE			0x01	// Receive data empty (0=DAV, 1=empty)
#define SIO_WDB			0x80	// Write data busy (0=ready, 1=not ready)
#define	SIO_RINT_ENABLE	0x01	// Receive interrupt enable
#define	SIO_XINT_ENABLE	0x02	// Transmit interrupt enable

// Status and control register defines for 88-2SIO
#define ACIA_RDRF			0x01	// Receive data register full (0=empty, 1=DAV)
#define	ACIA_TDRE			0x02	// Transmit data register empty (0=not ready, 1=ready)
#define	ACIA_INT_ACTIVE		0x80	// External INT asserted (1=INT asserted)
#define	ACIA_XINT_ENABLE	0x20	// Transmit interrupt enable (two bits)
#define	ACIA_XINT_MASK		0x60	// Mask to get xmit interrupt bits alone
#define	ACIA_RINT_ENABLE	0x80	// Receive interrupt enable
#define ACIA_RESET			0x03	// ACIA Master reset

// ASCII control characters
#define BEL 0x07	// console bell
#define BS	0x08	// backspace
#define HT  0x09	// horizontal tab
#define LF  0x0a	// line feed
#define FF  0x0c	// form feed; for VDT, clears screen
#define CR  0x0d	// carriage return
#define ESC 0x1b	// escape
#define DC1 0x11	// punch ON
#define DC2 0x12	// tape ON   (tape reader)
#define DC3 0x13	// punch OFF
#define DC4 0x14	// tape OFF  (tape reader)
#define RBO 0x7f	// rubout (backspace+space+backspace)


/**************************************************************\
*   External Routines  ****************************************|
***************************************************************/
/*--  Console_Init  ----------------------------------------------
	Initialize the console status and control register as:
	not ready to xmit, no receive data, interrupt out not
	asserted, no interrupts enabled.
-------------------------------------------------------------*/
void Console_Init( void )
{
	consoleStatus = 0;
	consoleControl = 0;
}

/*--  Check_Console ----------------------------------------------
	Called frequently from the scheduler to check console I/O
	to support interrupts on the console I/O ports.
	A common status register (consoleStatus) is updated. This
	common register is later examined by the I/O routines
	for the specific port type (e.g., SIO or 2SIO)
-------------------------------------------------------------*/
void Check_Console( void )
{

//  First, update the transmit and receive data flags
	consoleStatus &= ~CON_DATA_FLAGS;				// clear both receive and xmit data flags
	if (simstate.constato_func() != 0)				// transmit ready?
		consoleStatus |= CON_TDRE;					// yes
	if (simstate.constati_func() != 0) 				// receive data ready?
		consoleStatus |= CON_RDRF;					// yes, set receive data flag

// Update interrupt status.
	if ((consoleStatus & consoleControl & CON_DATA_FLAGS) != 0) {	// interrupt present ?
		consoleStatus |= CON_INT_ACTIVE;			// console device has interrupt asserted
		InterruptSources |= INTSRC_CONSOLE;			// assert the interrupt to 8080
	}
	else {
		consoleStatus &= ~CON_INT_ACTIVE;			// console device does not have interrupt asserted
		InterruptSources &= ~INTSRC_CONSOLE;		// remove interrupt from 8080
	}
}


/**************************************************************\
*   I/O Instruction Handlers  *********************************|
***************************************************************|
*
*	I/O instruction handlers are called from the CPU module
*  	when an IN or OUT instruction is issued.
*
*  	Each function is passed an 'io' flag, where 0 means a read
*  	from the port, and 1 means a write to the port.  On input,
*	the actual input is passed as the return value. On output,
*	'data' is written to the device.
\**************************************************************/

/*--  SIO00h  -------------------------------------------------
	Status/configuration register for console device. Status
	information is passed on simply as go/no-go. Ports 0-1 used
	to support Altair BASIC tape images and some other programs
	which assume a TTY device connected to an SIO card.	
-------------------------------------------------------------*/
int sio00h (int io, int data)
{
	int nSioStatus = 0;


	if (io == 0) {									// Input - create and return the status register
		Check_Console();							// get most recent status
		if ((consoleStatus & CON_RDRF) == 0)		// data register empty
			nSioStatus |= SIO_RDE;	
		if ((consoleStatus & CON_TDRE) == 0)		// transmit not ready
			nSioStatus |= SIO_WDB;
	}
	else {											// Output - write the control register
		consoleControl = 0;							// assume interrupts not enabled
		if ((data & SIO_RINT_ENABLE) != 0)	
			consoleControl |= CON_RINT_ENABLE;		// enable receive interrupts
		if ((data & SIO_XINT_ENABLE) != 0)
			consoleControl |= CON_XINT_ENABLE;		// enable xmit interrupts
		Check_Console();							// force possible interrupt status update
	}

	return nSioStatus;
}


/*--  SIO01h  -------------------------------------------------
	Data register for console device. ConStatIn returns 1 if
	data is	available, 0 if not. ConStatO returns 1 if data
	can be written,	0 if data can't be written (i.e., buffer
	full). On read, ConRead returns character or 0 if no data
	is available. ConWrite presently ignores errors.

	Ports 0-1 used to support Altair BASIC tape images which
	assume a TTY device connected to an SIO card.
-------------------------------------------------------------*/
int sio01h (int io, int data)
{

	int nData = 0xff;

	switch (io){
		case 0:									// READ
			nData = simstate.conread_func();	// get char from console
			if (nData == BEL){					// sound the terminal BELL from user
				PlaySound("BELL", winstate.hInst, SND_RESOURCE | SND_ASYNC | SND_NOWAIT);
			}
			Check_Console();					// update status and interrupts
			return nData;						// return character to user

		case 1:									// WRITE
			if (simstate.constato_func()){		// 1=OK to write, send char
				if (data == BEL){				// echo console BELL coming from the Altair
					PlaySound("BELL", winstate.hInst, SND_RESOURCE | SND_ASYNC | SND_NOWAIT);
				}
				simstate.conwrite_func((byte) data);
				Check_Console();				// update status and interrupts
			}
			return 0;

		default:
			return 0;
	}
}


/*--  SIO10h  -------------------------------------------------
	Status/configuration register for CP/M console device.
	Status information is passed on simply as go/no-go.

	Ports 10h-11h are used to support CP/M disks which assume
	that a console device lives on the first serial port of a
	SIO2 card or any other 88-SIO2 device.

-------------------------------------------------------------*/
int sio10h(int io, int data)
{
	int nSioStatus = 0;


	if (io == 0) {										// Input - form and return the status register
		Check_Console();								// get most recent status
		if ((consoleStatus & CON_RDRF) != 0)			// data register full
			nSioStatus |= ACIA_RDRF;	
		if ((consoleStatus & CON_TDRE) != 0)			// transmit ready
			nSioStatus |= ACIA_TDRE;
		if ((consoleStatus & CON_INT_ACTIVE) != 0)		// interrupt asserted
			nSioStatus |= ACIA_INT_ACTIVE;
	}
	else {												// Output - write the control register
		consoleControl = 0;								// assume interrupts not enabled
		if (data == ACIA_RESET) 
			consoleStatus = 0;
		else {
			if ((data & ACIA_RINT_ENABLE) != 0)	
				consoleControl |= CON_RINT_ENABLE;		// enable receive interrupts
			if ((data & ACIA_XINT_MASK) == ACIA_XINT_ENABLE)
				consoleControl |= CON_XINT_ENABLE;		// enable xmit interrupts
		}
		Check_Console();								// update status and interrupts
	}

	return nSioStatus;
}


/*--  SIO11h  -------------------------------------------------
	Data register for console device. ConStatO returns 1 if data
	can be written,	0 if data can't be written (i.e., buffer
	full). On read, ConRead returns character or 0 if no data
	is available. ConWrite presently ignores errors.
	
	Note that SIO01h and SIO11h are presently identical.
-------------------------------------------------------------*/
int sio11h(int io, int data)
{
	int nData = 0xff;

	switch (io){
		case 0:									// READ
			nData = simstate.conread_func();	// get char from console
			if (nData == 0x07){					// do BELL from user
				PlaySound("BELL", winstate.hInst, SND_RESOURCE | SND_ASYNC | SND_NOWAIT);
			}
			Check_Console();					// update status and interrupts
			return nData;						// return character

		case 1:									// WRITE
			if (simstate.constato_func()){		// 1=OK to write, send char
				if (data == 0x07){				// do BELL coming from the Altair
					PlaySound("BELL", winstate.hInst, SND_RESOURCE | SND_ASYNC | SND_NOWAIT);
				}
				simstate.conwrite_func((byte) data);
				Check_Console();				// update status and interrupts
			}
			return 0;

		default:
			return 0;
	}
}
