// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "corebase.h"
#include "container.h"
#include "common.h"
#include "ini.h"
#include "elf.h"
#include "messages.h"
#include "resource.h"
#include "hw_gx.h"
#include "stringCommons.h"
#include "main.h"
#include "dsp_core.h"
#include <math.h>
#include <Dbghelp.h>
#include <fstream>

using namespace stdext;

string g_errorstring;
static const char *gq_type[8] = { "single", "reserved", "reserved", "reserved",
"u8", "u16", "s8", "s16" };
CriticalSectionWithSpin css_interrupts;

Container<char>& strcpy(Container<char>& dest, const char*src) {
	strcpy_s(dest, dest.size(), src);
	return dest;
}

bool open_file(const char *filename, FILE **file) {
	DEGUB("Loading %s\n", filename);
	if(fopen_s(file, filename, "rb")) {
		DEGUB("Couldn't open file \"%s\"\n", filename);
		FAIL(UE_CANNOT_OPEN_FILE);
	}
	return true;
}

template<size_t size> void getline(istream& stream, char (&buffer)[size]) {
	/*//stream.getline(buffer, size);	//BUGGY PEICE OF SHIT! WON*T STOP READING ON \N!
	//stream.getline(buffer, size, '\xA');  //NEITHER WILL THIS
	//stream.get(buffer, size, '\xA');  //THIS TOO!
	int pos = 0;
	while(pos < size-1) {
	char c = (char)stream.get();
	if(c == 0xA)
	break;
	buffer[pos++] = c;
	}
	if(pos == size-1) {
	stream.setstate(ios_base::failbit);
	}
	buffer[pos] = 0;*/
	static size_t pos = 0;
	//bugs above explained; pos becomes pseudo-random between reads!
	//size_t temp = stream.tellg();
	//stream.seekg(temp);
	stream.seekg(pos);
	stream.getline(buffer, size);
	pos = stream.tellg();
}


#define DEFINE_UNIMPLEMENTED_DISASM(asm, pri, sm, sec) \
	void disasm_##asm(Container<char>& buffer) {\
	strcpy(buffer, " UNIMPLEMENTED!"); }

UNIMPLEMENTED_OPCODES(DEFINE_UNIMPLEMENTED_DISASM)

CoreBase::CoreBase(MemInterface &mem, CORE_ARGUMENTS a) : asmarr(0, 5),
interrupt(pending_interrupts, timed_events, p.event_time, cc),
reserve_bit(false), m(mem), r(mem), h(mem, interrupt, a, cc), symbol_pmap(0),
periods(0) {
	ZERO_OBJECT(p);
	p.hWnd = a.hWnd;

#define ADD_ASM_OPCODE(asm, pri, sm, sec) \
	asmarr.addOpcode(&disasm_##asm, #asm, pri, sm, sec);

	OPCODES(ADD_ASM_OPCODE);
	asmarr.addOpcode(&disasm_addic_, "addic.", 13, 0, 0);
	asmarr.addOpcode(&disasm_stwcx_, "stwcx.", 31, 21, 150);
	UNIMPLEMENTED_OPCODES(ADD_ASM_OPCODE);

	//Intel FPU Init; incomplete?
	//Set 64-bit precision, near rounding. Enable teh bad exceptions.
	//The other exceptions are Inexact and Denormal.
	/*_control87(_PC_64 | _RC_NEAR | _EM_INEXACT, // | _EM_DENORMAL
	_MCW_PC | _MCW_RC | _MCW_EM);*/
	//directx article "Top Issues for Windows Titles" says this is bad.
	//so we try not doing it.

	/*if(g::dual_run) {
	set_dec_event(0);
	}*/
}

CoreBase::~CoreBase() {
}

CycleCounterBase::CycleCounterBase() : cycles(ct), cpp(0), ct(0), ccp(0) { }
CycleCounter::CycleCounter() : cycles_current_period(ccp) {}
void CycleCounter::addCycles(DWORD c) { ct += c; ccp += c; }
void CycleCounter::endPeriod() { cpp += ct; ccp = 0; }

void CoreBase::dumpmem() const {
	m.dumpmem();
	h.dumpmem();
	DEGUB("Dumped memory.\n");
	Container<BYTE> rgb(640*480*3);
	for(DWORD address=0; address<=(MAIN_MEMORY_SIZE-640*480*2); address+=640*480*2) {
		convertYUYV2RGB(rgb, m.getp_physical(address, 640*480*2));
		ostringstream str;
		str << HEX08(address) << "-" << HEX08(address + 640*480*2 - 1) << ".raw";
		GLE(dumpToFile(rgb, 640*480*3, str.str().c_str()));
	}
}

#define WC_DEGUB_BUFFER_SIZE 128
void CoreBase::wc_debug() {
	/*if(g::verbose) {
	DEGUB("EmuDegub 0x%08X\n", r.gpr[3]);
	}*/
	const char* buffer = (const char*)m.getp_translated(r.gpr[3], WC_DEGUB_BUFFER_SIZE);
	DEGUB("EmuDegub: %s", buffer);
	emuDebug(buffer);
	/*if(g::verbose) {
	DEGUB("\n");
	}*/
	if(g::edo_osr)
		//Don't use PostMessage; multiple successive calls will cause
		//early texts to be replaced with duplicated of the later.
		SendMessage(p.hWnd, LM_DEBUG, EDO_ADD, (WPARAM)buffer);
}

void CoreBase::osreport() {
	if(g::verbose) {
		DEGUB("OSReport 0x%08X\n", r.gpr[3]);
	}

	const BYTE *source = m.getp_translated(r.gpr[3], WC_DEGUB_BUFFER_SIZE);
	ostringstream str;
	size_t i=0;
	int parameter=4;
	while(source[i] != 0) {
		if(source[i] != '%') {
			str << source[i];
		} else {
			i++;
			if(parameter > 9)
				throw generic_fatal_exception("Way too many OSReport parameters!");
			if(source[i]==0)
				throw generic_fatal_exception("Unexpected end in OSReport format string!");
			else if(source[i]=='%') {
				str << '%';
			} else if(source[i]=='s') {
				const char *substring = (const char *)
					m.getp_translated(r.gpr[parameter++], WC_DEGUB_BUFFER_SIZE);
				size_t sssize = strlen(substring);
				if(sssize > WC_DEGUB_BUFFER_SIZE)
					throw generic_fatal_exception("OSReport buffer size too small!");
				str.write(substring, sssize);
			} else if(source[i]=='d') {
				str << r.gpr[parameter++];
			} else if(source[i]=='x') {
				str << HEx(r.gpr[parameter++]);
			} else if(memcmp(source+i, "08x", 3) == 0) {
				str << HEx08(r.gpr[parameter++]);
				i += 2;
			} else {
				DEGUB("%s\n", source);
				throw generic_fatal_exception("Unhandled OSReport format type!");
			}
		}
		i++;
	}

	string s = str.str();
	const char* c = s.c_str();
	DEGUB("OSReport: %s", c);
	if(g::verbose) {
		DEGUB("\n");
		DEGUB("OSReport source: %s\n", source);
	}
	SendMessage(p.hWnd, LM_DEBUG, EDO_ADD, (LPARAM)c);
}

void CoreBase::do_oovpa() {
	if(g::hle) {
		for(size_t i=0; i<text_sections.size(); i++) {
			oovpa_do_section(text_sections[i].address, text_sections[i].size);
		}
	}
}

void CoreBase::dsp_quick_execute() {
	//const void* code, int codeSize, void* data, int entryPoint
	h.getDspCore().quickExecute(m.getp_translated(r.gpr[3], r.gpr[4]), r.gpr[4],
		m.getp_translated(r.gpr[5], 64), r.gpr[6], (WORD)r.gpr[7]);
}

bool CoreBase::load(const char *filename) {
	char drive[_MAX_DRIVE];
	char dir[_MAX_DIR];
	char fname[_MAX_FNAME];
	char ext[_MAX_EXT];
	_splitpath_s(filename, drive, dir, fname, ext);

	if(_stricmp(ext, ".elf") == 0) {
		TGLE(load_elf(filename));
	} else if(_stricmp(ext, ".dol") == 0) {
		TGLE(load_dol(filename));
	} else if(_stricmp(ext, ".gcm") == 0) {
		TGLE(load_gcm(filename));
	} else {
		FAIL(UE_INVALID_FILE_TYPE);
	}
	if(g::use_map) {
		char path[MAX_PATH];
		_makepath_s(path, drive, dir, fname, "map");
		TGLE(load_map(path));
	}
	if(_stricmp(ext, ".gcm") != 0)
		do_oovpa();

	TGLE(handle_load());

	return true;
}

bool CoreBase::load_map(const char *filename) {
	DEGUB("Loading %s\n", filename);
	ifstream mfile(filename);
	if(!mfile.good()) {
		DEGUB("Couldn't open file \"%s\"\n", filename);
		FAIL(UE_CANNOT_OPEN_FILE);
	}

#define EOF_BREAK(func)// if((func) == EOF) break;
#define MISSING_SIZE MAX_DWORD

	stdext::hash_map<DWORD, SYMBOL> symbol_map;
	DWORD max_paddress = 0;
	vector<string> vsymbols_ignore;
#define MAPFILETYPES(macro) macro(UNKNOWN), macro(SDK), macro(DEVKITPPC),\
	macro(DEVKITPPC_2), macro(LINUX)
	enum MapFileTypes { MAPFILETYPES(NOTHING), NMAPFILETYPES };
	const char *str_mapfiletypes[] = { MAPFILETYPES(STRINGIZE) };
	int nsymbols[NMAPFILETYPES] = {0};

	//read list of ignored symbols
	{
#define IGNORE_FILENAME "ignore.ini"
		ifstream ifile(IGNORE_FILENAME);
		if(!ifile.good()) {
			DEGUB("%s not found\n", IGNORE_FILENAME);
		} else {
			DEGUB("Loading %s...\n", IGNORE_FILENAME);
			while(ifile.good()) {
				string line;
				ifile >> line;
				vsymbols_ignore.push_back(line);
				DEGUB("Ignoring %s\n", line.c_str());
			}
			DEGUB("Ignore loading complete: %i symbols\n", vsymbols_ignore.size());
		}
	}

	//read symbol data
	while(mfile.good()) {
		char line[1024], name[1024];
		DWORD address, size;
		bool valid = false;

		getline(mfile, line);
		if(!mfile.good())
			break;
		{ //DolphinSDK type
			int res = sscanf_s(line, " %*x %x %x 4 %s", &size, &address, name, sizeof(name));
			EOF_BREAK(res);
			if(res == 3) {
				valid = true;
				nsymbols[SDK]++;
			}
		}
		if(!valid) { //DevkitPPC type 1
			int a=0, b=0, c=0;
			int res = sscanf_s(line, " 0x%x %n%s %n", &address, &a, name, sizeof(name), &b, &c);
			EOF_BREAK(res);
			if(res == 2 && a == 26 && c == 0) {
				size = MISSING_SIZE;
				valid = true;
				nsymbols[DEVKITPPC]++;
			}
		}
		if(!valid) { //DevkitPPC type 2
			int a=0, b=0, d=0;
			int res = sscanf_s(line, " %n0x%x %n%s %n", &d, &address,
				&a, name, sizeof(name), &b);
			EOF_BREAK(res);
			if(res == 2 && d == 16 && a == 42 && b == 0) {
				size = MISSING_SIZE;
				valid = true;
				nsymbols[DEVKITPPC_2]++;
			}
		}
		if(!valid) { //linux type
			int a=0, b=0;
			char c;
			int res = sscanf_s(line, " %x %n%c %n%s", &address, &a, &c, sizeof(c),
				&b, name, sizeof(name));
			EOF_BREAK(res);
			if(res == 3 && a == 9 && b == 11) {
				size = MISSING_SIZE;
				valid = true;
				nsymbols[LINUX]++;
			}
		}
		if(valid) {
			/*bool ignored = false;
			for(size_t i=0; i<vsymbols_ignore.size(); i++) {
			if(string(name) == vsymbols_ignore[i]) {
			ignored = true;
			break;
			}
			}
			if(!ignored) {*/
			stdext::hash_map<DWORD, SYMBOL>::iterator itr = symbol_map.find(address);
			if(itr == symbol_map.end()) {
				SYMBOL symbol = { PHYSICALIZE(address), size, name };
				DEGUB("Adding");
				if(size == MISSING_SIZE) {
					if(symbol.paddress > max_paddress)
						max_paddress = symbol.paddress;
				} else {
					if((symbol.paddress + symbol.size) > max_paddress)
						max_paddress = (symbol.paddress + symbol.size);
					DEGUB(" %06X", size);
				}
				DEGUB(" %08X %s\n", address, name);
				symbol_map.insert(pair<DWORD, SYMBOL>(address, symbol));
				symbol_array.push_back(symbol);
			} else {
				DEGUB("Doublette: %06X %08X %s\n", size, address, name);
				//FAIL(UE_GENERIC_ERROR);
			}
			//}
		}
	}
	mfile.close();

	if(symbol_array.size() > 0) {
		//handle map file types
		MapFileTypes type=UNKNOWN;
		int types=0;
		for(int i=0; i<NMAPFILETYPES; i++) {
			if(nsymbols[i] > 0) {
				types++;
				DEGUB("%s symbols: %i\n", str_mapfiletypes[i], nsymbols[i]);
				type = (MapFileTypes)i;
			}
		}
		if(types != 1) {
			FAIL(UE_GENERIC_ERROR);
		}

		//handle missing sizes
		for(size_t i=0; i<symbol_array.size() - 1; i++) {
			SYMBOL &cur = symbol_array[i], &next = symbol_array[i+1];
			if(cur.size == MISSING_SIZE) {
				if(type == LINUX) {
					cur.size = next.paddress - cur.paddress;
					if((cur.size & 3) != 0) {
						cur.size = 0;
					}
					DEGUB("Sized %08X %s: %i bytes\n", cur.paddress, cur.name.c_str(), cur.size);
				} else {
					cur.size = 4;
				}
			}
		}
		if(symbol_array[symbol_array.size()-1].size == MISSING_SIZE)
			symbol_array[symbol_array.size()-1].size = 0;
	}

	//fill in the physical address map
	symbol_pmap.resize(max_paddress >> 2);
	memset_dword(symbol_pmap.p(), BAD_ADDRESS, symbol_pmap.size());
	for(size_t i=0; i<symbol_array.size(); i++) {
		SYMBOL &sym = symbol_array[i];
		if((sym.paddress & 3) != 0)
			continue;
		bool ignored = false;
		for(size_t j=0; j<vsymbols_ignore.size(); j++) {
			if(string(sym.name) == vsymbols_ignore[j]) {
				ignored = true;
				break;
			}
		}
		if(!ignored) {
			//MYASSERT((sym.size & 3) == 0);
			//MYASSERT((sym.paddress & 3) == 0);
			for(DWORD pa=sym.paddress; pa<sym.paddress+sym.size; pa+=4) {
				symbol_pmap[pa >> 2] = i;
			}
		}
	}

	DEGUB("Mapping complete. Found %i symbols.\n", symbol_map.size());
	return true;
}

void CoreBase::do_symbol(DWORD address) {
	static size_t prev_index = BAD_ADDRESS;
	//DWORD paddress = PHYSICALIZE(address);
	size_t index = get_symbol_index(address);
	if(index != -1 && index != prev_index) {
		DEGUB("Symbol: ");
		degub_symbol(index, address);
		DEGUB(" @ cycle %I64i\n", getCycles());
		prev_index = index;
	}
}
void CoreBase::degub_symbol(size_t index, DWORD address) {
	MYASSERT(index < symbol_array.size());
	DWORD paddress = PHYSICALIZE(address);
	DEGUB("%s", symbol_array[index].name.c_str());
	if(paddress != symbol_array[index].paddress) {
		DEGUB(" + 0x%X", paddress - symbol_array[index].paddress);
	}
}

size_t CoreBase::get_symbol_index(DWORD address) {
	DWORD p_instr_num = PHYSICALIZE(address) >> 2;
	if(p_instr_num < symbol_pmap.size())
		return symbol_pmap[p_instr_num];
	else
		return BAD_ADDRESS;
}

struct GCM_header_1 {
	char game_id[3];
	char country_code;
	char makercode[2];
	char _padding1[22];
	DWORD magic;	//big endian...
	char gamename[GCMNAME_SIZE];
	DWORD db_discpos; //big endian...
	DWORD db_memaddr; //big endian...
	char _padding2[0x18];
	DWORD dol_discpos;  //big endian...
	DWORD fst_discpos;  //big endian...
	DWORD fst_size; //big endian...
	DWORD fst_maxsize;  //big endian...
};

struct GCM_header_apploader {
	char date[0x10];
	DWORD entrypoint;
	DWORD size;
	DWORD _unknown;
	DWORD _reserved;
};

//DOL stuff, Text aNd Data
#define TND(macro) \
	macro(text_filepos, 7)\
	macro(data_filepos, 11)\
	macro(text_address, 7)\
	macro(data_address, 11)\
	macro(text_size, 7)\
	macro(data_size, 11)
#define DECLARE_TND(id, size) DWORD id[size];
#define READ_TND(id, size) file.readbswapw(id, size);

const char *CoreBase::getGameName() {
	return p.gamename;
}
const char *CoreBase::getGameCode() {
	return p.gamecode;
}

bool CoreBase::load_gcm(const char *filename) {
	ReadOnlyFile file;
	GCM_header_1 gcmh_1;
	GCM_header_apploader gcmh_a;
	DWORD ArenaLo = 0;

	TGLE(file.open(filename));

	MYASSERT(sizeof(gcmh_1) == 0x430);
	TGLE(file.read(&gcmh_1, sizeof(gcmh_1)));

	for(int i=0; i<4; i++) {
		BYTE b = ((BYTE*)&gcmh_1)[i];
		p.gamecode[i] = (b == 0) ? ' ' : b;
	}
	p.gamecode[4] = 0;
	strcpy(p.gamename, gcmh_1.gamename);
	DEGUB("Loading GCM. Code: %s\nName: %s\n", p.gamecode, p.gamename);

#define SWAP_DWORD(i) { i = swapw(i); }
#define GCMH1_DWORDS(macro) \
	macro(gcmh_1.magic)\
	macro(gcmh_1.db_discpos)\
	macro(gcmh_1.db_memaddr)\
	macro(gcmh_1.dol_discpos)\
	macro(gcmh_1.fst_discpos)\
	macro(gcmh_1.fst_size)\
	macro(gcmh_1.fst_maxsize)\

	GCMH1_DWORDS(SWAP_DWORD);
	GCMH1_DWORDS(DUMPDWORD);

	TGLE(file.seek(0x2440));
	MYASSERT(sizeof(gcmh_a) == 0x20);
	TGLE(file.read(&gcmh_a, sizeof(gcmh_a)));

#define GCMHA_DWORDS(macro) \
	macro(gcmh_a.entrypoint)\
	macro(gcmh_a.size)\
	macro(gcmh_a._unknown)\
	macro(gcmh_a._reserved)\

	GCMHA_DWORDS(SWAP_DWORD);
	GCMHA_DWORDS(DUMPDWORD);

	uint pos;
	TGLE(file.tell(&pos));
	MYASSERT(pos == 0x2460);

	TGLE(file.read(m.getp_physical(0x01200000, gcmh_a.size), gcmh_a.size));

	{
		uint filesize;
		TGLE(file.size(&filesize));
		TGLE(file.seek(gcmh_1.dol_discpos));
		TND(DECLARE_TND);
		TND(READ_TND);
		for(int i=0; i<7; i++) {
			if(text_filepos[i] != 0 || text_address[i] != 0 || text_size[i] != 0) {
				DEGUB("TEXT %i:  filepos=%08X  address=%08X  size=%08X\n", i,
					text_filepos[i], text_address[i], text_size[i]);
				if(text_size[i] == 0)
					continue;
				if(text_size[i] % 4 != 0) {
					DEGUB("Unaligned text size!\n");
					FAIL(UE_INVALID_DOL);
				}
				if(text_filepos[i] >= filesize ||
					text_filepos[i] + text_size[i] > filesize)
				{
					DEGUB("Segment out of filesize bounds!\n");
					DEGUB("pos+size: %X | filesize: %X\n",
						data_filepos[i] + data_size[i], filesize);
					FAIL(UE_INVALID_DOL);
				}
				//Check for overlapped segments (?) to help fledgling DOL authors?
				//Nah, DOL making is automated nowadays.

				if(ArenaLo < text_address[i]+text_size[i])
					ArenaLo = text_address[i]+text_size[i];

				SECTION section = { text_address[i], text_size[i] };
				text_sections.push_back(section);
			}
		}
		for(int i=0; i<11; i++) {
			if(data_filepos[i] != 0 || data_address[i] != 0 || data_size[i] != 0) {
				DEGUB("DATA %i:  filepos=%08X  address=%08X  size=%08X\n", i,
					data_filepos[i], data_address[i], data_size[i]);
				if(data_size[i] == 0)
					continue;
				if(data_filepos[i] >= filesize ||
					data_filepos[i] + data_size[i] > filesize)
				{
					DEGUB("Segment out of filesize bounds!\n");
					DEGUB("pos+size: %X | filesize: %X\n",
						data_filepos[i] + data_size[i], filesize);
					FAIL(UE_INVALID_DOL);
				}

				if(ArenaLo < data_address[i]+data_size[i])
					ArenaLo = data_address[i]+data_size[i];

				SECTION section = { data_address[i], data_size[i] };
				data_sections.push_back(section);
			}
		}
	}

	h.unloadgcm();
	h.loadgcm(filename);
	h.setlid(false);
	//Load BS2
	{
		HRSRC hrsrc;
		HGLOBAL hRes;
		//TGLE(hrsrc = FindResource(NULL, MAKEINTRESOURCE(IDR_BS2), RT_RCDATA));
		TGLE(hrsrc = FindResource(NULL, MAKEINTRESOURCE(IDR_BS2), "Binary"));
		TGLE(hRes = LoadResource(NULL, hrsrc));
		m.write(0x81300000, SizeofResource(NULL, hrsrc), LockResource(hRes));
		TGLER(FreeResource(hRes));
	}
	//dependant on BS2.asm
	r.gpr[0] = gcmh_a.entrypoint;
	entry_point = 0x81300020;

	TGLE(dol_bios_emu(ArenaLo)); //good? bad? irrelevant?

	return true;
}

bool CoreBase::load_dol(const char *filename) {
	ReadOnlyFile file;
	TGLE(file.open(filename));

	//Container<BYTE> temp(0);
	DWORD ArenaLo = 0;//, ArenaHi = (MAIN_MEMORY_SIZE - 1*M) | 0x80000000;

	uint filesize;
	TGLE(file.size(&filesize));
	if(filesize < 256 + 4) {  //Smallest valid DOL imaginable
		DEGUB("File is too small to be a DOL (%i bytes)\n", filesize);
		FAIL(UE_INVALID_DOL);
	}

	TND(DECLARE_TND);
	TND(READ_TND);
	TGLE(file.readbswapw(&bss_address, 1));
	TGLE(file.readbswapw(&bss_size, 1));
	TGLE(file.readbswapw(&entry_point, 1));

	for(int i=0; i<7; i++) {
		if(text_filepos[i] != 0 || text_address[i] != 0 || text_size[i] != 0) {
			DEGUB("TEXT %i:  filepos=%08X  address=%08X  size=%08X\n", i,
				text_filepos[i], text_address[i], text_size[i]);
			if(text_size[i] == 0)
				continue;
			if(text_size[i] % 4 != 0) {
				DEGUB("Unaligned text size!\n");
				FAIL(UE_INVALID_DOL);
			}
			if(text_filepos[i] >= filesize ||
				text_filepos[i] + text_size[i] > filesize)
			{
				DEGUB("Segment out of filesize bounds!\n");
				DEGUB("pos+size: %X | filesize: %X\n", data_filepos[i] + data_size[i], filesize);
				FAIL(UE_INVALID_DOL);
			}
			//Check for overlapped segments (?) to help fledgling DOL authors?
			//Nah, DOL making is automated nowadays.

			TGLE(file.seek(text_filepos[i]));
			TGLE(file.read(m.getp_translated(text_address[i], text_size[i]), text_size[i]));
			if(ArenaLo < text_address[i]+text_size[i])
				ArenaLo = text_address[i]+text_size[i];

			SECTION section = { text_address[i], text_size[i] };
			text_sections.push_back(section);
		}
	}
	for(int i=0; i<11; i++) {
		if(data_filepos[i] != 0 || data_address[i] != 0 || data_size[i] != 0) {
			DEGUB("DATA %i:  filepos=%08X  address=%08X  size=%08X\n", i,
				data_filepos[i], data_address[i], data_size[i]);
			if(data_size[i] == 0)
				continue;
			if(data_filepos[i] >= filesize ||
				data_filepos[i] + data_size[i] > filesize)
			{
				DEGUB("Segment out of filesize bounds!\n");
				DEGUB("pos+size: %X | filesize: %X\n", data_filepos[i] + data_size[i], filesize);
				FAIL(UE_INVALID_DOL);
			}

			TGLE(file.seek(data_filepos[i]));
			TGLE(file.read(m.getp_translated(data_address[i], data_size[i]), data_size[i]));
			if(ArenaLo < data_address[i]+data_size[i])
				ArenaLo = data_address[i]+data_size[i];

			SECTION section = { data_address[i], data_size[i] };
			data_sections.push_back(section);
		}
	}
	DEGUB("BSS Address=%08X\tBSS Size=%08X\tEntry Point=%08X\n",
		bss_address, bss_size, entry_point);
	//memset(m.getp_translated(bss_address, bss_size), 0, bss_size);

	TGLE(dol_bios_emu(ArenaLo));

	return true;
}

bool CoreBase::load_elf(const char *filename) {
	ReadOnlyFile file;
	TGLE(file.open(filename));

	Elf32_Ehdr ehdr;
	TGLE(file.read(&ehdr, sizeof(ehdr)));

#define EIMAG_CHECK(nr) (ehdr.e_ident[EI_MAG##nr] == ELFMAG##nr)

	if(!(EIMAG_CHECK(0) && EIMAG_CHECK(1) && EIMAG_CHECK(2) && EIMAG_CHECK(3))) {
		FAIL(UE_INVALID_ELF);
	}

#define ELF_SWAPH(data) ((ehdr.e_ident[EI_DATA] == ELFDATA2MSB) ? swaph(data) : data)
#define ELF_SWAPW(data) ((ehdr.e_ident[EI_DATA] == ELFDATA2MSB) ? swapw(data) : data)

#define ELF_SWAP16(data) data = ELF_SWAPH(data)
#define ELF_SWAP32(data) data = ELF_SWAPW(data)


#define INVALID_INCOMPAT_CHECK(data, invalid_value, compatible_value) \
	if(data == invalid_value) { FAIL(UE_INVALID_ELF); }\
	if(data != compatible_value) { DEGUB("%s = %X\n", #data, data);\
	FAIL(UE_INCOMPATIBLE_ELF); }

	/*if(ehdr.e_ident[EI_CLASS] == ELFCLASSNONE)
	return InvalidELF;
	if(ehdr.e_ident[EI_CLASS] != ELFCLASS32)
	return IncompatibleELF;*/
	INVALID_INCOMPAT_CHECK(ehdr.e_ident[EI_CLASS], ELFCLASSNONE, ELFCLASS32);
	//INVALID_INCOMPAT_CHECK(ehdr.e_ident[EI_DATA], ELFDATANONE, ELFDATA2MSB);
	if(ehdr.e_ident[EI_DATA] != ELFDATA2MSB && ehdr.e_ident[EI_DATA] != ELFDATA2LSB) {
		FAIL(UE_INVALID_ELF);
	}

	INVALID_INCOMPAT_CHECK(ELF_SWAPH(ehdr.e_type), ET_NONE, ET_EXEC);
	INVALID_INCOMPAT_CHECK(ELF_SWAPH(ehdr.e_machine), EM_NONE, EM_PPC);
	INVALID_INCOMPAT_CHECK(ELF_SWAPW(ehdr.e_version), EV_NONE, EV_CURRENT);

	if(ELF_SWAPH(ehdr.e_ehsize) != sizeof(Elf32_Ehdr)) {
		FAIL(UE_INVALID_ELF);
	}
	if(ELF_SWAPH(ehdr.e_shentsize) != sizeof(Elf32_Shdr)) {
		FAIL(UE_INVALID_ELF);
	}
	if(ELF_SWAPH(ehdr.e_phentsize) != sizeof(Elf32_Phdr)) {
		FAIL(UE_INVALID_ELF);
	}

	entry_point = ELF_SWAPW(ehdr.e_entry);

	{ //Read Section Table
		ELF_SWAP16(ehdr.e_shstrndx);
		ELF_SWAP16(ehdr.e_shnum);
		ELF_SWAP32(ehdr.e_shoff);

		Container<char> strings(0);
		if(ehdr.e_shstrndx != 0) {
			Elf32_Shdr shdr;
			TGLE(file.seek(ehdr.e_shoff + ehdr.e_shstrndx * sizeof(Elf32_Shdr)));
			TGLE(file.read(&shdr, sizeof(shdr)));
			TGLE(file.seek(ELF_SWAPW(shdr.sh_offset)));
			ELF_SWAP32(shdr.sh_size);
			strings.resize(shdr.sh_size);
			TGLE(file.read(strings, shdr.sh_size));
		}

		DEGUB("%i sections, offset %X:\n", ehdr.e_shnum, ehdr.e_shoff);
		for(int i=0; i<ehdr.e_shnum; i++) {
			Elf32_Shdr shdr;
			TGLE(file.seek(ehdr.e_shoff + i * sizeof(Elf32_Shdr)));
			TGLE(file.read(&shdr, sizeof(shdr)));
			ELF_SWAP32(shdr.sh_name);
			ELF_SWAP32(shdr.sh_addr);
			ELF_SWAP32(shdr.sh_size);
			if(ehdr.e_shstrndx == 0 && shdr.sh_name != 0) {
				FAIL(UE_INVALID_ELF);
			}
			DEGUB("Name: %s(%i), Type: 0x%X, Flags: %08X, Address: %08X, Offset: 0x%X",
				(shdr.sh_name == 0) ? "" : (&strings[shdr.sh_name]), shdr.sh_name,
				ELF_SWAPW(shdr.sh_type), ELF_SWAPW(shdr.sh_flags),
				shdr.sh_addr, ELF_SWAPW(shdr.sh_offset));
			DEGUB(", Size: 0x%X, Link: 0x%X, Info: 0x%X, Addralign: %i, Entsize: %i\n",
				shdr.sh_size, ELF_SWAPW(shdr.sh_link), ELF_SWAPW(shdr.sh_info),
				ELF_SWAPW(shdr.sh_addralign), ELF_SWAPW(shdr.sh_entsize));

			//if(shdr.sh_name != 0) if(strcmp(&strings[shdr.sh_name], ".sbss") == 0 ||
			//strcmp(&strings[shdr.sh_name], ".bss") == 0)
			//memset(m.getp_translated(shdr.sh_addr, shdr.sh_size), 0, shdr.sh_size);
			//no need to zero bss, memeory's already zeroed.
		}
	}

	DWORD ArenaLo = 0;

	{ //Read Program Table
		ELF_SWAP16(ehdr.e_phnum);
		ELF_SWAP32(ehdr.e_phoff);

		DEGUB("%i segments, offset %X:\n", ehdr.e_phnum, ehdr.e_phoff);
		for(int i=0; i<ehdr.e_phnum; i++) {
			Elf32_Phdr phdr;
			TGLE(file.seek(ehdr.e_phoff + i * sizeof(Elf32_Phdr)));
			TGLE(file.read(&phdr, sizeof(phdr)));
			ELF_SWAP32(phdr.p_type);
			ELF_SWAP32(phdr.p_offset);
			ELF_SWAP32(phdr.p_vaddr);
			ELF_SWAP32(phdr.p_filesz);
			ELF_SWAP32(phdr.p_flags);
			DEGUB("Type: 0x%X, Offset: 0x%X, VAddress: %08X, PAddress: %08X, Filesize: 0x%X",
				phdr.p_type, phdr.p_offset, phdr.p_vaddr, ELF_SWAPW(phdr.p_paddr), phdr.p_filesz);
			DEGUB(", Memsize: 0x%X, Flags: %08X, Align: %i\n",
				ELF_SWAPW(phdr.p_memsz), phdr.p_flags, ELF_SWAPW(phdr.p_align));
			if(phdr.p_type == PT_NULL)
				continue;
			else if(phdr.p_type == PT_LOAD) {
				SECTION section = { phdr.p_vaddr, phdr.p_filesz };
				if(phdr.p_flags & PF_X) { //text
					/*if(phdr.p_filesz % 4 != 0) {
					DEGUB("Unaligned text size!\n");
					FAIL(UE_INVALID_ELF);
					}*/
					text_sections.push_back(section);
				} else {  //data
					data_sections.push_back(section);
				}

				TGLE(file.seek(phdr.p_offset));        
				TGLE(file.read(m.getp_translated(phdr.p_vaddr, phdr.p_filesz), phdr.p_filesz));

				if(ArenaLo < phdr.p_vaddr + phdr.p_filesz)
					ArenaLo = phdr.p_vaddr + phdr.p_filesz;
			} else {
				//FAIL(UE_INCOMPATIBLE_ELF);
			}

			/*else if(phdr.p_type == PT_NOTE) {
			Elf32_Nhdr nhdr;
			TGLE(file.seek(phdr.p_offset));
			TGLE(file.read(&nhdr, sizeof(nhdr)));
			ELF_SWAP32(nhdr.n_namesz);
			ELF_SWAP32(nhdr.n_descsz);
			ELF_SWAP32(nhdr.n_type);
			Container<char> name(nhdr.n_namesz), desc(nhdr.n_descsz);

			DEGUB("Note. Name size: %i, Desc size: %i, Type: %i, Name: %s, Desc: %s\n",

			}*/
		}
	}

	TGLE(dol_bios_emu(ArenaLo));

	return true;
}

bool CoreBase::dol_bios_emu(DWORD ArenaLo) {
	//BIOS/PSO emulation. This is junk. Make it better. Get a ROM/RAM dump or something.
	r.setmsr(0);

	DWORD ArenaHi = 0x817fe8c0; //from YAGCD

	if(g::gcube) { //gcube compat
		ArenaLo = 0x80403100;
		ArenaHi = 0x81700000;
	}
	ArenaLo += 0x70000; //arbitrary buffer size, seems to be enough for czn-mp6

	// console type
	//m.pww(0x2c, 3); //retail 3
	m.pww(0x2c, 0x10000006);  //dolphin-style devkit
	m.pww(0x20, 0x0D15EA5E);  //normal boot

	m.pww(0x30, ArenaLo);	//the lowest address where teh OS may allocate memory
	m.pww(0x34, ArenaHi);	//teh top. should be near 0x818000000. Don't know how near, exactly.

	//moved to Hardware::do_gcmload_hle
	//m.pww(0x38, 0x817fe8c0);  //yagcd
	//m.pww(0x3C, 0x24);

	//memsize, memtop and simmemsize
	m.pww(0x28, MAIN_MEMORY_SIZE);
	m.pww(0xEC, MAIN_MEMORY_SIZE);
	m.pww(0xf0, MAIN_MEMORY_SIZE);

	//purpose unknown
	m.pww(0xe4, 0xeef); //yagcd

	//clock speed (Hz)
	m.pww(0xf8, CPU_BUS_HZ);	//bus clock
	m.pww(0xfc, CPU_CORE_HZ);	//Gekko clock

	//unknown, exi or pad related
	m.pww(0x30e3, 6);

	//default interrupt handlers
	for(DWORD address=0x100; address<=0x1700; address += 0x100) {
		m.pww(address, 0x4C000064);	//rfi
	}

	//r.setmsr(MSR_EE);

	if(g::advanced_mode)
		r.setmsr(r.getmsr() | MSR_DR | MSR_IR);
	//BAT setup as done by PSO?
	/*for(int i=0; i<4; i++) {
	r.ibatu[i] = 0;
	r.ibatl[i] = 0;
	r.dbatu[i] = 0;
	r.dbatl[i] = 0;
	}*/
	r.ibatu[0] = 0x80001FFF;  //8
	r.ibatl[0] = 0x00000012;
	r.ibatu[1] = 0xC0001FFF;  //C
	r.ibatl[1] = 0x00000032;
	r.ibatu[2] = 0x00001FFF;  //0
	r.ibatl[2] = 0x00000012;
	r.ibatu[3] = 0x00001FFC;  //0, disabled
	r.ibatl[3] = 0x00000012;
	r.dbatu[0] = 0x80001FFF;  //8
	r.dbatl[0] = 0x0000001A;
	r.dbatu[1] = 0xC0001FFF;  //C
	r.dbatl[1] = 0x0000007A;
	r.dbatu[2] = 0x00001FFF;  //0
	r.dbatl[2] = 0x0000001A;
	r.dbatu[3] = 0x00001FFC;  //0, disabled
	r.dbatl[3] = 0x0000001A;
	//r.dbatu[3] = 0xCC000003;
	//r.dbatl[3] = 0x0C00007A;
	m.bat_notify_all();

	//from gcube
	r.gpr[1] = ArenaHi;	//stack pointer
	r.lr = INTERRUPT_SYSTEMRESET;

	TGLE(h.do_gcmload_hle());
	return true;
}

bool CoreBase::mfspr_requires_processing(DWORD spr) const {
	r.mfspr_is_valid(spr);
	switch(spr) {
	case 1: //xer
	case 8: //lr
	case 9: //ctr
	case 25:  //sdr1
	case 26:  //srr0
	case 27:  //srr1
	case 272: //sprg0
	case 273: //sprg1
	case 274: //sprg2
	case 275: //sprg3
	case 912: //gqr0, hack-added for newlibogc
	case 913: //gqr1, hack-added for costis' modplayer (tetris)
	case 914: //gqr2
	case 915: //gqr3
	case 916: //gqr4
	case 917: //gqr5
	case 918: //gqr6
	case 919: //gqr7
	case 1008:  //hid0, hack-added for costis' modplayer (tetris)
	case 528: //dbat0l, hack-added for gc-linux
	case 529:
	case 530:
	case 531:
	case 532:
	case 533:
	case 534:
	case 535:
	case 536:
	case 537:
	case 538:
	case 539:
	case 540:
	case 541:
	case 542:
	case 543: //ibat3u, hack-added for gc-linux
	case 287: //pvr
	case 18:  //dsisr
	case 19:  //dar
		return false;
	default:
		return true;
	}
}

//may be called by recompiled code
void CoreBase::mfspr_with_processing(DWORD opcode) {
	DWORD spr = getbitsw(opcode, 11, 15) | (getbitsw(opcode, 16, 20) << 5);
	DWORD rD = getbitsw(opcode, 6, 10);

	r.mfspr_is_valid(spr);
	switch(spr) {
	case 22:  //DEC //so far only costis' mp3 player reads the DEC.
		r.gpr[rD] = getDEC();
		break;
	case 920: //HID2
		DEGUB("HID2 read: %08X\n", r.hid2.word);
		r.gpr[rD] = r.hid2.word;
		if(r.hid2.dmaql > 0)
			r.hid2.dmaql--;
		break;
	case 921: //WPAR
		r.wpar = (r.wpar & makemaskw(0, 26)) | (h.wgp_empty() ? 0 : 1);  //important!
		break;
	case 1017:  //l2cr
	case 952: //mmcr0
	case 956: //mmcr0
	case 953: //pmc1
	case 954: //pmc2
	case 957: //pmc3
	case 958: //pmc4
	case 1009:  //hid1
		r.gpr[rD] = *r.getmfspr(spr);
		DEGUB("Ignored SPR read: %s (%i) %08X\n",
			r.getsprasm(spr).c_str(), spr, *r.getmfspr(spr));
		break;
	default:
		r.gpr[rD] = *r.getmfspr(spr);
		DEGUB("Unhandled SPR read: %s (%i) %08X\n",
			r.getsprasm(spr).c_str(), spr, *r.getmfspr(spr));
		//if(g::bouehr)
		throw bouehr_exception("Unhandled SPR read");
	}
}

bool CoreBase::mtspr_requires_processing(DWORD spr) const {
	r.mtspr_is_valid(spr);
	switch(spr) {
	case 1: //xer
	case 8: //lr
	case 9: //ctr
	case 26:  //srr0
	case 27:  //srr1
	case 272: //sprg0
	case 273: //sprg1
	case 274: //sprg2
	case 275: //sprg3
	case 1008:  //hid0, hack-added for costis' modplayer (tetris) and gc-linux
	case 1009:  //hid1, hack-added for gc-linux
		return false;
	default:
		return true;
	}
}
void CoreBase::mtspr_with_processing(DWORD opcode) {
	DWORD spr = getbitsw(opcode, 11, 15) | (getbitsw(opcode, 16, 20) << 5);
	DWORD rS = getbitsw(opcode, 6, 10);

	r.mtspr_is_valid(spr);
	switch(spr) {
	case 921: //WPAR
		//r.wpar = r.gpr[rS] & makemaskw(0, 26);
		if(r.gpr[rS] != 0x0C008000) {
			DEGUB("mtWPAR error: r%i = 0x%08X. Well, screw you, cause I'm setting it to 0x0C008000 anyway! ;)",
				rS, r.gpr[rS]);
			throw bouehr_exception("Invalid mtWPAR!");
		}
		r.wpar = 0x0C008000;
		DEGUB("mtWPAR 0x0C008000\n");
		break;
	case 22:  //DEC
		r.dec = r.gpr[rS];
		set_dec_event(getCT());
		if(g::log_interrupts)
			DEGUB("mtDEC 0x%08X\n", r.dec);
		break;
	case 284: //TBL
		r.tbl = r.gpr[rS];
		setCT(&p.tbl_wt);
		if(g::log_interrupts)
			DEGUB("mtTBL 0x%08X\n", r.tbl);
		break;
	case 285: //TBU
		r.tbu = r.gpr[rS];
		setCT(&p.tbu_wt);
		if(g::log_interrupts)
			DEGUB("mtTBU 0x%08X\n", r.tbu);
		break;
	case 528: //ibat0u
	case 529:
	case 530:
	case 531:
	case 532:
	case 533:
	case 534:
	case 535: //ibat3l
	case 536: //dbat0u
	case 537:
	case 538:
	case 539:
	case 540:
	case 541:
	case 542:
	case 543: //dbat3l
		//if(!g::advanced_mode && g::bouehr)
		//throw bouehr_exception("BAT write with /amode off!");
		DEGUB("%s(spr %i) write: %08X\n", r.getsprasm(spr).c_str(), spr, r.gpr[rS]);
		if(*r.getmtspr(spr) != r.gpr[rS]) {
			*r.getmtspr(spr) = r.gpr[rS];
			m.bat_notify(spr);
		}
		break;
	case 25:  //sdr1
		DEGUB("mtSDR1 0x%08X\n", r.gpr[rS]);
		//if(r.getmsr() & (MSR_IR | MSR_DR))
		//throw generic_fatal_exception("SDR1 write while MSR[IR] or [DR]!");
		r.sdr1 = r.gpr[rS];
		if((getbitsw(r.sdr1, 7, 15) & getbitsw(r.sdr1, 23, 31)) != 0)
			throw generic_fatal_exception("Invalid SDR1, type 1!");

		{
			int first_one = 32;
			bool one=false;
			for(int i=23; i<=31; i++) {
				if(one && !getbit(r.sdr1, i))
					throw generic_fatal_exception("Invalid SDR1, type 2!");
				if(getbit(r.sdr1, i))
					one = true;
			}
			DEGUB("Page table size: 2^%i groups, 2^%i pages, %s\n",
				42 - first_one, 45 - first_one, spbytes(1 << (48 - first_one)).c_str());
		}
		//m.sdr1_notify();
		break;
	case 912: //gqr0
	case 913: //gqr1
	case 914: //gqr2
	case 915: //gqr3
	case 916: //gqr4
	case 917: //gqr5
	case 918: //gqr6
	case 919: //gqr7
		spr -= 912;
		r.gqr[spr] = r.gpr[rS];
		gqrp[spr].ld_type = (BYTE)getbitsw(r.gqr[spr], 13, 15);
		gqrp[spr].ld_scale = pow(2.0f, -exts6((char)getbitsw(r.gqr[spr], 2, 7)));
		gqrp[spr].st_type = (BYTE)getbitsw(r.gqr[spr], 29, 31);
		gqrp[spr].st_scale = pow(2.0f, exts6((char)getbitsw(r.gqr[spr], 18, 23)));
		/*if(gqrp[spr].ld_type != 0 || gqrp[spr].st_type != 0) {
		DEGUB("mtGQR%i: %08X (LD_TYPE: %s, LD_SCALE: %g, ST_TYPE: %s, ST_SCALE: %g)\n",
		spr, r.gpr[rS], gq_type[gqrp[spr].ld_type], gqrp[spr].ld_scale,
		gq_type[gqrp[spr].st_type], gqrp[spr].st_scale);
		}*/
		break;
	case 920: //hid2
		{
			HID2 source;
			source.word = r.gpr[rS];
			DEGUB("HID2 write: %08X\n", source.word);
			if(source.wpe && !r.hid2.wpe) {
				DEGUB("Write Pipe Enabled\n");
				HwGX::check_wpe(&h);
			} else if(!source.wpe && r.hid2.wpe) {
				DEGUB("Write Pipe Disabled\n");
			}
			source.dmaql = r.hid2.dmaql;
			r.hid2 = source;
		}
		break;
	case 922:
		DEGUB("DMAU write: %08X\n", r.gpr[rS]);
		r.dmau = r.gpr[rS];
		break;
	case 923:
		{
			DWORD data = r.gpr[rS];
			DEGUB("DMAL write: %08X\n", data);
			if(getbit(data, 31))
				throw generic_fatal_exception("Locked Cache DMA Flush unemulated!");
			if(getbit(data, 30)) {  //command start
				bool mode = getbit(data, 27);
				DWORD mem = r.dmau & makemaskw(0, 26);
				DWORD lc = data & makemaskw(0, 26);
				DWORD len = (getbitsw(r.dmau, 27, 31) << 2) | getbitsw(data, 28, 29);
				if(len == 0) len = 128;
				DWORD size = len * 32;
				DEGUB("Locked Cache %s | mem 0x%08X | lc 0x%08X | 0x%X(%i) bytes\n",
					mode ? "Load" : "Store", mem, lc, size, size);
				if(lc < LOCKED_CACHE_BASE)
					throw generic_fatal_exception("Locked Cache DMA: bad LC address!");
				if(mode)
					m.read_physical(mem, size, m.getp_physical(lc, size));
				else
					m.write_physical(mem, size, m.getp_physical(lc, size));
				data &= ~makeflag(30);
				r.dmal = data;
				if(r.hid2.dmaql == 15)
					throw generic_fatal_exception("Locked Cache DMA overflow!");
				r.hid2.dmaql++;
			}
		}
		break;
	case 1017:  //l2cr
	case 952: //mmcr0
	case 956: //mmcr0
	case 953: //pmc1
	case 954: //pmc2
	case 957: //pmc3
	case 958: //pmc4
	case 1019:  //ictc
#ifdef WII
	case 1011:  //hid4
#endif	//WII
		*r.getmtspr(spr) = r.gpr[rS];
		DEGUB("Ignored SPR write: %s (%i) <- %08X\n",
			r.getsprasm(spr).c_str(), spr, r.gpr[rS]);
		break;
	default:
		*r.getmtspr(spr) = r.gpr[rS];
		DEGUB("Unhandled SPR write: %s (%i) <- %08X\n",
			r.getsprasm(spr).c_str(), spr, r.gpr[rS]);
		//if(g::bouehr)
		throw bouehr_exception("Unhandled SPR write");
	}

}

inline void CoreBase::setCT(ULI *uli) {
	if(g::timing_mode == g::TM_REALTIME) {
		GET_RTC(uli);
	} else {
		MYASSERT(g::timing_mode == g::TM_EXACT_FAST || g::timing_mode == g::TM_EXACT_REAL);
		*uli = getCycles();
	}
}

inline ULI CoreBase::getCT() {
	ULI uli;
	if(g::timing_mode == g::TM_REALTIME) {
		GET_RTC(&uli);
		return uli;
	} else {
		MYASSERT(g::timing_mode == g::TM_EXACT_FAST || g::timing_mode == g::TM_EXACT_REAL);
		return getCycles();
	}
}

/*#define RETURNDEGUB(name, value) { int temp = (value);\
DEGUB("%s: %i\n", name, temp); return temp; }*/
#define RETURNDEGUB(name, value) return (value)
//#define RETURNDEGUB(name, value) { DWORD temp = (value); if(g::log_interrupts) {\
//DEGUB("%s\n"/*: 0x%08X\n"*/, name, temp); } return temp; }
//degub-hack

DWORD CoreBase::getDEC() {
	if(g::timing_mode == g::TM_REALTIME) {
		ULI uli;
		GET_RTC(&uli);
		RETURNDEGUB("getDEC", r.dec + RTC_2_TBL(uli - p.dec_wt));
	} else if(g::timing_mode == g::TM_EXACT_FAST) {
		ULI uli;
		uli = getCycles() - p.dec_wt;
		RETURNDEGUB("getDEC", r.dec + uli.LowPart);
	} else if(g::timing_mode == g::TM_EXACT_REAL) {
		RETURNDEGUB("getDEC", r.dec + CYCLES_2_TBL(getCycles() - p.dec_wt));
	} else
		throw generic_fatal_exception("BFE!");
}
DWORD CoreBase::getTBL() {
	p.tbl_reads++;
	if(g::timing_mode == g::TM_REALTIME) {
		ULI uli;
		GET_RTC(&uli);
		RETURNDEGUB("getTBL", r.tbl + RTC_2_TBL(uli - p.tbl_wt));
	} else if(g::timing_mode == g::TM_EXACT_FAST) {
		ULI uli;
		uli = getCycles() - p.tbl_wt;
		RETURNDEGUB("getTBL", r.tbl + uli.LowPart);
	} else if(g::timing_mode == g::TM_EXACT_REAL) {
		RETURNDEGUB("getTBL", r.tbl + CYCLES_2_TBL(getCycles() - p.tbl_wt));
	} else
		throw generic_fatal_exception("BFE!");
}
DWORD CoreBase::getTBU() {
	if(g::timing_mode == g::TM_REALTIME) {
		ULI uli;
		GET_RTC(&uli);
		RETURNDEGUB("getTBU", r.tbu + RTC_2_TBU(uli - p.tbu_wt));
	} else if(g::timing_mode == g::TM_EXACT_FAST) {
		ULI uli;
		uli = getCycles() - p.tbu_wt;
		RETURNDEGUB("getTBU", r.tbu + uli.HighPart);
	} else if(g::timing_mode == g::TM_EXACT_REAL) {
		RETURNDEGUB("getTBU", r.tbu + CYCLES_2_TBU(getCycles() - p.tbu_wt));
	} else
		throw generic_fatal_exception("BFE!");
}

#undef RETURNDEGUB

bool CoreBase::run(DWORD runstop, bool degub, QWORD degub_skip, bool dumpmem) {
	MYASSERT(p.q == false);
	/*if(!g::dual_run)*/ {
		GET_RTC(&p.rtc_start);

		if(g::timing_mode == g::TM_REALTIME) {
			CSSHandler csshandler(css_interrupts);
			list<TIMED_EVENT>::iterator itr;
			for(itr = timed_events.begin(); itr != timed_events.end(); itr++) {
				itr->time += p.rtc_start;
			}
		}
		interrupt.dump_events("Runstart");

		ULI ct = getCT();
		p.tbl_wt = p.tbu_wt = ct;
		set_dec_event(ct);
		p.running = true;

		DEGUB("Run period %i\n", ++periods);
		TGLE(h.startPeriodicTasks());
	}
	bool result = _run(runstop, degub, degub_skip, dumpmem);
	/*if(!g::dual_run)*/ {
		ULI rtc_stop;
		GET_RTC(&rtc_stop);
		p.rtc_completed += rtc_stop - p.rtc_start;
		p.running = false;
		r.tbl = getTBL();
		r.tbu = getTBU();
		r.dec = getDEC();
		p.q = false;
		DEGUB("Quit signal reset.\n");

		if(g::timing_mode == g::TM_REALTIME) {
			CSSHandler csshandler(css_interrupts);
			list<TIMED_EVENT>::iterator itr;
			for(itr = timed_events.begin(); itr != timed_events.end(); itr++) {
				itr->time -= rtc_stop;
			}
		}
		interrupt.dump_events("Runstop");
	}
	return result;
}

void CoreBase::getVerboseText(ostream& str) {
	str << "TBL: "<<p.tbl_reads<<"\n";
}

void CoreBase::getTimings(std::vector<TIMING> &timings) {
	const size_t core = timings.size();
	timings.push_back(TIMING("Core", p.rtc_completed));
	h.getTimings(timings);
	for(size_t i=core+1; i<timings.size(); i++) {
		timings[core].t -= timings[i].t;
	}
	if(p.running) {
		ULI rtc;
		GET_RTC(&rtc);
		timings[core].t += rtc - p.rtc_start;
	}
}

void CoreBase::set_dec_event(ULI event_time) {
	p.dec_wt = event_time;

	//calculate
	ULI new_time;
	if(g::timing_mode == g::TM_REALTIME) {
		new_time = event_time + TB_2_RTC(r.dec);
	} else if(g::timing_mode == g::TM_EXACT_FAST) {
		new_time = event_time + r.dec;
	} else if(g::timing_mode == g::TM_EXACT_REAL) {
		new_time = event_time + TB_2_CYCLES(r.dec);
	}
	//find & erase
	interrupt.remove_events(event_test<(event_callback)dec_event>());
	//add
	TIMED_EVENT event;
	event.function = (event_callback)dec_event;
	event.arg = this;
	event.time = new_time;
	event.desc = "DEC";
	interrupt.add_event(event);
}

void dec_event(CoreBase *cb) {
	CSSHandler csshandler(css_interrupts);
	//if(r.getmsr() & MSR_EE)
	cb->interrupt.raise_unless_pending(INTERRUPT_DEC, "DEC");
	cb->r.dec = 0xFFFFFFFF;
	cb->set_dec_event(cb->interrupt.event_time());
}

bool CoreBase::do_interrupts() {
	//VDEGUB("CoreBase::do_interrupts() @ %I64i cycles\n", cc.cycles);
	CSSHandler csshandler(css_interrupts);
	ULI et;
	setCT(&et);

	if(et.isAfter(p.event_time)) {
		event_callback function = timed_events.front().function;
		void *arg = timed_events.front().arg;
		/*if(g::log_interrupts && g::verbose) {
		DEGUB("Taking event: arg:0x%08X func:0x%08X time:0x%016I64X(%I64i) desc: %s\n",
		arg, function, timed_events.front().time.QuadPart,
		timed_events.front().time.QuadPart, timed_events.front().desc.c_str());
		}*/
		interrupt.dump_events("Event taken");
		timed_events.pop_front();
		function(arg);	//may add new events
		MYASSERT(timed_events.size() >= 1);
		p.event_time = timed_events.front().time;
	}

	h.do_interrupts();

	if(pending_interrupts.empty())
		return true;

	//Iterate through the list of pending interrupts, break on the first one that can be taken.
	//Order is important.
	std::list<PENDING_INTERRUPT>::iterator it = pending_interrupts.begin();
	while(it != pending_interrupts.end()) {
		bool takeInterrupt = true;

		switch((*it).vector) {
			//case INTERRUPT_SYSTEMRESET:
		case INTERRUPT_MACHINECHECK:
		case INTERRUPT_DSI:
		case INTERRUPT_ISI:
		case INTERRUPT_SC:
		case INTERRUPT_PROGRAM:
			break;
		case INTERRUPT_EXTERNAL:
		case INTERRUPT_DEC:
			if(!(r.getmsr() & MSR_EE)) {// || !last_instruction_was_branch) {
				takeInterrupt = false;
				if(g::log_interrupts && g::verbose) {
					DEGUB("Interrupt 0x%04X @ 0x%08X blocked by %s | cycle %I64i\n",
						(*it).vector, getNIA(), !(r.getmsr() & MSR_EE) ? "MSR_EE" : "liwb",
						cc.cycles.QuadPart);
				}
				//if(r.getmsr() & MSR_EE)
				//DEGUB("liwb\n");
			}
			break;
		default:
			DEGUB("Unsupported Interrupt 0x%04X @ 0x%08X | cycle %I64i\n",
				(*it).vector, getNIA(), cc.cycles.QuadPart);
			FAIL(UE_GENERIC_ERROR);
		}
		if(takeInterrupt) {
			if(g::log_interrupts) {
				DEGUB("Taking Interrupt 0x%04X @ 0x%08X | cycle %I64i\n",
					(*it).vector, getNIA(), cc.cycles.QuadPart);
				if(g::verbose) {
					DELTA_TIMING;
				}
			}

			if((*it).callback != NULL)
				(*it).callback(&h);
			set_interrupt_registers((*it).vector, getNIA());
			TGLE(take_interrupt((r.getmsr() & MSR_IP) ?
				((*it).vector | 0xFFF00000) : ((*it).vector & 0x000FFFFF)));
			pending_interrupts.erase(it);
			break;
		}

		it++;
	}
	return true;
}

void CoreBase::set_interrupt_registers(WORD vector, DWORD srr0) {
	//r.setmsr(r.getmsr() | MSR_RI);  //This may be a hack
	DWORD msr = r.getmsr() | MSR_RI;

	r.srr0 = srr0;
	DWORD msrcopymask, srr1clearmask, msrclearmask=0;
	switch(vector) {
	case INTERRUPT_EXTERNAL:
	case INTERRUPT_DSI:
	case INTERRUPT_DEC:
		srr1clearmask = makemaskw(1, 4) | makemaskw(10, 15);
		msrcopymask = makemaskw(16, 23) | makemaskw(25, 27) | makemaskw(30, 31);
		break;
	case INTERRUPT_SC:
		srr1clearmask = makemaskw(1, 4) | makemaskw(10, 15);
		msrcopymask = makeflag(0) | makemaskw(5, 9) | makemaskw(16, 23) |
			makemaskw(25, 27) | makemaskw(30, 31);
		break;
	case INTERRUPT_ISI:
		srr1clearmask = makemaskw(10, 15);
		msrcopymask = makemaskw(16, 23) | makemaskw(25, 27) | makemaskw(30, 31);
		break;
	case INTERRUPT_PROGRAM:
		srr1clearmask = makemaskw(1, 4) | makeflag(10);
		msrcopymask = makemaskw(16, 23) | makemaskw(25, 27) | makemaskw(30, 31);
		break;
	case INTERRUPT_MACHINECHECK:
		srr1clearmask = makemaskw(0, 15);
		msrcopymask = makemaskw(16, 31);
		msrclearmask = MSR_ME;
		break;
	default:
		DEGUB("Unhandled interrupt 0x%04X @ 0x%08X\n", vector, srr0);
		throw generic_fatal_exception("Unhandled interrupt!");
	}
	r.srr1 &= ~srr1clearmask;
	r.srr1 = (msrcopymask & msr) | (~msrcopymask & r.srr1);

	msr &= ~(MSR_POW | MSR_EE | MSR_PR | MSR_FP | MSR_FE0 | MSR_SE |
		MSR_BE | MSR_FE1 | MSR_IR | MSR_DR | MSR_RI | msrclearmask);
	msr = getflag(msr, MSR_ILE) ? (msr | MSR_LE) : (msr & ~MSR_LE);
	r.setmsr(msr);
}

//#define QUANT_LOAD_FACTOR pow(2.0f, -gqrp[I].ld_scale)
#define QUANT_LOAD_FACTOR gqrp[I].ld_scale

union PPC_HWORD {
	WORD half;
	struct { BYTE lobyte, hibyte; };
};
union PPC_WORD {
	DWORD word;
	struct { WORD lohalf, hihalf; };
};
union PPC_DWORD {
	QWORD dword;
	struct { DWORD loword, hiword; };
	struct { float ps1, ps0; };
};

//may be called by recompiled code
void CoreBase::quantized_load_W0(DWORD frD, DWORD address, DWORD I) {
	MYASSERT(I < 8);
	switch(gqrp[I].ld_type) {
	case 0: //ignore scale
		PPC_DWORD dword;
		dword.dword = m.rd(address);
		PS0(frD) = MAKE(float, dword.hiword);
		PS1(frD) = MAKE(float, dword.loword);
		break;
	case 4:
		{
			PPC_HWORD hword;
			hword.half = m.rh(address);
			PS0(frD) = float(hword.hibyte) * QUANT_LOAD_FACTOR;
			PS1(frD) = float(hword.lobyte) * QUANT_LOAD_FACTOR;
		}
		break;
	case 5:
		{
			PPC_WORD word;
			word.word = m.rw(address);
			PS0(frD) = float(word.hihalf) * QUANT_LOAD_FACTOR;
			PS1(frD) = float(word.lohalf) * QUANT_LOAD_FACTOR;
		}
		break;
	case 6:
		{
			PPC_HWORD hword;
			hword.half = m.rh(address);
			PS0(frD) = float((char)hword.hibyte) * QUANT_LOAD_FACTOR;
			PS1(frD) = float((char)hword.lobyte) * QUANT_LOAD_FACTOR;
		}
		break;
	case 7:
		{
			PPC_WORD word;
			word.word = m.rw(address);
			PS0(frD) = float((short)word.hihalf) * QUANT_LOAD_FACTOR;
			PS1(frD) = float((short)word.lohalf) * QUANT_LOAD_FACTOR;
		}
		break;
	default:
		throw generic_fatal_exception("Big Phat Error in quantized_load()");
	}
}
void CoreBase::quantized_load_W1(DWORD frD, DWORD address, DWORD I) {
	MYASSERT(I < 8);
	switch(gqrp[I].ld_type) {
	case 0: //ignore scale
		{
			DWORD word = m.rw(address);
			PS0(frD) = MAKE(float, word);
		}
		break;
	case 4:
		PS0(frD) = float(m.rb(address)) * QUANT_LOAD_FACTOR;
		break;
	case 5:
		PS0(frD) = float(m.rh(address)) * QUANT_LOAD_FACTOR;
		break;
	case 6:
		PS0(frD) = float((char)m.rb(address)) * QUANT_LOAD_FACTOR;
		break;
	case 7:
		PS0(frD) = float((short)m.rh(address)) * QUANT_LOAD_FACTOR;
		break;
	default:
		throw generic_fatal_exception("Big Phat Error in quantized_load()");
	}
	PS1(frD) = 1.0;
}

//#define QUANT_STORE_FACTOR pow(2.0f, gqrp[I].st_scale)
#define QUANT_STORE_FACTOR gqrp[I].st_scale

#define CLAMP_U8(f) BYTE(MAX(0, MIN((f), 255)))
#define CLAMP_S8(f) char(MAX(-128, MIN((f), 127)))
#define CLAMP_U16(f) WORD(MAX(0, MIN((f), 65535)))
#define CLAMP_S16(f) short(MAX(-32768, MIN((f), 32767)))

void CoreBase::quantized_store_W0(DWORD address, DWORD I, DWORD frS) {
	MYASSERT(I < 8);
	switch(gqrp[I].st_type) {
	case 0: //ignore scale
		{
			PPC_DWORD dword;
			dword.ps0 = (float)PS0(frS);
			dword.ps1 = (float)PS1(frS);
			m.wd(address, dword.dword);
			break;
		}
	case 4:
		{
			PPC_HWORD hword;
			hword.hibyte = CLAMP_U8(PS0(frS) * QUANT_STORE_FACTOR);
			hword.lobyte = CLAMP_U8(PS1(frS) * QUANT_STORE_FACTOR);
			m.wh(address, hword.half);
			break;
		}
	case 5:
		{
			PPC_WORD word;
			word.hihalf = CLAMP_U16(PS0(frS) * QUANT_STORE_FACTOR);
			word.lohalf = CLAMP_U16(PS1(frS) * QUANT_STORE_FACTOR);
			m.ww(address, word.word);
			break;
		}
	case 6:
		{
			PPC_HWORD hword;
			hword.hibyte = CLAMP_S8(PS0(frS) * QUANT_STORE_FACTOR);
			hword.lobyte = CLAMP_S8(PS1(frS) * QUANT_STORE_FACTOR);
			m.wh(address, hword.half);
			break;
		}
	case 7:
		{
			PPC_WORD word;
			word.hihalf = CLAMP_S16(PS0(frS) * QUANT_STORE_FACTOR);
			word.lohalf = CLAMP_S16(PS1(frS) * QUANT_STORE_FACTOR);
			m.ww(address, word.word);
			break;
		}
	default:
		throw generic_fatal_exception("Big Phat Error in quantized_store()");
	}
}
void CoreBase::quantized_store_W1(DWORD address, DWORD I, DWORD frS) {
	MYASSERT(I < 8);
	switch(gqrp[I].st_type) {
	case 0: //ignore scale (I think)
		{
			float f = (float)PS0(frS);
			m.ww(address, MAKE(DWORD, f));
		}
		break;
	case 4:
		m.wb(address, CLAMP_U8(PS0(frS) * QUANT_STORE_FACTOR));
		break;
	case 5:
		m.wh(address, CLAMP_U16(PS0(frS) * QUANT_STORE_FACTOR));
		break;
	case 6:
		m.wb(address, CLAMP_S8(PS0(frS) * QUANT_STORE_FACTOR));
		break;
	case 7:
		m.wh(address, CLAMP_S16(PS0(frS) * QUANT_STORE_FACTOR));
		break;
	default:
		throw generic_fatal_exception("Big Phat Error in quantized_store()");
	}
}

//We might want to put the interrupt mask check here...
bool InterruptRaiser::raiseAsync(WORD vector, hw_callback callback,
																 const std::string &desc)
{
	CSSHandler csshandler(css_interrupts);
	//We should perhaps check if there's another interrupt of the same vector in the queue
	//But since the emulator will never always run at full speed, that could cause
	//real-time interrupts (like audio) to be lost
	if(g::log_interrupts) {
		if(pending_interrupts.size() > 0) {
			DEGUB("%i pending, ", pending_interrupts.size());
		}
		DEGUB("Interrupt 0x%04X raised: %s\n", vector, desc.c_str());
	}
	PENDING_INTERRUPT pi = { vector, callback, desc };
	pending_interrupts.push_back(pi);
	return true;
}

bool InterruptRaiser::raiseAsync_unless_pending(WORD vector, hw_callback callback,
																								const std::string &desc)
{
	CSSHandler csshandler(css_interrupts);
	std::list<PENDING_INTERRUPT>::iterator it = pending_interrupts.begin();
	while(it != pending_interrupts.end()) {
		if((*it).vector == vector) {
			if(g::log_interrupts) {
				DEGUB("Interrupt 0x%04X blocked: %s\n", vector, desc.c_str());
			}
			return false;
		}
		it++;
	}
	return raiseAsync(vector, callback, desc);
}

void InterruptRaiser::add_event(const TIMED_EVENT &event) {
	CSSHandler csshandler(css_interrupts);
	if(event.desc == "")
		throw generic_fatal_exception("Internal error: Empty event description!");
	list<TIMED_EVENT>::iterator itr = timed_events.begin();
	//we want to find the iterator which points to the event just after the new one
	while(itr != timed_events.end()) {
		if(itr->time.isAfter(event.time))
			break;
		itr++;
	}
	timed_events.insert(itr, event);
	evt = timed_events.front().time;

	dump_events("Event added");
}

void InterruptRaiser::dump_events(const char *intro) {
	if(g::verbose && g::log_interrupts) {
		CSSHandler csshandler(css_interrupts);
		ULI ct;
		if(g::timing_mode == g::TM_REALTIME) {
			GET_RTC(&ct);
		} else {
			MYASSERT(g::timing_mode == g::TM_EXACT_FAST || g::timing_mode == g::TM_EXACT_REAL);
			ct = m_cc.cycles;
		}
		DEGUB("%s. CT:0x%016I64X(%I64i). ET:0x%016I64X(%I64i) %i events:\n", intro, ct, ct,
			evt, evt, timed_events.size());
		list<TIMED_EVENT>::iterator itr;
		for(itr = timed_events.begin(); itr != timed_events.end(); itr++) {
			/*#define SYMSIZE 10000
			Container<SYMBOL_INFO> pSym(SYMSIZE);
			char buffer[MAX_PATH];
			ZeroMemory(pSym, SYMSIZE);
			pSym->SizeOfStruct = SYMSIZE;
			pSym->MaxNameLen = SYMSIZE - sizeof(SYMBOL_INFO);
			GLE(SymFromAddr(GetCurrentProcess(), (ULONG)itr->function, NULL, pSym));
			GLE(UnDecorateSymbolName(pSym->Name, buffer, MAX_PATH,
			UNDNAME_COMPLETE | UNDNAME_NO_THISTYPE | UNDNAME_NO_SPECIAL_SYMS |
			UNDNAME_NO_MEMBER_TYPE | UNDNAME_NO_MS_KEYWORDS | UNDNAME_NO_ACCESS_SPECIFIERS));*/
			DEGUB("arg:0x%08X func:0x%08X time:0x%016I64X(%I64i) desc: %s\n", itr->arg,
				itr->function, itr->time.QuadPart, itr->time.QuadPart, itr->desc.c_str());
		}
	}
}
