// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "cpu.h"
#include "container.h"
#include "ini.h"
#include "resource.h"
#include "messages.h"
#include "stringCommons.h"
#include <stdlib.h>
#include <sys/timeb.h>
#include <float.h>

#define DEFINE_UNIMPLEMENTED_OPCODE(asm, pri, sm, sec) \
	void CPU::_##asm() {\
	throw interp_fatal_exception("Unimplemented opcode encountered: " #asm); }

UNIMPLEMENTED_OPCODES(DEFINE_UNIMPLEMENTED_OPCODE)

CPU::CPU(CORE_ARGUMENTS a)
: CoreBase(m, a), m(h, r, interrupt), priarr(0, 5), fetch_pointer(NULL),
last_branch_unconditional(false) {
	DWORD count=0;

#define ADD_CPU_OPCODE(asm, pri, sm, sec) count++;\
	priarr.addOpcode(&CPU::_##asm, #asm, pri, sm, sec);

	OPCODES(ADD_CPU_OPCODE);
	priarr.addOpcode(&CPU::_addic_, "addic.", 13, 0, 0); count++;
	priarr.addOpcode(&CPU::_stwcx_, "stwcx.", 31, 21, 150); count++;
	DEGUB("Supported opcodes: %i\n", count);
	UNIMPLEMENTED_OPCODES(ADD_CPU_OPCODE);
	DEGUB("Defined opcodes: %i\n", count);

	degub_changes(g::degub_run, false);
}

bool CPU::handle_load() {
	SET_FETCH(entry_point);
	m.load_cache(); //This is a hack. Emulate cache properly. Someday. Maybe. ;)
	if(g::degub_run) {
		degub_changes(g::degub_run, false);	//We want to show the changes from Gekko boot settings
	}

	return true;
}

string CPU::getdasma(DWORD address) {
	return getdasmo(address, m.rw(address));
}
string CPU::getdasmo(DWORD cia, DWORD opcode) {
	disassembly::opcode = opcode;
	disassembly::cia = cia;
	const char *opasm = priarr.getOpAsm(disassembly::opcode);
	ostringstream str;
	if(opasm) {
		Container<char> buffer(90);
		(*asmarr.getOpFunc(disassembly::opcode))(buffer);
		str << opasm << buffer;
	}
	return str.str();
}

bool CPU::take_interrupt(DWORD requested_nia) {
	SET_FETCH(requested_nia);
	return true;
}

bool CPU::disassemble() {
	for(size_t i=0; i<text_sections.size(); i++) {
		DEGUB("section:  address=%08X  size=%X\n", i,
			text_sections[i].address, text_sections[i].size);
		SET_FETCH(text_sections[i].address);
		TGLE(run(text_sections[i].size / 4, true, 0, false));
	}
	return true;
}

//Returns false if an error caused the stop
bool CPU::_run(DWORD runstop, bool degub, QWORD degub_skip, bool dumpmem) {
	m_run_degub = degub;
	bool temp_degub;
	__timeb64 start_time, stop_time;

	error=false;

	//Main loop
	_ftime64_s(&start_time);  //Record start time
	while(!error) {
		if(!g::disassembly_mode)
			if(quit())
				break;

		temp_degub = (m_run_degub && getCycles() >= degub_skip) ||
			(g::lowmem_log && (PHYSICALIZE(nia) < 0x3100 ||
			(g::advanced_mode && (r.getmsr() & (MSR_IR | MSR_DR)))));// && !loop
		m.setDegub(temp_degub);
		if(getCycles() == degub_skip)
			degub_changes(temp_degub, false);

		cia = nia;
		nia += 4;

		try {
			opcode = swapw(*(fetch_pointer++));
			//m.clear_degub();

			cpu_popf opfunc = priarr.getOpFunc(opcode);
			if(temp_degub) {
				DEGUB("%08X  %08X  ", cia, opcode);
			}

			if(opfunc) {
				if(temp_degub) {
					DEGUB(priarr.getOpAsm(opcode));
					disassembly::opcode = opcode;
					disassembly::cia = cia;
					Container<char> buffer(80);
					(*asmarr.getOpFunc(opcode))(buffer);
					DEGUB("%s\n", buffer);
				}
				if(!g::disassembly_mode)
					(this->*opfunc)();
			} else {
				ostringstream str;
				str << "Undefined opcode at " << HEX08(cia) << ": " << HEX0x08(opcode) <<
					" Primary:" << getbitsw(opcode, 0, 5) << " Sec21:" <<
					getbitsw(opcode, 21, 30) << " Sec22:" << getbitsw(opcode, 22, 30) <<
					" Sec25:" << getbitsw(opcode, 25, 30) << " Sec26:" << getbitsw(opcode, 26, 30);
				g_errorstring = str.str();
				DEGUB("%s\n", g_errorstring.c_str());
				if(!g::disassembly_mode) {
					error = true;
				}
			}
		} catch(page_fault_exception &e) {
			DEGUB("@ %I64i cycles:\n", getCycles());
			DEGUB("page_fault_exception @ 0x%08X: %s\n", cia, e.what());
			SET_FETCH(cia);
		} catch(interp_fatal_exception &e) {
			degub_changes(temp_degub, false);
			DEGUB("interp_fatal_exception @ 0x%08X: %s\n", cia, e.what());
			if(!g::disassembly_mode) {
				error = true;
				g_errorstring = e.what();
			}
		} catch(generic_fatal_exception &e) {
			degub_changes(temp_degub, false);
			DEGUB("generic_fatal_exception @ 0x%08X: %s\n", cia, e.what());
			if(!g::disassembly_mode) {
				error = true;
				g_errorstring = e.what();
			}
		} catch(hardware_fatal_exception &e) {
			degub_changes(temp_degub, false);
			DEGUB("hardware_fatal_exception @ 0x%08X: %s\n", cia, e.what());
			if(!g::disassembly_mode) {
				error = true;
				g_errorstring = e.what();
			}
		} catch(bouehr_exception &e) {
			degub_changes(temp_degub, false);
			DEGUB("bouehr_exception @ 0x%08X: %s\n", cia, e.what());
			if(!g::disassembly_mode) {
				error = true;
				g_errorstring = e.what();
			}
		} catch(emulation_stop_exception &e) {
			degub_changes(temp_degub, false);
			DEGUB("emulation_stop_exception @ 0x%08X: %s\n", cia, e.what());
			if(!g::disassembly_mode) {
				error = true;
				g_errorstring = e.what();
			}
		}

		cc.addCycles(1);

		if(!error) {
			//bool branch = MASKED_EQUAL(opcode, 0x48000001, 0xFC000000);
			//if(branch) {	//hacky
			TGLE(do_interrupts());
			//}
		}

		degub_changes(temp_degub, false);

		{ //degub/loop stuff
			if(g::disassembly_mode) {
				SET_FETCH(cia + 4);
			} else if(nia != cia + 4) {
				//infinite loop detection
				if(g::iloop && last_branch_unconditional && nia == cia) {
					ostringstream str;
					str << "Infinite loop detected at " << HEX0x08(cia);
					g_errorstring = str.str();
					//stop = true;
					error = true;
				}
			}
		}
		if(RUNSTOP && cc.cycles_current_period >= runstop)
			break;
	}
	_ftime64_s(&stop_time);	//Record stop time

	TGLE(h.killPeriodicTasks(error ? NULL : &g_errorstring));
	if(!g_errorstring.empty()) {
		error = true;
	}
	DWORD elapsed_time = DWORD((stop_time.time - start_time.time) * 1000 +
		(stop_time.millitm - start_time.millitm));
	if(!m.is_ok()) {
		//sprintf(errorstring, "Uninitialized memory access.");
		DEGUB("Memory error.\n");
	}
	if(error) {
		DEGUB("An error ocurred.\n");
	}
	//if(!g::dual_run) {
	DEGUB("Period %i complete. Ran %I64i cycles in %i ms (%s).\n",
		periods, cc.cycles_current_period, elapsed_time,
		ips_string(cc.cycles_current_period, elapsed_time).c_str());
	//}

	cc.endPeriod();

	//error = error || !m.is_ok();
	if(dumpmem)
		m.dumpmem();

	SetLastError(NO_ERROR); //This signifies that errorstring has been loaded
	return !error;
}

void CPU::degub_changes(bool on, bool all) const {
	if(!on)
		m.clear_degub();
	else {
		m.degub();
		r.degub(all, old_registers);
	}
}
