// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef DSP_CORE
#define DSP_CORE

#include "meminterface.h"
#include "array.h"
#include <sstream>
/*  macro(JNE)\
macro(JEQ)\
macro(JZR)\
macro(JNZ)\
macro(RETEQ)\
macro(RETNZ)\
macro(CALLNE)\
*/

#define DSP_OPCODES(macro) \
	macro(NOP)\
	macro(HALT)\
	macro(RET)\
	macro(RTI)\
	macro(CALL)\
	macro(JMP)\
	macro(DAR)\
	macro(IAR)\
	macro(CALLR)\
	macro(JMPR)\
	macro(SBCLR)\
	macro(SBSET)\
	macro(LSL)\
	macro(LSR)\
	macro(ASL)\
	macro(ASR)\
	macro(LRI)\
	macro(LR)\
	macro(SR)\
	macro(MRR)\
	macro(SI)\
	macro(LRS)\
	macro(SRS)\
	macro(LRIS)\
	macro(ADDIS)\
	macro(CMPIS)\
	macro(ANDI)\
	macro(ANDF)\
	macro(XORI)\
	macro(ANDCF)\
	macro(ORI)\
	macro(ORF)\
	macro(ADDI)\
	macro(CMPI)\
	macro(ILRR)\
	macro(ILRRD)\
	macro(ILRRI)\
	macro(ILRRN)\
	macro(LRRI)\
	macro(LRRD)\
	macro(LRRN)\
	macro(LRR)\
	macro(SRRI)\
	macro(SRRD)\
	macro(SRRN)\
	macro(SRR)\
	macro(LOOPI)\
	macro(BLOOPI)\
	macro(LOOP)\
	macro(BLOOP)\
	macro(NX)\
	macro(S40)\
	macro(S16)\
	macro(M2)\
	macro(M0)\
	macro(CLR15)\
	macro(SET15)\
	macro(DECM)\
	macro(INCM)\
	macro(DEC)\
	macro(INC)\
	macro(NEG)\
	macro(TST)\
	macro(TSTAXH)\
	macro(CMP)\
	macro(CMPAXH)\
	macro(CLR)\
	macro(CLRP)\
	macro(MOV)\
	macro(MOVAX)\
	macro(MOVR)\
	macro(MOVP)\
	macro(MOVPZ)\
	macro(ADDPAXZ)\
	macro(ADDP)\
	macro(LSL16)\
	macro(LSR16)\
	macro(ASR16)\
	macro(XORR)\
	macro(ORR)\
	macro(ANDR)\
	macro(MULX)\
	macro(MULXAC)\
	macro(MULXMV)\
	macro(MULXMVZ)\
	macro(MUL)\
	macro(MULAC)\
	macro(MULMV)\
	macro(MULMVZ)\
	macro(MULC)\
	macro(MULCAC)\
	macro(MULCMV)\
	macro(MULCMVZ)\
	macro(ADDR)\
	macro(ADDAX)\
	macro(ADD)\
	macro(ADDAXL)\
	macro(SUBR)\
	macro(SUBAX)\
	macro(SUB)\
	macro(MADD)\
	macro(MSUB)\
	macro(MADDX)\
	macro(MSUBX)\
	macro(MADDC)\
	macro(MSUBC)\

#define DSP_EXT_OPCODES(macro)\
	macro(L)\
	macro(LN)\
	macro(LS)\
	macro(LSN)\
	macro(LSM)\
	macro(LSNM)\
	macro(SL)\
	macro(SLN)\
	macro(SLM)\
	macro(SLNM)\
	macro(S)\
	macro(SN)\
	macro(LD)\
	macro(LDN)\
	macro(LDM)\
	macro(LDNM)\
	macro(MV)\
	macro(DR)\
	macro(IR)\
	macro(NR)\
	macro(NOP)
/*macro(LDX)\
macro(LDXN)\
macro(LDXM)\
macro(LDXNM)\*/

class DSPInterpreter;

typedef void (DSPInterpreter::*DSPOpfunc)();

//enum DSPState { DSPS_RESET, DSPS_RUNNING, DSPS_UNKNOWN };

#define P_NONE (0x0000)
#define P_VAL (0x0001)
#define P_IMM (0x0002)
#define P_MEM (0x0003)
#define P_STR (0x0004)
#define P_AXX (0x0005)
#define P_ACC (0x0006)
#define P_PRG (0x0007)
#define P_CC (0x0008)
#define P_REG (0x8000)
#define P_REG08 (P_REG | 0x0800)
#define P_REG10 (P_REG | 0x1000)
#define P_REG18 (P_REG | 0x1800)
#define P_REG19 (P_REG | 0x1900)
#define P_REG1A (P_REG | 0x1a00)
#define P_REG1C (P_REG | 0x1c00)
#define P_REG1E (P_REG | 0x1e00)
#define P_REGS_MASK (0x1f00)
//P_REF P_REG | 0x4000,
//P_PRG P_REF | P_REG,

template<class T> struct OPCODE_PARAM {
	WORD type;
	BYTE size, loc, lshift;
	T mask;
};
template<class T> struct OPCODE_INIT {
	DSPOpfunc opfunc;
	const char *name;
	T code, mask;
	BYTE size, nparams;
	OPCODE_PARAM<T> params[3];
};

//if/when a dsp recompiler is created, part of this class will become DSPCore.
class DSPInterpreter {
public:
	DSPInterpreter(MemInterface &mi, Hardware& h);
	~DSPInterpreter();
	void turn_on();
	void reset();
	void run();
	void pause();  //in cooperative mode, only DSP can halt itself.
	u64 getCycles() { return m.cycles; }
	void quickExecute(const void* code, int codeSize, void* data, int entryPoint,
		WORD gprFill);
private:
	DSPInterpreter(const DSPInterpreter&);
	DSPInterpreter& operator=(const DSPInterpreter&);

	MemInterface& mainmem;
	Hardware& h;

	//ordinary private functions
	bool load();	//GLE
	void disassemble(WORD base_address, WORD size);
	void execute_ext(BYTE code);
	void disassemble_one(ostream& str);
	template<class T> void disassemble_params(const OPCODE_INIT<T>& pop, ostream& str);
	void dump_changes();
	bool isHalted() const;
	void setHalt();
	const OPCODE_INIT<WORD>& findOpcode(WORD opcode);

	//opcode arrays
#define OP_MAIN_SIZE 64*K
#define OP_EXT_SIZE 256
	DSPOpfunc op_main[OP_MAIN_SIZE], op_ext[OP_EXT_SIZE];

	static const OPCODE_INIT<WORD> m_opar[];
	static const OPCODE_INIT<BYTE> m_extar[];
	static const size_t m_nops, m_nexts;

	//opcode functions
#define DECLARE_DSP_OPFUNC(name) void _##name();
#define DECLARE_DSP_EXT_OPFUNC(name) void _x##name();
	DSP_OPCODES(DECLARE_DSP_OPFUNC);
	DSP_EXT_OPCODES(DECLARE_DSP_EXT_OPFUNC);
#undef DECLARE_DSP_OPFUNC
#undef DECLARE_DSP_EXT_OPFUNC

	void __undefined();
	void _x_undefined();


	//-----------------------------------MEMORY-------------------------------------
	//class Memory {
	//private:
	//the unit of sizes and addresses are 16-bit words.
#define DSP_IRAMSIZE (4*K)
#define DSP_IROMSIZE (4*K)
#define DSP_IROMBASE 0x8000 //Duddie's guess
#define DSP_DRAMSIZE (4*K)
#define DSP_DROMSIZE (2*K)
#define DSP_DROMBASE 0x8000 //My guess
	Array<WORD> iram, dram;
	Array<WORD> irom, drom;
#define DSP_HWMEMBASE 0xFFC0
#define DSP_HWMEMSIZE (0x10000-DSP_HWMEMBASE)
	Array<WORD> hwmem;	//part of data memory
	//public:
	//Memory();
	WORD iRead(WORD address);
	WORD dRead(WORD address);
	WORD hwRead(WORD address);
	void dWrite(WORD address, WORD data);
	void hwWrite(WORD address, WORD data);
	void iDmaWrite(WORD dst, const void* src, WORD size);
	void iDmaRead(WORD src, void* dst, WORD size);
	void dDmaWrite(WORD dst, const void* src, WORD size);
	void dDmaRead(WORD src, void* dst, WORD size);
	//} mem;

	//---------------------------------EXCEPTIONS-----------------------------------
	enum Exception { RESET, STOVF, E2, E3, E4, ACCOV, E6, E7 };
	static void throw_exception(Exception e);

	//-------------------------------REGISTER ACCESS--------------------------------
	void wReg(BYTE reg, WORD data);
	WORD rReg(BYTE reg);

	//-------------------------------OPCODE HELPERS---------------------------------
	bool cond(BYTE pCond);
	s64 getAc(uint i);
	void setAc(uint i, s64 ac);
	void lsl(uint pAc, uint pI);
	void setFlags(s64 ac, bool fullSignFlag=true);
	void doLogicM0(s64 ac, uint pAc);
	void doCmp(s64 ac0, s64 ac1);
	void setAcHM(uint i, s32 achm);
	s32 getAcHM(uint i);
	s32 getProdHM();

	struct ZEROABLE {
		u64 cycles;
		//registers
		WORD cpc, npc;  //current/next program counter
		WORD gpr[32];
		//BYTE ac[2][5];  //accumulator //mapped to gpr
		//WORD st[4];	//stack pointers	//mapped to gpr
		//WORD cr;  //config  //mapped to gpr
		//WORD sr;  //status  //mapped to gpr
		//BYTE prod[5]; bool prod_overflow;	//mapped to gpr
		uint bloopCount;
		bool pause;
		BYTE init_mask;
#define DSP_INIT_RESET 1
#define DSP_INIT_ON 2
#define DSP_INITCOMPLETE 3

#define R_ST(i) m.gpr[0x0c + i]

		template<uint SIZE> class Stack {
		public:
			void push(WORD w) {
				if(sp >= SIZE)
					throw_exception(STOVF);
				mData[sp++] = w;
			}
			WORD pop() {
				if(sp == 0)
					throw_exception(STOVF);
				return mData[--sp];
			}
			WORD& top() {
				MYASSERT(sp >= 0);
				return mData[sp - 1];
			}
			void reset() {
				sp = 0;
			}
		private:
			uint sp;
			WORD mData[SIZE];
		};

#define STACKS(m) m(call,8,0) m(data,4,1) m(loop_address,4,2) m(loop_counter,4,3)
#define DECLARE_STACK(name,size,reg) Stack<size> name;
		STACKS(DECLARE_STACK);
	} m;

	struct ZEROABLE_OLD {
		WORD gpr[32];
	} old;
	//const WORD m_rom[4*K];
};

#define DSPDEGUB if(g::dsp_log) DEGUB

#endif	//DSP_CORE
