// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "dsp_core.h"
#include "dsp_opdefs.h"

void DSPInterpreter::setFlags(s64 ac, bool fullSignFlag) {
	m.sr &= ~0x3F;
	ac &= 0xFFFFFFFFFF;
	if(ac == 0)
		m.sr |= SR_EQ;
	if(fullSignFlag ? sign40(ac) : sign32((s32)ac))
		m.sr |= SR_S;
	u8 uk_nibble = (u8)((ac >> 28) & 0xF);
	if(uk_nibble < 4 || uk_nibble > 0xB)
		m.sr |= SR_UK;	//weird
	u8 top_byte = (u8)((ac >> 32) & 0xFF);
	if((top_byte != 0 || ((ac >> 31) & 1)) && ((ac & 0xFF80000000) != 0xFF80000000))
		m.sr |= SR_AS;	//this one just baffles me.
}

bool DSPInterpreter::cond(BYTE pCond) {
	switch(pCond) {
	case 1:	return !(m.sr & SR_UK) && (m.sr & SR_S);
	case 2: return !(m.sr & SR_S);
	case 4: return !(m.sr & SR_EQ);
	case 5: return !!(m.sr & SR_EQ);
	case 0xC: return !(m.sr & SR_LZ);
	case 0xD: return !!(m.sr & SR_LZ);
	case 0xF: return true;
	default:
		throw hardware_fatal_exception("DSP unemulated CC");
	}
	/*if(pCond == 0xF)
		return true;
	if(pCond < 0xE) {
		return ((m.sr & (1 << (pCond >> 1))) == 0) != bool(pCond & 1);
	}
	throw hardware_fatal_exception("DSP unemulated CC");*/
}

s64 DSPInterpreter::getAc(uint i) {
	MYASSERT(i <= 1);
	s64 res;
	u32 low;
	res = SEXT8(m.ACH(i));
	res <<= 32;
	low = m.ACM(i);
	low <<= 16;
	low |= m.ACL(i);
	res |= low;
	return res;
}
void DSPInterpreter::setAc(uint i, s64 ac) {
	m.ACL(i) = (u16)ac;
	ac >>= 16;
	m.ACM(i) = (u16)ac;
	ac >>= 16;
	m.ACH(i) = (s16)(s8)ac;
}

void DSPInterpreter::setAcHM(uint i, s32 achm) {
	m.ACL(i) = 0;
	m.ACM(i) = (u16)achm;
	achm >>= 16;
	m.ACH(i) = (s16)(s8)achm;
}

s32 DSPInterpreter::getAcHM(uint i) {
	return ((u32)m.ACH(i) << 16) | m.ACM(i);
}

s32 DSPInterpreter::getProdHM() {
	s32 res;
	MYASSERT(m.PRODH < 0x100);
	res = m.PRODH;
	res <<= 16;
	res += m.PRODM1;
	res += m.PRODM2;	//experimental
	res += m.PRODL >> 15;
	return res;
}

void DSPInterpreter::lsl(uint pAc, uint pI) {
	u64 ac = getAc(pAc);
	ac <<= pI;
	setAc(pAc, ac);
	setFlags(ac);
	//doLogicM0(ac, pAc);
}

void DSPInterpreter::doLogicM0(s64 ac, uint pAc) {
	if((m.sr & SR_M0) && (ac & 0xFF80000000) != 0 && (ac & 0xFF80000000) != 0xFF80000000) {
		if(sign40(ac))
			m.ACM(pAc) = 0x8000;
		else
			m.ACM(pAc) = 0x7FFF;
	}
}

void DSPInterpreter::doCmp(s64 ac0, s64 ac1) {
	s64 result = ac0 - ac1;
	setFlags(result);
	bool overflow = (u64)ac1 <= (u64)ac0;
	SETFLAGS(m.sr, SR_OV, overflow);
	if((sign40(ac0) && !sign40(ac1) && !sign40(result)) ||
		(!sign40(ac0) && sign40(ac1) && sign40(result)))
		m.sr |= 0x82;
}
