// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "dsp_core.h"
#include "dsp_opdefs.h"
#include "stringCommons.h"
#include "hw_ai.h"

static const char *s_regnames[32] = { "ar0", "ar1", "ar2", "ar3",
"ix0", "ix1", "ix2", "ix3", NULL, NULL, NULL, NULL, "st0", "st1", "st2", "st3",
"ac0.h", "ac1.h", "config", "sr", "prod.l", "prod.m1", "prod.h", "prod.m2",
"ax0.l", "ax1.l", "ax0.h", "ax1.h", "ac0.l", "ac1.l", "ac0.m", "ac1.m" };


#define INIT_PARAM(name, type, size, loc, lshift, mask) {\
	P_##type, size, loc, lshift, mask }

#define INIT_PARAM_ARRAY(type, name)\
	DSP##type##_##name(INIT_PARAM_ARRAY_BASE)(type, name)
#define INIT_PARAM_ARRAY_BASE(code, mask, size, nparams)\
	INIT_##nparams##PARAM_ARRAY
#define INIT_0PARAM_ARRAY(type, name) 0
#define INIT_1PARAM_ARRAY(type, name) { DSP##type##_##name##_PARAM1(INIT_PARAM) }
#define INIT_2PARAM_ARRAY(type, name) { DSP##type##_##name##_PARAM1(INIT_PARAM),\
	DSP##type##_##name##_PARAM2(INIT_PARAM) }
#define INIT_3PARAM_ARRAY(type, name) { DSP##type##_##name##_PARAM1(INIT_PARAM),\
	DSP##type##_##name##_PARAM2(INIT_PARAM), DSP##type##_##name##_PARAM3(INIT_PARAM) }

#define PASS_4_PARAMS(code, mask, size, nparams) code, mask, size, nparams

#define INIT_OPCODE_INIT(name)\
	,{ &DSPInterpreter::_##name, #name, DSPOP_##name(PASS_4_PARAMS),\
	INIT_PARAM_ARRAY(OP, name) }

#define INIT_EXT_OPCODE_INIT(name)\
	,{ &DSPInterpreter::_x##name, #name, DSPEXT_##name(PASS_4_PARAMS),\
	INIT_PARAM_ARRAY(EXT, name) }

const OPCODE_INIT<WORD> DSPInterpreter::m_opar[] = { {NULL,0,0,0,0}\
DSP_OPCODES(INIT_OPCODE_INIT) };

const OPCODE_INIT<BYTE> DSPInterpreter::m_extar[] = { {NULL,0,0,0,0}\
DSP_EXT_OPCODES(INIT_EXT_OPCODE_INIT) };

const size_t DSPInterpreter::m_nops = sizeof(m_opar)/sizeof(OPCODE_INIT<WORD>);
const size_t DSPInterpreter::m_nexts = sizeof(m_extar)/sizeof(OPCODE_INIT<BYTE>);

template<class T, size_t NOPS, const OPCODE_INIT<T>* ARR>
const OPCODE_INIT<T>& findOpcode(WORD opcode) {
	const OPCODE_INIT<T> *pop = NULL;
	for(size_t i=1; i<NOPS; i++) {
		if(MASKED_EQUAL(opcode, ARR[i].code, ARR[i].mask)) {
			pop = &ARR[i];
			break;
		}
	}
	if(pop == NULL)
		throw disasm_exception("Unknown opcode!");
	return *pop;
}

const OPCODE_INIT<WORD>& DSPInterpreter::findOpcode(WORD opcode) {
	return ::findOpcode<WORD, m_nops, m_opar>(opcode);
}

template<class T> void init_opcodes(bool ext, DSPOpfunc me_uninitialized, size_t me_nops,
																		const OPCODE_INIT<T> *me_ar, DSPOpfunc *target_ar)
{
	//T fullmask = ext ? 0xff : 0xffff;
	//const char *me_string = ext ? "ext" : "main";
	for(size_t i=1; i<me_nops; i++) {
		//opcode coherency check
		//we check that parameters don't overlap,
		//and that every bit of the opcode is accounted for.
		CONST OPCODE_INIT<T> &op = me_ar[i];
		VDEGUB("%s ", op.name);
		T fillmask = op.mask;
		for(size_t p=0; p<op.nparams; p++) {
			if(op.params[p].loc == 1) {
				if(op.params[p].mask != 0xffff)
					throw generic_fatal_exception("DSP param bad mask!");
				if(ext)
					throw generic_fatal_exception("DSP param bad loc!");
			} else {
				if(op.params[p].loc != 0)
					throw generic_fatal_exception("DSP param bad loc!");
				if((op.params[p].mask & fillmask) != 0)
					throw generic_fatal_exception("DSP param overlap!");
				fillmask |= op.params[p].mask;
			}
		}
		if(op.size & P_EXT) {
			if(fillmask & 0xff)
				throw generic_fatal_exception("DSP extensible overlap!");
			fillmask |= 0xff;
		}
		if(fillmask != (ext ? 0xff : 0xffff))
			throw generic_fatal_exception("DSP opcode has unused bits!");

		//opfunc array init
		for(size_t j=0; j<(ext ? OP_EXT_SIZE : OP_MAIN_SIZE); j++) {
			if(MASKED_EQUAL(j, op.code, op.mask)) {
				if(target_ar[j] != me_uninitialized)
					throw generic_fatal_exception("DSP opcode overlap!");
				target_ar[j] = op.opfunc;
			}
		}
	}
	VDEGUB("\n");
	DEGUB("%i %s opcodes\n", me_nops-1, ext ? "ext" : "main");
}

DSPInterpreter::DSPInterpreter(MemInterface &mi, Hardware& _h)
: mainmem(mi), h(_h), iram(DSP_IRAMSIZE), irom(DSP_IROMSIZE),
dram(DSP_DRAMSIZE), drom(DSP_DROMSIZE), hwmem(DSP_HWMEMSIZE)
{
	DEGUB("Initializing DSP interpreter...\n");
	ZERO_OBJECT(m);
	ZERO_OBJECT(old);

	{
		DSPOpfunc opfunc = &DSPInterpreter::__undefined;
		memset_dword(op_main, MAKE(DWORD, opfunc), OP_MAIN_SIZE);
		opfunc = &DSPInterpreter::_x_undefined;
		memset_dword(op_ext, MAKE(DWORD, opfunc), OP_EXT_SIZE);
	}

	init_opcodes<WORD>(false, &DSPInterpreter::__undefined, m_nops, m_opar, op_main);
	init_opcodes<BYTE>(true, &DSPInterpreter::_x_undefined, m_nexts, m_extar, op_ext);

	HWGLE(load());

	m.pause = true;
	setHalt();
}

DSPInterpreter::~DSPInterpreter() {
	DEGUB("DSP ran %I64i cycles\n", m.cycles);
}

bool DSPInterpreter::load() {
	ReadOnlyFile file;
	TGLE(file.open("gc_dsp_irom.bin"));
	TGLE(file.readbswaph(irom.p(), DSP_IROMSIZE));
	//disassemble(DSP_IROMBASE, DSP_IROMSIZE);

	ZeroMemory(dram.p(), dram.size() * 2);
	return true;
}

void DSPInterpreter::reset() {
	MYASSERT(isHalted());
	MYASSERT(m.pause);

	ZERO_ARRAY(m.gpr);

	//m.gpr[19] = 0x2060;	//Status register, HLE HACK.

	m.npc = DSP_IROMBASE;
	m.init_mask |= DSP_INIT_RESET;
}
void DSPInterpreter::turn_on() {
	MYASSERT(isHalted());
	MYASSERT(m.pause);
	m.init_mask |= DSP_INIT_ON;
}
void DSPInterpreter::pause() {
	//VDSPDEBUG("DSPInterpreter::halt();\n");
	m.pause = true;
}
bool DSPInterpreter::isHalted() const {
	return (h.hrh(PR_DSPCR) & DSPCR_HALT) != 0;
}
void DSPInterpreter::setHalt() {
	h.hwh(PR_DSPCR, h.hrh(PR_DSPCR) | DSPCR_HALT);	//unsure, test
}

void DSPInterpreter::quickExecute(const void* code, int codeSize, void* data,
																	int entryPoint, WORD /*gprFill*/)
{
	DSPLOG("DSPquickExecute 0x%x bytes @ 0x%x\n", codeSize, entryPoint);
	MYASSERT(entryPoint >= 0 && entryPoint < DSP_IRAMSIZE);
	MYASSERT(codeSize > 0 && codeSize < DSP_IRAMSIZE);
	memcpy_swaph(iram.p(), code, codeSize);
	//PrintPacket((BYTE*)code, codeSize);
	m.npc = (WORD)entryPoint;
	h.hwh(PR_DSPCR, h.hrh(PR_DSPCR) & ~DSPCR_HALT);	//unhalt
	m.init_mask = DSP_INITCOMPLETE;	//for safety
#define RESET_STACK(name,size,reg) m.name.reset();
	STACKS(RESET_STACK);

	//memset_dword(m.gpr, (((DWORD)gprFill) << 16) | gprFill, 16);

	run();	//should exit after halt

	memcpy_swaph(data, m.gpr, sizeof(m.gpr));	//copy registers
}

void DSPInterpreter::run() {
	if(isHalted())
		return;
	MYASSERT(m.pause);
	if(m.init_mask != DSP_INITCOMPLETE)
		throw hardware_fatal_exception("DSP Unhalted without being properly initialized!");

	bool error = false;
	DSPLOG("DSP running from 0x%04X\n", m.npc);

	m.pause = false;
	while(!m.pause) {
		m.cpc = m.npc;
		WORD opcode = iRead(m.cpc);
		m.cycles++;
		try {
			if(g::dsp_log) {  //disassemble
				ostringstream str;
				try {
					WORD prev_npc = m.npc;
					disassemble_one(str);
					m.npc = prev_npc;
					DEGUB("0x%04X: %04X %s\n", m.cpc, opcode, str.str().c_str());
				} catch(stop_exception_base &e) {
					DEGUB("0x%04X: %04X %s  %s\n", m.cpc, opcode, str.str().c_str(), e.what());
					throw;	//rethrows exception
				}
			}
			//execute opcode
			(this->*op_main[opcode])();
			if(m.bloopCount > 0) {
				if(m.cpc == m.loop_address.top()) {
					if(--m.loop_counter.top() > 0) {
						//continue bloop
						m.npc = m.call.top();
					} else {
						//end of bloop
						m.bloopCount--;
						m.call.pop();
						m.loop_address.pop();
						m.loop_counter.pop();
					}
				}
			}
		} catch(stop_exception_base &e) {
			DEGUB("Exception @ 0x%04X: %s\n", m.cpc, e.what());
			error = m.pause = true;
			throw;
		}
		if(g::dsp_log)
			dump_changes();
	}
	if(m.pause && !error) {
		DSPLOG("DSP paused @ 0x%04X", m.cpc);
	}
	DSPLOG("\n");
}

void DSPInterpreter::disassemble(WORD base_address, WORD size) {
	DEGUB("DSP disassembling 0x%04X words @ 0x%04X\n", size, base_address);

	m.npc = base_address;
	try { while(m.npc < base_address + size) {
		m.cpc = m.npc;
		WORD opcode = iRead(m.cpc);
		ostringstream str;
		try {
			disassemble_one(str);
			DEGUB("0x%04X: %04X %s\n", m.cpc, opcode, str.str().c_str());
		} catch(stop_exception_base& e) {
			DEGUB("0x%04X: %04X %s  %s\n", m.cpc, opcode, str.str().c_str(), e.what());
			//throw;
		}
	} } catch(stop_exception_base& ) {
	}
	DEGUB("\n");
}

//WARNING: modifies m.npc!
void DSPInterpreter::disassemble_one(ostream& str) {
	m.npc++;
	const OPCODE_INIT<WORD>& op(::findOpcode<WORD, m_nops, m_opar>(iRead(m.cpc)));
	BYTE opsize_words = (op.size & ~P_EXT);
	//two-word opcodes
	if(opsize_words == 2) {
		m.npc++;
		str << HEX0(4, iRead(m.cpc+1)) << "  ";
	} else {
		MYASSERT(opsize_words == 1);
		str << "      ";
	}
	//sprint
	str << op.name;
	disassemble_params(op, str);
	//ext opcode
	if(op.size & P_EXT) {
		//find
		const OPCODE_INIT<BYTE>& ext(::findOpcode<BYTE, m_nexts, m_extar>(iRead(m.cpc)));
		str << (op.nparams ? "  " : "") << "'" << ext.name;
		disassemble_params(ext, str);
	}
}

static const char* cc_names[] = {
	"NOV","OV","2","3",
	"NEQ","EQ","NS","S",
	"NAS","AS","NUK","UK",
	"NLZ","LZ",NULL,"A"
};

template<class T> void DSPInterpreter::disassemble_params(const OPCODE_INIT<T>& op,
																													ostream& str)
{
	for(size_t i=0; i<op.nparams; i++) {
		const OPCODE_PARAM<T> &param = op.params[i];
		WORD type = param.type;
		WORD value = (iRead(m.cpc+param.loc) & param.mask) >> param.lshift;
		if(type & P_REG) {
			value += (type & P_REGS_MASK) >> 8;
			type &= ~P_REGS_MASK;
		}

		str << (i > 0 ? "," : "") << " ";
		switch(type) {
		case P_IMM:
			str << "#" << HEx0(param.size*2, value);
			break;
		case P_CC:
			str << "#" << HEx(value);
			str << "(" << (cc_names[value] == NULL ? "NULL" : cc_names[value]) << ")";
			if(cc_names[value] == NULL)
				throw disasm_exception("DSP disasm, unemulated CC!");
			break;
		case P_VAL:
			if(param.size != 2)
				throw disasm_exception("DSP disasm, bad param size!");
			str << HEx0x0(4, value);
			break;
		case P_REG:
			MYASSERT(value < 32);
			str << "$" << HEx0(2, value);
			if(s_regnames[value] != NULL)
				str << "(" << s_regnames[value] << ")";
			//if(use_register_names)
			//buffer += sprintf(buffer, "$%s", s_regnames[value]);
			//else
			//buffer += sprintf(buffer, "$%02x", value);
			break;
		case P_MEM:
			if(param.size == 1)
				value = SEXT8(value);
			str << "@" << HEx0(4, value);
			break;
		case P_ACC:
			str << "$ac" << value;
			break;
		case P_PRG:
			str << "@$ar" << value;
			break;
		case P_AXX:
			str << "$ax" << value;
			break;
		default:
			throw disasm_exception("DSP disasm, unhandled param type!");
		}
	}
}

//this should only be executed in op_main context
void DSPInterpreter::execute_ext(BYTE code) {
	(this->*op_ext[code])();
}

void DSPInterpreter::dump_changes() {
	for(size_t i=0; i<32; i++) {
		if(old.gpr[i] != m.gpr[i]) {
			DEGUB("r%02x", i);
			if(s_regnames[i] != NULL) {
				DEGUB("(%s)", s_regnames[i]);
			}
			DEGUB(" = %04X\n", m.gpr[i]);
			old.gpr[i] = m.gpr[i];
		}
	}
}

void DSPInterpreter::throw_exception(Exception e) {
	std::string msg = "DSP Exception, level "+std::string(1, '0'+e);
	throw emulation_stop_exception(msg);
}

WORD DSPInterpreter::iRead(WORD address) {
	WORD data;
	if(address & 0x8000) {
		data = irom[address & 0x7FFF];
	} else {
		data = iram[address & 0x7FFF];
	}
	//DSPLOG("iRead(0x%04X): 0x%04X\n", address, data);
	return data;
}
WORD DSPInterpreter::dRead(WORD address) {
	WORD data;
	DSPLOG("dRead(0x%04X): ", address);
	switch(address >> 12) {
	case 0:
		data = dram[address & 0x7FFF];
		break;
	case 8:
		throw hardware_fatal_exception("DSP dRead rom");
		//data = drom[address & 0x7FFF];
	case 0xF:
		data = hwRead(address);
		break;
	default:
		throw hardware_fatal_exception("DSP dRead");
	}
	DSPLOG("0x%04X\n", data);
	return data;
}
void DSPInterpreter::dWrite(WORD address, WORD data) {
	DSPLOG("dWrite(0x%04X): 0x%04X\n", address, data);
	switch(address >> 12) {
	case 0:
		dram[address & 0x7FFF] = data;
		break;
	case 8:
		throw hardware_fatal_exception("DSP dWrite rom");
	case 0xF:
		hwWrite(address, data);
		break;
	default:
		throw hardware_fatal_exception("DSP dWrite");
	}
}
void DSPInterpreter::iDmaWrite(WORD dst, const void* src, WORD size) {
	if(DWORD(dst) + (size >> 1) > DSP_IRAMSIZE || (size & 1) || (size == 0))
		throw hardware_fatal_exception("DSP iDmaWrite");
	memcpy_swaph(iram.p() + dst, src, size);
	if(g::dsp_log) {
		File dump;
		dump.open("dsp.bin", File::CreateAlways);
		dump.write(src, size);
	}
}
void DSPInterpreter::iDmaRead(WORD src, void* dst, WORD size) {
	if(DWORD(src) + (size >> 1) > DSP_IRAMSIZE || (size & 1))
		throw hardware_fatal_exception("DSP iDmaRead");
	memcpy_swaph(dst, iram.p() + src, size);
}
void DSPInterpreter::dDmaWrite(WORD dst, const void* src, WORD size) {
	if(DWORD(dst) + (size >> 1) > DSP_DRAMSIZE || (size & 1))
		throw hardware_fatal_exception("DSP dDmaWrite");
	memcpy_swaph(dram.p() + dst, src, size);
}
void DSPInterpreter::dDmaRead(WORD src, void* dst, WORD size) {
	if(DWORD(src) + (size >> 1) > DSP_DRAMSIZE || (size & 1))
		throw hardware_fatal_exception("DSP dDmaRead");
	memcpy_swaph(dst, dram.p() + src, size);
}

void DSPInterpreter::wReg(BYTE reg, WORD data) {
	MYASSERT(reg < 32);
	switch(reg) {
	case 0x0c: m.call.push(data); break;
	case 0x0d: m.data.push(data); break;
	case 0x0e: m.loop_address.push(data); break;
	case 0x0f: m.loop_counter.push(data); break;
	case 0x10:	//ac0.h
	case 0x11:	//ac1.h
		m.gpr[reg] = (short)(char)data;	//sign-extended 8-bit
		break;
	case 0x12:	//config
		if(data != 0xFF)
			throw hardware_fatal_exception("DSP config");
		m.gpr[reg] = data;
		break;
	case 0x13:	//sr
		m.gpr[reg] = data & 0xFEFF;
		break;
	case 0x16:	//prod.h
		m.gpr[reg] = data & 0x00FF;
		break;
	case 0x1e:	//ac0.m
	case 0x1f:	//ac1.m
		if(m.sr & SR_M0) {
			if(sign16(data))
				m.gpr[reg - 0xe] = 0xFFFF;	//acX.h
			else {
				m.gpr[reg - 0xe] = 0;
			}
			m.gpr[reg - 2] = 0;	//acX.l
		}
	default:
		m.gpr[reg] = data;
	}
}
WORD DSPInterpreter::rReg(BYTE reg) {
	MYASSERT(reg < 32);
	switch(reg) {
	case 0x0c: return m.call.pop();
	case 0x0d: return m.data.pop();
	case 0x0e: return m.loop_address.pop();
	case 0x0f: return m.loop_counter.pop();
	default:
		return m.gpr[reg];
	}
}
