// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef DSP_INTERNAL
#define DSP_INTERNAL

//helpers
#define SEXT8(a) ((signed char)a)
#define SETFLAGS(d, flags, set) { if(set) (d) |= (flags); else (d) &= ~(flags); }

__inline bool sign40(s64 ac) {
	return (ac & 0x8000000000) != 0;
}
__inline bool sign32(s32 ac) {
	return (ac & 0x80000000) != 0;
}
__inline bool sign24(s32 ac) {
	return (ac & 0x800000) != 0;
}
__inline bool sign16(u16 ac) {
	return (ac & 0x8000) != 0;
}
__inline bool sign8(u16 ac) {
	return (ac & 0x80) != 0;
}

const OPCODE_INIT<WORD>& findOpcode(WORD opcode);

//register aliases
#define config gpr[0x12]
#define sr gpr[0x13]
#define AR(i) gpr[i]
#define ACM(i) gpr[0x1e + (i)]
#define ACH(i) gpr[0x10 + (i)]
#define ACL(i) gpr[0x1c + (i)]
#define AXH(i) gpr[0x1a + (i)]
#define AXL(i) gpr[0x18 + (i)]
#define PRODH gpr[0x16]
#define PRODM1 gpr[0x15]
#define PRODM2 gpr[0x17]
#define PRODL gpr[0x14]

//Status Register flags
#define SR_EQ 4	//Zero
#define SR_LZ 0x40	//Logic zero, set only by a few boolean arithmetic instructions.
#define SR_UK 0x20	//Unknown, not documented by Duddie.
#define SR_AS 0x10	//"AS" according to Duddie. I don't know what it means.
#define SR_S 8	//Sign
#define SR_OV 1	//Overflow

#define SR_M0 0x4000	//Arithmetic control flag. Makes all kinds of weird stuff happen.


#define P_EXT	0x80

//STANDARD OPCODES
#define DSPOP_NOP(macro) macro(0x0000, 0xffff, 1, 0)
/*
:Wh\{:Wh"{:i}",:Wh{0x:h+},:Wh{0x:h+},:Wh{:d},:Wh{:d},:Wh\{\},:Wh\},$
#define DSPOP_\1(macro) macro(\2, \3, \4, \5)
*/
#define DSPOP_HALT(macro) macro(0x0021, 0xffff, 1, 0)
#define DSPOP_RET(macro) macro(0x02df, 0xffff, 1, 0)
//#define DSPOP_RETEQ(macro) macro(0x02d5, 0xffff, 1, 0)
//#define DSPOP_RETNZ(macro) macro(0x02dd, 0xffff, 1, 0)
#define DSPOP_RTI(macro) macro(0x02ff, 0xffff, 1, 0)

//everything comes from GC DSP.pdf
#define DSPOP_CALL(macro) macro(0x02bf, 0xffff, 2, 1)
#define DSPOP_CALL_PARAM1(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)
/*
:Wh\{:Wh"{:i}",:Wh{0x:h+,:Wh0x:h+,:Wh:d,:Wh:d},:Wh\{\{P_{:i,:Wh:d,:Wh:d,:Wh:d,:Wh0x:h+}\}\},:Wh\},$
#define DSPOP_\1(macro) macro(\2)\n#define DSPOP_\1_PARAM1(macro) macro(\3)
*/
//#define DSPOP_CALLNE(macro) macro(0x02b4, 0xffff, 2, 1)
//#define DSPOP_CALLNE_PARAM1(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)

/*#define DSPOP_JNE(macro) macro(0x0294, 0xffff, 2, 1)
#define DSPOP_JNE_PARAM1(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)
#define DSPOP_JEQ(macro) macro(0x0295, 0xffff, 2, 1)
#define DSPOP_JEQ_PARAM1(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)
#define DSPOP_JZR(macro) macro(0x029c, 0xffff, 2, 1)
#define DSPOP_JZR_PARAM1(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)
#define DSPOP_JNZ(macro) macro(0x029d, 0xffff, 2, 1)
#define DSPOP_JNZ_PARAM1(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)*/
//#define DSPOP_JMP(macro) macro(0x029f, 0xffff, 2, 1)
//#define DSPOP_JMP_PARAM1(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)
#define DSPOP_JMP(macro) macro(0x0290, 0xfff0, 2, 2)
#define DSPOP_JMP_PARAM1(macro) macro(Cond, CC, 1, 0, 0, 0x000f)
#define DSPOP_JMP_PARAM2(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)

#define DSPOP_DAR(macro) macro(0x0004, 0xfffc, 1, 1)
#define DSPOP_DAR_PARAM1(macro) macro(AR, REG, 1, 0, 0, 0x0003)
#define DSPOP_IAR(macro) macro(0x0008, 0xfffc, 1, 1)
#define DSPOP_IAR_PARAM1(macro) macro(AR, REG, 1, 0, 0, 0x0003)

#define DSPOP_CALLR(macro) macro(0x171f, 0xff1f, 1, 1)
#define DSPOP_CALLR_PARAM1(macro) macro(R, REG, 1, 0, 5, 0x00e0)
#define DSPOP_JMPR(macro) macro(0x170f, 0xff1f, 1, 1)
#define DSPOP_JMPR_PARAM1(macro) macro(R, REG, 1, 0, 5, 0x00e0)

#define DSPOP_SBCLR(macro) macro(0x1200, 0xfff8, 1, 1)
#define DSPOP_SBCLR_PARAM1(macro) macro(I, IMM, 1, 0, 0, 0x0007)
#define DSPOP_SBSET(macro) macro(0x1300, 0xfff8, 1, 1)
#define DSPOP_SBSET_PARAM1(macro) macro(I, IMM, 1, 0, 0, 0x0007)

#define DSPOP_M2(macro) macro(0x8a00, 0xff00, 1 | P_EXT, 0)
#define DSPOP_M0(macro) macro(0x8b00, 0xff00, 1 | P_EXT, 0)
#define DSPOP_CLR15(macro) macro(0x8c00, 0xff00, 1 | P_EXT, 0)
#define DSPOP_SET15(macro) macro(0x8d00, 0xff00, 1 | P_EXT, 0)
#define DSPOP_S40(macro) macro(0x8e00, 0xff00, 1 | P_EXT, 0)
#define DSPOP_S16(macro) macro(0x8f00, 0xff00, 1 | P_EXT, 0)

/*
:Wh\{:Wh"{:i}",:Wh{0x:h+,:Wh0x:h+,:Wh:d,:Wh:d},:Wh\{\{P_{:i,:Wh:d,:Wh:d,:Wh:d,:Wh0x:h+}\},:Wh\{P_{:i,:Wh:d,:Wh:d,:Wh:d,:Wh0x:h+}\}\},:Wh\},$
#define DSPOP_\1(macro) macro(\2)\n#define DSPOP_\1_PARAM1(macro) macro(\3)\n#define DSPOP_\1_PARAM2(macro) macro(\4)
*/
#define DSPOP_LSL(macro) macro(0x1400, 0xfec0, 1, 2)
#define DSPOP_LSL_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_LSL_PARAM2(macro) macro(I, IMM, 1, 0, 0, 0x003f)
#define DSPOP_LSR(macro) macro(0x1440, 0xfec0, 1, 2)
#define DSPOP_LSR_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_LSR_PARAM2(macro) macro(I, IMM, 1, 0, 0, 0x003f)
#define DSPOP_ASL(macro) macro(0x1480, 0xfec0, 1, 2)
#define DSPOP_ASL_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ASL_PARAM2(macro) macro(I, IMM, 1, 0, 0, 0x003f)
#define DSPOP_ASR(macro) macro(0x14c0, 0xfec0, 1, 2)
#define DSPOP_ASR_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ASR_PARAM2(macro) macro(I, IMM, 1, 0, 0, 0x003f)

#define DSPOP_LRI(macro) macro(0x0080, 0xffe0, 2, 2)
#define DSPOP_LRI_PARAM1(macro) macro(D, REG, 1, 0, 0, 0x001f)
#define DSPOP_LRI_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)
#define DSPOP_LR(macro) macro(0x00c0, 0xffe0, 2, 2)
#define DSPOP_LR_PARAM1(macro) macro(D, REG, 1, 0, 0, 0x001f)
#define DSPOP_LR_PARAM2(macro) macro(M, MEM, 2, 1, 0, 0xffff)
#define DSPOP_SR(macro) macro(0x00e0, 0xffe0, 2, 2)
#define DSPOP_SR_PARAM1(macro) macro(M, MEM, 2, 1, 0, 0xffff)
#define DSPOP_SR_PARAM2(macro) macro(S, REG, 1, 0, 0, 0x001f)

#define DSPOP_MRR(macro) macro(0x1c00, 0xfc00, 1, 2)
#define DSPOP_MRR_PARAM1(macro) macro(D, REG, 1, 0, 5, 0x03e0)
#define DSPOP_MRR_PARAM2(macro) macro(S, REG, 1, 0, 0, 0x001f)

#define DSPOP_SI(macro) macro(0x1600, 0xff00, 2, 2)
#define DSPOP_SI_PARAM1(macro) macro(M8, MEM, 1, 0, 0, 0x00ff)
#define DSPOP_SI_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)

#define DSPOP_LRS(macro) macro(0x2000, 0xf800, 1, 2)
#define DSPOP_LRS_PARAM1(macro) macro(D18, REG18, 1, 0, 8, 0x0700)
#define DSPOP_LRS_PARAM2(macro) macro(M8, MEM, 1, 0, 0, 0x00ff)
#define DSPOP_SRS(macro) macro(0x2800, 0xf800, 1, 2)
#define DSPOP_SRS_PARAM1(macro) macro(M8, MEM, 1, 0, 0, 0x00ff)
#define DSPOP_SRS_PARAM2(macro) macro(S18, REG18, 1, 0, 8, 0x0700)

#define DSPOP_LRIS(macro) macro(0x0800, 0xf800, 1, 2)
#define DSPOP_LRIS_PARAM1(macro) macro(D18, REG18, 1, 0, 8, 0x0700)
#define DSPOP_LRIS_PARAM2(macro) macro(I, IMM, 1, 0, 0, 0x00ff)

#define DSPOP_ADDIS(macro) macro(0x0400, 0xfe00, 1, 2)
#define DSPOP_ADDIS_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ADDIS_PARAM2(macro) macro(I, IMM, 1, 0, 0, 0x00ff)
#define DSPOP_CMPIS(macro) macro(0x0600, 0xfe00, 1, 2)
#define DSPOP_CMPIS_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_CMPIS_PARAM2(macro) macro(I, IMM, 1, 0, 0, 0x00ff)

#define DSPOP_ANDI(macro) macro(0x0240, 0xfeff, 2, 2)
#define DSPOP_ANDI_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ANDI_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)
#define DSPOP_ANDF(macro) macro(0x02a0, 0xfeff, 2, 2)
#define DSPOP_ANDF_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ANDF_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)

#define DSPOP_XORI(macro) macro(0x0220, 0xfeff, 2, 2)
#define DSPOP_XORI_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_XORI_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)
#define DSPOP_ANDCF(macro) macro(0x02c0, 0xfeff, 2, 2)
#define DSPOP_ANDCF_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ANDCF_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)

#define DSPOP_ORI(macro) macro(0x0260, 0xfeff, 2, 2)
#define DSPOP_ORI_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ORI_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)
#define DSPOP_ORF(macro) macro(0x02e0, 0xfeff, 2, 2)
#define DSPOP_ORF_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ORF_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)

#define DSPOP_ADDI(macro) macro(0x0200, 0xfeff, 2, 2)
#define DSPOP_ADDI_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ADDI_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)
#define DSPOP_CMPI(macro) macro(0x0280, 0xfeff, 2, 2)
#define DSPOP_CMPI_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_CMPI_PARAM2(macro) macro(I, IMM, 2, 1, 0, 0xffff)

#define DSPOP_ILRR(macro) macro(0x0210, 0xfefc, 1, 2)
#define DSPOP_ILRR_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ILRR_PARAM2(macro) macro(ArS, PRG, 1, 0, 0, 0x0003)
#define DSPOP_ILRRD(macro) macro(0x0214, 0xfefc, 1, 2)
#define DSPOP_ILRRD_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ILRRD_PARAM2(macro) macro(ArS, PRG, 1, 0, 0, 0x0003)
#define DSPOP_ILRRI(macro) macro(0x0218, 0xfefc, 1, 2)
#define DSPOP_ILRRI_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ILRRI_PARAM2(macro) macro(ArS, PRG, 1, 0, 0, 0x0003)
#define DSPOP_ILRRN(macro) macro(0x021C, 0xfefc, 1, 2)
#define DSPOP_ILRRN_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ILRRN_PARAM2(macro) macro(ArS, PRG, 1, 0, 0, 0x0003)

// load and store value pointed by indexing reg and increment; LRR/SRR variants
#define DSPOP_LRRI(macro) macro(0x1900, 0xff80, 1, 2)
#define DSPOP_LRRI_PARAM1(macro) macro(RD, REG, 1, 0, 0, 0x001f)
#define DSPOP_LRRI_PARAM2(macro) macro(ArS, PRG, 1, 0, 5, 0x0060)
#define DSPOP_LRRD(macro) macro(0x1880, 0xff80, 1, 2)
#define DSPOP_LRRD_PARAM1(macro) macro(RD, REG, 1, 0, 0, 0x001f)
#define DSPOP_LRRD_PARAM2(macro) macro(ArS, PRG, 1, 0, 5, 0x0060)
#define DSPOP_LRRN(macro) macro(0x1980, 0xff80, 1, 2)
#define DSPOP_LRRN_PARAM1(macro) macro(RD, REG, 1, 0, 0, 0x001f)
#define DSPOP_LRRN_PARAM2(macro) macro(ArS, PRG, 1, 0, 5, 0x0060)
#define DSPOP_LRR(macro) macro(0x1800, 0xff80, 1, 2)
#define DSPOP_LRR_PARAM1(macro) macro(RD, REG, 1, 0, 0, 0x001f)
#define DSPOP_LRR_PARAM2(macro) macro(ArS, PRG, 1, 0, 5, 0x0060)
#define DSPOP_SRRI(macro) macro(0x1b00, 0xff80, 1, 2)
#define DSPOP_SRRI_PARAM1(macro) macro(ArD, PRG, 1, 0, 5, 0x0060)
#define DSPOP_SRRI_PARAM2(macro) macro(RS, REG, 1, 0, 0, 0x001f)
#define DSPOP_SRRD(macro) macro(0x1a80, 0xff80, 1, 2)
#define DSPOP_SRRD_PARAM1(macro) macro(ArD, PRG, 1, 0, 5, 0x0060)
#define DSPOP_SRRD_PARAM2(macro) macro(RS, REG, 1, 0, 0, 0x001f)
#define DSPOP_SRRN(macro) macro(0x1b80, 0xff80, 1, 2)
#define DSPOP_SRRN_PARAM1(macro) macro(ArD, PRG, 1, 0, 5, 0x0060)
#define DSPOP_SRRN_PARAM2(macro) macro(RS, REG, 1, 0, 0, 0x001f)
#define DSPOP_SRR(macro) macro(0x1a00, 0xff80, 1, 2)
#define DSPOP_SRR_PARAM1(macro) macro(ArD, PRG, 1, 0, 5, 0x0060)
#define DSPOP_SRR_PARAM2(macro) macro(RS, REG, 1, 0, 0, 0x001f)

#define DSPOP_LOOPI(macro) macro(0x1000, 0xff00, 1, 1)
#define DSPOP_LOOPI_PARAM1(macro) macro(I, IMM, 1, 0, 0, 0x00ff)
#define DSPOP_BLOOPI(macro) macro(0x1100, 0xff00, 2, 2)
#define DSPOP_BLOOPI_PARAM1(macro) macro(I, IMM, 1, 0, 0, 0x00ff)
#define DSPOP_BLOOPI_PARAM2(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)
#define DSPOP_LOOP(macro) macro(0x0040,	0xffe0, 1, 1)
#define DSPOP_LOOP_PARAM1(macro) macro(R, REG, 1, 0, 0, 0x001f)
#define DSPOP_BLOOP(macro) macro(0x0060, 0xffe0, 2, 2)
#define DSPOP_BLOOP_PARAM1(macro) macro(R, REG, 1, 0, 0, 0x001f)
#define DSPOP_BLOOP_PARAM2(macro) macro(Addr, VAL, 2, 1, 0, 0xffff)


/*
:Wh\{:Wh"{:i}",:Wh*{0x:h+,:Wh0x:h+,.#,:Wh:z},:Wh\{\{P_{:i,:Wh:z,:Wh:z,:Wh:z,:Wh0x:h+}\},:Wh\{P_{:i,:Wh:z,:Wh:z,:Wh:z,:Wh0x:h+}\},:Wh\{P_{:i,:Wh:z,:Wh:z,:Wh:z,:Wh0x:h+}\}\},:Wh\},$
#define DSPOP_\1(macro) macro(\2)\n#define DSPOP_\1_PARAM1(macro) macro(\3)\n#define DSPOP_\1_PARAM2(macro) macro(\4)\n#define DSPOP_\1_PARAM3(macro) macro(\5)
*/
/*
:Wh\{:Wh"{:i}",:Wh*{0x:h+,:Wh0x:h+,.#,:Wh:z},:Wh\{\},:Wh\},$
#define DSPOP_\1(macro) macro(\2)
*/

//EXTENSIBLE OPCODES
#define DSPOP_NX(macro) macro(0x8000, 0xff00, 1 | P_EXT, 0)

#define DSPOP_DECM(macro) macro(0x7800, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_DECM_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_INCM(macro) macro(0x7400, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_INCM_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_DEC(macro) macro(0x7a00, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_DEC_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_INC(macro) macro(0x7600, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_INC_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)

#define DSPOP_NEG(macro) macro(0x7c00, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_NEG_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)

#define DSPOP_TST(macro) macro(0xb100, 0xf700, 1 | P_EXT, 1)
#define DSPOP_TST_PARAM1(macro) macro(Ac, ACC, 1, 0, 11, 0x0800)
#define DSPOP_TSTAXH(macro) macro(0x8600, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_TSTAXH_PARAM1(macro) macro(Ax, REG1A, 1, 0, 8, 0x0100)
#define DSPOP_CMP(macro) macro(0x8200, 0xff00, 1 | P_EXT, 0)
#define DSPOP_CMPAXH(macro) macro(0xc100, 0xe700, 1 | P_EXT, 2)
#define DSPOP_CMPAXH_PARAM1(macro) macro(Ac, ACC, 1, 0, 11, 0x0800)
#define DSPOP_CMPAXH_PARAM2(macro) macro(Ax, REG1A, 1, 0, 12, 0x1000)

#define DSPOP_CLR(macro) macro(0x8100, 0xf700, 1 | P_EXT, 1)
#define DSPOP_CLR_PARAM1(macro) macro(Ac, ACC, 1, 0, 11, 0x0800)
#define DSPOP_CLRP(macro) macro(0x8400, 0xff00, 1 | P_EXT, 0)

#define DSPOP_MOV(macro) macro(0x6c00, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_MOV_PARAM1(macro) macro(Axx, AXX, 1, 0, 8, 0x0100)
#define DSPOP_MOVAX(macro) macro(0x6800, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_MOVAX_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MOVAX_PARAM2(macro) macro(AxS, REG18, 1, 0, 9, 0x0200)
#define DSPOP_MOVR(macro) macro(0x6000, 0xf800, 1 | P_EXT, 2)
#define DSPOP_MOVR_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MOVR_PARAM2(macro) macro(AxS, REG18, 1, 0, 9, 0x0600)
#define DSPOP_MOVP(macro) macro(0x6e00, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_MOVP_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MOVPZ(macro) macro(0xfe00, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_MOVPZ_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)

#define DSPOP_ADDPAXZ(macro) macro(0xf800, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_ADDPAXZ_PARAM1(macro) macro(Magic, IMM, 1, 0, 9, 0x0200)
#define DSPOP_ADDPAXZ_PARAM2(macro) macro(Axc, REG1A, 1, 0, 8, 0x0100)
#define DSPOP_ADDP(macro) macro(0x4e00, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_ADDP_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)

#define DSPOP_LSL16(macro) macro(0xf000, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_LSL16_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_LSR16(macro) macro(0xf400, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_LSR16_PARAM1(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ASR16(macro) macro(0x9100, 0xf700, 1 | P_EXT, 1)
#define DSPOP_ASR16_PARAM1(macro) macro(Ac, ACC, 1, 0, 11, 0x0800)

#define DSPOP_XORR(macro) macro(0x3000, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_XORR_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_XORR_PARAM2(macro) macro(AxS, REG, 1, 0, 9, 0x0200)
#define DSPOP_ORR(macro) macro(0x3800, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_ORR_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ORR_PARAM2(macro) macro(AxS, REG, 1, 0, 9, 0x0200)
#define DSPOP_ANDR(macro) macro(0x3400, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_ANDR_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ANDR_PARAM2(macro) macro(AxS, REG, 1, 0, 9, 0x0200)

#define DSPOP_MULX(macro) macro(0xa000, 0xe700, 1 | P_EXT, 2)
#define DSPOP_MULX_PARAM1(macro) macro(S, REG18, 1, 0, 11, 0x1000)
#define DSPOP_MULX_PARAM2(macro) macro(T, REG19, 1, 0, 10, 0x0800)
#define DSPOP_MULXAC(macro) macro(0xa400, 0xe600, 1 | P_EXT, 3)
#define DSPOP_MULXAC_PARAM1(macro) macro(S, REG18, 1, 0, 11, 0x1000)
#define DSPOP_MULXAC_PARAM2(macro) macro(T, REG19, 1, 0, 10, 0x0800)
#define DSPOP_MULXAC_PARAM3(macro) macro(AcR, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MULXMV(macro) macro(0xa600, 0xe600, 1 | P_EXT, 3)
#define DSPOP_MULXMV_PARAM1(macro) macro(S, REG18, 1, 0, 11, 0x1000)
#define DSPOP_MULXMV_PARAM2(macro) macro(T, REG19, 1, 0, 10, 0x0800)
#define DSPOP_MULXMV_PARAM3(macro) macro(AcR, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MULXMVZ(macro) macro(0xa200, 0xe600, 1 | P_EXT, 3)
#define DSPOP_MULXMVZ_PARAM1(macro) macro(S, REG18, 1, 0, 11, 0x1000)
#define DSPOP_MULXMVZ_PARAM2(macro) macro(T, REG19, 1, 0, 10, 0x0800)
#define DSPOP_MULXMVZ_PARAM3(macro) macro(AcR, ACC, 1, 0, 8, 0x0100)

#define DSPOP_MUL(macro) macro(0x9000, 0xf700, 1 | P_EXT, 1)
#define DSPOP_MUL_PARAM1(macro) macro(AxS, AXX, 1, 0, 11, 0x0800)
#define DSPOP_MULAC(macro) macro(0x9400, 0xf600, 1 | P_EXT, 2)
#define DSPOP_MULAC_PARAM1(macro) macro(AxS, AXX, 1, 0, 11, 0x0800)
#define DSPOP_MULAC_PARAM2(macro) macro(AcR, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MULMV(macro) macro(0x9600, 0xf600, 1 | P_EXT, 2)
#define DSPOP_MULMV_PARAM1(macro) macro(AxS, AXX, 1, 0, 11, 0x0800)
#define DSPOP_MULMV_PARAM2(macro) macro(AcR, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MULMVZ(macro) macro(0x9200, 0xf600, 1 | P_EXT, 2)
#define DSPOP_MULMVZ_PARAM1(macro) macro(Ax, AXX, 1, 0, 11, 0x0800)
#define DSPOP_MULMVZ_PARAM2(macro) macro(Ac, ACC, 1, 0, 8, 0x0100)

#define DSPOP_MULC(macro) macro(0xc000, 0xe700, 1 | P_EXT, 2)
#define DSPOP_MULC_PARAM1(macro) macro(AxT, REG1A, 1, 0, 11, 0x0800)
#define DSPOP_MULC_PARAM2(macro) macro(AcS, ACC, 1, 0, 12, 0x1000)
#define DSPOP_MULCAC(macro) macro(0xc400, 0xe600, 1 | P_EXT, 3)
#define DSPOP_MULCAC_PARAM1(macro) macro(AxT, REG1A, 1, 0, 11, 0x0800)
#define DSPOP_MULCAC_PARAM2(macro) macro(AcS, ACC, 1, 0, 12, 0x1000)
#define DSPOP_MULCAC_PARAM3(macro) macro(AcR, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MULCMV(macro) macro(0xc600, 0xe600, 1 | P_EXT, 3)
#define DSPOP_MULCMV_PARAM1(macro) macro(AxT, REG1A, 1, 0, 11, 0x0800)
#define DSPOP_MULCMV_PARAM2(macro) macro(AcS, ACC, 1, 0, 12, 0x1000)
#define DSPOP_MULCMV_PARAM3(macro) macro(AcR, ACC, 1, 0, 8, 0x0100)
#define DSPOP_MULCMVZ(macro) macro(0xc200, 0xe600, 1 | P_EXT, 3)
#define DSPOP_MULCMVZ_PARAM1(macro) macro(AxT, REG1A, 1, 0, 11, 0x0800)
#define DSPOP_MULCMVZ_PARAM2(macro) macro(AcS, ACC, 1, 0, 12, 0x1000)
#define DSPOP_MULCMVZ_PARAM3(macro) macro(AcR, ACC, 1, 0, 8, 0x0100)

#define DSPOP_ADDR(macro) macro(0x4000, 0xf800, 1 | P_EXT, 2)
#define DSPOP_ADDR_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ADDR_PARAM2(macro) macro(S18, REG18, 1, 0, 9, 0x0600)
#define DSPOP_ADDAX(macro) macro(0x4800, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_ADDAX_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ADDAX_PARAM2(macro) macro(AxS, REG18, 1, 0, 9, 0x0200)
#define DSPOP_ADD(macro) macro(0x4c00, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_ADD_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ADDAXL(macro) macro(0x7000, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_ADDAXL_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_ADDAXL_PARAM2(macro) macro(AxS, REG18, 1, 0, 9, 0x0200)

#define DSPOP_SUBR(macro) macro(0x5000, 0xf800, 1 | P_EXT, 2)
#define DSPOP_SUBR_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_SUBR_PARAM2(macro) macro(S18, REG18, 1, 0, 9, 0x0600)
#define DSPOP_SUBAX(macro) macro(0x5800, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_SUBAX_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)
#define DSPOP_SUBAX_PARAM2(macro) macro(AxS, REG18, 1, 0, 9, 0x0200)
#define DSPOP_SUB(macro) macro(0x5c00, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_SUB_PARAM1(macro) macro(AcD, ACC, 1, 0, 8, 0x0100)

#define DSPOP_MADD(macro) macro(0xf200, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_MADD_PARAM1(macro) macro(AxS, AXX, 1, 0, 8, 0x0100)
#define DSPOP_MSUB(macro) macro(0xf600, 0xfe00, 1 | P_EXT, 1)
#define DSPOP_MSUB_PARAM1(macro) macro(AxS, AXX, 1, 0, 8, 0x0100)
#define DSPOP_MADDX(macro) macro(0xe000, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_MADDX_PARAM1(macro) macro(S, REG18, 1, 0, 8, 0x0200)
#define DSPOP_MADDX_PARAM2(macro) macro(T, REG19, 1, 0, 7, 0x0100)
#define DSPOP_MSUBX(macro) macro(0xe400, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_MSUBX_PARAM1(macro) macro(S, REG18, 1, 0, 8, 0x0200)
#define DSPOP_MSUBX_PARAM2(macro) macro(T, REG19, 1, 0, 7, 0x0100)
#define DSPOP_MADDC(macro) macro(0xe800, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_MADDC_PARAM1(macro) macro(AcS, ACC, 1, 0, 9, 0x0200)
#define DSPOP_MADDC_PARAM2(macro) macro(AxT, REG19, 1, 0, 7, 0x0100)
#define DSPOP_MSUBC(macro) macro(0xec00, 0xfc00, 1 | P_EXT, 2)
#define DSPOP_MSUBC_PARAM1(macro) macro(AcS, ACC, 1, 0, 9, 0x0200)
#define DSPOP_MSUBC_PARAM2(macro) macro(AxT, REG19, 1, 0, 7, 0x0100)


//EXTENDED OPCODES
#define DSPEXT_L(macro) macro(0x40, 0xc4, 1, 2)
#define DSPEXT_L_PARAM1(macro) macro(D18, REG18, 1, 0, 3, 0x38)
#define DSPEXT_L_PARAM2(macro) macro(RS, PRG, 1, 0, 0, 0x03)
#define DSPEXT_LN(macro) macro(0x44, 0xc4, 1, 2)
#define DSPEXT_LN_PARAM1(macro) macro(D18, REG18, 1, 0, 3, 0x38)
#define DSPEXT_LN_PARAM2(macro) macro(RS, PRG, 1, 0, 0, 0x03)
#define DSPEXT_LS(macro) macro(0x80, 0xce, 1, 2)
#define DSPEXT_LS_PARAM1(macro) macro(D18, REG18, 1, 0, 4, 0x30)
#define DSPEXT_LS_PARAM2(macro) macro(AcS, ACC, 1, 0, 0, 0x01)
#define DSPEXT_LSN(macro) macro(0x84, 0xce, 1, 2)
#define DSPEXT_LSN_PARAM1(macro) macro(D18, REG18, 1, 0, 4, 0x30)
#define DSPEXT_LSN_PARAM2(macro) macro(AcS, ACC, 1, 0, 0, 0x01)
#define DSPEXT_LSM(macro) macro(0x88, 0xce, 1, 2)
#define DSPEXT_LSM_PARAM1(macro) macro(D18, REG18, 1, 0, 4, 0x30)
#define DSPEXT_LSM_PARAM2(macro) macro(AcS, ACC, 1, 0, 0, 0x01)
#define DSPEXT_LSNM(macro) macro(0x8c, 0xce, 1, 2)
#define DSPEXT_LSNM_PARAM1(macro) macro(D18, REG18, 1, 0, 4, 0x30)
#define DSPEXT_LSNM_PARAM2(macro) macro(AcS, ACC, 1, 0, 0, 0x01)
#define DSPEXT_SL(macro) macro(0x82, 0xce, 1, 2)
#define DSPEXT_SL_PARAM1(macro) macro(AcS, ACC, 1, 0, 0, 0x01)
#define DSPEXT_SL_PARAM2(macro) macro(D18, REG18, 1, 0, 4, 0x30)
#define DSPEXT_SLN(macro) macro(0x86, 0xce, 1, 2)
#define DSPEXT_SLN_PARAM1(macro) macro(AcS, ACC, 1, 0, 0, 0x01)
#define DSPEXT_SLN_PARAM2(macro) macro(D18, REG18, 1, 0, 4, 0x30)
#define DSPEXT_SLM(macro) macro(0x8a, 0xce, 1, 2)
#define DSPEXT_SLM_PARAM1(macro) macro(AcS, ACC, 1, 0, 0, 0x01)
#define DSPEXT_SLM_PARAM2(macro) macro(D18, REG18, 1, 0, 4, 0x30)
#define DSPEXT_SLNM(macro) macro(0x8e, 0xce, 1, 2)
#define DSPEXT_SLNM_PARAM1(macro) macro(AcS, ACC, 1, 0, 0, 0x01)
#define DSPEXT_SLNM_PARAM2(macro) macro(D18, REG18, 1, 0, 4, 0x30)
#define DSPEXT_S(macro) macro(0x20, 0xe4, 1, 2)
#define DSPEXT_S_PARAM1(macro) macro(RD, PRG, 1, 0, 0, 0x03)
#define DSPEXT_S_PARAM2(macro) macro(S1c, REG1C, 1, 0, 3, 0x18)
#define DSPEXT_SN(macro) macro(0x24, 0xe4, 1, 2)
#define DSPEXT_SN_PARAM1(macro) macro(RD, PRG, 1, 0, 0, 0x03)
#define DSPEXT_SN_PARAM2(macro) macro(S1c, REG1C, 1, 0, 3, 0x18)
#define DSPEXT_LDX(macro) macro(0xc0, 0xcf, 1, 3)
#define DSPEXT_LDX_PARAM1(macro) macro(R18, REG18, 1, 0, 4, 0x10)
#define DSPEXT_LDX_PARAM2(macro) macro(R1a, REG1A, 1, 0, 4, 0x10)
#define DSPEXT_LDX_PARAM3(macro) macro(RS, PRG, 1, 0, 5, 0x20)
#define DSPEXT_LDXN(macro) macro(0xc4, 0xcf, 1, 3)
#define DSPEXT_LDXN_PARAM1(macro) macro(R18, REG18, 1, 0, 4, 0x10)
#define DSPEXT_LDXN_PARAM2(macro) macro(R1a, REG1A, 1, 0, 4, 0x10)
#define DSPEXT_LDXN_PARAM3(macro) macro(RS, PRG, 1, 0, 5, 0x20)
#define DSPEXT_LDXM(macro) macro(0xc8, 0xcf, 1, 3)
#define DSPEXT_LDXM_PARAM1(macro) macro(R18, REG18, 1, 0, 4, 0x10)
#define DSPEXT_LDXM_PARAM2(macro) macro(R1a, REG1A, 1, 0, 4, 0x10)
#define DSPEXT_LDXM_PARAM3(macro) macro(RS, PRG, 1, 0, 5, 0x20)
#define DSPEXT_LDXNM(macro) macro(0xcc, 0xcf, 1, 3)
#define DSPEXT_LDXNM_PARAM1(macro) macro(R18, REG18, 1, 0, 4, 0x10)
#define DSPEXT_LDXNM_PARAM2(macro) macro(R1a, REG1A, 1, 0, 4, 0x10)
#define DSPEXT_LDXNM_PARAM3(macro) macro(RS, PRG, 1, 0, 5, 0x20)
#define DSPEXT_LD(macro) macro(0xc0, 0xcc, 1, 3)
#define DSPEXT_LD_PARAM1(macro) macro(R18, REG18, 1, 0, 4, 0x20)
#define DSPEXT_LD_PARAM2(macro) macro(R19, REG19, 1, 0, 3, 0x10)
#define DSPEXT_LD_PARAM3(macro) macro(RS, PRG, 1, 0, 0, 0x03)
#define DSPEXT_LDN(macro) macro(0xc4, 0xcc, 1, 3)
#define DSPEXT_LDN_PARAM1(macro) macro(R18, REG18, 1, 0, 4, 0x20)
#define DSPEXT_LDN_PARAM2(macro) macro(R19, REG19, 1, 0, 3, 0x10)
#define DSPEXT_LDN_PARAM3(macro) macro(RS, PRG, 1, 0, 0, 0x03)
#define DSPEXT_LDM(macro) macro(0xc8, 0xcc, 1, 3)
#define DSPEXT_LDM_PARAM1(macro) macro(R18, REG18, 1, 0, 4, 0x20)
#define DSPEXT_LDM_PARAM2(macro) macro(R19, REG19, 1, 0, 3, 0x10)
#define DSPEXT_LDM_PARAM3(macro) macro(RS, PRG, 1, 0, 0, 0x03)
#define DSPEXT_LDNM(macro) macro(0xcc, 0xcc, 1, 3)
#define DSPEXT_LDNM_PARAM1(macro) macro(R18, REG18, 1, 0, 4, 0x20)
#define DSPEXT_LDNM_PARAM2(macro) macro(R19, REG19, 1, 0, 3, 0x10)
#define DSPEXT_LDNM_PARAM3(macro) macro(RS, PRG, 1, 0, 0, 0x03)
#define DSPEXT_MV(macro) macro(0x10, 0xf0, 1, 2)
#define DSPEXT_MV_PARAM1(macro) macro(D18, REG18, 1, 0, 2, 0x0c)
#define DSPEXT_MV_PARAM2(macro) macro(S1c, REG1C, 1, 0, 0, 0x03)
#define DSPEXT_DR(macro) macro(0x04, 0xfc, 1, 1)
#define DSPEXT_DR_PARAM1(macro) macro(Ar, REG, 1, 0, 0, 0x03)
#define DSPEXT_IR(macro) macro(0x08, 0xfc, 1, 1)
#define DSPEXT_IR_PARAM1(macro) macro(Ar, REG, 1, 0, 0, 0x03)
#define DSPEXT_NR(macro) macro(0x0c, 0xfc, 1, 1)
#define DSPEXT_NR_PARAM1(macro) macro(Ar, REG, 1, 0, 0, 0x03)

#define DSPEXT_NOP(macro) macro(0x00, 0xff, 1, 0)

#endif	//DSP_INTERNAL
