// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "file.h"
#include "degub.h"
#include "common.h"

ReadOnlyFile::ReadOnlyFile() : mHandle(INVALID_HANDLE_VALUE) {
}
ReadOnlyFile::~ReadOnlyFile() {
	GLE(close());
}
bool ReadOnlyFile::isOpen() {
	return mHandle != INVALID_HANDLE_VALUE;
}
bool ReadOnlyFile::close() {
	if(isOpen()) {
		TGLE(CloseHandle(mHandle));
		mHandle = INVALID_HANDLE_VALUE;
	}
	return true;
}
bool ReadOnlyFile::open(const char* filename) {
	MYASSERT(!isOpen());
	mHandle = CreateFile(filename, GENERIC_READ, FILE_SHARE_READ, NULL,
		OPEN_EXISTING, 0, NULL);
	if(mHandle == INVALID_HANDLE_VALUE) {
		DEGUB("Failed to open \"%s\"\n", filename);
		DumpGLE(); RD(0);
	}
	return true;
}
bool ReadOnlyFile::read(void* dst, uint size) {
	DWORD nBR;
	TGLE(ReadFile(mHandle, dst, size, &nBR, NULL));
	MYASSERT(nBR == size);
	return true;
}
bool ReadOnlyFile::size(uint* result) {
	*result = GetFileSize(mHandle, NULL);
	TGLECUSTOM(*result, INVALID_FILE_SIZE);
	return true;
}
bool ReadOnlyFile::seek(uint pos) {
	TGLECUSTOM(INVALID_SET_FILE_POINTER, SetFilePointer(mHandle, pos, NULL, FILE_BEGIN));
	return true;
}
bool ReadOnlyFile::tell(uint* result) {
	*result = SetFilePointer(mHandle, 0, NULL, FILE_CURRENT);
	TGLECUSTOM(INVALID_SET_FILE_POINTER, *result);
	return true;
}
bool ReadOnlyFile::readbswapw(DWORD* dst, size_t count) {
	TGLE(read(dst, 4 * count));
	for(size_t i=0; i<count; i++) {
		dst[i] = swapw(dst[i]);
	}
	return true;
}
bool ReadOnlyFile::readbswaph(WORD* dst, size_t count) {
	TGLE(read(dst, 2 * count));
	for(size_t i=0; i<count; i++) {
		dst[i] = swaph(dst[i]);
	}
	return true;
}

bool File::open(const char* filename, OpenMode mode) {
	MYASSERT(!isOpen());
	DWORD creationDisposition;
	switch(mode) {
	case OpenExisting: creationDisposition = OPEN_EXISTING; break;
	case CreateAlways: creationDisposition = CREATE_ALWAYS; break;
	case CreateNew: creationDisposition = CREATE_NEW; break;
	default:
		FAIL(UE_GENERIC_ERROR);
	}
	mHandle = CreateFile(filename, GENERIC_READ|GENERIC_WRITE, 0, NULL,
		creationDisposition, 0, NULL);
	TGLECUSTOM(mHandle, INVALID_HANDLE_VALUE);
	return true;
}
bool File::write(const void* src, uint size) {
	DWORD nBW;
	TGLE(WriteFile(mHandle, src, size, &nBW, NULL));
	MYASSERT(nBW == size);
	return true;
}
