// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "gp_texture.h"
#include "gp_internal.h"
#include "main.h"
#include "stringCommons.h"

void conv_i4_ai8(BYTE *dest, const BYTE *src) {
	dest[0] = dest[1] = HIGH_FOUR_TO_8BIT(*src);
	dest[2] = dest[3] = LOW_FOUR_TO_8BIT(*src);
}
void conv_i8_ai8(BYTE *dest, const BYTE *src) {
	dest[0] = dest[1] = *src;
}
void conv_ia4_ai8(BYTE *dest, const BYTE *src) {
	dest[0] = LOW_FOUR_TO_8BIT(*src);
	dest[1] = HIGH_FOUR_TO_8BIT(*src);
}
void conv_ia8_ai8(BYTE *dest, const BYTE *src) {
	dest[0] = src[1];
	dest[1] = src[0];
}
void conv_16(BYTE *dest, const BYTE *src) {
	*MAKEP(WORD, dest) = swaph(*MAKEP(WORD, src));
}
void conv_rgb5a3_bgra8(BYTE *dest, const BYTE *src) {
	WORD word = swaph(*MAKEP(WORD, src));
	if((word & 0x8000) == 0) {  //A3RGB4
		BYTE b = (BYTE)getbitsh(word, 12, 15);
		dest[0] = FOUR_TO_8BIT(b);
		BYTE g = (BYTE)getbitsh(word, 8, 11);
		dest[1] = FOUR_TO_8BIT(g);
		BYTE r = (BYTE)getbitsh(word, 4, 7);
		dest[2] = FOUR_TO_8BIT(r);
		BYTE a = (BYTE)getbitsh(word, 1, 3);
		dest[3] = THREE_TO_8BIT(a);
	} else {	//X1RGB5
		BYTE b = (BYTE)getbitsh(word, 11, 15);
		dest[0] = FIVE_TO_8BIT(b);
		BYTE g = (BYTE)getbitsh(word, 6, 10);
		dest[1] = FIVE_TO_8BIT(g);
		BYTE r = (BYTE)getbitsh(word, 1, 5);
		dest[2] = FIVE_TO_8BIT(r);
		dest[3] = 0xFF;
	}
}
void conv_rgba8_bgra8(BYTE *dest, const BYTE *src) {
	dest[0] = src[33];
	dest[1] = src[32];  //relies on tile_multiplier
	dest[2] = src[1];
	dest[3] = src[0];
}

void dump_i4(BYTE *color, BYTE* /*alpha*/, const BYTE *src) {
	color[0] = HIGH_FOUR_TO_8BIT(*src);
	color[1] = LOW_FOUR_TO_8BIT(*src);
}
void dump_i8(BYTE *color, BYTE* /*alpha*/, const BYTE *src) {
	*color = *src;
}
void dump_ia4(BYTE *color, BYTE *alpha, const BYTE *src) {
	*color = LOW_FOUR_TO_8BIT(*src);
	*alpha = HIGH_FOUR_TO_8BIT(*src);
}
void dump_ia8(BYTE *color, BYTE *alpha, const BYTE *src) {
	*color = src[1];
	*alpha = src[0];
}
void dump_rgb565(BYTE *color, BYTE* /*alpha*/, const BYTE *src) {
	WORD word = swaph(*MAKEP(WORD, src));
	BYTE r = (BYTE)getbitsh(word, 0, 4);
	color[0] = (r << 3) | (r >> 2);
	BYTE g = (BYTE)getbitsh(word, 5, 10);
	color[1] = (g << 2) | (g >> 4);
	BYTE b = (BYTE)getbitsh(word, 11, 15);
	color[2] = (b << 3) | (b >> 2);
}
void dump_rgb5a3(BYTE *color, BYTE *alpha, const BYTE *src) {
	BYTE bgra[4];
	conv_rgb5a3_bgra8(bgra, src);
	color[0] = bgra[2];
	color[1] = bgra[1];
	color[2] = bgra[0];
	*alpha = bgra[3];
}
void dump_rgba8(BYTE *color, BYTE *alpha, const BYTE *src) {
	color[0] = src[1];
	color[1] = src[32];
	color[2] = src[33];
	*alpha = src[0];
}

//These two are set by createTexture
static BYTE *s_tlut;
static GXTlutFmt s_tlutfmt;
void conv_c4(BYTE *dest, const BYTE *src) {
	BYTE i = (*src) >> 4;
	txfc[s_tlutfmt].conversion(dest, s_tlut + i*2);
	i = (*src) & 0x0F;
	txfc[s_tlutfmt].conversion(dest + txtf[txfc[s_tlutfmt].tf].bytes, s_tlut + i*2);
}
void conv_c8(BYTE *dest, const BYTE *src) {
	BYTE i = *src;
	txfc[s_tlutfmt].conversion(dest, s_tlut + i*2);
}
void conv_c14(BYTE *dest, const BYTE *src) {
	WORD i = swaph(*MAKEP(WORD, src)) & 0x3FFF;
	txfc[s_tlutfmt].conversion(dest, s_tlut + i*2);
}
void dump_c4(BYTE *color, BYTE *alpha, BYTE *index, const BYTE *src) {
	BYTE i = (*src) >> 4;
	*index = i;
	txfc[s_tlutfmt].dump(color, alpha, s_tlut + i*2);
	*(index + 1) = i = (*src) & 0x0F;
	txfc[s_tlutfmt].dump(color + txfc[s_tlutfmt].dump_color,
		alpha + txfc[s_tlutfmt].dump_alpha, s_tlut + i*2);
}
void dump_c8(BYTE *color, BYTE *alpha, BYTE *index, const BYTE *src) {
	BYTE i = *src;
	*index = i;
	txfc[s_tlutfmt].dump(color, alpha, s_tlut + i*2);
}
void dump_c14(BYTE *color, BYTE *alpha, BYTE *index, const BYTE *src) {
	WORD i = swaph(*MAKEP(WORD, src)) & 0x3FFF;
	*MAKEP(WORD, index) = i;
	txfc[s_tlutfmt].dump(color, alpha, s_tlut + i*2);
}


LPDIRECT3DTEXTURE9 GP::createTexture(DWORD address, DWORD width, DWORD height,
																		 DWORD format, DWORD tlutformat, DWORD tlutoffset,
																		 int nchanges)
{
	LPDIRECT3DTEXTURE9 pTex = NULL;

	MYASSERT(format < 16);
	if(txf[format].pixelsize == INVALID_PS)
		throw hardware_fatal_exception("GP Texture illegal format!");
	if(width > 1024 || height > 1024)
		throw hardware_fatal_exception("GP Texture illegal size!");

	TargetFormat targetformat = TX_ISCI(format) ?
		txfc[tlutformat].tf : txf[format].targetformat;
	size_t tile_width = txps[txf[format].pixelsize].tile_width,
		tile_height = txps[txf[format].pixelsize].tile_height;
	size_t tiles_x = 1 + (width - 1) / tile_width,
		tiles_y = 1 + (height - 1) / tile_height;
	size_t tiled_width = tiles_x*tile_width, tiled_height = tiles_y*tile_height;
	size_t tiled_npixels = tiled_width * tiled_height;
	size_t tf_bytes = txtf[targetformat].bytes;
	size_t sf_bits = txps[txf[format].pixelsize].bits;
	size_t src_size = tiled_npixels * sf_bits / 8;
	size_t tile_multiplier = (sf_bits*tile_width*tile_height / 8) / 32;
	MYASSERT(tile_multiplier == 1 || tile_multiplier == 2);

	//if(src_size > ((txf[format].pixelsize == PS_32) ? 1*M : 512*K))
	//throw hardware_fatal_exception("GP Texture is too big to fit in tmem!");

	Container<BYTE> dump_color(0);
	Container<BYTE> dump_alpha(0);
	Container<BYTE> dump_index(0);
	size_t mul_dc, mul_da, mul_di;
	void (*func_dump_indexed)(BYTE *, BYTE *, BYTE *, const BYTE *) = NULL;
	if(TX_ISCI(format)) {
		s_tlutfmt = (GXTlutFmt)tlutformat;
		s_tlut = m.tmem + tlutoffset;
		mul_dc = txfc[tlutformat].dump_color;
		mul_da = txfc[tlutformat].dump_alpha;
		mul_di = format == GX_TF_C14 ? 2 : 1;

		switch(format) {
		case GX_TF_C4:
			func_dump_indexed = dump_c4;
			break;
		case GX_TF_C8:
			func_dump_indexed = dump_c8;
			break;
		case GX_TF_C14:
			func_dump_indexed = dump_c14;
			break;
		default:
			throw hardware_fatal_exception("Internal error: Texture conversion no. 1!");
		}
	} else {
		mul_dc = txf[format].dump_color;
		mul_da = txf[format].dump_alpha;
		mul_di = 0;
	}
	dump_color.resize(tiled_npixels * mul_dc);
	dump_alpha.resize(tiled_npixels * mul_da);
	dump_index.resize(tiled_npixels * mul_di);

	Container<BYTE> dest(tiled_npixels * tf_bytes);
	BYTE *source = mem.getp_physical(address, src_size);

	for(size_t ty=0; ty<tiles_y; ty++) {
		for(size_t tx=0; tx<tiles_x; tx++) {
			size_t tile = ty*tiles_x + tx;
			if(format == GX_TF_CMPR) {
				//each block is 8 bytes, 16(4x4) texels
				//each tile is 4(2x2) blocks
				size_t blocks_x = tiled_width / 4;//, blocks_y = tiles_y * 8 / 4;
				for(size_t y=0; y<2; y++) {
					for(size_t x=0; x<2; x++) {
						size_t src = (tile*4 + y*2 + x) * 8;
						size_t dst = ((ty*2 + y)*blocks_x + tx*2 + x) * 8;
						//VGPDEGUB("src %i => dst %i\n", src / 8, dst / 8);
						MYASSERT(src < tiled_npixels / 2);
						MYASSERT(dst < tiled_npixels * 4);
						struct {
							WORD col0, col1;
							BYTE b[4];
						} block;
						MAKE(QWORD, block) = MAKE(QWORD, source[src]);
						block.col0 = swaph(block.col0);
						block.col1 = swaph(block.col1);

#define GC_G(num) FIVE_TO_8BIT(getbitsh(block.col##num, 0, 4))
#define GC_B(num) SIX_TO_8BIT(getbitsh(block.col##num, 5, 10))
#define GC_R(num) FIVE_TO_8BIT(getbitsh(block.col##num, 11, 15))
						//VGPDEGUB("Block is 0x%04X, 0x%04X, %02X %02X %02X %02X\n",
						//block.col0, block.col1, block.b[0], block.b[1], block.b[2], block.b[3]);
						//VGPDEGUB("0: r %f, b %f, g %f\n1: r %f, b %f, g %f\n",
						//GC_R(0), GC_B(0), GC_G(0), GC_R(1), GC_B(1), GC_G(1));

						bool opaque = block.col0 > block.col1;
						//opaque = true;
						for(size_t by=0; by<4; by++) {
							for(size_t bx=0; bx<4; bx++) {
								size_t bdst = (ty*8 + y*4 + by)*tiled_width + tx*8 + x*4 + bx;
								DWORD t = getbitsb(block.b[by], bx*2, bx*2+1);
#define GC_F2B(fl) (BYTE(fl))
#define GC_RGB(macro) macro(GC_R) macro(GC_B) macro(GC_G)
								BYTE bGC_R, bGC_B, bGC_G;
								bool alpha = true;
								switch(t) {
		case 0:
#define GC_C0(color) b##color = GC_F2B(color(0));
			GC_RGB(GC_C0);
			break;
		case 1:
#define GC_C1(color) b##color = GC_F2B(color(1));
			GC_RGB(GC_C1);
			break;
		case 2:
#define GC_C2(color) b##color = GC_F2B(opaque ?\
	((2 * color(0) + color(1) + 1) / 3) : ((color(0) + color(1)) / 2));
			GC_RGB(GC_C2);
			break;
		case 3:
#define GC_C3(color) b##color = GC_F2B(opaque ?\
	((color(0) + 2 * color(1) + 1) / 3) : 0);
			GC_RGB(GC_C3);
			if(!opaque)
				alpha = false;
			break;
		default:
			throw hardware_fatal_exception(
				"Internal error: Texture conversion no. 2!");
								}
								//VGPDEGUB("bdst = %i\n", bdst);
								size_t temp = bdst * 4;
								dest[temp + 3] = alpha ? 0xFF : 0x00;
#define GC_SET(color) dest[temp++] = b##color;
								GC_RGB(GC_SET);
								if(g::gp_dtex || g::gp_dtexraw) {
									dump_color[bdst*3+0] = dest[bdst*4+2];
									dump_color[bdst*3+1] = dest[bdst*4+1];
									dump_color[bdst*3+2] = dest[bdst*4+0];
									dump_alpha[bdst] = dest[bdst*4+3];
								}
							}
						}
					}
				}
			} else for(size_t y=0; y<tile_height; y++) {  //uncompressed
				for(size_t x=0; x<tile_width; x++) {
					size_t dst = (ty*tile_height + y)*tiled_width + tx*tile_width + x;
					size_t src = (((tile*tile_multiplier*tile_height + y)*tile_width + x)*
						sf_bits)/8/tile_multiplier;
					MYASSERT(dst < dest.size());
					MYASSERT(src < src_size);
					txf[format].conversion(dest + dst*tf_bytes, source + src);
					if(g::gp_dtex || g::gp_dtexraw) {
						if(TX_ISCI(format)) {
							func_dump_indexed(dump_color + dst*mul_dc, dump_alpha + dst,
								dump_index + dst*mul_di, source + src);
						} else {
							txf[format].dump(dump_color + dst*mul_dc,
								dump_alpha + dst, source + src);
						}
					}
					if(sf_bits == 4)  //kinda hacky, but less so than the alternative
						x++;
				}
			}
		}
	}

	createLoadTexture(&pTex, width, height, txtf[targetformat].format,
		tiled_width*txtf[targetformat].bytes, dest);

	const char *str_dump_color[4] = { TROGDOR, "i", TROGDOR, "rgb" };
	if(g::gp_dtex || g::gp_dtexraw) {
		DEGUB("Dumping texture: width %d | height %d", width, height);
		if(tiled_width != width || tiled_height != height) {
			DEGUB(" | tiled_width %d | tiled_height %d", tiled_width, tiled_height);
		}
		DEGUB(" | format %x\n", format);
	}
	MYASSERT(mul_dc == 1 || mul_dc == 3);

	if(g::gp_dtex || g::gp_dtexraw) {
		string basename;

		HWGLE(prepareTexDumpBaseName(basename, address, nchanges));
		ostringstream str;
		str << basename << "_" << HEx(format);
		basename = str.str();

#define BASE2(str) CONCAT(basename, str)
		if(g::gp_dtex) {
			HWGLE(writePNGEx(BASE2(str_dump_color[mul_dc]),
				mul_dc - 1,	//hella-hack
				tiled_width, tiled_height, dump_color));
		}
		if(g::gp_dtexraw) {
#define RAWNAME(str) CONCAT(BASE2(str), ".raw")
			HWGLE(dumpToFile(source, src_size, RAWNAME('t')));
			HWGLE(dumpToFile(dump_color, dump_color.size(), RAWNAME(str_dump_color[mul_dc])));
			if(mul_da)
				HWGLE(dumpToFile(dump_alpha, dump_alpha.size(), RAWNAME('a')));
			if(mul_di)
				HWGLE(dumpToFile(dump_index, dump_index.size(), RAWNAME('c')));
		}
	}
	return pTex;
}

bool GP::prepareTexDumpBaseName(string& s, DWORD address, int nchanges) {
	if(mDumpTexDir.empty()) {  //fix directory
		const string& exepath = g_capp.getExeName();
		size_t pos = exepath.find_last_of('\\');
		string exefilename = (pos == string::npos) ? exepath : exepath.substr(pos + 1);
		TGLE(getNextNumberedName(mDumpTexDir, (exefilename + ".").c_str(), 3, ""));
		TGLE(CreateDirectory(mDumpTexDir.c_str(), NULL));
	}
	ostringstream str;
	str << mDumpTexDir << "\\" << HEX08(address) << "_" << setw(3) << nchanges;
	s = str.str();
	return true;
}
