// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#define _WIN32_WINNT 0x0500

#include "hw_internal.h"
#include "degub.h"
#include "hardware.h"
#include "container.h"
#include "ini.h"
#include "registers.h"
#include "interrupt.h"
#include "messages.h"
#include "hw_ai.h"
#include "hw_di.h"
#include "hw_ei.h"
#include "hw_pi.h"
#include "hw_si.h"
#include "hw_vi.h"
#include "hw_gx.h"
#include "gp.h"
#include "memcard.h"
#include "exi_01.h"
#include "exi_ad16.h"
#include "exi_bba.h"
#include "exi_memcard.h"
#include "dsp_core.h"
#include "main.h"
#include "stringCommons.h"
#include <crtdbg.h>
#include <fstream>

class HwBase {
public:
	//Read Byte/Word/Dword
	static BYTE direct_rb(Hardware *h, WORD offset) { return h->hrb(offset); }
	static WORD direct_rh(Hardware *h, WORD offset) { return h->hrh(offset); }
	static DWORD direct_rw(Hardware *h, WORD offset) { return h->hrw(offset); }
	static QWORD direct_rd(Hardware *h, WORD offset) { return h->hrd(offset); }
	//Write Byte/Word/Dword
	static void direct_wb(Hardware *h, WORD offset, BYTE data) { h->hwb(offset, data); }
	static void direct_wh(Hardware *h, WORD offset, WORD data) { h->hwh(offset, data); }
	static void direct_ww(Hardware *h, WORD offset, DWORD data) { h->hww(offset, data); }
	static void direct_wd(Hardware *h, WORD offset, QWORD data) { h->hwd(offset, data); }
	//Read Byte/Word/Dword
	static BYTE unhandled_rb(Hardware *h, WORD offset);
	static WORD unhandled_rh(Hardware *h, WORD offset);
	static DWORD unhandled_rw(Hardware *h, WORD offset);
	static QWORD unhandled_rd(Hardware *h, WORD offset);
	//Write Byte/Word/Dword
	static void unhandled_wb(Hardware *h, WORD offset, BYTE data);
	static void unhandled_wh(Hardware *h, WORD offset, WORD data);
	static void unhandled_ww(Hardware *h, WORD offset, DWORD data);
	static void unhandled_wd(Hardware *h, WORD offset, QWORD data);

	static WORD rb_log(Hardware *h, WORD offset);
	static WORD rh_log(Hardware *h, WORD offset);
	static DWORD rw_log(Hardware *h, WORD offset);
	static void wb_log(Hardware *h, WORD offset, BYTE data);
	static void wh_log(Hardware *h, WORD offset, WORD data);
	static void ww_log(Hardware *h, WORD offset, DWORD data);
};

#pragma warning( disable : 4355 )

Hardware::Hardware(MemInterface &mem, InterruptRaiser &i, CORE_ARGUMENTS a,
									 CycleCounterBase &ccb) :
m(mem), hWnd(a.hWnd), m_pdiDevice(a.pdiDevice), m_visi_limit(0),
m_lidopen(false), m_dvdstatus(0),
eir_mcs1(this, 0), eir_mcs2(this, 1), eir_bba(this, 2),
m_cc(ccb), interrupt(this), m_visis(0), m_old_visis(0),
gp(new GP(mem, a.pd3dDevice, this)), m_window_active(true),
m_gp_activated(false), m_aram(ARAM_SIZE), gekko_interrupt(i), m_pisr_reads(0),
m_old_pisr_reads(0), m_hwt(N_HWTIMINGS),
dsp_core(g::dsp_hle ? NULL : new DSPInterpreter(mem, *this))
{
	ZeroMemory(hwrm, HARDWARE_MEMSIZE);
	ZeroMemory(m_aram, ARAM_SIZE);
	ZERO_OBJECT(vi);
	ZERO_OBJECT(ai);
	ZERO_OBJECT(dsp);
	ZERO_OBJECT(exi);
	ZeroMemory(m_hwt.p(), m_hwt.size()*sizeof(ULI));
	ZERO_OBJECT(num);

	ZERO_OBJECT(fifo);
	fifo.gp.idle = true;	//hack, needed due to bypass of warning 4355

	vi.minSleepsPerFrame = INT_MAX;

	memset_dword(arr_rb, (DWORD)&HwBase::unhandled_rb, RHOOK_NUM_ARRAY_ENTRIES);
	memset_dword(arr_rh, (DWORD)&HwBase::unhandled_rh, RHOOK_NUM_ARRAY_ENTRIES);
	memset_dword(arr_rw, (DWORD)&HwBase::unhandled_rw, RHOOK_NUM_ARRAY_ENTRIES);
	memset_dword(arr_rd, (DWORD)&HwBase::unhandled_rd, RHOOK_NUM_ARRAY_ENTRIES);
	memset_dword(arr_wb, (DWORD)&HwBase::unhandled_wb, WHOOK_NUM_ARRAY_ENTRIES);
	memset_dword(arr_wh, (DWORD)&HwBase::unhandled_wh, WHOOK_NUM_ARRAY_ENTRIES);
	memset_dword(arr_ww, (DWORD)&HwBase::unhandled_ww, WHOOK_NUM_ARRAY_ENTRIES);
	memset_dword(arr_wd, (DWORD)&HwBase::unhandled_wd, WHOOK_NUM_ARRAY_ENTRIES);

	//not completely accurate (R/W assert), but I don't care right now.
#define SET_HOOK(type, offset, function) { MYASSERT((offset) < WHOOK_NUM_ARRAY_ENTRIES);\
	MAKE(DWORD, arr_##type[offset]) = (DWORD)(function); }
#define SET_HOOK_DIRECT(type, offset) SET_HOOK(type, offset, HwBase::direct_##type)
#define SET_HOOK_LOG(condition, type, offset) if(condition) {\
	SET_HOOK(type, offset, HwBase::type##_log); } else {\
	SET_HOOK_DIRECT(type, offset); }

	//Video
	for(int i=0x2000; i<0x2080; i+=2) {
		SET_HOOK_LOG(g::gp_log, wh, i);
		SET_HOOK_LOG(g::gp_log, rh, i);
	}
	for(int i=0x2000; i<0x2080; i+=4) {
		SET_HOOK_LOG(g::gp_log, ww, i);
		SET_HOOK_LOG(g::gp_log, rw, i);
	}
	SET_HOOK(wh, 0x2000, HwVI::wh_vtr);
	SET_HOOK(wh, 0x2002, HwVI::wh_dcr);
	SET_HOOK(ww, 0x2000, HwVI::ww_cr);
	SET_HOOK(ww, 0x201C, HwVI::ww_xfb);
	SET_HOOK(ww, 0x2024, HwVI::ww_xfb);
	SET_HOOK(rh, 0x202C, HwVI::rh_vct);
	SET_HOOK(rh, 0x202E, HwVI::rh_hct);

#define SET_VIDI_HOOKS(number)\
	SET_HOOK_LOG(g::gp_log, rw, VI_DI(number));\
	SET_HOOK(ww, VI_DI(number),\
	((void (*)(Hardware*, WORD, DWORD))HwVI::ww_di<number>));\
	SET_HOOK(wh, VI_DI(number),\
	((void (*)(Hardware*, WORD, WORD))HwVI::wh_di_high<number>));\
	SET_HOOK(wh, VI_DI(number) + 2,\
	((void (*)(Hardware*, WORD, WORD))HwVI::wh_di_low<number>));\
	SET_HOOK_DIRECT(rh, VI_DI(number));

	VIDEO_INTERRUPTS(SET_VIDI_HOOKS);

	SET_HOOK(wh, 0x201C, HwVI::wh_xfb_high);
	SET_HOOK(wh, 0x201E, HwVI::wh_xfb_low);
	SET_HOOK(wh, 0x2024, HwVI::wh_xfb_high);
	SET_HOOK(wh, 0x2026, HwVI::wh_xfb_low);

	//proper, I think.
	SET_HOOK_DIRECT(rh, 0x206C);

	SET_HOOK_LOG(g::gp_log, rb, 0x2074); //s3-f106b.dol
	SET_HOOK_LOG(g::gp_log, rb, 0x2075); //s3-f106b.dol
	SET_HOOK_LOG(g::gp_log, rb, 0x2001); //s3-f106b.dol

	//SI
	for(WORD i=0; i<PAD_NPADS; i++) {
		//Default values for unconnected pads?
		hww(0x6400 + i*0xC, 0x00400300);	//Command? See CZN info.
		hwh(0x6404 + i*0xC, 0x0000);  //No buttons pressed
		hwh(0x6406 + i*0xC, 0x8080);  //Analog sticks in middle
		hwh(0x6408 + i*0xC, 0x8080);
		hwh(0x640A + i*0xC, 0x0000);  //Analog triggers released

		SET_HOOK(ww, 0x6400 + i*0xC, HwSI::ww_out);
		SET_HOOK_LOG(g::si_log, rw, 0x6400 + i*0xC);
		SET_HOOK(rw, 0x6404 + i*0xC, HwSI::rw_in);
		SET_HOOK(rw, 0x6408 + i*0xC, HwSI::rw_in);
	}
	SET_HOOK(ww, SI_POLL, HwSI::ww_poll);
	SET_HOOK(rw, SI_POLL, HwSI::rw_poll);
	SET_HOOK(ww, SI_CR, HwSI::ww_cr);
	SET_HOOK(rw, SI_CR, HwSI::rw_cr);
	SET_HOOK(ww, SI_SR, HwSI::ww_sr);
	SET_HOOK(rw, SI_SR, HwSI::rw_sr);
	SET_HOOK_LOG(g::si_log, rw, SI_EXILK);
	SET_HOOK(ww, SI_EXILK, HwSI::ww_exilk);
	for(int i=0; i<0x80; i+=4) {
		SET_HOOK_LOG(g::si_log, rw, 0x6480+i);
		SET_HOOK_LOG(g::si_log, ww, 0x6480+i);
	}

	//GX
	SET_HOOK(wb, 0x8000, HwGX::wb_fifo);
	SET_HOOK(wh, 0x8000, HwGX::wh_fifo);
	SET_HOOK(ww, 0x8000, HwGX::ww_fifo);
	SET_HOOK(wd, 0x8000, HwGX::wd_fifo);

	for(WORD i=0x0000; i<0x0100; i+=4) {
		SET_HOOK_LOG(g::gp_log, ww, i);	//kinderkram
	}
	for(WORD i=0x0000; i<0x0100; i+=2) {
		SET_HOOK_LOG(g::gp_log, wh, i);	//gx_demo.tmbinc.dol
	}

	SET_HOOK(rh, 0x0000, HwGX::rh_cp_sr);
	SET_HOOK(wh, 0x0002, HwGX::wh_cp_cr);
	SET_HOOK_LOG(g::gp_log, rh, 0x0002);
	//be ware! breakpoint (and possibly others) change state without writing to hwmem.
	//SET_HOOK(rh, 0x0002, HwGX::rh_cp_cr);
	SET_HOOK(wh, 0x0004, HwGX::wh_cp_clear);
	SET_HOOK(wh, 0x0006, HwGX::wh_cp_metric);

	SET_HOOK_LOG(g::gp_log, wh, 0x000A);	//unknown, unrelled\mainGB.dol
	SET_HOOK_LOG(g::gp_log, rh, 0x0040);
	SET_HOOK_LOG(g::gp_log, rh, 0x0042);
	SET_HOOK_LOG(g::gp_log, rh, 0x0044);
	SET_HOOK_LOG(g::gp_log, rh, 0x0046);
	SET_HOOK_LOG(g::gp_log, rh, 0x0048);
	SET_HOOK_LOG(g::gp_log, rh, 0x004A);
	SET_HOOK_LOG(g::gp_log, rh, 0x004C);
	SET_HOOK_LOG(g::gp_log, rh, 0x004E);

	for(WORD i=0x1000; i<0x1100; i++) { //weird, unrelled\mainGB.dol
		SET_HOOK_LOG(g::gp_log, wb, i);
	}
	SET_HOOK_LOG(g::gp_log, wh, 0x1000);
	SET_HOOK_LOG(g::gp_log, wh, 0x1002);
	SET_HOOK_LOG(g::gp_log, rh, 0x1002);
	SET_HOOK_LOG(g::gp_log, wh, 0x1004);
	SET_HOOK_LOG(g::gp_log, wh, 0x1006);
	SET_HOOK_LOG(g::gp_log, wh, 0x1008);
	SET_HOOK_LOG(g::gp_log, rh, 0x100A);
	SET_HOOK(wh, 0x100A, HwGX::wh_pe_isr);
	SET_HOOK_LOG(g::gp_log, wh, 0x100E);
	SET_HOOK_LOG(g::gp_log, rh, 0x100E);
	SET_HOOK_LOG(g::gp_log, rh, 0x1010);
	SET_HOOK_LOG(g::gp_log, rh, 0x1012);
	SET_HOOK_LOG(g::gp_log, rh, 0x1014);
	SET_HOOK_LOG(g::gp_log, rh, 0x1016);
	for(WORD i=0x1018; i<=0x102E; i+=2) {	//ReadPixMetric
		SET_HOOK_LOG(g::gp_log, rh, i);
	}

#define HOOK_GXI_CPU(cap, gem, address) SET_HOOK(ww, address, HwGX::ww_cpufifo_##gem);\
	SET_HOOK(rw, address, HwGX::rw_cpufifo_##gem);
	GXI_CPUS(HOOK_GXI_CPU);
#define HOOK_GXI_GP(cap, gem, address, rhlo)\
	SET_HOOK(wh, address, HwGX::wh_gpfifo_##gem##_lo);\
	SET_HOOK(wh, address + 2, HwGX::wh_gpfifo_##gem##_hi);\
	SET_HOOK(rh, address, HwGX::rh_gpfifo_##gem##_lo);\
	SET_HOOK(rh, address + 2, HwGX::rh_gpfifo_##gem##_hi);
	GXI_GPS(HOOK_GXI_GP);

	//AI, DSP, ARAM
	SET_HOOK(rh, 0x503A, HwAI::rh_dma_bl);
	SET_HOOK(wh, 0x5000, HwAI::wh_mail_dsp_high);
	SET_HOOK(rh, 0x5000, HwAI::rh_mail_dsp_high);
	SET_HOOK(wh, 0x5002, HwAI::wh_mail_dsp_low);
	SET_HOOK(rh, 0x5004, HwAI::rh_mail_cpu_high);
	SET_HOOK(rh, 0x5006, HwAI::rh_mail_cpu_low);
	SET_HOOK(wh, 0x500A, HwAI::wh_dspcr);
	SET_HOOK_LOG(g::dsp_log, rh, 0x500A);
	SET_HOOK_LOG(g::dsp_log, rh, 0x5002);
	SET_HOOK_LOG(g::dsp_log, wh, 0x5012);
	SET_HOOK_LOG(g::dsp_log, rh, 0x5012);
	SET_HOOK(rh, 0x5016, HwAI::rh_ar_mode);
	SET_HOOK_LOG(g::dsp_log, rh, 0x501A);
	SET_HOOK_LOG(g::dsp_log, wh, 0x501A);
	SET_HOOK_DIRECT(rh, 0x5030);
	SET_HOOK_DIRECT(rh, 0x5032);
	SET_HOOK_DIRECT(rh, 0x5036);
	SET_HOOK(wh, 0x5036, HwAI::wh_dma_cr);
	SET_HOOK_DIRECT(wh, 0x5030);
	SET_HOOK_DIRECT(wh, 0x5032);
	SET_HOOK_DIRECT(ww, 0x5020);
	SET_HOOK_DIRECT(ww, 0x5024);
	SET_HOOK(ww, 0x5028, HwAI::ww_ar_dma);
	SET_HOOK_DIRECT(wh, 0x5020);
	SET_HOOK_DIRECT(wh, 0x5022);
	SET_HOOK_DIRECT(wh, 0x5024);
	SET_HOOK_DIRECT(wh, 0x5026);
	//both must be written for a transfer to take place (?)
	SET_HOOK(wh, 0x5028, HwAI::wh_ar_dma_high);
	SET_HOOK(wh, 0x502A, HwAI::wh_ar_dma_low);
	SET_HOOK_DIRECT(rh, 0x5020);
	SET_HOOK_DIRECT(rh, 0x5022);
	SET_HOOK_DIRECT(rh, 0x5024);
	SET_HOOK_DIRECT(rh, 0x5026);
	SET_HOOK_DIRECT(rh, 0x5028);
	SET_HOOK_DIRECT(rh, 0x502A);

	SET_HOOK(ww, 0x6C00, HwAI::ww_aicr);
	SET_HOOK(rw, 0x6C00, HwAI::rw_aicr);
	SET_HOOK_LOG(g::log_audio, rw, 0x6C04);
	//SET_HOOK(ww, 0x6C04, HwAI::ww_aivr);
	SET_HOOK_LOG(g::log_audio, ww, 0x6C04);
	SET_HOOK(rw, 0x6C08, HwAI::rw_aiscnt);
	SET_HOOK_LOG(g::log_audio, ww, 0x6C0C);

	hwh(0x500A, DSPCR_HALT);  //DSP is initially halted(!)

	HR(DirectSoundCreate8(NULL, &ai.lpDirectSound, NULL));
	HR(ai.lpDirectSound->SetCooperativeLevel(hWnd, DSSCL_NORMAL));//DSSCL_PRIORITY));

	ai.wfx.wFormatTag = WAVE_FORMAT_PCM;
	ai.wfx.nChannels = 2;
	ai.wfx.wBitsPerSample = 16;
	ai.wfx.nBlockAlign = ai.wfx.nChannels * ai.wfx.wBitsPerSample / 8;
	ai.wfx.nSamplesPerSec = 48000;  //AICR is zeroed at first, which means 48kHz
	ai.wfx.nAvgBytesPerSec = ai.wfx.nSamplesPerSec * ai.wfx.nBlockAlign;
	ai.wfx.cbSize = 0;
	ZeroMemory(&ai.dsbdesc, sizeof(DSBUFFERDESC)); 
	ai.dsbdesc.dwSize = sizeof(DSBUFFERDESC); 
	ai.dsbdesc.dwFlags = DSBCAPS_CTRLPAN | DSBCAPS_CTRLVOLUME |
		DSBCAPS_CTRLFREQUENCY | DSBCAPS_GLOBALFOCUS | //DSBCAPS_CTRLPOSITIONNOTIFY |
		DSBCAPS_GETCURRENTPOSITION2 | DSBCAPS_STICKYFOCUS | DSBCAPS_LOCSOFTWARE;
	ai.dsbdesc.lpwfxFormat = &ai.wfx; 

	if(g::dump_audio) {
		//open, truncate and close, all in one line :)
		ofstream::basic_ofstream("audio.raw");
	}

	//DVD
	hww(PR_DISR,  0x0000002A); //From OpenGC

	SET_HOOK(ww, 0x6000, HwDI::ww_sr);
	SET_HOOK_LOG(g::dvd_log, rw, 0x6000);
	SET_HOOK(ww, 0x6004, HwDI::ww_cvr);
	SET_HOOK(rw, 0x6004, HwDI::rw_cvr);
	SET_HOOK_LOG(g::dvd_log, ww, 0x6008);
	SET_HOOK_LOG(g::dvd_log, rw, 0x6008);
	SET_HOOK_LOG(g::dvd_log, ww, 0x600C);
	SET_HOOK_LOG(g::dvd_log, ww, 0x6010);
	SET_HOOK_LOG(g::dvd_log, ww, 0x6014);
	SET_HOOK_LOG(g::dvd_log, ww, 0x6018);
	SET_HOOK_LOG(g::dvd_log, rw, 0x6018);
	SET_HOOK(ww, 0x601C, HwDI::ww_cr);
	SET_HOOK_LOG(g::dvd_log, rw, 0x601C);
	SET_HOOK(rw, 0x6020, HwBase::rw_log);
	SET_HOOK(rw, 0x6024, HwDI::rw_cfg);

	//EXI
	ZeroMemory(exi_devices, sizeof(exi_devices));
	exi_devices[0][2] = new EXIDevice_BBA(eir_bba);
	exi_devices[0][1] = new EXIDevice_01(hWnd);
	exi_devices[2][0] = new EXIDevice_AD16();

	for(int i=0; i<3; i++) {
		SET_HOOK_DIRECT(ww, EXI_IMMDAT(i));
		SET_HOOK_DIRECT(ww, EXI_DMABUF(i));
		SET_HOOK_DIRECT(ww, EXI_DMALEN(i));
		SET_HOOK(ww, EXI_STATUS(i), HwEI::ww_sr);
		SET_HOOK(ww, EXI_CONTROL(i), HwEI::ww_cr);
		SET_HOOK_DIRECT(rw, EXI_CONTROL(i));
		SET_HOOK_DIRECT(rw, EXI_IMMDAT(i));
		SET_HOOK(rw, EXI_STATUS(i), HwEI::rw_sr);
		SET_HOOK(rh, EXI_STATUS(i) + 2, HwEI::rh_sr_low);
	}
	for(int i=0; i<2; i++) {
		if(g::memcardslot_info[i].content != MSC_Empty) {
			exi_devices[i][0] = new EXIDevice_Memcard((i) == 0 ? eir_mcs1 : eir_mcs2,
				MSC2MCT(g::memcardslot_info[i].content), g::memcardslot_info[i].filename,
				g::memcardslot_info[i].caching);
		}
	}
	SET_HOOK_LOG(g::exi_log, rw, 0x6830);
	SET_HOOK_LOG(g::exi_log, ww, 0x683C);	//unknown, tetrisAuto
	SET_HOOK_LOG(g::exi_log, rw, 0x6840);	//unknown, cardtool
	SET_HOOK_LOG(g::exi_log, rw, 0x6844);	//unknown, lsdoom
	SET_HOOK_LOG(g::exi_log, rw, 0x6848);	//unknown, tetrisAuto
	SET_HOOK_LOG(g::exi_log, rw, 0x6854);	//unknown, cardtool
	SET_HOOK_LOG(g::exi_log, rw, 0x6858);	//unknown, lsdoom
	for(WORD i=0x6880; i<0x6C00; i+=0x40) { //unknown, cardtool
		SET_HOOK_LOG(g::exi_log, rw, i);
		SET_HOOK_LOG(g::exi_log, rw, i+0x14);
	}

	//PI
	SET_HOOK(ww, 0x3024, HwPI::ww_reset);
	SET_HOOK(rw, 0x3024, HwPI::rw_reset);
	SET_HOOK(rw, 0x3000, HwPI::rw_intsr);
	SET_HOOK(ww, 0x3000, HwPI::ww_intsr);
	SET_HOOK(ww, 0x3004, HwPI::ww_intmr);
	SET_HOOK_DIRECT(rw, 0x3004);
	SET_HOOK(ww, 0x3018, HwBase::ww_log);
	SET_HOOK(rw, 0x302C, HwPI::rw_console_type);

	//MI
	/*SET_HOOK(wh, 0x4010, HwBase::wh_log);
	SET_HOOK(wh, 0x401C, HwBase::wh_log);
	SET_HOOK(wh, 0x4020, HwBase::wh_log);
	SET_HOOK(wh, 0x4028, HwBase::wh_log);	//unknown, lsdoom*/
	for(WORD i=0x4000; i<0x4100; i+=2) {
		SET_HOOK(wh, i, HwBase::wh_log);
		SET_HOOK(rh, i, HwBase::rh_log);
	}
}

struct TOP_ITEM {
	const char *type;
	WORD offset;
	DWORD value;
};
void do_at(TOP_ITEM *top, DWORD *narr, const char *str, WORD nae, int i);

Hardware::~Hardware() {
	for(int i=0; i<3; i++) {
		for(int j=0; j<3; j++) {
			SAFE_DELETE(exi_devices[i][j]);
		}
	}

	unloadgcm();

	delete gp;
	delete dsp_core;

	SAFE_RELEASE(ai.pDsb8);
	SAFE_RELEASE(ai.lpDirectSound);

#ifndef PUBLIC_RELEASE
	if(ai.dma_start_ndeltas != 0) {
		DUMPINT(ai.dma_start_ndeltas);
		__int64 mean = ai.dma_start_delta_sum / ai.dma_start_ndeltas;
		DEGUB("Mean: %i ms (%I64i ticks)\n", RTC_2_MS(mean), mean);
		DEGUB("Sum: %i ms (%I64i ticks)\n", RTC_2_MS(ai.dma_start_delta_sum),
			ai.dma_start_delta_sum.QuadPart);
	}
	DUMPINT(vi.nSleepyFrames);
	DUMPINT(vi.nSleeps);
	if(vi.nSleeps != 0) {
		DUMPINT(vi.maxSleepsPerFrame);
		DUMPINT(vi.minSleepsPerFrame);
		DEGUB("vi mediumSleepsPerFrame: %g\n", double(vi.nSleeps) / vi.nSleepyFrames);
	}
	{
		ostringstream str;
		getMAHR(str);
		DEGUB("%s", str.str().c_str());
	}
#endif
}

HRESULT Hardware::gp_clear() {
	return gp->clear();
}
HRESULT Hardware::gp_endScene() {
	return gp->endScene();
}

void Hardware::setQuitSignal() {
	if(!g::dsp_hle)
		dsp_core->pause();
}

void Hardware::getMAHR(ostream& str) {
#define TOP_NUMBER 10
#define READ_ACCESS_TYPES(macro) macro(rb) macro(rh) macro(rw) macro(rd)
#define WRITE_ACCESS_TYPES(macro) macro(wb) macro(wh) macro(ww) macro(wd)
#define ALL_ACCESS_TYPES(macro) READ_ACCESS_TYPES(macro) WRITE_ACCESS_TYPES(macro)
#define DEFINE_AT_STRING(at) const char *str_##at = #at;
	ALL_ACCESS_TYPES(DEFINE_AT_STRING);
	str << TOP_NUMBER << " most accessed hardware registers:\n";
	TOP_ITEM top[TOP_NUMBER];
	ZERO_ARRAY(top);
	for(int i=0; i<TOP_NUMBER; i++) {
#define DO_AT(at, nae) do_at(top, num.at, str_##at, nae, i);
#define DO_RAT(at) DO_AT(at, RHOOK_NUM_ARRAY_ENTRIES)
#define DO_WAT(at) DO_AT(at, WHOOK_NUM_ARRAY_ENTRIES)
		READ_ACCESS_TYPES(DO_RAT);
		WRITE_ACCESS_TYPES(DO_WAT);
		str << top[i].type << " " << HEX0x0(4, top[i].offset) << ": " << top[i].value <<
			" " << (top[i].type[0] == 'r' ? "read" : "write") << s(top[i].value) << "\n";
	}
}

//nae = Number of Array Entries
void do_at(TOP_ITEM *top, DWORD *narr, const char *str, WORD nae, int i) {
	for(WORD j=0; j<nae; j++) {
		if(narr[j] >= top[i].value) {
			bool entry_already_in_top = false;
			for(int k=0; k<i; k++) {
				if(strcmp(top[k].type, str) == 0 && top[k].offset == j) {
					entry_already_in_top = true;
					break;
				}
			}
			if(!entry_already_in_top) {
				top[i].type = str;
				top[i].offset = j;
				top[i].value = narr[j];
			}
		}
	}
}

//this is executed on every pause. Make sure all periodic tasks are re-entrant.
bool Hardware::killPeriodicTasks(std::string* errString) {
	for(int i=0; i<3; i++) {
		for(int j=0; j<3; j++) {
			EXIDevice* dev = exi_devices[i][j];
			if(dev)
				TGLE(dev->pause());
		}
	}

	if(ai.audio_playing) {
		HR2GLE(UE_GENERIC_ERROR, ai.pDsb8->Stop());
	}

	if(!errString) {
		TGLE(gp->endThread());
	}
	VGPDEGUB("Waiting for GP in killPeriodicTasks()\n");
	DWORD res;
	TGLE(GetExitCodeThread(gp->getThread(), &res));
	VGPDEGUB("GP thread exit code: %i\n", res);
	switch(res) {
	case STILL_ACTIVE:
		while(!fifo.gp.idle) Sleep(0);
	case GPTR_SUCCESS:
		if(errString)
			errString->clear();
		break;
	case GPTR_ERROR:
		if(errString)
			*errString = gp->getErrorString();
		break;
	default:
		DEGUB("Unknown GPTR: %i\n", res);
		throw generic_fatal_exception("Internal error: Unknown GPTR!");
	}
	return true;
}

bool Hardware::startPeriodicTasks() {
	for(int i=0; i<3; i++) {
		for(int j=0; j<3; j++) {
			EXIDevice* dev = exi_devices[i][j];
			if(dev)
				GLE(dev->resume());
		}
	}
	if(ai.audio_playing) {
		HR(ai.pDsb8->Play(0, 0, DSBPLAY_LOOPING));
	}
	return true;
}

void Hardware::reset_button_pressed() {
	interrupt.raise(INTEX_RSW, "Reset button pressed");
}

bool Hardware::updateMemcardSlot(int slot, const MEMCARDSLOT_INFO &info) {
	MYASSERT(slot == 0 || slot == 1);
	EXIDevice_Memcard *MC = (EXIDevice_Memcard*)exi_devices[slot][0];

	if(MC) {
		if(MC->getType() != MSC2MCT(info.content) || MC->getFilename() != info.filename) {
			MC->interrupt.raiseEXT("Memcard removed");
			SAFE_DELETE(exi_devices[slot][0]);
		} else {
			TGLE(MC->setCaching(info.caching));
		}
	} else if(info.content != MSC_Empty) {
		exi_devices[slot][0] = MC = new EXIDevice_Memcard(slot == 0 ? eir_mcs1 : eir_mcs2,
			MSC2MCT(info.content), info.filename, info.caching);
		MC->interrupt.raiseEXI("Memcard inserted");
	}
	return true;
}

void Hardware::setCT(ULI *uli) {
	if(g::timing_mode == g::TM_REALTIME) {
		GET_RTC(uli);
	} else {
		MYASSERT(g::timing_mode == g::TM_EXACT_FAST || g::timing_mode == g::TM_EXACT_REAL);
		*uli = m_cc.cycles;
	}
}

void HwVI::VISleep(Hardware *h) {
	static ULI then=0;
	ULI now;
	//ULI &_now = now;
	DWORD ms;
	int nSleepsThisFrame = 0;

	GET_RTC(&now);
	ms = RTC_2_MS(then - now);
	/*DUMPQWORD(now);
	DUMPQWORD(then);
	DUMPINT(DWORD(then - now));
	DUMPINT(ms);*/
	while(ms + 2 < h->getVDelay()) { //see notes.txt for the +2 thing
		GET_RTC(&now);
		ms = RTC_2_MS(then - now);
#define WCSLEEP_INTERVAL 1
		if(g::verbose) {
			Timing timing("Sleep");
			Sleep(WCSLEEP_INTERVAL);
		} else {
			Sleep(WCSLEEP_INTERVAL);
		}
		nSleepsThisFrame++;
	}
	h->vi.nSleepyFrames++;
	h->vi.nSleeps += nSleepsThisFrame;
	if(nSleepsThisFrame > h->vi.maxSleepsPerFrame)
		h->vi.maxSleepsPerFrame = nSleepsThisFrame;
	if(nSleepsThisFrame < h->vi.minSleepsPerFrame)
		h->vi.minSleepsPerFrame = nSleepsThisFrame;
	if(then == 0) {
		then = now;
		//DUMPQWORD(then);
	} else {
		then += g::rtcf / (h->vi.mode == 1 ? 50 : 60);
		//DUMPQWORD(then);
		while(then > now && then - now < g::rtcf) {
			GET_RTC(&now);
			//DUMPQWORD(now);
		}
	}
	//DEGUB("%i sleeps this frame\n", nSleepsThisFrame);
}

void Hardware::postVsync(VSyncType type) {
	//Do the sleepy thing (frame limiting)
	if(g::timing_mode != g::TM_REALTIME && g::frame_limit) {
		if(type == VST_EFB_COPY) {
			//VDEGUB("Adding Sleep event\n");
			interrupt.add_event_delay(HwVI::VISleep, 0, "VISleep");
			//DEGUB("Suspending...\n");
			//g_capp.SuspendEmuThread();
			//DEGUB("Suspended\n");
			//HwVI::VISleep(this);
			//DEGUB("Resuming...\n");
			//g_capp.ResumeEmuThread();
			//DEGUB("Resumed\n");
		} else {
			//VDEGUB("Calling Sleep\n");
			HwVI::VISleep(this);
		}
	}

	//Then post the message
	GLE(PostMessage(hWnd, LM_VSYNC, type, 0));
	vi.xfb_changed_since_last_vsync = false;
}

void Hardware::generic_interrupt_callback(WORD ei) {
	SETHWFLAGS(0x3000, ei);
}

//WORD g_prev = 1;

template<WORD number> void HwVI::visimple_event(Hardware *h) {
	CSSHandler csshandler(css_interrupts);
	DWORD di = h->hrw(VI_DI(number));
	//the interrupt may have been disabled since the event was added
	if(getflag(di, VI_DIENB) && !getflag(di, VI_DIINT)) {
		if(VISI_LOG) {
			DEGUB("Raising VISI VII %i @ %I64u\n", number, h->m_cc.cycles);
		}
		/*if(g_prev == number) {
		//DUMPINT(g_prev);
		//DUMPINT(number);
		VDEGUB("Internal warning: VISI order discrepancy! (%i)\n", number);
		//return;
		throw hardware_fatal_exception("Internal error: VISI order discrepancy!");
		}
		if(number > 1)
		throw hardware_fatal_exception("Unemulated VISI!");*/
		//g_prev = number;
		h->m_visis++;
		raiseVI(h, di, number);
	}
	h->vi.visi_done--;
	if(VISI_LOG) {
		DUMPINT(h->vi.visi_done);
	}
}

void HwVI::visi_timeout_event(Hardware *h) {
	if(VISI_LOG) {
		DEGUB("Timeout VISI @ cycle %I64u\n", h->m_cc.cycles);
	}
	h->vi_simple_interrupt();
}

void Hardware::vi_simple_interrupt() {
	CSSHandler csshandler(css_interrupts);
	//DEGUB("Got CSS\n");
	MYASSERT(g::timing_mode == g::TM_EXACT_FAST);
	if(VISI_LOG) {
		DEGUB("vi_simple_interrupt called @ cycle %I64u\n", m_cc.cycles);
	}
	//if(vi_enabled)
	if(hrh(0x2002) & 0x0001) {
		//VDEGUB("VISI test: 0x%016I64x(%I64u) >= 0x%016I64x(%I64u)\n", m_cc.cycles, m_cc.cycles,
		//m_visi_limit, m_visi_limit);
		interrupt.remove_events(
			event_test<(event_callback)(hw_callback)HwVI::visi_timeout_event >());
		if(VISI_LOG) {
			DUMPINT(vi.visi_done);
		}
		if(m_cc.cycles >= m_visi_limit && vi.visi_done == 0) {
			int ints=0;

#define DO_VISI(number) { DWORD di = hrw(VI_DI(number));\
	if(getflag(di, VI_DIENB) && !getflag(di, VI_DIINT)) { ints++;\
	interrupt.add_event_delay(HwVI::visimple_event<number>, (number+1)*32*K,\
	"VISI "#number); } }

			VIDEO_INTERRUPTS(DO_VISI);
			vi.visi_done += ints;

			m_visi_limit = m_cc.cycles + 128*K;
			if(VISI_LOG) {
				VDEGUB("%i VI event(s) generated\n", ints);
			}
			//if(ints != 2)
			//throw hardware_fatal_exception("Unemulated VISI number!");
			interrupt.add_event(HwVI::visi_timeout_event, m_cc.cycles + VISI_TIMEOUT,
				"VISI Timeout");
		} else {
			interrupt.add_event(HwVI::visi_timeout_event, m_cc.cycles + 128*K,
				"VISI Timeout");
		}
	}

}

void HwVI::raiseVI(Hardware *h, DWORD di, WORD number) {
	MYASSERT(number < 4);
	h->hww(VI_DI(number), di | VI_DIINT);
	h->interrupt.raise(INTEX_VI, STRING_PLUS_DIGIT("VI ", number));

	//hacking in the serial poll interrupt here...
	bool bRaise = false;
	DWORD sicr = h->hrw(SI_CR);
	DWORD sisr = h->hrw(SI_SR);
	DWORD new_sr = sisr;
	for(int i=0; i<4; i++) {
		if(g::pad_on[i] && (h->hrw(SI_POLL) & SIPOLL_EN(i))) {
			new_sr |= SISR_RDST(i);
			bRaise = true;
		}
	}
	if(bRaise) {
		h->hww(SI_SR, new_sr);
		if(!((sisr & SI_ALLCHANS(SISR_RDST)))) {
			//hww(0x6434, sicr | SICR_RDSTINT); //not here. rw_cr takes care of it.
			if(sicr & SICR_RDSTMSK) {
				if(g::log_interrupts || g::si_log) {
					DEGUB("SI interrupt generated\n");
				}
				h->interrupt.raise(INTEX_SI, "SI");
			}
		}
	}
}

void Hardware::do_interrupts() {
	CSSHandler csshandler(css_interrupts);
	//ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	if(pending_interrupts.size() != 0 && (m.getmsr() & MSR_EE) && hrw(0x3000) == 0) {
		hww(0x3000, pending_interrupts.front().ei);
		gekko_interrupt.raiseAsync(INTERRUPT_EXTERNAL, pending_interrupts.front().callback,
			pending_interrupts.front().desc);
		pending_interrupts.pop_front();
	}
}

void GenericHWInterruptRaiser::raise(WORD ei, const std::string &desc) {
	CSSHandler csshandler(css_interrupts);
	if(g::log_interrupts) {
		if(h->pending_interrupts.size() > 0) {
			DEGUB("%i pending, ", h->pending_interrupts.size());
		}
		DEGUB("HWInterrupt 0x%02X raised\n", ei);
	}
	EXTERNAL_INTERRUPT e = { ei, NULL, desc };
	h->pending_interrupts.push_back(e);
}
void GenericHWInterruptRaiser::raiseAsync(hw_callback callback, const std::string &desc) {
	CSSHandler csshandler(css_interrupts);
	if(g::log_interrupts) {
		if(h->pending_interrupts.size() > 0) {
			DEGUB("%i pending, ", h->pending_interrupts.size());
		}
		DEGUB("HWInterrupt Async raised\n");
	}
	EXTERNAL_INTERRUPT e = { 0, callback, desc };
	h->pending_interrupts.push_back(e);
}
void GenericHWInterruptRaiser::add_event(hw_callback function, ULI time,
																				 const std::string &desc)
{
	TIMED_EVENT event;
	event.arg = h;
	event.function = (event_callback)function;
	event.time = time;
	event.desc = desc;
	h->gekko_interrupt.add_event(event);
}
void GenericHWInterruptRaiser::add_event_delay(hw_callback function, DWORD delay_tb,
																							 const std::string &desc)
{
	ULI time;
	if(g::timing_mode == g::TM_REALTIME) {
		GET_RTC(&time);
		time += TB_2_RTC(delay_tb);
	} else if(g::timing_mode == g::TM_EXACT_FAST) {
		time = h->m_cc.cycles + delay_tb;
	} else if(g::timing_mode == g::TM_EXACT_REAL) {
		time = h->m_cc.cycles + TB_2_CYCLES(delay_tb);
	}
	add_event(function, time, desc);
}

void Hardware::dumpmem() const {
	dumpToFile(hwrm, HARDWARE_MEMSIZE, "hwmem.bin");
}

void Hardware::getTimings(std::vector<TIMING> &timings) {
	Array<ULI> temp_hwt(m_hwt.size(), m_hwt.p());
	temp_hwt[HWT_FIFO] -= temp_hwt[HWT_GX];
	for(size_t i=HWT_HW+1; i<temp_hwt.size(); i++) {
		temp_hwt[HWT_HW] -= temp_hwt[i];
	}
#define PUSH_HWT(id) timings.push_back(TIMING(#id, temp_hwt[HWT_##id]));
	HWTIMINGS(PUSH_HWT);
}

void Hardware::getVerboseText(ostream& str) {
	str << (m_visis - m_old_visis) << " VISI VIIs last second\n" << m_visis <<
		" VISI VIIs total\n";
	m_old_visis = m_visis;
	getMAHR(str);
	//gp->getVerboseText(buffer);
}

BYTE Hardware::rb(WORD offset) {
	ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	MYASSERT(offset < RHOOK_NUM_ARRAY_ENTRIES);
	num.rb[offset]++;
	return arr_rb[offset](this, offset);
}
BYTE HwBase::unhandled_rb(Hardware *h, WORD offset) {
	DEGUB("Unhandled hardware rb: 0x%04X, 0x%02X\n", offset, h->hrb(offset));
	//if(g::bouehr)
	throw bouehr_exception("Unhandled hardware read");
	//return h->hrb(offset);
}

WORD Hardware::rh(WORD offset) {
	ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	MYASSERT(offset < RHOOK_NUM_ARRAY_ENTRIES);
	num.rh[offset]++;
	return arr_rh[offset](this, offset);
}
WORD HwBase::unhandled_rh(Hardware *h, WORD offset) {
	DEGUB("Unhandled hardware rh: 0x%04X, 0x%04X\n", offset, h->hrh(offset));
	//if(g::bouehr)
	throw bouehr_exception("Unhandled hardware read");
	//return h->hrh(offset);
}

DWORD Hardware::rw(WORD offset) {
	ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	MYASSERT(offset < RHOOK_NUM_ARRAY_ENTRIES);
	num.rw[offset]++;
	return arr_rw[offset](this, offset);
}
DWORD HwBase::unhandled_rw(Hardware *h, WORD offset) {
	DEGUB("Unhandled hardware rw: 0x%04X, 0x%08X\n", offset, h->hrw(offset));
	//if(g::bouehr)
	throw bouehr_exception("Unhandled hardware read");
	//return h->hrw(offset);
}

QWORD Hardware::rd(WORD offset) {
	ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	MYASSERT(offset < RHOOK_NUM_ARRAY_ENTRIES);
	num.rd[offset]++;
	return arr_rd[offset](this, offset);
}
QWORD HwBase::unhandled_rd(Hardware *h, WORD offset) {
	DEGUB("Unhandled hardware rd: 0x%04X, 0x%016I64X\n", offset, h->hrd(offset));
	//if(g::bouehr)
	throw bouehr_exception("Unhandled hardware read");
	//return h->hrd(offset);
}

void Hardware::wb(WORD offset, BYTE data) {
	ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	MYASSERT(offset < WHOOK_NUM_ARRAY_ENTRIES);
	num.wb[offset]++;
	arr_wb[offset](this, offset, data);
}
void HwBase::unhandled_wb(Hardware *, WORD offset, BYTE data) {
	DEGUB("Unhandled hardware wb: 0x%04X, 0x%02X\n", offset, data);
	//if(g::bouehr)
	throw bouehr_exception("Unhandled hardware write");
	//h->hwb(offset, data);
}

void Hardware::wh(WORD offset, WORD data) {
	ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	MYASSERT(offset < WHOOK_NUM_ARRAY_ENTRIES);
	num.wh[offset]++;
	arr_wh[offset](this, offset, data);
}
void HwBase::unhandled_wh(Hardware *, WORD offset, WORD data) {
	DEGUB("Unhandled hardware wh: 0x%04X, 0x%04X\n", offset, data);
	//if(g::bouehr)
	throw bouehr_exception("Unhandled hardware write");
	//h->hwh(offset, data);
}

void Hardware::ww(WORD offset, DWORD data) {
	ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	MYASSERT(offset < WHOOK_NUM_ARRAY_ENTRIES);
	num.ww[offset]++;
	arr_ww[offset](this, offset, data);
}
void HwBase::unhandled_ww(Hardware *, WORD offset, DWORD data) {
	DEGUB("Unhandled hardware ww: 0x%04X, 0x%08X\n", offset, data);
	//if(g::bouehr)
	throw bouehr_exception("Unhandled hardware write");
	//h->hww(offset, data);
}

void Hardware::wd(WORD offset, QWORD data) {
	ADDITIVE_TIMING(&m_hwt[HWT_HW]);
	MYASSERT(offset < WHOOK_NUM_ARRAY_ENTRIES);
	num.wd[offset]++;
	arr_wd[offset](this, offset, data);
}
void HwBase::unhandled_wd(Hardware *, WORD offset, QWORD data) {
	DEGUB("Unhandled hardware wd: 0x%04X, 0x%016I64X\n", offset, data);
	//if(g::bouehr)
	throw bouehr_exception("Unhandled hardware write");
	//h->hwd(offset, data);
}

WORD HwBase::rb_log(Hardware *h, WORD offset) {
	DEGUB("Hardware rh: 0x%04X, 0x%02X\n", offset, h->hrb(offset));
	return h->hrb(offset);
}
WORD HwBase::rh_log(Hardware *h, WORD offset) {
	DEGUB("Hardware rh: 0x%04X, 0x%04X\n", offset, h->hrh(offset));
	return h->hrh(offset);
}
DWORD HwBase::rw_log(Hardware *h, WORD offset) {
	DEGUB("Hardware rw: 0x%04X, 0x%08X\n", offset, h->hrw(offset));
	return h->hrw(offset);
}
void HwBase::wb_log(Hardware *h, WORD offset, BYTE data) {
	DEGUB("Hardware wb: 0x%04X, 0x%02X\n", offset, data);
	h->hwb(offset, data);
}
void HwBase::wh_log(Hardware *h, WORD offset, WORD data) {
	DEGUB("Hardware wh: 0x%04X, 0x%04X\n", offset, data);
	h->hwh(offset, data);
}
void HwBase::ww_log(Hardware *h, WORD offset, DWORD data) {
	DEGUB("Hardware ww: 0x%04X, 0x%08X\n", offset, data);
	h->hww(offset, data);
}
