// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef HARDWARE_H
#define HARDWARE_H

#include "config.h"

#include "container.h"
#include "common.h"
#include "meminterface.h"
#include "interrupt.h"
#include "array.h"
#include "emutime.h"
#include <d3d9.h>
#include <Dsound.h>
#define DIRECTINPUT_VERSION  0x0800
#include <Dinput.h>
#include <queue>

struct EXTERNAL_INTERRUPT {
	WORD ei;
	hw_callback callback;
	std::string desc;
};

class EXIDevice;
class MemcardSlot;
class DSPInterpreter;
class MemInterface;

void convertYUYV2RGB(BYTE *screen_rgb, const BYTE *screen_yuyv);  //slow
void convertYUYV2XRGB(BYTE *screen_xrgb, const BYTE *screen_yuyv);  //slow

enum VSyncType { VST_VCT_READ, VST_HCT_READ, VST_FBA_WRITE, VST_INTERRUPT_READ,
VST_EFB_COPY, _VST_MAX };

struct CORE_ARGUMENTS {
	HWND hWnd;
	LPDIRECT3DDEVICE9 pd3dDevice;
	LPDIRECTINPUTDEVICE pdiDevice;
};

class GenericHWInterruptRaiser {
public:
	GenericHWInterruptRaiser(Hardware *hw) : h(hw) {}
	void raise(WORD ei, const std::string &desc);
	void raiseAsync(hw_callback callback, const std::string &desc);
	void add_event(hw_callback function, ULI time, const std::string &desc);
	void add_event_delay(hw_callback function, DWORD delay_tb, const std::string &desc);
	template<class Pr1> void remove_events(Pr1 pred) {
		h->gekko_interrupt.remove_events<Pr1>(pred); }
protected:
	Hardware *h;
};

class EXIInterruptRaiser : protected GenericHWInterruptRaiser {
public:
	EXIInterruptRaiser(Hardware *hw, int channel)
		: GenericHWInterruptRaiser(hw), mChannel(channel) {}
	void raiseEXI(const std::string& desc);
	void raiseEXT(const std::string& desc);
protected:
	const int mChannel;
private:
	EXIInterruptRaiser(const EXIInterruptRaiser&);
	EXIInterruptRaiser& operator=(const EXIInterruptRaiser&);
};


//DSP HLE
class MailQueue {
public:
	MailQueue();
	void push(DWORD data);
	DWORD pop();
	bool empty();
private:
	std::queue<DWORD> mails;
	int pop_ctr;
};

struct TIMING {
	const char *str;
	__int64 t;
	TIMING(const char *_s, __int64 _t) : str(_s), t(_t) {}
};

struct GP_FIFO {
	DWORD base, end, hiwater, lowater, dist, writeptr, readptr, breakpoint;
	bool idle, quit, bp_enabled, read_enabled, in_command, breakpoint_reached;
};

class Hardware {
public:
	Hardware(MemInterface &mem, InterruptRaiser &i, CORE_ARGUMENTS a,
		CycleCounterBase &ccb);
	~Hardware();

	//GFX functions
	bool getScreen(BYTE *screen_yuyv);  //fast
	DWORD getVDelay();
	const char *getGfxModeString();
	bool gp_activated() { return m_gp_activated; }
	HRESULT gp_clear();
	HRESULT gp_endScene();
	bool wgp_empty() { return !(((size_t)fifo.pTempWriteptr) & 31); }

	//GCM functions
	bool loadgcm(const char *gcmname);
	void unloadgcm();
	void setlid(bool open);
	bool getlid() { return m_lidopen; }
	bool do_gcmload_hle();

	//emulation control functions
	void do_interrupts();
	bool killPeriodicTasks(std::string* errString);
	bool startPeriodicTasks();
	void setQuitSignal();

	//miscellaneous
	void dumpmem() const;
	void setWindowActive(bool active) { m_window_active = active; }
	bool updateMemcardSlot(int slot, const MEMCARDSLOT_INFO &info);
	void getTimings(std::vector<TIMING> &timings);
	void doVISI() { MYASSERT(g::timing_mode == g::TM_EXACT_FAST); vi_simple_interrupt(); }
	void reset_button_pressed();
	u64 getDSPCycles();
	DSPInterpreter& getDspCore() { return *dsp_core; }

	//should be called once per second if emu is running && g::verbose
	void getVerboseText(ostream& str);

	//Read
	template<class T> T tr(WORD offset);
#define DECLARE_HW_READ(name, type) type r##name(WORD offset);\
	template<> type tr<type>(WORD offset) { return r##name(offset); }
	MEM_ACCESS_TYPES(DECLARE_HW_READ);
	//Write
	template<class T> void tw(WORD offset, T data);
#define DECLARE_HW_WRITE(name, type) void w##name(WORD offset, type data);\
	template<> void tw<type>(WORD offset, type data) { return w##name(offset, data); }
	MEM_ACCESS_TYPES(DECLARE_HW_WRITE);

#define _HA(type) MYASSERT(offset < 0x8000)
	//Read without hw interference
#define DEFINE_HHW_READ(name, type) type hr##name(WORD offset) const {\
	_HA(type); return tswap(MAKE(type, hwrm[offset])); }
	MEM_ACCESS_TYPES(DEFINE_HHW_READ);
	//Write without hw interference
#define DEFINE_HHW_WRITE(name, type) void hw##name(WORD offset, type data) {\
	_HA(type); MAKE(type, hwrm[offset]) = tswap(data); }
	MEM_ACCESS_TYPES(DEFINE_HHW_WRITE);
#undef _HA
private:
	//General
	MemInterface &m;
	bool m_window_active;
	HWND hWnd;
	void postVsync(VSyncType type);
	CycleCounterBase &m_cc;
	Array<ULI> m_hwt;
	DWORD m_pisr_reads, m_old_pisr_reads, m_visis, m_old_visis;
	void setCT(ULI *uli);
	void getMAHR(ostream& str);
	ULI m_visi_limit;

	//Interrupts
	friend class GenericHWInterruptRaiser;
	friend class EXIInterruptRaiser;
	GenericHWInterruptRaiser interrupt;
	EXIInterruptRaiser eir_mcs1, eir_mcs2, eir_bba;
	void generic_interrupt_callback(WORD ei);
	list<EXTERNAL_INTERRUPT> pending_interrupts;
	InterruptRaiser &gekko_interrupt;

	//Pad/SI
	void update_pads();
	LPDIRECTINPUTDEVICE m_pdiDevice;
	DWORD si_start(DWORD data);

	//VI
	struct VI_ZEROABLE {
		DWORD xfb[2];
		bool xfb_changed_since_last_vsync;
		int mode;
		ULI next_vct_vsync;
		int nSleeps, maxSleepsPerFrame, minSleepsPerFrame, nSleepyFrames;
		int visi_done;
	} vi;
	bool gfx_move_halfline();
	void vi_simple_interrupt();

	//3D GX
	friend class GP;
	GP * const gp;
	bool m_gp_activated;
	struct FIFO_ZEROABLE {
		BYTE *pTempWriteptr, *pBase, *pEnd;
		struct CPU_FIFO {
			DWORD base, end, writeptr;
			bool wrap;
		} cpu;
		struct COPY_FIFO {
			DWORD base, end, writeptr;
			BYTE *pTempWriteptr, *pBase, *pEnd;
		} copy;
		volatile GP_FIFO gp;
		bool linked, overflowed, copy_fifo;
	} fifo;

	//DVD
	ReadOnlyFile gcmfile;
	void do_dvd();
	bool m_lidopen;
	DWORD m_dvdstatus;

	//Audio
	void do_audio_dma();
	bool waitingForAIDMAInterrupt();
	struct AI_ZEROABLE {
		LPDIRECTSOUND8 lpDirectSound;
		LPDIRECTSOUNDBUFFER8 pDsb8;
		WAVEFORMATEX wfx;
		DSBUFFERDESC dsbdesc;
		bool audio_interrupted, audio_playing;
		ULI dma_start_time, dma_stop_time, audio_play_start_time, scnt_start_time;
		WORD audio_length;
		UINT audio_slots, audio_wp, audio_pp, audio_repeats;
		bool ar_dma_high_set, ar_dma_low_set;
		ULI dma_start_delta_sum;
		int dma_start_ndeltas;
	} ai;

	//DSP
	Container<BYTE> m_aram;
	DSPInterpreter * const dsp_core;

	//DSP HLE
	MailQueue mail_from_dsp;
	struct DSP_ZEROABLE {
		bool mail_from_high_read, mail_from_low_read;
		bool mail_to_high_written, mail_to_low_written;
		int state;
		int step;
#define DSP_MAXPARAMS 10
		DWORD parameters[DSP_MAXPARAMS];
		int init_state;
	} dsp;

	//EXI
#define NUM_EXI_CHANNELS 3
#define NUM_EXI_DEVICES_PER_CHANNEL 3
	EXIDevice *exi_devices[NUM_EXI_CHANNELS][NUM_EXI_DEVICES_PER_CHANNEL];
	struct EXI_ZEROABLE {
		bool exi_interrupt_waiting[NUM_EXI_CHANNELS];
		bool ext_interrupt_waiting[NUM_EXI_CHANNELS];
	} exi;
	void do_exi_rw(BYTE channel, DWORD controlreg);
	void do_exi_select(BYTE channel, DWORD statusreg);
	void do_exi_rw_end(BYTE channel, DWORD controlreg, DWORD status, int erv);

	//function containers
	friend class HwBase;
	friend class HwSI;
	friend class HwVI;
	friend class HwGX;
	friend class HwDI;
	friend class HwAI;
	friend class HwEI;
	friend class HwPI;
#define RHOOK_NUM_ARRAY_ENTRIES 0x8000
#define WHOOK_NUM_ARRAY_ENTRIES 0x8001
	//Read Byte/Word/Dword
	BYTE (*arr_rb[RHOOK_NUM_ARRAY_ENTRIES])(Hardware *h, WORD offset);
	WORD (*arr_rh[RHOOK_NUM_ARRAY_ENTRIES])(Hardware *h, WORD offset);
	DWORD (*arr_rw[RHOOK_NUM_ARRAY_ENTRIES])(Hardware *h, WORD offset);
	QWORD (*arr_rd[RHOOK_NUM_ARRAY_ENTRIES])(Hardware *h, WORD offset);
	//Write Byte/Word/Dword
	void (*arr_wb[WHOOK_NUM_ARRAY_ENTRIES])(Hardware *h, WORD offset, BYTE data);
	void (*arr_wh[WHOOK_NUM_ARRAY_ENTRIES])(Hardware *h, WORD offset, WORD data);
	void (*arr_ww[WHOOK_NUM_ARRAY_ENTRIES])(Hardware *h, WORD offset, DWORD data);
	void (*arr_wd[WHOOK_NUM_ARRAY_ENTRIES])(Hardware *h, WORD offset, QWORD data);
	struct NUM {
		DWORD rb[RHOOK_NUM_ARRAY_ENTRIES];
		DWORD rh[RHOOK_NUM_ARRAY_ENTRIES];
		DWORD rw[RHOOK_NUM_ARRAY_ENTRIES];
		DWORD rd[RHOOK_NUM_ARRAY_ENTRIES];
		DWORD wb[WHOOK_NUM_ARRAY_ENTRIES];
		DWORD wh[WHOOK_NUM_ARRAY_ENTRIES];
		DWORD ww[WHOOK_NUM_ARRAY_ENTRIES];
		DWORD wd[WHOOK_NUM_ARRAY_ENTRIES];
	} num;

	BYTE hwrm[HARDWARE_MEMSIZE];
};

#endif	//HARDWARE_H
