// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "hw_ai.h"
#include "hw_emutime.h"
#include "dsp_core.h"
#include <fstream>

u64 Hardware::getDSPCycles() {
	return g::dsp_hle ? 0 : dsp_core->getCycles();
}

QWORD AISCNT_2_ET(DWORD aiscnt) {
	if(g::timing_mode == g::TM_REALTIME) {
		return AISCNT_2_RTC(aiscnt);
	} else if(g::timing_mode == g::TM_EXACT_REAL) {
		return AISCNT_2_CYCLES(aiscnt);
	} else {  //TM_EXACT_FAST
		return aiscnt;
	}
}

#define DSPM_INIT	0xDCD10000
#define DSPM_RESUME	0xDCD10001
#define DSPM_YIELD	0xDCD10002
#define DSPM_DONE	0xDCD10003
#define DSPM_SYNC	0xDCD10004
#define DSPM_UNKNOWN	0xDCD10005

MailQueue::MailQueue() : pop_ctr(0) {}
void MailQueue::push(DWORD data) {
	mails.push(data);
	VAIDEGUB("Pushed mail from DSP: 0x%08X\n", data);
}
DWORD MailQueue::pop() {
	MYASSERT(!empty());
	DWORD data = mails.front();
	mails.pop();
	VAIDEGUB(" mail no.%i from DSP: 0x%08X\n", ++pop_ctr, data);
	return data;
}
bool MailQueue::empty() {
	return mails.empty();
}

void HwAI::wh_dspcr(Hardware *h, WORD offset, WORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x500A);
	WORD original = h->hrh(offset), changed_data = data;
	/*//not documented, from gcube.
	changed_data = CLEARINTFLAGS(original, data,
	(DSPCR_DSPINT | DSPCR_ARINT | DSPCR_AIDINT | DSPCR_DMAINT));
	#define DSPCR_WRITEMASK (DSPCR_DSPINTMSK | DSPCR_ARINTMSK | DSPCR_AIDINTMASK |\
	DSPCR_DSPINT | DSPCR_ARINT | DSPCR_AIDINT | DSPCR_DMAINT)
	changed_data = (changed_data & DSPCR_WRITEMASK) | (original & ~DSPCR_WRITEMASK);*/
	//from dolphin
	//I'm unsure about the interrupts, maybe they should be handled traditionally.
	changed_data = data & ~(DSPCR_RESET | DSPCR_DMASTATE |
		DSPCR_DSPINT | DSPCR_ARINT | DSPCR_AIDINT);
	AIDEGUB("DSPCR wh: 0x%04X + 0x%04X => 0x%04X\n", original, data, changed_data);
	if(data & DSPCR_RESET) {
		AIDEGUB("DSP Reset\n");
		if(g::dsp_hle) {
			if(h->dsp.init_state == DSPIS_1)
				h->mail_from_dsp.push(0x8071FEED);	//from dolphin
		} else {
			h->dsp_core->reset();

			//hack?
#if 0
			changed_data &= ~DSPCR_HALT;
			h->hwh(offset, changed_data);
			h->dsp_core->run();
//#else
			h->hwh(PR_CMBH, 0x8071);
			h->hwh(PR_CMBL, 0xFEED);
#endif
		}
	}
	if((data & DSPCR_ON) && !(original & DSPCR_ON)) {
		AIDEGUB("DSP Turned on\n");
		if(g::dsp_hle) {
			//h->dsp.do_init_hack = true;
			//h->dsp_cpumail.push(0x80544348);  //from dolphin (hack?)
		} else {
			h->dsp_core->turn_on();
		}
	}
	if((data & DSPCR_HALT) != (original & DSPCR_HALT)) {
		AIDEGUB("DSP %salt\n", (data & DSPCR_HALT) ? "H" : "Unh");
		if(g::dsp_hle) {
			if(!(data & DSPCR_HALT) && h->dsp.init_state == DSPIS_2) {
				h->mail_from_dsp.push(0x8071FEED);
				h->dsp.init_state = DSPIS_INIT;
			}
		} else {
			if(!(data & DSPCR_HALT))
				h->dsp_core->run();
		}
	}
	if(data & DSPCR_PIINT) {  //I don't know much about this
		h->interrupt.raise(INTEX_DSP, "DSP PIINT");
	}
	h->hwh(offset, changed_data);
}

WORD HwAI::rh_mail_cpu_high(Hardware *h, WORD offset) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x5004);
	WORD data;
	if(g::dsp_hle) {
		DWORD mail = get_mail(h);
		data = HIWORD(mail);
		h->dsp.mail_from_high_read = true;
	} else {
		data = h->hrh(offset);
	}
	AIDEGUB("Hardware rh: 0x%04X, 0x%04X\n", offset, data);
	if(!g::dsp_hle && !(data & 0x8000))
		h->dsp_core->run();
	return data;
}
WORD HwAI::rh_mail_cpu_low(Hardware *h, WORD offset) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x5006);
	WORD data;
	if(g::dsp_hle) {
		DWORD mail = get_mail(h);
		data = LOWORD(mail);
		h->dsp.mail_from_low_read = true;
	} else {
		data = h->hrh(offset);
		h->hwh(0x5004, h->hrh(0x5004) & ~0x8000);
	}
	AIDEGUB("Hardware rh: 0x%04X, 0x%04X\n", offset, data);
	return data;
}

DWORD HwAI::get_mail(Hardware *h) {
	MYASSERT(g::dsp_hle);
	DWORD mail = h->hrw(0x5004);
	if((h->dsp.mail_from_low_read && h->dsp.mail_from_high_read) || mail == NO_MAIL) {
		h->dsp.mail_from_low_read = h->dsp.mail_from_high_read = false;
		if(!h->mail_from_dsp.empty()) {
			h->hww(0x5004, mail = h->mail_from_dsp.pop());
			//raise_dspint(h);
		} else
			h->hww(0x5004, mail = NO_MAIL);//^= 0x80000000);
	}
	return mail;
}

WORD HwAI::rh_mail_dsp_high(Hardware *h, WORD offset) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x5000);
	WORD data = h->hrh(offset);
	AIDEGUB("Hardware rh: 0x%04X, 0x%04X\n", offset, data);
	if(!g::dsp_hle && (data & 0x8000))
		h->dsp_core->run();
	return data;
}
void HwAI::wh_mail_dsp_high(Hardware *h, WORD offset, WORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == PR_DMBH);
	AIDEGUB("Hardware wh: 0x%04X, 0x%04X\n", offset, data);
	if(g::dsp_hle) {
		if(h->dsp.mail_to_low_written) {
			handle_mail(h, data, h->hrh(PR_DMBL));
			h->dsp.mail_to_low_written = false;
		} else {
			h->hwh(offset, data);
			h->dsp.mail_to_high_written = true;
		}
	} else {
		h->hwh(offset, data & ~0x8000);
	}
}
void HwAI::wh_mail_dsp_low(Hardware *h, WORD offset, WORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == PR_DMBL);
	AIDEGUB("Hardware wh: 0x%04X, 0x%04X\n", offset, data);
	if(g::dsp_hle) {
		if(h->dsp.mail_to_high_written) {
			handle_mail(h, h->hrh(PR_DMBH), data);
			h->dsp.mail_to_high_written = false;
		} else {
			h->hwh(offset, data);
			h->dsp.mail_to_low_written = true;
		}
	} else {
		h->hwh(offset, data);
		h->hwh(PR_DMBH, h->hrh(PR_DMBH) | 0x8000);
		DSPLOG("Mail to DSP detected: 0x%08X\n", h->hrw(PR_DMBH));
		//h->dsp_core->unhalt();
	}
}

//simple stupid crc invented by ector :P
DWORD ector_crc(const BYTE *iram, int length) {
	DWORD crc=0;
	for(int i=0; i<length; i++) {
		crc ^= iram[i];
		//let's rol
		crc=(crc<<3) | (crc>>29);
	}
	return crc;
}
void HwAI::set_state(Hardware *h, DWORD crc) {
	MYASSERT(g::dsp_hle);
	switch(crc) {
	case 0x4e8a8b21:  //SDK\axstreamD.elf
		h->dsp.state = DSPS_AX;
		h->mail_from_dsp.push(DSPM_INIT);
		assert_dsp_interrupt(h);
		break;
	case 0x65d6cc6f:  //SDK\formatD.elf
		h->dsp.state = DSPS_CARD;
		h->mail_from_dsp.push(DSPM_INIT);
		assert_dsp_interrupt(h);
		//m_Mails.push(0x80000000);
		break;
	/*case 0xd73338cf:  //PAL IPL
		h->dsp.state = DSPS_JAC;
		break;*/
	case 0x07f88145:  //BAM3K
	default:
		//throw hardware_fatal_exception("Unemulated DSP crc!");
		h->dsp.state = DSPS_AX;
		h->mail_from_dsp.push(DSPM_INIT);
		//assert_dsp_interrupt(h);
		h->mail_from_dsp.push(0x80000000);
		assert_dsp_interrupt(h);
		break;
	}
}

#define CONCAT_WORDS(a, b) (((a) << 16) | (b))

void HwAI::handle_mail(Hardware *h, WORD high, WORD low) {
#define MAIL CONCAT_WORDS(high, low)
	VAIDEGUB("Handling mail %04X %04X\n", high, low);
	MYASSERT(g::dsp_hle);
	h->hww(0x5000, 0);  //ack

	switch(h->dsp.state) {
	case DSPS_UNINIT:
#define DSPU_NPARAMS 10
		h->dsp.parameters[h->dsp.step] = MAIL;
		h->dsp.step++;
		if(h->dsp.step >= DSPU_NPARAMS) {
			DWORD main_addr=0, iram_addr=0, size_bytes=0, init_vector=0, unknown=0;
			BYTE bitfield=0;
			for(int i=0; i<DSPU_NPARAMS; i+=2) {
				switch(h->dsp.parameters[i]) {
				case 0x80F3A001: main_addr = h->dsp.parameters[i+1]; bitfield |= 1; break;
				case 0x80F3A002: size_bytes = h->dsp.parameters[i+1]; bitfield |= 2; break;
				case 0x80F3B002: iram_addr = h->dsp.parameters[i+1]; bitfield |= 4; break;
				case 0x80F3C002: unknown = h->dsp.parameters[i+1]; bitfield |= 8; break;
				case 0x80F3D001: init_vector = h->dsp.parameters[i+1]; bitfield |= 0x10; break;
				default:
					throw hardware_fatal_exception("DSP Init, Unknown parameter id");
				}
			}
			if(bitfield != 0x1F)
				throw hardware_fatal_exception("DSP Init, Not all parameters set");
			if(size_bytes == 0)
				throw hardware_fatal_exception("DSP Init, Zero size");
			//size_bytes = 1024;  //arbitrary hack from dolphin
			if(unknown != 0)
				throw hardware_fatal_exception("DSP Init, Unknown set");
			DWORD crc = ector_crc(h->m.getp_translated(main_addr, size_bytes), size_bytes);
			if(g::dsp_log) {
				DUMPDWORD(main_addr);
				DUMPDWORD(size_bytes);
				DUMPDWORD(iram_addr);
				DUMPDWORD(init_vector);
				DUMPDWORD(unknown);

				//change this to print words instead of bytes
				//PrintPacket(h->m.getp_translated(main_addr, size_bytes), size_bytes);
				DEGUB("CRC: 0x%08X\n", crc);

				//dsp load/disasm/run
				/*if(iram_addr > 0xFFFF || init_vector > 0xFFFF)
					throw hardware_fatal_exception("DSP BFE");
				h->dsp_core->load(h->m.getp_translated(main_addr, size_bytes), size_bytes,
					(WORD)iram_addr, (WORD)init_vector);*/
				set_state(h, crc);
			}
			h->dsp.step = 0;
			set_state(h, crc);
		}
		break;
	case DSPS_CARD:
#define DSPI_NPARAMS 2
		h->dsp.parameters[h->dsp.step] = MAIL;
		h->dsp.step++;
		if(h->dsp.step >= DSPI_NPARAMS) {
			VAIDEGUB("DSP Unlock sequence complete. Parameters:\n");
			for(int i=0; i<DSPI_NPARAMS; i++) {
				VAIDEGUB("%i: 0x%08X\n", i, h->dsp.parameters[i]);
			}
			h->mail_from_dsp.push(DSPM_DONE);	//0x80000000//DSPM_RESUME//DSPM_SYNC
			assert_dsp_interrupt(h);
			h->dsp.state = DSPS_UNKNOWN;
		}
		break;
	case DSPS_AX:
		if(high == 0xBABE) {
			VAIDEGUB("BABE 0x%04X\n", low);
		} else {
			h->mail_from_dsp.push(DSPM_RESUME);
			assert_dsp_interrupt(h);
		}
		break;
	case DSPS_UNKNOWN:
		VAIDEGUB("DSP State is unknown. Ignoring mail.\n");
		break;
	default:
		MYASSERT(false);
	}
}

void HwAI::assert_dsp_interrupt(Hardware *h) {
	WORD dspcr = h->hrh(0x500A);
	if(getflag(dspcr, DSPCR_DSPINTMSK)) {
		h->interrupt.raiseAsync(dsp_interrupt_callback, "DSP interrupt");
		VAIDEGUB("DSP interrupt asserted\n");
	} else {
		h->hwh(0x500A, dspcr | DSPCR_DSPINT);
		VAIDEGUB("DSP interrupt disabled\n");
	}
}
void HwAI::dsp_interrupt_callback(Hardware *h) {
	VAIDEGUB("dsp_interrupt_callback\n");
	h->generic_interrupt_callback(INTEX_DSP);
	h->hwh(0x500A, h->hrh(0x500A) | DSPCR_DSPINT);
}

WORD HwAI::rh_ar_mode(Hardware *h, WORD offset) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x5016);
	//return 0xFFFF;  //hack
	return h->hrh(offset) | 1;
}

WORD HwAI::rh_dma_bl(Hardware *h, WORD offset) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x503A);
	WORD data = 0;
	if(h->ai.pDsb8 != NULL) {
		if(h->ai.audio_length != 0 ) {
			DWORD aicr = h->hrw(PR_AICR);
			if(g::timing_mode == g::TM_REALTIME) {
				ULI time_since_start;
				GET_RTC(&time_since_start);
				time_since_start -= h->ai.dma_start_time;
				if(FREQED(RTC_2_LENGTH)(time_since_start.QuadPart) < h->ai.audio_length)
					data = h->ai.audio_length - WORD(FREQED(RTC_2_LENGTH)(time_since_start));
				//else data = 0;  //data is already 0
			} else if(g::timing_mode == g::TM_EXACT_REAL) {
				ULI cycles_since_start = h->m_cc.cycles - h->ai.dma_start_time;
				if(FREQED(CYCLES_2_LENGTH)(cycles_since_start) < h->ai.audio_length)
					data = h->ai.audio_length - WORD(FREQED(CYCLES_2_LENGTH)(cycles_since_start));
			} else if(g::timing_mode == g::TM_EXACT_FAST) {
				ULI cycles_since_start = h->m_cc.cycles - h->ai.dma_start_time;
				if(cycles_since_start < h->ai.audio_length)
					data = h->ai.audio_length - WORD(cycles_since_start);
			}
		}
	}
	AIDEGUB("Hardware rh: 0x%04X, 0x%04X\n", offset, data);
	return data;
}

void HwAI::ww_aicr(Hardware *h, WORD offset, DWORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x6C00);
	DWORD prev = h->hrw(offset);
	if(g::log_audio) {
		DEGUB("AICR write: 0x%08X\n", data);
		DEGUB("DSP Sample Rate: %ikHz", (data & AICR_DSR) ? 32 : 48);
		if(data & AICR_SCRESET)
			DEGUB(", Sample Counter Reset");
		DEGUB(", AIINTVLD: %s", getflag(data, AICR_AIINTVLD) ? "hold" : "valid");
		if(data & AICR_AIINT)
			DEGUB(", AIINT cleared");
		DEGUB(", AIINT %s", abled(getflag(data, AICR_AIINTMSK)));
		DEGUB(", AFR: %i kHz", (data & AICR_AFR) ? 32 : 48);
		DEGUB(", PSTAT: %i", data & AICR_PSTAT);
		DEGUB("\n");
	}
	if((data & AICR_DSR) != (prev & AICR_DSR)) {
		h->ai.audio_repeats = 0;
		if(h->ai.pDsb8 != NULL) {
			AIDEGUB("Mid-play frequency change!\n");
			HWHR(h->ai.pDsb8->SetFrequency((data & AICR_DSR) ? 32000 : 48000));
		}
	}
	if(data & AICR_SCRESET) {
		h->hww(PR_AISCNT, 0);
		//h->ai.audio_interrupted = false;
		data &= ~AICR_SCRESET;
	}
	if((data & AICR_PSTAT) && ((data & AICR_SCRESET) || !(prev & AICR_PSTAT))) {
		h->setCT(&h->ai.scnt_start_time);
	}
	if(!(data & AICR_PSTAT) && (prev & AICR_PSTAT)) {
		h->hww(PR_AISCNT, calculateAISCNT(h));
	}

	data = CLEARINTFLAGS(prev, data, AICR_AIINT) & ~AICR_SCRESET;

#define SCI (AICR_PSTAT | AICR_AIINTVLD)  //Sample Counter Interrupt
	if(MASKED_NEQUAL(data, prev, SCI | AICR_AIINT)) {
		if(ALL_FLAGS(prev, SCI) && !(prev & AICR_AIINT)) {
			//event was on, now we turn it off
			h->interrupt.remove_events(event_test<(event_callback)sci_event>());
			AIDEGUB("SCI off\n");
		} else if(ALL_FLAGS(data, SCI) && !(data & AICR_AIINT)) {
			//event was off, now we turn it on
			h->interrupt.add_event(sci_event,
				h->ai.scnt_start_time + AISCNT_2_ET(h->hrw(PR_AIIT) - h->hrw(PR_AISCNT)),
				"AI SCI");
			AIDEGUB("SCI on\n");
		}
	}
	if(g::log_audio) {
		DEGUB("Result: 0x%08X\n", data);
	}
	h->hww(offset, data);
}

void HwAI::sci_event(Hardware *h) {
	AIDEGUB("AIINT raised\n");
	DWORD aicr = h->hrw(PR_AICR);
	if(aicr & AICR_AIINTMSK)
		h->interrupt.raise(INTEX_AI, "AI SCI");
	h->hww(PR_AICR, aicr | AICR_AIINT);
}

void HwAI::ww_aivr(Hardware *, WORD offset, DWORD /*data*/) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == PR_AIVR);
}

DWORD HwAI::rw_aiscnt(Hardware *h, WORD offset) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == PR_AISCNT);
	DWORD aiscnt;
	DWORD aicr = h->hrw(PR_AICR);
	if(aicr & AICR_PSTAT)
		aiscnt = calculateAISCNT(h);
	else
		aiscnt = h->hrw(offset);
	AIDEGUB("AISCNT read: 0x%08X\n", aiscnt);
	return aiscnt;
}

DWORD HwAI::rw_aicr(Hardware *h, WORD offset) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x6C00);
	DWORD aicr = h->hrw(PR_AICR);
	/*if(ALL_FLAGS(aicr, AICR_PSTAT | AICR_AIINTVLD)) {
	DWORD aiscnt = calculateAISCNT(h);
	if(aiscnt >= h->hrw(PR_AIIT)) {
	if(!(aicr & AICR_AIINT)) {
	AIDEGUB("AIINT match!\n");
	}
	h->hww(PR_AICR, aicr |= AICR_AIINT);
	}
	}*/
	AIDEGUB("AICR read: 0x%08X\n", aicr);
	return aicr;
}

void HwAI::ww_ar_dma(Hardware *h, WORD offset, DWORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x5028);
	do_aram_dma_transfer(h, data);
	h->hww(offset, data);
}

void HwAI::assert_aram_interrupt(Hardware *h) {
	WORD dspcr = h->hrh(0x500A);
	if(getflag(dspcr, DSPCR_ARINTMSK)) {
		h->interrupt.raiseAsync(aram_interrupt_callback, "ARAM");
		VAIDEGUB("ARAM interrupt asserted\n");
	} else {
		h->hwh(0x500A, dspcr | DSPCR_ARINT);
		VAIDEGUB("ARAM interrupt disabled\n");
	}
}
void HwAI::aram_interrupt_callback(Hardware *h) {
	VAIDEGUB("aram_interrupt_callback\n");
	h->generic_interrupt_callback(INTEX_DSP);
	h->hwh(0x500A, h->hrh(0x500A) | DSPCR_ARINT);
}

void HwAI::do_aram_dma_transfer(Hardware *h, DWORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	//memory addresses are unclear
	DWORD mma = PHYSICALIZE(h->hrw(0x5020)), ara = h->hrw(0x5024), count = data & 0x7FFFFFFF;
	ara &= (ARAM_SIZE - 1);
	bool read = getbit(data, 0);
	DSPDEGUB("%s ARAM. MMADDR: %08X, ARADDR: %08X, count: 0x%X\n",
		read ? "Read from" : "Write to", mma, ara, count);
	if(mma >= MAIN_MEMORY_SIZE || mma + count > MAIN_MEMORY_SIZE ||
		ara >= ARAM_SIZE || ara + count > ARAM_SIZE)	//we only accept physical addresses
		throw hardware_fatal_exception("Bad bad ARAM transfer! ;)");
	if(g::dsp_hle) {
		if((h->dsp.init_state == DSPIS_1) &&
			!read && mma == 0x01000000 && ara == 0 && count == 0x20)
		{
			DSPDEGUB("DSP_Fake_OSInitAudioSystem\n");
			h->mail_from_dsp.push(0x80544348);  //from dolphin (hack.)
			h->dsp.init_state = DSPIS_2;
		}
	}

	if(read)
		h->m.write_physical(mma, count, h->m_aram + ara);
	else
		h->m.read_physical(mma, count, h->m_aram + ara);
	assert_aram_interrupt(h);
	h->m_visi_limit = MAX(h->m_visi_limit, h->m_cc.cycles + 128*K); //arbitrary

	/*if(g::log_audio && g::verbose)
	PrintPacket(h->m_aram + ara, count);*/
}

void HwAI::wh_ar_dma_high(Hardware *h, WORD offset, WORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x5028);
	if(h->ai.ar_dma_low_set) {
		h->ai.ar_dma_low_set = false;
		do_aram_dma_transfer(h, h->hrh(0x502A) | (DWORD(data) << 16));
	} else {
		h->ai.ar_dma_high_set = true;
	}
}
void HwAI::wh_ar_dma_low(Hardware *h, WORD offset, WORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x502A);
	if(h->ai.ar_dma_high_set) {
		h->ai.ar_dma_high_set = false;
		do_aram_dma_transfer(h, data | (DWORD(h->hrh(0x5028)) << 16));
	} else {
		h->ai.ar_dma_low_set = true;
	}
}

void HwAI::wh_dma_cr(Hardware *h, WORD offset, WORD data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_DSP]);
	MYASSERT(offset == 0x5036);
	WORD old_data = h->hrh(offset);
	h->hwh(offset, data);

#define AUDIO_WP_SLOT (h->ai.audio_wp % h->ai.audio_slots)
#define length data
	DWORD aicr = h->hrw(0x6C00);

	if(RISE_FLAG(data, old_data, 0x8000)) {  //Start
		WORD new_length = length & 0x7FFF;
		DWORD address = h->hrw(0x5030);
		DWORD len_bytes = new_length * 32;
		if(g::dump_audio) { //dump to file
			char filename[MAX_PATH] = "audio.raw";
			if(g::log_audio) {
				DEGUB("Dumping from 0x%08X to %s\n", address, filename);
				DELTA_TIMING;
			}

			Container<char> locked(len_bytes);
			ofstream file(filename, ios_base::binary);
			memcpy_swaph(locked, h->m.getp_translated(address, len_bytes), len_bytes);
			file.write(locked, len_bytes);
		}
		if(g::log_audio) {
			DEGUB("Audio DMA sample starting. Address: 0x%08X  Size: 0x%X  Frequency: %s\n",
				address, (length & 0x7FFF) * 32,
				(aicr & AICR_DSR) ? "32KHz" : "48KHz");
			//16 bits, 2 channels => 4 bytes per sample
			DEGUB("Estimated runtime: %i ms\n",
				(new_length * (32/4)) / ((aicr & AICR_DSR) ? 32 : 48));
			//DUMPDWORD(pDsb8);
			//DUMPINT(audio_slots);
			//DUMPINT(new_length);
			//DUMPINT(audio_length);
		}

		//Reset
		UINT slots = 1 + (g::audio_buffer_min_size_ms-1) /
			MAX(((new_length * (32/4)) / ((aicr & AICR_DSR) ? 32 : 48)), 1);
		if((h->ai.pDsb8 != NULL)) {
			DWORD play, write;
			HWHR(h->ai.pDsb8->GetCurrentPosition(&play, &write));
			if((new_length != h->ai.audio_length) ||
				(AUDIO_WP_SLOT >= play / len_bytes && h->ai.audio_playing &&
				AUDIO_WP_SLOT <= (write + len_bytes - 1) / len_bytes))
			{
				if(g::log_audio && g::verbose && new_length == h->ai.audio_length) {
					DEGUB("In slot %i (bytes %i to %i). Play-Write: %i-%i (slots %i-%i, inclusive)\n",
						AUDIO_WP_SLOT, AUDIO_WP_SLOT * len_bytes, (AUDIO_WP_SLOT + 1) * len_bytes,
						play, write, play / len_bytes, (write + len_bytes - 1) / len_bytes);
					DEGUB("Collision!\n");
				}
				HWHR(h->ai.pDsb8->Stop());
				h->ai.audio_playing = false;
				SAFE_RELEASE(h->ai.pDsb8);
				h->ai.audio_repeats = 0;
			} else if(h->ai.audio_slots == 1) {
				HWHR(h->ai.pDsb8->Stop());
				h->ai.audio_playing = false;
				if(h->ai.audio_interrupted) {
					h->ai.audio_repeats++;
					VAIDEGUB("Repeat %i\n", h->ai.audio_repeats);
				}
				if(h->ai.audio_repeats == slots) {
					SAFE_RELEASE(h->ai.pDsb8);
				} else if(h->ai.audio_repeats < slots) {
					//Reset position (?)
					HWHR(h->ai.pDsb8->SetCurrentPosition(0));
				}
			}
		}
		h->ai.audio_length = new_length;
		VAIDEGUB("AI Length: %i (%i ms, %I64i ticks)\n", h->ai.audio_length,
			(h->ai.audio_length * 1000) / LENGTH_PER_SECOND,
			FREQED(LENGTH_2_RTC)(h->ai.audio_length));
		if(h->ai.pDsb8 == NULL) {
			if(h->ai.audio_repeats == slots)
				h->ai.audio_slots = slots;
			else
				h->ai.audio_slots = 1;
			h->ai.audio_wp = 0;
			VAIDEGUB("%i slots created\n", h->ai.audio_slots);
			LPDIRECTSOUNDBUFFER pDsb = NULL;
			h->ai.dsbdesc.dwBufferBytes = len_bytes * h->ai.audio_slots; 
			HWHR(h->ai.lpDirectSound->CreateSoundBuffer(&h->ai.dsbdesc, &pDsb, NULL));
			HWHR(pDsb->QueryInterface(IID_IDirectSoundBuffer8, (LPVOID*)&h->ai.pDsb8));
			pDsb->Release();
			//Reset position (?)
			HWHR(h->ai.pDsb8->SetCurrentPosition(0));
			//Set frequency
			HWHR(h->ai.pDsb8->SetFrequency((aicr & AICR_DSR) ? 32000 : 48000));
		}

		//Lock
		BYTE *locked;
		DWORD lSize;
		//Byteswap the entire buffer at once
		if(g::log_audio && g::verbose) {
			DWORD play, write;
			HWHR(h->ai.pDsb8->GetCurrentPosition(&play, &write));
			DEGUB("Writing slot %i (bytes %i to %i). Play-Write: %i-%i (slots %i-%i, inclusive)\n",
				AUDIO_WP_SLOT, AUDIO_WP_SLOT * len_bytes, (AUDIO_WP_SLOT + 1) * len_bytes,
				play, write, play / len_bytes, (write + len_bytes - 1) / len_bytes);
			if(AUDIO_WP_SLOT >= play / len_bytes && h->ai.audio_playing &&
				AUDIO_WP_SLOT <= (write + len_bytes - 1) / len_bytes)
				DEGUB("Collision!\n");
		}
		HWHR(h->ai.pDsb8->Lock(AUDIO_WP_SLOT * len_bytes, len_bytes, (LPVOID*)&locked,
			&lSize, NULL, NULL, 0));
		MYASSERT(lSize == len_bytes);
		if(g::quiet) {
			ZeroMemory(locked, len_bytes);	//Quiet...
		} else {
			memcpy_swaph(locked, h->m.getp_physical(PHYSICALIZE(address), len_bytes),
				len_bytes);	
		}
		HWHR(h->ai.pDsb8->Unlock(locked, lSize, NULL, 0));

		VAIDEGUB("Wrote slot %i\n", AUDIO_WP_SLOT);
		h->ai.audio_wp++;
		//if(audio_wp == audio_slots)
		//audio_wp = 0;

		assert_dma_interrupt(h);
		h->ai.audio_interrupted = false;

		if(g::timing_mode == g::TM_REALTIME) {
			ULI rtc;
			GET_RTC(&rtc);
			//strictly speaking, there is no invalid rtc at all, but this one should almost
			//never be used and is the default init value.
#define INVALID_RTC 0
			//we should use a bool instead.
			if(h->ai.dma_start_time == INVALID_RTC) {
				VAIDEGUB("This is the first AIStart.\n");
			} else {
				ULI diff = rtc - h->ai.dma_start_time;
				VAIDEGUB("AIStartdelta: %i ms (%I64i ticks)\n", RTC_2_MS(diff), diff.QuadPart);
				h->ai.dma_start_delta_sum += diff;
				h->ai.dma_start_ndeltas++;
			}
			h->ai.dma_start_time = rtc;
		} else {  //exact
			VAIDEGUB("AIStartdelta: %I64i cycles\n",
				h->m_cc.cycles - h->ai.dma_start_time);
			h->ai.dma_start_time = h->m_cc.cycles;
		}

		if((!h->ai.audio_playing && h->ai.audio_wp == (h->ai.audio_slots / 2) + 1) ||
			h->ai.audio_slots == 1)
		{
			h->ai.audio_playing = true;
			h->ai.audio_pp = 0;
			VAIDEGUB("Starting play!\n");
			h->ai.audio_play_start_time = h->ai.dma_start_time;
			HWHR(h->ai.pDsb8->Play(0, 0, DSBPLAY_LOOPING));
		}
		if(h->ai.audio_playing) {
			DWORD len_till_event = h->ai.audio_length * (h->ai.audio_pp + 1);
			ULI event_time;
			if(g::timing_mode == g::TM_REALTIME) {
				event_time = h->ai.audio_play_start_time +
					FREQED(LENGTH_2_RTC)(len_till_event);
			} else if(g::timing_mode == g::TM_EXACT_REAL) {
				event_time = h->ai.audio_play_start_time +
					FREQED(LENGTH_2_CYCLES)(len_till_event);
			} else {	//exact fast
				event_time = h->ai.audio_play_start_time + len_till_event;
			}
			h->interrupt.add_event(dma_playing_event, event_time, "AI DMA playing");
		} else {
			ULI event_time;
			if(g::timing_mode == g::TM_REALTIME) {
				event_time = h->ai.dma_start_time + FREQED(LENGTH_2_RTC)(h->ai.audio_length);
			} else if(g::timing_mode == g::TM_EXACT_REAL) {
				event_time = h->ai.dma_start_time + FREQED(LENGTH_2_CYCLES)(h->ai.audio_length);
			} else {	//exact fast
				event_time = h->ai.dma_start_time + h->ai.audio_length;
			}
			h->interrupt.add_event(dma_buffering_event, event_time, "AI DMA buffering");
		}
	} else if(FALL_FLAG(data, old_data, 0x8000)) {  //Stop
		//we should perhaps use this to set a stop marker of some sort, but I think
		//it's better to stop when we run out of data, that is,
		//when the play pointer catches up with the write pointer.
		//that check should be made in do_events().
		AIDEGUB("AI Stop\n");

		//yes, but we also need a stop marker. see ctr-snd.
		if(h->ai.audio_playing && h->ai.audio_slots == 1) {
			HWHR(h->ai.pDsb8->Stop());
			h->ai.audio_playing = false;
			//h->audio_interrupted = false;
			//h->audio_length = 0;

			if(g::log_audio && g::verbose) {
				if(g::timing_mode == g::TM_REALTIME) {
					ULI uli;
					GET_RTC(&uli);
					uli -= h->ai.dma_start_time;
					DEGUB("AIStopdelta: %i ms (%I64i ticks)\n", RTC_2_MS(uli), uli.QuadPart);
				} else {  //exact
					DEGUB("AIStopdelta: %I64i cycles\n",
						h->m_cc.cycles - h->ai.dma_start_time);
				}
			}
		} /*else if(audio_length != 0) {
			GLE(QueryPerformanceCounter(&ai_dma_stop_time));
			audio_length = 0;
			audio_interrupted = false;
			//audio_stop_timer_set = true;
			}*/
	} else {  //No start/stop flag change
		AIDEGUB("AIDMACR write ignored: 0x%04X\n", data);
	}
}

void AIDumpET(ULI et, const char *name) {
	if(g::timing_mode == g::TM_REALTIME) {
		DEGUB("%s: %i ms (%I64i ticks)\n", name, RTC_2_MS(et), et.QuadPart);
	} else {
		DEGUB("%s: %I64i cycles\n", name, et.QuadPart);
	}
}

void HwAI::dma_buffering_event(Hardware *h) {
	WORD cr = h->hrh(0x5036);
	if(g::log_audio && g::verbose) {
		AIDumpET(h->gekko_interrupt.event_time() - h->ai.dma_start_time, "AIDINTdelta");
		DEGUB("Clearing DMA bit: 0x%04X->0x%04X\n", cr, cr & ~0x8000);
	}
	h->hwh(0x5036, cr & ~0x8000);
	assert_dma_interrupt(h);
	h->ai.audio_interrupted = true;
}

void HwAI::dma_playing_event(Hardware *h) {
	if(g::log_audio && g::verbose) {
		AIDumpET(h->gekko_interrupt.event_time() - h->ai.audio_play_start_time, "Play time");
		DUMPINT(h->ai.audio_pp);
	}
	assert_dma_interrupt(h);
	h->ai.audio_interrupted = true;
	h->ai.audio_pp++;
}

DWORD HwAI::calculateAISCNT(Hardware *h) {
	if(g::timing_mode == g::TM_REALTIME) {
		ULI rtc;
		GET_RTC(&rtc);
		return h->hrw(PR_AISCNT) + RTC_2_AISCNT(rtc - h->ai.scnt_start_time);
	} else if(g::timing_mode == g::TM_EXACT_REAL) {
		return h->hrw(PR_AISCNT) + CYCLES_2_AISCNT(h->m_cc.cycles - h->ai.scnt_start_time);
	} else {  //TM_EXACT_FAST
		DWORD aiscnt = h->hrw(PR_AISCNT) + 1;
		h->hww(PR_AISCNT, aiscnt);
		return aiscnt;
	}
}

void HwAI::assert_dma_interrupt(Hardware *h) {
	WORD dspcr = h->hrh(0x500A);
	if(getflag(dspcr, DSPCR_AIDINTMASK)) {
		h->interrupt.raiseAsync(ai_dma_interrupt_callback, "AI DMA");
		VAIDEGUB("AIDMA interrupt raised\n");
	} else {
		h->hwh(0x500A, dspcr | DSPCR_AIDINT);
		VAIDEGUB("AIDMA interrupt asserted, not raised\n");
	}
}

void HwAI::ai_dma_interrupt_callback(Hardware *h) {
	VAIDEGUB("ai_dma_interrupt_callback\n");
	h->generic_interrupt_callback(INTEX_DSP);
	h->hwh(0x500A, h->hrh(0x500A) | DSPCR_AIDINT);
}

bool Hardware::waitingForAIDMAInterrupt() {
	//a sound is playing, it hasn't been interrupted, but will be soon.
	return getflag(hrh(0x5036), 0x8000) && !ai.audio_interrupted &&
		getflag(hrh(0x500A), DSPCR_AIDINTMASK);
}
